/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <X11/Xmd.h>
#include <X11/X.h>
#include <X11/Xproto.h>

#include "Misc.h"
#include "Unpack.h"

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

#ifndef True
#define True  (1)
#define False (0)
#endif
#define Error (-1)

typedef unsigned char uint8;
typedef unsigned short uint16;
typedef unsigned int uint32;

#define CVAL(p)   (*(p++))

#undef NEED_ALIGN

#ifdef NEED_ALIGN
    #ifdef L_ENDIAN
    #define CVAL2(p, v) { v = (*(p++)); v |= (*(p++)) << 8; }
    #else
    #define CVAL2(p, v) { v = (*(p++)) << 8; v |= (*(p++)); }
    #endif /* L_ENDIAN */
#else
    #define CVAL2(p, v) { v = (*((uint16*)p)); p += 2; }
#endif /* NEED_ALIGN */

#define UNROLL8(exp) { exp exp exp exp exp exp exp exp }

#define REPEAT(statement) \
{ \
  while ((count & ~0x7) && ((x+8) < width)) \
    UNROLL8( statement; count--; x++; ); \
\
  while ((count > 0) && (x < width)) { statement; count--; x++; } \
}

#define MASK_UPDATE() \
{ \
  mixmask <<= 1; \
  if (mixmask == 0) \
  { \
    mask = fom_mask ? fom_mask : CVAL(input); \
    mixmask = 1; \
  } \
}

//
// Defines used for Hextile encoding
//

#define rfbHextileRaw                  (1 << 0)
#define rfbHextileBackgroundSpecified  (1 << 1)
#define rfbHextileForegroundSpecified  (1 << 2)
#define rfbHextileAnySubrects          (1 << 3)
#define rfbHextileSubrectsColoured     (1 << 4)

#define rfbHextileExtractX(byte) ((byte) >> 4)
#define rfbHextileExtractY(byte) ((byte) & 0xf)
#define rfbHextileExtractW(byte) (((byte) >> 4) + 1)
#define rfbHextileExtractH(byte) (((byte) & 0xf) + 1)

#define GET_PIXEL8(pix, ptr) ((pix) = *(ptr)++)

#define GET_PIXEL16(pix, ptr) (((CARD8*)&(pix))[0] = *(ptr)++, \
                              ((CARD8*)&(pix))[1] = *(ptr)++)

#define GET_PIXEL32(pix, ptr) (((CARD8*)&(pix))[0] = *(ptr)++, \
                              ((CARD8*)&(pix))[1] = *(ptr)++, \
                              ((CARD8*)&(pix))[2] = *(ptr)++, \
                              ((CARD8*)&(pix))[3] = *(ptr)++)

static CARD32 bg32, fg32;
static CARD16 bg16, fg16;
static CARD8  bg8 , fg8;

//
// Get bits per pixel set by client
// according to display geometry.
//

int GetBitsPerPixel(T_geometry *geometry, unsigned int depth)
{
  switch (depth)
  {
    case 1:
    {
      return geometry -> depth1_bpp;
    }
    case 4:
    {
      return geometry -> depth4_bpp;
    }
    case 8:
    {
      return geometry -> depth8_bpp;
    }
    case 15:
    case 16:
    {
      return geometry -> depth16_bpp;
    }
    case 24:
    {
      return geometry -> depth24_bpp;
    }
    case 32:
    {
      return geometry -> depth32_bpp;
    }
    default:
    {
      return 0;
    }
  }
}

int UnpackAlpha(T_alpha *alpha, unsigned char *dst_data,
                    int dst_size, int big_endian)
{
  unsigned int *next = (unsigned int *) dst_data;
  unsigned int count = dst_size >> 2;

  unsigned int mask;

  #ifdef TEST
  *logofs << "UnpackAlpha: Destination pixels are " << count
          << " with " << alpha -> entries << " unpack alpha "
          << "data entries.\n" << logofs_flush;
  #endif

  if (count == alpha -> entries)
  {
    unsigned char *data = alpha -> data;

    while (count--)
    {
      mask = big_endian ? *data : (*data) << 24;

      *next++ |= mask;

      data++;
    }

    return 1;
  }

  return 0;
}

int Unpack8To8(const T_colormask *colormask, const unsigned char *data,
                    unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack8To8: Unpacking " << end - out
          << " bytes of data.\n" << logofs_flush;
  #endif

  memcpy(out, data, end - out);

  return 1;
}

int Unpack8To16(const T_colormask *colormask, const unsigned char *data,
                    unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack8To16: Unpacking " << end - out
          << " bytes of data.\n" << logofs_flush;
  #endif

  unsigned short *out16 = (unsigned short *) out;
  unsigned short *end16 = (unsigned short *) end;
  
  while (out16 < end16)
  {
    if (*data == 0)
    {
      *out16 = 0x0;
    }
    else if (*data == 0xff)
    {
      *out16 = 0xffff;
    }
    else
    {
      //
      // Pixel layout:
      //
      // 8bits 00RRGGBB -> 16bits RR000GG0 000BB000.
      //

      *out16 = (((((*data & 0x30) << 2) | colormask -> correction_mask) << 8) & 0xf800) |
                   (((((*data & 0xc) << 4) | colormask -> correction_mask) << 3) & 0x7e0) |
                       (((((*data & 0x3) << 6) | colormask -> correction_mask) >> 3) & 0x1f);
    }

    out16++;
    data++;
  }

  return 1;
}

int Unpack8To24(const T_colormask *colormask, const unsigned char *data,
                    unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack8To24: Unpacking " << end - out
          << " bytes of data.\n" << logofs_flush;
  #endif

  while (out < (end - 2))
  {
    if (*data == 0x00)
    {
      out[0] = out[1] = out[2] = 0x00;
    }
    else if (*data == 0xff)
    {
      out[0] = out[1] = out[2] = 0xff;
    }
    else
    {
      //
      // Pixel layout:
      //
      // 8bits 00RRGGBB -> 24bits RR000000 GG00000 BB000000.
      //

      out[0] = (((*data & 0x30) << 2) | colormask -> correction_mask);
      out[1] = (((*data & 0x0c) << 4) | colormask -> correction_mask);
      out[2] = (((*data & 0x03) << 6) | colormask -> correction_mask);
    }

    out  += 3;
    data += 1;
  }

  return 1;
}

int Unpack8To32(const T_colormask *colormask, const unsigned char *data,
                    unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack8To32: Unpacking " << end - out
          << " bytes of data.\n" << logofs_flush;
  #endif

  unsigned int *out32 = (unsigned int *) out;
  unsigned int *end32 = (unsigned int *) end;

  while (out32 < end32)
  {
    if (*data == 0)
    {
      *out32 = 0x0;
    }
    else if (*data == 0xff)
    {
      *out32 = 0xffffff;
    }
    else
    {
      *out32 = ((((*data & 0x30) << 2) | colormask -> correction_mask) << 16) |
                   ((((*data & 0xc) << 4) | colormask -> correction_mask) << 8) |
                       (((*data & 0x3) << 6) | colormask -> correction_mask);
    }

    out32++;
    data++;
  }

  return 1;
}

int Unpack8(T_geometry *geometry, const T_colormask *colormask, int src_depth, int src_width,
                int src_height, unsigned char *src_data, int src_size, int dst_depth,
                    int dst_width, int dst_height, unsigned char *dst_data, int dst_size)
{
  int dst_bpp = GetBitsPerPixel(geometry, dst_depth);

  int (*unpack)(const T_colormask *colormask, const unsigned char *data,
                    unsigned char *out, unsigned char *end);

  switch (dst_bpp)
  {
    case 8:
    {
      unpack = Unpack8To8;

      break;
    }
    case 16:
    {
      unpack = Unpack8To16;

      break;
    }
    case 24:
    {
      unpack = Unpack8To24;

      break;
    }
    case 32:
    {
      unpack = Unpack8To32;

      break;
    }
    default:
    {
      #ifdef PANIC
      *logofs << "Unpack8: PANIC! Bad destination bits per pixel "
              << dst_bpp << ". Only 16/24/32 are supported.\n"
              << logofs_flush;
      #endif

      return -1;
    }
  }

  if (dst_bpp == 24)
  {
    unsigned char *dst_end = dst_data;

    #ifdef TEST
    *logofs  << "Unpack8: Handling 24 bits with dst_size "
             << dst_size << ".\n" << logofs_flush;
    #endif

    for (int y = 0; y < dst_height; y++)
    {
      dst_data = dst_end;

      dst_end += RoundUp4(dst_width * 3);

      (*unpack)(colormask, src_data, dst_data, dst_end);

      src_data += src_width;
    }
  }
  else
  {
    unsigned char *dst_end = dst_data + dst_size;

    (*unpack)(colormask, src_data, dst_data, dst_end);
  }

  return 1;
}

int Unpack16To16(const T_colormask *colormask, const unsigned char *data,
                     unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack16To16: Unpacking " << end - out
          << " bytes of data.\n" << logofs_flush;
  #endif

  if (colormask -> correction_mask)
  {
    unsigned short *data16 = (unsigned short *) data;

    unsigned short *out16 = (unsigned short *) out;
    unsigned short *end16 = (unsigned short *) end;

    while (out16 < end16)
    {
      if (*data16 == 0x0000)
      {
        *out16 = 0x0000;
      }
      else if (*data16 == 0xffff)
      {
        *out16 = 0xffff;
      }
      else
      {
        //
        // Pixel layout:
        //
        // 16bit RRRRRGGG GG0BBBBB  -> RRRRRGGG GGGBBBBB.
        //

        *out16 = (((((*data16 & 0xf100) >> 8) | colormask -> correction_mask) << 8) & 0xf800) |
                     (((((*data16 & 0x7c0) >> 3) | colormask -> correction_mask) << 3) & 0x7e0) |
                         (((((*data16 & 0x1f) << 3) | colormask -> correction_mask) >> 3) & 0x1f);
      }

      out16++;
      data16++;
    }
  }
  else
  {
    #ifdef TEST
    *logofs << "Unpack16To16: Using bitwise copy due to null correction mask.\n"
            << logofs_flush;
    #endif

    memcpy((unsigned char *) out, (unsigned char *) data, end - out);
  }

  return 1;
}

int Unpack16To24(const T_colormask *colormask, const unsigned char *data,
                     unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack16To24: Unpacking " << end - out
          << " bytes of data.\n" << logofs_flush;
  #endif

  unsigned short *data16 = (unsigned short *) data;


  while (out < end - 2)
  {
    if (*data16 == 0x0)
    {
      out[0] = 0x00;
      out[1] = 0x00;
      out[2] = 0x00;
    }
    else if (*data16 == 0xffff)
    {
      out[0] = 0xff;
      out[1] = 0xff;
      out[2] = 0xff;
    }
    else
    {
      #ifdef TEST
      *logofs << "Unpack16To24: Pixel [" << *data16 << "]\n"
              << logofs_flush;
      #endif

      //
      // Pixel layout:
      //
      //  16bit 0RRRRRGG GGGBBBBB  -> 24 bit RRRRR000 GGGGG000 BBBBB000
      //

      out[0] = (((*data16 & 0x7c00) >> 7) | colormask -> correction_mask);
      out[1] = (((*data16 & 0x03e0) >> 2) | colormask -> correction_mask);
      out[2] = (((*data16 & 0x001f) << 3) | colormask -> correction_mask);
    }

    out    += 3;
    data16 += 1;
  }

  return 1;
}

int Unpack16To32(const T_colormask *colormask, const unsigned char *data,
                     unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack16To32: Unpacking " << end - out
          << " bytes of data.\n" << logofs_flush;
  #endif

  unsigned short *data16 = (unsigned short *) data;

  unsigned int *out32 = (unsigned int *) out;
  unsigned int *end32 = (unsigned int *) end;

  while (out32 < end32)
  {
    if (*data16 == 0x0)
    {
      *out32 = 0x0;
    }
    else if (*data16 == 0xffff)
    {
      *out32 = 0xffffff;
    }
    else
    {
      *out32 = ((((*data16 & 0x7c00) >> 7) | colormask -> correction_mask) << 16) |
                   ((((*data16 & 0x3e0) >> 2) | colormask -> correction_mask) << 8) |
                       (((*data16 & 0x1f) << 3) | colormask -> correction_mask);
    }

    out32++;
    data16++;
  }

  return 1;
}

int Unpack16(T_geometry *geometry, const T_colormask *colormask, int src_depth, int src_width,
                 int src_height, unsigned char *src_data, int src_size, int dst_depth,
                     int dst_width, int dst_height, unsigned char *dst_data, int dst_size)
{
  int dst_bpp = GetBitsPerPixel(geometry, dst_depth);

  int (*unpack)(const T_colormask *colormask, const unsigned char *data,
                    unsigned char *out, unsigned char *end);

  switch (dst_bpp)
  {
    case 16:
    {
      unpack = Unpack16To16;

      break;
    }
    case 24:
    {
      unpack = Unpack16To24;

      break;
    }
    case 32:
    {
      unpack = Unpack16To32;

      break;
    }
    default:
    {
      #ifdef PANIC
      *logofs << "Unpack16: PANIC! Bad destination bits per pixel "
              << dst_bpp << ". Only 24/32 are supported.\n"
              << logofs_flush;
      #endif

      return -1;
    }
  }

  if (dst_bpp == 24)
  {
    unsigned char *dst_end = dst_data;

    for (int y = 0; y < dst_height; y++)
    {
      dst_data = dst_end;

      dst_end += RoundUp4(dst_width * 3);

      (*unpack)(colormask, src_data, dst_data, dst_end);

      src_data += (src_width * 2);
    }

  }
  else
  {
    unsigned char *dst_end = dst_data + dst_size;

    (*unpack)(colormask, src_data, dst_data, dst_end);
  }
  
  return 1;
}

int Unpack24To24(const T_colormask *colormask, const unsigned char *data,
                     unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack24To24: Unpacking " << end - out
          << " bytes of data.\n" << logofs_flush;
  #endif

  if (colormask -> correction_mask)
  {
    while (out < end)
    {
      if (data[0] == data[1] == data[2] == 0x00)
      {
        out[0] = out[1] = out[2] = 0x00;
      }
      else if (data[0] == data[1] == data[2] == 0xff)
      {
        out[0] = out[1] = out[2] = 0xff;
      }
      else
      {
        out[0] = (data[0] | colormask -> correction_mask);
        out[1] = (data[1] | colormask -> correction_mask);
        out[2] = (data[2] | colormask -> correction_mask);
      }

      out  += 3;
      data += 3;
    }
  }
  else
  {
    #ifdef TEST
    *logofs << "Unpack24To24: Using bitwise copy due to null correction mask.\n"
            << logofs_flush;
    #endif

    memcpy((unsigned char *) out, (unsigned char *) data, end - out);
  }

  return 1;
}

int Unpack24To32(const T_colormask *colormask, const unsigned char *data,
                     unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack24To32: Unpacking " << end - out
          << " bytes of data.\n" << logofs_flush;
  #endif

  unsigned int *out32 = (unsigned int *) out;
  unsigned int *end32 = (unsigned int *) end;

  while (out32 < end32)
  {
    if (colormask -> color_mask == 0xff)
    {
      *out32 = (data[0] << 16) | (data[1] << 8) | data[2];
    }
    else
    {
      if (data[0] == 0x0 && data[1] == 0x0 && data[2] == 0x0)
      {
        *out32 = 0x0;
      }
      else if (data[0] == 0xff && data[1] == 0xff && data[2] == 0xff)
      {
        *out32 = 0xffffff;
      }
      else
      {
        *out32 = (((unsigned int) data[0] | colormask -> correction_mask) << 16) |
                   (((unsigned int) data[1] | colormask -> correction_mask) << 8) |
                       ((unsigned int) data[2] | colormask -> correction_mask);
      }
    }

    out32 += 1;
    data  += 3;
  }

  return 1;
}

int Unpack24(T_geometry *geometry, const T_colormask *colormask, int src_depth, int src_width,
                 int src_height, unsigned char *src_data, int src_size, int dst_depth,
                     int dst_width, int dst_height, unsigned char *dst_data, int dst_size)
{
  int dst_bpp = GetBitsPerPixel(geometry, dst_depth);

  int (*unpack)(const T_colormask *colormask, const unsigned char *data,
                    unsigned char *out, unsigned char *end);

  switch (dst_bpp)
  {
    case 24:
    {
      unpack = Unpack24To24;

      break;
    }
    case 32:
    {
      unpack = Unpack24To32;

      break;
    }
    default:
    {
      #ifdef PANIC
      *logofs << "Unpack24: PANIC! Bad destination bits per pixel "
              << dst_bpp << ". Only 32 is supported.\n"
              << logofs_flush;
      #endif

      return -1;
    }
  }

  if (dst_bpp == 24)
  {
    unsigned char *dst_end;
    unsigned long scanline_size = RoundUp4(dst_width * dst_bpp / 8);

    dst_end = dst_data;

    #ifdef TEST
    *logofs  << "Unpack24: Handling 24 bits with dst_height "
             << dst_height << " scanline_size " << scanline_size
             << " dst_size " << dst_size << ".\n" << logofs_flush;
    #endif

    for (int y = 0; y < dst_height; y++)
    {
      dst_data = dst_end;

      dst_end += scanline_size;

      (*unpack)(colormask, src_data, dst_data, dst_end);

      src_data += scanline_size;
    }
  }
  else
  {
    unsigned char *dst_end = dst_data + dst_size;

    (*unpack)(colormask, src_data, dst_data, dst_end);
  }

  return 1;
}

int Unpack8To8(T_colormap *colormap, const unsigned char *data,
                   unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack8To8: Unpacking " << end - out
          << " bytes of colormapped data.\n"
          << logofs_flush;
  #endif

  while (out < end)
  {
    *(out++) = (unsigned char) colormap -> data[*(data++)];
  }

  return 1;
}

int Unpack8To16(T_colormap *colormap, const unsigned char *data,
                    unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack8To16: Unpacking " << end - out
          << " bytes of colormapped data.\n"
          << logofs_flush;
  #endif

  unsigned short *out16 = (unsigned short *) out;
  unsigned short *end16 = (unsigned short *) end;

  while (out16 < end16)
  {
    *(out16++) = (unsigned short) colormap -> data[*(data++)];
  }

  return 1;
}

int Unpack8To24(T_colormap *colormap, const unsigned char *data,
                    unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack8To24: Unpacking " << end - out
          << " bytes of colormapped data.\n"
          << logofs_flush;
  #endif

  unsigned int value;

  while (out < end)
  {
    value = colormap -> data[*(data++)];

    *(out++) = value;
    *(out++) = value >> 8;
    *(out++) = value >> 16;
  }

  return 1;
}

int Unpack8To32(T_colormap *colormap, const unsigned char *data,
                    unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack8To32: Unpacking " << end - out
          << " bytes of colormapped data.\n"
          << logofs_flush;
  #endif

  unsigned int *out32 = (unsigned int *) out;
  unsigned int *end32 = (unsigned int *) end;

  while (out32 < end32)
  {
    *(out32++) = colormap -> data[*(data++)];
  }

  return 1;
}

int Unpack8(T_geometry *geometry, T_colormap *colormap, int src_depth, int src_width, int src_height,
                 unsigned char *src_data, int src_size, int dst_depth, int dst_width,
                     int dst_height, unsigned char *dst_data, int dst_size)
{
  if (src_depth != 8)
  {
    #ifdef PANIC
    *logofs << "Unpack8: PANIC! Cannot unpack colormapped image of source depth "
            << src_depth << ".\n" << logofs_flush;
    #endif

    return -1;
  }

  int (*unpack)(T_colormap *colormap, const unsigned char *data,
                    unsigned char *out, unsigned char *end);

  int dst_bpp = GetBitsPerPixel(geometry, dst_depth);

  switch (dst_bpp)
  {
    case 8:
    {
      unpack = Unpack8To8;

      break;
    }
    case 16:
    {
      unpack = Unpack8To16;

      break;
    }
    case 24:
    {
      unpack = Unpack8To24;

      break;
    }
    case 32:
    {
      unpack = Unpack8To32;

      break;
    }
    default:
    {
      #ifdef PANIC
      *logofs << "Unpack8: PANIC! Bad destination bits per pixel "
              << dst_bpp << ". Only 8/16/24/32 are supported.\n"
              << logofs_flush;
      #endif

      return -1;
    }
  }

  if (src_width == dst_width &&
          src_height == dst_height)
  {
    unsigned char *dst_end = dst_data + dst_size;

    (*unpack)(colormap, src_data, dst_data, dst_end);
  }
  else if (src_width >= dst_width &&
             src_height >= dst_height)
  {
    unsigned char *dst_end = dst_data;

    for (int y = 0; y < dst_height; y++)
    {
      dst_data = dst_end;

      dst_end += RoundUp4(dst_width * dst_bpp / 8);

      (*unpack)(colormap, src_data, dst_data, dst_end);

      src_data += src_width;
    }
  }
  else
  {
    #ifdef PANIC
    *logofs << "Unpack8: PANIC! Cannot unpack image. "
            << "Destination area " << dst_width << "x"
            << dst_height << " is not fully contained in "
            << src_width << "x" << src_height << " source.\n"
            << logofs_flush;
    #endif

    return -1;
  }

  return 1;
}

int Unpack15To16(const unsigned char *data, unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack15To16: Unpacking " << end - out
          << " bytes of colormapped data.\n"
          << logofs_flush;
  #endif
  
  unsigned short *data16 = (unsigned short *) data;
  unsigned short *out16 = (unsigned short *) out;
  unsigned short *end16 = (unsigned short *) end;

  while (out16 < end16)
  {
    if (*data16 == 0x0)
    {
      *out16 = 0x0;
    }
    else if (*data16 == 0x7fff)
    {
      *out16 = 0xffff;
    }
    else
    {
      #ifdef TEST
      *logofs << "Unpack15To16: Pixel [" << *data16 << "]\n"
              << logofs_flush;
      #endif

      *out16 = (((*data16 & 0x7fe0) >> 1) |
                 (*data16 & 0x001f));
    }

    out16  += 1;
    data16 += 1;
  }
  
  return 1;
}

int Unpack15To24(const unsigned char *data, unsigned char *out, unsigned char *end)

{
  #ifdef TEST
  *logofs << "Unpack15To24: Unpacking " << end - out
          << " bytes of data.\n" << logofs_flush;
  #endif

  unsigned short *data16 = (unsigned short *) data;

  while (out < end - 2)
  {
    if (*data16 == 0x0)
    {
      out[0] = 0x00;
      out[1] = 0x00;
      out[2] = 0x00;
    }
    else if (*data16 == 0x7fff)
    {
      out[0] = 0xff;
      out[1] = 0xff;
      out[2] = 0xff;
    }
    else
    {
      #ifdef TEST
      *logofs << "Unpack15To24: Pixel [" << *data16 << "]\n"
              << logofs_flush;
      #endif

      out[0] = ((*data16 >> 7) & 0xf8) | ((*data16 >> 12) & 0x07);
      out[1] = ((*data16 >> 2) & 0xf8) | ((*data16 >> 8) & 0x07);
      out[2] = ((*data16 << 3) & 0xf8) | ((*data16 >> 2) & 0x07);
    }

    out    += 3;
    data16 += 1;
  }

  return 1;
}

int Unpack15To32(const unsigned char *data, unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack15To32: Unpacking " << end - out
          << " bytes of data.\n"
          << logofs_flush;
  #endif

  unsigned short *data16 = (unsigned short *) data;
  unsigned int *out32 = (unsigned int *) out;
  unsigned int *end32 = (unsigned int *) end;

  while (out32 < end32)
  {
    if (*data16 == 0x0)
    {
      *out32 = 0x0;
    }
    else if (*data16 == 0xffff)
    {
      *out32 = 0xffffff;
    }
    else
    {
        *out32 = ((((*data16 >> 7) & 0xf8) | ((*data16 >> 12) & 0x07)) << 16) | 
                 ((((*data16 >> 2) & 0xf8) | ((*data16 >> 8) & 0x07)) << 8) | 
                  (((*data16 << 3) & 0xf8) | ((*data16 >> 2) & 0x07));
    }

    out32++;
    data16++;
  }

  return 1;
}

int Unpack15(T_geometry *geometry, int src_depth, int src_width, int src_height,
                 unsigned char *src_data, int src_size, int dst_depth, int dst_width,
                     int dst_height, unsigned char *dst_data, int dst_size)
{
  if (src_depth != 16)
  {
    #ifdef PANIC
    *logofs << "Unpack15: PANIC! Cannot unpack colormapped image of source depth "
            << src_depth << ".\n" << logofs_flush;
    #endif

    return -1;
  }

  int (*unpack)(const unsigned char *data, unsigned char *out, unsigned char *end);

  int dst_bpp = GetBitsPerPixel(geometry, dst_depth);

  switch (dst_bpp)
  {
    case 16:
    {
      unpack = Unpack15To16;

      break;
    }
    case 24:
    {
      unpack = Unpack15To24;

      break;
    }
    case 32:
    {
      unpack = Unpack15To32;

      break;
    }
    default:
    {
      #ifdef PANIC
      *logofs << "Unpack15: PANIC! Bad destination bits per pixel "
              << dst_bpp << ". Only 16/24/32 are supported.\n"
              << logofs_flush;
      #endif

      return -1;
    }
  }

  if (src_width == dst_width && src_height == dst_height)
  {
    unsigned char *dst_end = dst_data + dst_size;

    (*unpack)(src_data, dst_data, dst_end);
  }
  else if (src_width >= dst_width && src_height >= dst_height)
  {
    unsigned char *dst_end = dst_data;

    for (int y = 0; y < dst_height; y++)
    {
      dst_data = dst_end;
      dst_end += RoundUp4(dst_width * dst_bpp / 8);

      (*unpack)(src_data, dst_data, dst_end);

      src_data += src_width * 2;
    }
  }
  else
  {
    #ifdef PANIC
    *logofs << "Unpack15: PANIC! Cannot unpack image. "
            << "Destination area " << dst_width << "x"
            << dst_height << " is not fully contained in "
            << src_width << "x" << src_height << " source.\n"
            << logofs_flush;
    #endif

    return -1;
  }

  return 1;
}

int Unpack16To16(const unsigned char *data,
                    unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack16To16: Unpacking " << end - out
          << " bytes of colormapped data.\n"
          << logofs_flush;
  #endif
  
  unsigned short *data16 = (unsigned short *) data;
  unsigned short *out16 = (unsigned short *) out;
  unsigned short *end16 = (unsigned short *) end;

  while (out16 < end16)
  {
    *(out16++) = *(data16++);
  }

  return 1;
}

int Unpack16To24(const unsigned char *data, unsigned char *out, unsigned char *end)

{
  #ifdef TEST
  *logofs << "Unpack16To24: Unpacking " << end - out
          << " bytes of data.\n" << logofs_flush;
  #endif

  unsigned short *data16 = (unsigned short *) data;

  while (out < end - 2)
  {
    if (*data16 == 0x0)
    {
      out[0] = 0x00;
      out[1] = 0x00;
      out[2] = 0x00;
    }
    else if (*data16 == 0xffff)
    {
      out[0] = 0xff;
      out[1] = 0xff;
      out[2] = 0xff;
    }
    else
    {
      #ifdef TEST
      *logofs << "Unpack16To24: Pixel [" << *data16 << "]\n"
              << logofs_flush;
      #endif

      out[0] = ((*data16 >> 8) & 0xf8) | ((*data16 >> 13) & 0x07);
      out[1] = ((*data16 >> 3) & 0xfc) | ((*data16 >> 9) & 0x03);
      out[2] = ((*data16 << 3) & 0xf8) | ((*data16 >> 2) & 0x07);
    }

    out    += 3;
    data16 += 1;
  }

  return 1;
}


int Unpack16To32(const unsigned char *data, unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack16To32: Unpacking " << end - out
          << " bytes of data.\n"
          << logofs_flush;
  #endif

  unsigned short *data16 = (unsigned short *) data;
  unsigned int *out32 = (unsigned int *) out;
  unsigned int *end32 = (unsigned int *) end;

  while (out32 < end32)
  {
    if (*data16 == 0x0)
    {
      *out32 = 0x0;
    }
    else if (*data16 == 0xffff)
    {
      *out32 = 0xffffff;
    }
    else
    {
      
      *out32 = ((((*data16 >> 8) & 0xf8) | ((*data16 >> 13) & 0x07)) << 16) | 
               ((((*data16 >> 3) & 0xfc) | ((*data16 >> 9) & 0x03)) << 8) | 
               (((*data16 << 3) & 0xf8) | ((*data16 >> 2) & 0x07));
      
    }
    out32++;
    data16++;
  }
  return 1;
}

int Unpack16(T_geometry *geometry, int src_depth, int src_width, int src_height,
                 unsigned char *src_data, int src_size, int dst_depth, int dst_width,
                     int dst_height, unsigned char *dst_data, int dst_size)
{
  if (src_depth != 16)
  {
    #ifdef PANIC
    *logofs << "Unpack16: PANIC! Cannot unpack colormapped image of source depth "
            << src_depth << ".\n" << logofs_flush;
    #endif

    return -1;
  }

  int (*unpack)(const unsigned char *data, unsigned char *out, unsigned char *end);

  int dst_bpp = GetBitsPerPixel(geometry, dst_depth);
  
  switch (dst_bpp)
  {
    case 16:
    {
      unpack = Unpack16To16;

      break;
    }
    case 24:
    {
      unpack = Unpack16To24;

      break;
    }
    case 32:
    {
      unpack = Unpack16To32;

      break;
    }
    default:
    {
      #ifdef PANIC
      *logofs << "Unpack16: PANIC! Bad destination bits per pixel "
              << dst_bpp << ". Only 16/24/32 are supported.\n"
              << logofs_flush;
      #endif

      return -1;
    }
  }

  if (src_width == dst_width && src_height == dst_height)
  {
    unsigned char *dst_end = dst_data + dst_size;

    (*unpack)(src_data, dst_data, dst_end);
  }
  else if (src_width >= dst_width && src_height >= dst_height)
  {
    unsigned char *dst_end = dst_data;

    for (int y = 0; y < dst_height; y++)
    {
      dst_data = dst_end;
      dst_end += RoundUp4(dst_width * dst_bpp / 8);

      (*unpack)(src_data, dst_data, dst_end);

      src_data += src_width * 2;
    }
  }
  else
  {
    #ifdef PANIC
    *logofs << "Unpack16: PANIC! Cannot unpack image. "
            << "Destination area " << dst_width << "x"
            << dst_height << " is not fully contained in "
            << src_width << "x" << src_height << " source.\n"
            << logofs_flush;
    #endif

    return -1;
  }

  return 1;
}

int Unpack24To24(const unsigned char *data,
                    unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack124To24: Unpacking " << end - out
          << " bytes of colormapped data.\n"
          << logofs_flush;
  #endif

  while (out < end)
  {
    *(out++) = *(data++);
  }

  return 1;
}

int Unpack24To32(const unsigned char *data, unsigned char *out, unsigned char *end)
{
  #ifdef TEST
  *logofs << "Unpack24To32: Unpacking " << end - out
          << " bytes of colormapped data.\n"
          << logofs_flush;
  #endif

  unsigned int *out32 = (unsigned int *) out;
  unsigned int *end32 = (unsigned int *) end;

  while (out32 < end32)
  {
    if (data[0] == 0x0 && data[1] == 0x0 && data[2] == 0x0)
    {
      *out32 = 0x0;
    }
    else if (data[0] == 0xff && data[1] == 0xff && data[2] == 0xff)
    {
      *out32 = 0xffffff;
    }
    else
    {
      *out32 = (data[2] << 16) | (data[1] << 8) | data[0];  
    }

    out32 += 1;
    data  += 3;
  }

  return 1;
}

int Unpack24(T_geometry *geometry, int src_depth, int src_width, int src_height,
                 unsigned char *src_data, int src_size, int dst_depth, int dst_width,
                     int dst_height, unsigned char *dst_data, int dst_size)
                     
{
  if (src_depth != 24)
  {
    #ifdef PANIC
    *logofs << "Unpack24: PANIC! Cannot unpack colormapped image of source depth "
            << src_depth << ".\n" << logofs_flush;
    #endif

    return -1;
  }

  int (*unpack)(const unsigned char *data, unsigned char *out, unsigned char *end);

  int dst_bpp = GetBitsPerPixel(geometry, dst_depth);

  switch (dst_bpp)
  {
    case 24:
    {
      unpack = Unpack24To24;

      break;
    }
    case 32:
    {
      unpack = Unpack24To32;

      break;
    }
    default:
    {
      #ifdef PANIC
      *logofs << "Unpack24: PANIC! Bad destination bits per pixel "
              << dst_bpp << ". Only 24/32 are supported.\n"
              << logofs_flush;
      #endif

      return -1;
    }
  }

  if (src_width == dst_width && src_height == dst_height)
  {
    unsigned char *dst_end = dst_data + dst_size;

    (*unpack)(src_data, dst_data, dst_end);
  }
  else if (src_width >= dst_width && src_height >= dst_height)
  {
    unsigned char *dst_end = dst_data;

    for (int y = 0; y < dst_height; y++)
    {
      dst_data = dst_end;
      dst_end += RoundUp4(dst_width * dst_bpp / 8);

      (*unpack)(src_data, dst_data, dst_end);

      src_data += src_width * 3;
    }
  }
  else
  {
    #ifdef PANIC
    *logofs << "Unpack16: PANIC! Cannot unpack image. "
            << "Destination area " << dst_width << "x"
            << dst_height << " is not fully contained in "
            << src_width << "x" << src_height << " source.\n"
            << logofs_flush;
    #endif

    return -1;
  }

  return 1;
}

int Unpack32To32(const T_colormask *colormask, const unsigned int *data,
                     unsigned int *out, unsigned int *end)
{
  #ifdef TEST
  *logofs << "Unpack32To32: Unpacking " << end - out
          << " bytes of data.\n" << logofs_flush;
  #endif

  if (colormask -> correction_mask)
  {
    while (out < end)
    {
      if (*data == 0x00000000)
      {
        *out = 0x00000000;
      }
      else if (*data == 0xFFFFFFFF)
      {
        *out = 0xFFFFFFFF;
      }
      else
      {
        *out = *data | ((colormask -> correction_mask << 16) |
                            (colormask -> correction_mask << 8) |
                                colormask -> correction_mask);
      }

      out  += 1;
      data += 1;
    }
  }
  else
  {
    #ifdef TEST
    *logofs << "Unpack32To32: Using bitwise copy due to null correction mask.\n"
            << logofs_flush;
    #endif

    memcpy((unsigned char *) out, (unsigned char *) data, end - out);
  }

  return 1;
}

//
// The following three functions are borrowed from
// the rdesktop code. They unpack a RDP bitmap to
// a X image buffer.
//

static uint32
cvalx(uint8 **input, int Bpp)
{
        uint32 rv = 0;
        memcpy(&rv, *input, Bpp);
        *input += Bpp;
        return rv;
}

static void
setli(uint8 *input, int offset, uint32 value, int Bpp)
{
        input += offset * Bpp;
        memcpy(input, &value, Bpp);
}

static uint32
getli(uint8 *input, int offset, int Bpp)
{
        uint32 rv = 0;
        input += offset * Bpp;
        memcpy(&rv, input, Bpp);
        return rv;
}

int
UnpackRDP(unsigned char *output, int width, int height, unsigned char *input, int size, int Bpp)
{
        uint8 *end = input + size;
        uint8 *prevline = NULL, *line = NULL;
        int opcode, count, offset, isfillormix, x = width;
        int lastopcode = -1, insertmix = False, bicolour = False;
        uint8 code;
        uint32 colour1 = 0, colour2 = 0;
        uint8 mixmask, mask = 0;
        uint32 mix = 0xffffffff;
        int fom_mask = 0;
        
        while (input < end)
        {
                fom_mask = 0;
                code = CVAL(input);
                opcode = code >> 4;

                /* Handle different opcode forms */
                switch (opcode)
                {
                        case 0xc:
                        case 0xd:
                        case 0xe:
                                opcode -= 6;
                                count = code & 0xf;
                                offset = 16;
                                break;

                        case 0xf:
                                opcode = code & 0xf;
                                if (opcode < 9)
                                {
                                        count = CVAL(input);
                                        count |= CVAL(input) << 8;
                                }
                                else
                                {
                                        count = (opcode < 0xb) ? 8 : 1;
                                }
                                offset = 0;
                                break;

                        default:
                                opcode >>= 1;
                                count = code & 0x1f;
                                offset = 32;
                                break;
                }

                /* Handle strange cases for counts */
                if (offset != 0)
                {
                        isfillormix = ((opcode == 2) || (opcode == 7));

                        if (count == 0)
                        {
                                if (isfillormix)
                                        count = CVAL(input) + 1;
                                else
                                        count = CVAL(input) + offset;
                        }
                        else if (isfillormix)
                        {
                                count <<= 3;
                        }
                }

                /* Read preliminary data */
                switch (opcode)
                {
                        case 0:        /* Fill */
                                if ((lastopcode == opcode) && !((x == width) && (prevline == NULL)))
                                        insertmix = True;
                                break;
                        case 8:        /* Bicolour */
                                colour1 = cvalx(&input, Bpp);
                        case 3:        /* Colour */
                                colour2 = cvalx(&input, Bpp);
                                break;
                        case 6:        /* SetMix/Mix */
                        case 7:        /* SetMix/FillOrMix */
                                mix = cvalx(&input, Bpp);
                                opcode -= 5;
                                break;
                        case 9:        /* FillOrMix_1 */
                                mask = 0x03;
                                opcode = 0x02;
                                fom_mask = 3;
                                break;
                        case 0x0a:        /* FillOrMix_2 */
                                mask = 0x05;
                                opcode = 0x02;
                                fom_mask = 5;
                                break;

                }

                lastopcode = opcode;
                mixmask = 0;

                /* Output body */
                while (count > 0)
                {
                        if (x >= width)
                        {
                                if (height <= 0)
                                        return Error;

                                x = 0;
                                height--;

                                prevline = line;
                                line = output + height * width * Bpp;
                        }

                        switch (opcode)
                        {
                                case 0:        /* Fill */
                                        if (insertmix)
                                        {
                                                if (prevline == NULL)
                                                        setli(line, x, mix, Bpp);
                                                else
                                                        setli(line, x,
                                                              getli(prevline, x, Bpp) ^ mix, Bpp);

                                                insertmix = False;
                                                count--;
                                                x++;
                                        }

                                        if (prevline == NULL)
                                        {
                                        REPEAT(setli(line, x, 0, Bpp))}
                                        else
                                        {
                                                REPEAT(setli
                                                       (line, x, getli(prevline, x, Bpp), Bpp));
                                        }
                                        break;

                                case 1:        /* Mix */
                                        if (prevline == NULL)
                                        {
                                                REPEAT(setli(line, x, mix, Bpp));
                                        }
                                        else
                                        {
                                                REPEAT(setli
                                                       (line, x, getli(prevline, x, Bpp) ^ mix,
                                                        Bpp));
                                        }
                                        break;

                                case 2:        /* Fill or Mix */
                                        if (prevline == NULL)
                                        {
                                                REPEAT(MASK_UPDATE();
                                                       if (mask & mixmask) setli(line, x, mix, Bpp);
                                                       else
                                                       setli(line, x, 0, Bpp););
                                        }
                                        else
                                        {
                                                REPEAT(MASK_UPDATE();
                                                       if (mask & mixmask)
                                                       setli(line, x, getli(prevline, x, Bpp) ^ mix,
                                                             Bpp);
                                                       else
                                                       setli(line, x, getli(prevline, x, Bpp),
                                                             Bpp););
                                        }
                                        break;

                                case 3:        /* Colour */
                                        REPEAT(setli(line, x, colour2, Bpp));
                                        break;

                                case 4:        /* Copy */
                                        REPEAT(setli(line, x, cvalx(&input, Bpp), Bpp));
                                        break;

                                case 8:        /* Bicolour */
                                        REPEAT(if (bicolour)
                                               {
                                               setli(line, x, colour2, Bpp); bicolour = False;}
                                               else
                                               {
                                               setli(line, x, colour1, Bpp); bicolour = True;
                                               count++;}
                                        );
                                        break;

                                case 0xd:        /* White */
                                        REPEAT(setli(line, x, 0xffffffff, Bpp));
                                        break;

                                case 0xe:        /* Black */
                                        REPEAT(setli(line, x, 0, Bpp));
                                        break;

                                default:
                                        return Error;
                        }
                }
        }

        return True;
}


int UnpackRDPTo8(unsigned char *output, int width, int height, unsigned char *input, int size)
{
    uint8 *end = input + size;
    uint8 *prevline = NULL, *line = NULL;
    int opcode, count, offset, isfillormix, x = width;
    int lastopcode = -1, insertmix = False, bicolour = False;
    uint8 code;
    uint8 colour1 = 0, colour2 = 0;
    uint8 mixmask, mask = 0;
    uint8 mix = 0xff;
    int fom_mask = 0;

    while (input < end)
    {
        fom_mask = 0;
        code = CVAL(input);
        opcode = code >> 4;
        /* Handle different opcode forms */
        switch (opcode)
        {
            case 0xc:
            case 0xd:
            case 0xe:
                opcode -= 6;
                count = code & 0xf;
                offset = 16;
                break;
            case 0xf:
                opcode = code & 0xf;
                if (opcode < 9)
                {
                    count = CVAL(input);
                    count |= CVAL(input) << 8;
                }
                else
                {
                    count = (opcode < 0xb) ? 8 : 1;
                }
                offset = 0;
                break;
            default:
                opcode >>= 1;
                count = code & 0x1f;
                offset = 32;
                break;
        }
        /* Handle strange cases for counts */
        if (offset != 0)
        {
            isfillormix = ((opcode == 2) || (opcode == 7));
            if (count == 0)
            {
                if (isfillormix)
                        count = CVAL(input) + 1;
                else
                        count = CVAL(input) + offset;
            }
            else 
            if (isfillormix)
            {
                count <<= 3;
            }
        }
        /* Read preliminary data */
        switch (opcode)
        {
            case 0:        /* Fill */
                if ((lastopcode == opcode) && !((x == width) && (prevline == NULL)))
                insertmix = True;
                break;
            
            case 8:        /* Bicolour */
                colour1 = CVAL(input);
            
            case 3:        /* Colour */
                colour2 = CVAL(input);
                break;
            
            case 6:        /* SetMix/Mix */
            
            case 7:        /* SetMix/FillOrMix */
                mix = CVAL(input);
                opcode -= 5;
                break;
            
            case 9:        /* FillOrMix_1 */
                mask = 0x03;
                opcode = 0x02;
                fom_mask = 3;
                break;
            
            case 0x0a:        /* FillOrMix_2 */
                mask = 0x05;
                opcode = 0x02;
                fom_mask = 5;
                break;
        }
        lastopcode = opcode;
        mixmask = 0;
        /* Output body */
        while (count > 0)
        {
            if (x >= width)
            {
                if (height <= 0)
                        return False;
                x = 0;
                height--;
                prevline = line;
                line = output + height * width;
            }
            switch (opcode)
            {
                
                case 0:        /* Fill */
                    if (insertmix)
                    {
                        if (prevline == NULL)
                            line[x] = mix;
                        else
                            line[x] = prevline[x] ^ mix;
                        insertmix = False;
                        count--;
                        x++;
                    }
                    if (prevline == NULL)
                    {
                        REPEAT(line[x] = 0)
                    }
                    else
                    {
                        REPEAT(line[x] = prevline[x])
                    }
                    break;
                
                case 1:        /* Mix */
                    if (prevline == NULL)
                    {
                        REPEAT(line[x] = mix)
                    }
                    else
                    {
                        REPEAT(line[x] = prevline[x] ^ mix)
                    }
                    break;
                
                case 2:        /* Fill or Mix */
                    if (prevline == NULL)
                    {
                        REPEAT(
                                MASK_UPDATE();
                                if (mask & mixmask)
                                    line[x] = mix;
                                else
                                    line[x] = 0;
                              )
                    }
                    else
                    {
                        REPEAT(
                                MASK_UPDATE();
                                if (mask & mixmask)
                                    line[x] = prevline[x] ^ mix;
                                else
                                    line[x] = prevline[x];
                               )
                    }
                    break;
                
                case 3:        /* Colour */
                    REPEAT(line[x] = colour2)
                break;
                
                case 4:        /* Copy */
                    REPEAT(line[x] = CVAL(input))
                break;
                
                case 8:        /* Bicolour */
                    REPEAT(
                            if (bicolour)
                            {
                                line[x] = colour2;
                                bicolour = False;
                            }
                            else
                            {
                                line[x] = colour1;
                                bicolour = True; count++;
                            }
                          )
                break;
                
                case 0xd:        /* White */
                    REPEAT(line[x] = 0xff)
                break;
                
                case 0xe:        /* Black */
                    REPEAT(line[x] = 0)
                break;
                
                default:
                    #ifdef PANIC
                    *logofs << "UnpackRDPTo8: PANIC! Unimplemented bitmap opcode "
                                << (void *) opcode << ".\n" << logofs_flush;
                    #endif
                    cerr << "Error" << ": Unimplemented bitmap opcode "
                             << (void *) opcode << " in RDP image.\n";
                    return False;
                }
                
        }
    }

    return True;
}                     

int UnpackRDPTo16(unsigned char *output, int width, int height,
                     unsigned char *input, int size)
{
    uint8 *end = input + size;
    uint16 *prevline = NULL, *line = NULL;
    int opcode = -1;
    int count, offset, isfillormix, x = width;
    int lastopcode = -1, insertmix = False, bicolour = False;
    uint8 code;
    uint16 colour1 = 0, colour2 = 0;
    uint8 mixmask, mask = 0;
    uint16 mix = 0xffff;
    int fom_mask = 0;
        
    while (input < end)
    {
        fom_mask = 0;
        code = CVAL(input);
        opcode = code >> 4;
        /* Handle different opcode forms */
        switch (opcode)
        {
            case 0xc:
        
            case 0xd:
            
            case 0xe:
                opcode -= 6;
                count = code & 0xf;
                offset = 16;
            break;
        
            case 0xf:
                opcode = code & 0xf;
                if (opcode < 9)
                {
                    count = CVAL(input);
                    count |= CVAL(input) << 8;
                }
                else
                {
                    count = (opcode < 0xb) ? 8 : 1;
                }
                offset = 0;
            break;
            
            default:
                opcode >>= 1;
                count = code & 0x1f;
                offset = 32;
            break;
        }
        /* Handle strange cases for counts */
        if (offset != 0)
        {
            isfillormix = ((opcode == 2) || (opcode == 7));
            if (count == 0)
            {
                if (isfillormix)
                    count = CVAL(input) + 1;
                else
                    count = CVAL(input) + offset;
            }
            else 
            if (isfillormix)
            {
                count <<= 3;
            }
        }
        /* Read preliminary data */
        switch (opcode)
        {
            case 0:        /* Fill */
                if ((lastopcode == opcode) && !((x == width) && (prevline == NULL)))
                insertmix = True;
            break;
        
            case 8:        /* Bicolour */
                CVAL2(input, colour1);
        
            case 3:        /* Colour */
                CVAL2(input, colour2);
            break;
        
            case 6:        /* SetMix/Mix */
            
            case 7:        /* SetMix/FillOrMix */
                CVAL2(input, mix);
                opcode -= 5;
            break;
        
            case 9:        /* FillOrMix_1 */
                mask = 0x03;
                opcode = 0x02;
                fom_mask = 3;
            break;
            
            case 0x0a:        /* FillOrMix_2 */
                mask = 0x05;
                opcode = 0x02;
                fom_mask = 5;
            break;
        }
        
        lastopcode = opcode;
        mixmask = 0;
        /* Output body */
        while (count > 0)
        {
            if (x >= width)
            {
                if (height <= 0)
                {
                    #ifdef PANIC
                      *logofs << "UnpackRDPTo16: PANIC! Bad height value " << height
                            << " in RDP image.\n" << logofs_flush;
                    #endif
                    cerr << "Error" << ": Bad height value " << height
                                        << " in RDP image.\n";
                    return Error;
                }
                x = 0;
                height--;
                prevline = line;
                line = ((uint16 *) output) + height * width;
            }
            switch (opcode)
            {
                case 0:        /* Fill */
                    if (insertmix)
                    {
                        if (prevline == NULL)
                            line[x] = mix;
                        else
                            line[x] = prevline[x] ^ mix;
                        insertmix = False;
                        count--;
                        x++;
                    }
                    if (prevline == NULL)
                    {
                        REPEAT(line[x] = 0)
                    }
                    else
                    {
                        REPEAT(line[x] = prevline[x])
                    }
                break;
                
                case 1:        /* Mix */
                    if (prevline == NULL)
                    {
                        REPEAT(line[x] = mix)
                    }
                        else
                    {
                        REPEAT(line[x] = prevline[x] ^ mix)
                    }
                break;
                
                case 2:        /* Fill or Mix */
                    if (prevline == NULL)
                    {
                        REPEAT
                        (
                            MASK_UPDATE();
                            if (mask & mixmask)
                                line[x] = mix;
                            else
                                line[x] = 0;
                        )
                    }
                    else
                    {
                        REPEAT
                        (
                            MASK_UPDATE();
                            if (mask & mixmask)
                                line[x] = prevline[x] ^ mix;
                            else
                                line[x] = prevline[x];
                        )
                    }
                break;
                
                case 3:        /* Colour */
                    REPEAT(line[x] = colour2)
                break;
                
                case 4:        /* Copy */
                    REPEAT(CVAL2(input, line[x]))
                break;
                
                case 8:        /* Bicolour */
                    REPEAT
                    (
                        if (bicolour)
                        {
                            line[x] = colour2;
                            bicolour = False;
                        }
                        else
                        {
                            line[x] = colour1;
                            bicolour = True;
                            count++;
                        }
                    )
                break;
                
                case 0xd:        /* White */
                    REPEAT(line[x] = 0xffff)
                break;
                
                case 0xe:        /* Black */
                    REPEAT(line[x] = 0)
                break;
                
                default:
                    #ifdef PANIC
                    *logofs << "UnpackRDPTo16: PANIC! Unimplemented bitmap opcode "
                                << (void *) opcode << ".\n" << logofs_flush;
                    #endif
                    cerr << "Error" << ": Unimplemented bitmap opcode "
                            << (void *) opcode << " in RDP image.\n";
                return Error;
            }
        }
    }

    return True;
}

int UnpackRDPTo24(unsigned char *output, int width, int height,
                     unsigned char *input, int size)
{
    uint8 *end = input + size;
    uint8 *prevline = NULL, *line = NULL;
    int opcode, count, offset, isfillormix, x = width;
    int lastopcode = -1, insertmix = False, bicolour = False;
    uint8 code;
    uint8 colour1[3] = {0, 0, 0}, colour2[3] = {0, 0, 0};
    uint8 mixmask, mask = 0;
    uint8 mix[3] = {0xff, 0xff, 0xff};
    int fom_mask = 0;

    while (input < end)
    {
        fom_mask = 0;
        code = CVAL(input);
        opcode = code >> 4;
        /* Handle different opcode forms */
        switch (opcode)
        {
            case 0xc:
            
            case 0xd:
            
            case 0xe:
                opcode -= 6;
                count = code & 0xf;
                offset = 16;
            break;
            
            case 0xf:
                opcode = code & 0xf;
                if (opcode < 9)
                {
                    count = CVAL(input);
                    count |= CVAL(input) << 8;
                }
                else
                {
                    count = (opcode < 0xb) ? 8 : 1;
                }
                offset = 0;
            break;
            
            default:
                opcode >>= 1;
                count = code & 0x1f;
                offset = 32;
            break;
        }
        /* Handle strange cases for counts */
        if (offset != 0)
        {
            isfillormix = ((opcode == 2) || (opcode == 7));
            if (count == 0)
            {
                if (isfillormix)
                    count = CVAL(input) + 1;
                else
                    count = CVAL(input) + offset;
            }
            else 
            if (isfillormix)
            {
                count <<= 3;
            }
        }
        /* Read preliminary data */
        switch (opcode)
        {
            case 0:        /* Fill */
                if ((lastopcode == opcode) && !((x == width) && (prevline == NULL)))
                insertmix = True;
            break;
        
            case 8:        /* Bicolour */
                colour1[0] = CVAL(input);
                colour1[1] = CVAL(input);
                colour1[2] = CVAL(input);
            
            case 3:        /* Colour */
                colour2[0] = CVAL(input);
                colour2[1] = CVAL(input);
                colour2[2] = CVAL(input);
            break;
            
            case 6:        /* SetMix/Mix */
            
            case 7:        /* SetMix/FillOrMix */
                mix[0] = CVAL(input);
                mix[1] = CVAL(input);
                mix[2] = CVAL(input);
                opcode -= 5;
            break;
        
            case 9:        /* FillOrMix_1 */
                mask = 0x03;
                opcode = 0x02;
                fom_mask = 3;
            break;
        
            case 0x0a:        /* FillOrMix_2 */
                mask = 0x05;
                opcode = 0x02;
                fom_mask = 5;
            break;
        }
        lastopcode = opcode;
        mixmask = 0;
        /* Output body */
        while (count > 0)
        {
            if (x >= width)
            {
                if (height <= 0)
                {
                    #ifdef PANIC
                      *logofs << "UnpackRDPTo24: PANIC! Bad height value " << height
                            << " in RDP image.\n" << logofs_flush;
                    #endif
                    cerr << "Error" << ": Bad height value " << height
                                        << " in RDP image.\n";

                    return Error;
                }
                x = 0;
                height--;
                prevline = line;
                line = output + height * (width * 3);
            }
            switch (opcode)
            {
                case 0:        /* Fill */
                    if (insertmix)
                    {
                        if (prevline == NULL)
                        {
                            line[x * 3] = mix[0];
                            line[x * 3 + 1] = mix[1];
                            line[x * 3 + 2] = mix[2];
                        }
                        else
                        {
                            line[x * 3] =
                            prevline[x * 3] ^ mix[0];
                            line[x * 3 + 1] =
                            prevline[x * 3 + 1] ^ mix[1];
                            line[x * 3 + 2] =
                            prevline[x * 3 + 2] ^ mix[2];
                        }
                        insertmix = False;
                        count--;
                        x++;
                    }
                    if (prevline == NULL)
                    {
                        REPEAT
                        (
                            line[x * 3] = 0;
                            line[x * 3 + 1] = 0;
                            line[x * 3 + 2] = 0;
                        )
                    }
                    else
                    {
                        REPEAT
                        (
                            line[x * 3] = prevline[x * 3];
                            line[x * 3 + 1] = prevline[x * 3 + 1];
                            line[x * 3 + 2] = prevline[x * 3 + 2];
                        )
                    }
                break;
                
                case 1:        /* Mix */
                    if (prevline == NULL)
                    {
                        REPEAT
                        (
                            line[x * 3] = mix[0];
                            line[x * 3 + 1] = mix[1];
                            line[x * 3 + 2] = mix[2];
                        )
                    }
                    else
                    {
                        REPEAT
                        (
                            line[x * 3] =
                            prevline[x * 3] ^ mix[0];
                            line[x * 3 + 1] =
                            prevline[x * 3 + 1] ^ mix[1];
                            line[x * 3 + 2] =
                            prevline[x * 3 + 2] ^ mix[2];
                        )
                    }
                break;
                
                case 2:        /* Fill or Mix */
                    if (prevline == NULL)
                    {
                        REPEAT
                        (
                            MASK_UPDATE();
                            if (mask & mixmask)
                            {
                                line[x * 3] = mix[0];
                                line[x * 3 + 1] = mix[1];
                                line[x * 3 + 2] = mix[2];
                            }
                            else
                            {
                                line[x * 3] = 0;
                                line[x * 3 + 1] = 0;
                                line[x * 3 + 2] = 0;
                            }
                        )
                    }
                    else
                    {
                        REPEAT
                        (
                            MASK_UPDATE();
                            if (mask & mixmask)
                            {
                                line[x * 3] =  prevline[x * 3] ^ mix [0];
                                line[x * 3 + 1] = prevline[x * 3 + 1] ^ mix [1];
                                line[x * 3 + 2] = prevline[x * 3 + 2] ^ mix [2];
                            }
                            else
                            {
                                line[x * 3] = prevline[x * 3];
                                line[x * 3 + 1] = prevline[x * 3 + 1];
                                line[x * 3 + 2] = prevline[x * 3 + 2];
                            }
                        )
                    }
                break;
                
                case 3:        /* Colour */
                    REPEAT
                    (
                        line[x * 3] = colour2 [0];
                        line[x * 3 + 1] = colour2 [1];
                        line[x * 3 + 2] = colour2 [2];
                    )
                break;
                
                case 4:        /* Copy */
                    REPEAT
                    (
                        line[x * 3] = CVAL(input);
                        line[x * 3 + 1] = CVAL(input);
                        line[x * 3 + 2] = CVAL(input);
                    )
                break;
                
                case 8:        /* Bicolour */
                    REPEAT
                    (
                        if (bicolour)
                        {
                            line[x * 3] = colour2[0];
                            line[x * 3 + 1] = colour2[1];
                            line[x * 3 + 2] = colour2[2];
                            bicolour = False;
                        }
                        else
                        {
                            line[x * 3] = colour1[0];
                            line[x * 3 + 1] = colour1[1];
                            line[x * 3 + 2] = colour1[2];
                            bicolour = True;
                            count++;
                        }
                    )
                break;
                
                case 0xd:        /* White */
                    REPEAT
                    (
                        line[x * 3] = 0xff;
                        line[x * 3 + 1] = 0xff;
                        line[x * 3 + 2] = 0xff;
                    )
                break;
                
                case 0xe:        /* Black */
                    REPEAT
                    (
                        line[x * 3] = 0;
                        line[x * 3 + 1] = 0;
                        line[x * 3 + 2] = 0;
                    )
                break;
                
                default:
                    #ifdef PANIC
                    *logofs << "UnpackRDPTo24: PANIC! Unimplemented bitmap opcode "
                                << (void *) opcode << ".\n" << logofs_flush;
                    #endif
                    cerr << "Error" << ": Unimplemented bitmap opcode "
                        << (void *) opcode << " in RDP image.\n";

                    return Error;
            }
        }
        
    }

    return True;
}

int UnpackRDPText(unsigned int drawable, unsigned int gcontext, int big_endian,
                      unsigned char *src_data, unsigned int src_size,
                          unsigned char *dst_data, unsigned int dst_size)
{
  unsigned char *buffer = dst_data;

  //
  // Number of glyphs.
  //

  unsigned int elements = GetULONG(src_data + 12, big_endian);

  if (elements != (src_size - 16) / 20)
  {
    #ifdef PANIC
    *logofs << "UnpackRDPText: PANIC! Text elements mismatch. "
            << (src_size - 16) / 20 << " expected " << elements
            << " found.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": RDP text elements mismatch. "
         << (src_size - 16) / 20 << " expected " << elements
         << " found.\n";

    return -1;
  }

  #ifdef TEST
  *logofs << "UnpackRDPText: Using drawable " << (void *) drawable
          << " and gcontext " << (void *) gcontext << ".\n"
          << logofs_flush;
  #endif

  //
  // First X_ChangeGC sets background
  // foreground and fill style.
  //

  *buffer = (unsigned char) X_ChangeGC;

  PutUINT(6, buffer + 2, big_endian);

  PutULONG(gcontext, buffer + 4,  big_endian);

  PutULONG(GCForeground | GCBackground | GCFillStyle, buffer + 8,  big_endian);

  PutULONG(GetULONG(src_data, big_endian), buffer + 12,  big_endian);
  PutULONG(GetULONG(src_data + 4, big_endian), buffer + 16,  big_endian);
  PutULONG(GetULONG(src_data + 8, big_endian), buffer + 20,  big_endian);

  //
  // Now src_data points to the
  // first glyph element.
  //

  src_data += 16;
  buffer   += 24;

  for (unsigned int i = 0; i < elements; i++)
  {
    //
    // X_ChangeGC sets x, y and pixmap
    // for the stipple.
    //

    unsigned int pixmap, x, y, width, height;

    pixmap = GetULONG(src_data, big_endian);
    x      = GetULONG(src_data + 4, big_endian);
    y      = GetULONG(src_data + 8, big_endian);
    width  = GetULONG(src_data + 12, big_endian);
    height = GetULONG(src_data + 16, big_endian);

    *buffer = (unsigned char) X_ChangeGC;

    PutUINT(6, buffer + 2, big_endian);

    PutULONG(gcontext, buffer + 4,  big_endian);

    PutULONG(GCStipple | GCTileStipXOrigin | GCTileStipYOrigin, buffer + 8,  big_endian);

    #ifdef TEST
    *logofs << "UnpackRDPText: Building X_ChangeGC element "
            << i << " with pixmap " << (void *) pixmap
            << ".\n" << logofs_flush;
    #endif

    //
    // Stipple, tile-stipple-x-origin
    // and tile-stipple-y-origin.
    //

    PutULONG(pixmap, buffer + 12, big_endian);
    PutULONG(x,      buffer + 16, big_endian);
    PutULONG(y,      buffer + 20, big_endian);

    buffer += 24;

    //
    // Draw a rectangle using the stipple.
    //

    *buffer = (unsigned char) X_PolyFillRectangle;

    PutUINT(5, buffer + 2, big_endian);

    PutULONG(drawable, buffer + 4, big_endian);
    PutULONG(gcontext, buffer + 8, big_endian);

    PutUINT(x,      buffer + 12, big_endian);
    PutUINT(y,      buffer + 14, big_endian);
    PutUINT(width,  buffer + 16, big_endian);
    PutUINT(height, buffer + 18, big_endian);

    //
    // Go to next glyph.
    //

    buffer   += 20;
    src_data += 20;
  }

  return 1;
}

//
// The following are from the VNC client software.
//

int UnpackHextileTo8(int src_width, int src_height, unsigned char *src_data,
                         int src_size, int dst_depth, int dst_width, int dst_height,
                             unsigned char *dst_data, int dst_size)
{
    int i, j, k, x, y, w, h, sx, sy, sw, sh;
    CARD8 *ptr;
    CARD8 subencoding;
    CARD8 nSubrects;
    CARD8 bg, fg;
    unsigned int bytesPerPixel, bytesPerLine;

    unsigned char *compData = src_data;
    unsigned char *decompData = dst_data;

    if (dst_height == 0)
    {
      #ifdef TEST
      *logofs << "UnpackHextile8: WARNING! Spurious hextile "
              << "packet with dst_height " << dst_height
              << " and dst_size " << dst_size << ".\n"
              << logofs_flush;
      #endif

      bytesPerLine = 0;
    }
    else
    {
      bytesPerLine = dst_size / dst_height;
    }

    bytesPerPixel = dst_depth / 8;

    bg = bg8;
    fg = fg8;

    for (y = 0; y < dst_height; y += 16)
    {
        for (x = 0; x < dst_width; x += 16)
        {
            w = h = 16;
            if (dst_width - x < 16)
                w = dst_width - x;
            if (dst_height - y < 16)
                h = dst_height - y;

            // read the subencoding
            subencoding = *(CARD8 *)compData;
            compData ++;

            // raw Hextile, just copy everything in the right place
            if (subencoding & rfbHextileRaw)
            {
                decompData = dst_data + y * bytesPerLine + x * bytesPerPixel;
                for (j = 0; j < h; j++)
                {
                    if ((int) ((decompData - dst_data) + w * bytesPerPixel) > dst_size)
                    {
                        #ifdef TEST
                        *logofs << "UnpackHextile8: PANIC! Can't write at pixel "
                                << (decompData - dst_data) + bytesPerPixel
                                << " with size " << dst_size << " in context "
                                << "[A].\n" << logofs_flush;
                        #endif

                        continue;
                    }

                    memcpy(decompData, compData, w * bytesPerPixel);
                    decompData += bytesPerLine;
                    compData += w * bytesPerPixel;
                }

                continue;
            }

            // backgroundSpecified Hextile, read the new background
            if (subencoding & rfbHextileBackgroundSpecified)
            {
                    bg = *(CARD8 *)compData;
                    compData += bytesPerPixel;
            }

            // fill the tile with the background color
            decompData = dst_data + y * bytesPerLine + x * bytesPerPixel;
            for (j = 0; j < h; j++)
            {
                for (k = 0; k < w; k++)
                {
                    if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                    {
                        #ifdef TEST
                        *logofs << "UnpackHextile8: PANIC! Can't write at pixel "
                                << (decompData - dst_data) + bytesPerPixel
                                << " with size " << dst_size << " in context "
                                << "[B].\n" << logofs_flush;
                        #endif

                        continue;
                    }

                    *(CARD8 *)decompData = bg;
                    decompData += bytesPerPixel;
                }
                decompData += bytesPerLine - w * bytesPerPixel;
            }

            // foregroundSpecified Hextile, read the new foreground
            if (subencoding & rfbHextileForegroundSpecified)
            {
                    fg = *(CARD8 *)compData;
                    compData += bytesPerPixel;
            }

            if (!(subencoding & rfbHextileAnySubrects))
            {
                continue;
            }

            nSubrects = *(CARD8 *)compData;
            compData++;
            ptr = (CARD8 *)compData;

            // handle the subrectangles of the tile
            if (subencoding &rfbHextileSubrectsColoured)
            {
                for (i = 0; i < nSubrects; i++)
                {
                    GET_PIXEL8(fg, ptr);
                    sx = rfbHextileExtractX(*ptr);
                    sy = rfbHextileExtractY(*ptr);
                    ptr++;
                    sw = rfbHextileExtractW(*ptr);
                    sh = rfbHextileExtractH(*ptr);
                    ptr++;

                    decompData = dst_data + (y + sy) * bytesPerLine + (x + sx) * bytesPerPixel;
                    for (j = 0; j < sh; j++)
                    {
                        for (k = 0; k < sw; k++)
                        {
                            if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                            {
                                #ifdef TEST
                                *logofs << "UnpackHextile8: PANIC! Can't write at pixel "
                                        << (decompData - dst_data) + bytesPerPixel
                                        << " with size " << dst_size << " in context "
                                        << "[C].\n" << logofs_flush;
                                #endif

                                continue;
                            }

                            *(CARD8 *)decompData = fg;
                            decompData += bytesPerPixel;
                        }
                        decompData += bytesPerLine - sw * bytesPerPixel;
                    }
                }
            }
            else
            {
                for (i = 0; i < nSubrects; i++)
                {
                    sx = rfbHextileExtractX(*ptr);
                    sy = rfbHextileExtractY(*ptr);
                    ptr++;
                    sw = rfbHextileExtractW(*ptr);
                    sh = rfbHextileExtractH(*ptr);
                    ptr++;

                    decompData = dst_data + (y + sy) * bytesPerLine + (x + sx) * bytesPerPixel;
                    for (j = 0; j < sh; j++)
                    {
                        for (k = 0; k < sw; k++)
                        {
                            if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                            {
                                #ifdef TEST
                                *logofs << "UnpackHextile8: PANIC! Can't write at pixel "
                                        << (decompData - dst_data) + bytesPerPixel
                                        << " with size " << dst_size << " in context "
                                        << "[D].\n" << logofs_flush;
                                #endif

                                continue;
                            }

                            *(CARD8 *)decompData = fg;
                            decompData += bytesPerPixel;
                        }
                        decompData += bytesPerLine - sw * bytesPerPixel;
                    }
                }
                
            }
            compData = ptr;
        }
    }

    bg8 = bg;
    fg8 = fg;

    #ifdef DEBUG
    *logofs << "UnpackHextileTo8. Incoming " << src_size << " run "
            << compData - src_data << " outgoing " << dst_size
            << " next offset " << decompData - dst_data << ".\n"
            << logofs_flush;
    #endif

    return True;
}

int UnpackHextileTo16(int src_width, int src_height, unsigned char *src_data,
                          int src_size, int dst_depth, int dst_width, int dst_height,
                              unsigned char *dst_data, int dst_size)
{
    int i, j, k, x, y, w, h, sx, sy, sw, sh;
    CARD8 *ptr;
    CARD8 subencoding;
    CARD8 nSubrects;
    CARD16 bg, fg;
    unsigned int bytesPerPixel, bytesPerLine;

    unsigned char *compData = src_data;
    unsigned char *decompData = dst_data;

    if (dst_height == 0)
    {
      #ifdef TEST
      *logofs << "UnpackHextile16: WARNING! Spurious hextile "
              << "packet with dst_height " << dst_height
              << " and dst_size " << dst_size << ".\n"
              << logofs_flush;
      #endif

      bytesPerLine = 0;
    }
    else
    {
      bytesPerLine = dst_size / dst_height;
    }

    bytesPerPixel = dst_depth / 8;

    bg = bg16;
    fg = fg16;

    for (y = 0; y < dst_height; y += 16)
    {
        for (x = 0; x < dst_width; x += 16)
        {
            w = h = 16;
            if (dst_width - x < 16)
                w = dst_width - x;
            if (dst_height - y < 16)
                h = dst_height - y;

            // read the subencoding
            subencoding = *(CARD8 *)compData;
            compData++;

            // raw Hextile, just copy everything in the right place
            if (subencoding & rfbHextileRaw)
            {
                decompData = dst_data + y * bytesPerLine + x * bytesPerPixel;
                for (j = 0; j < h; j++)
                {
                    if ((int) ((decompData - dst_data) + w * bytesPerPixel) > dst_size)
                    {
                        #ifdef TEST
                        *logofs << "UnpackHextile16: PANIC! Can't write at pixel "
                                << (decompData - dst_data) + bytesPerPixel
                                << " with size " << dst_size << " in context "
                                << "[A].\n" << logofs_flush;
                        #endif

                        continue;
                    }

                    memcpy(decompData, compData, w * bytesPerPixel);
                    decompData += bytesPerLine;
                    compData += w * bytesPerPixel;
                }

                continue;
            }

            // backgroundSpecified Hextile, read the new background
            if (subencoding & rfbHextileBackgroundSpecified)
            {
                    bg = *(CARD16 *)compData;
                    compData += bytesPerPixel;
            }

            // fill the tile with the background color
            decompData = dst_data + y * bytesPerLine + x * bytesPerPixel;
            for (j = 0; j < h; j++)
            {
                for (k = 0; k < w; k++)
                {
                    if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                    {
                        #ifdef TEST
                        *logofs << "UnpackHextile16: PANIC! Can't write at pixel "
                                << (decompData - dst_data) + bytesPerPixel
                                << " with size " << dst_size << " in context "
                                << "[B].\n" << logofs_flush;
                        #endif

                        continue;
                    }

                    *(CARD16 *)decompData = bg;
                    decompData += bytesPerPixel;
                }
                decompData += bytesPerLine - w * bytesPerPixel;
            }

            // foregroundSpecified Hextile, read the new foreground
            if (subencoding & rfbHextileForegroundSpecified)
            {
                    fg = *(CARD16 *)compData;
                    compData += bytesPerPixel;
            }

            if (!(subencoding & rfbHextileAnySubrects))
            {
                continue;
            }

            nSubrects = *(CARD8 *)compData;
            compData++;
            ptr = (CARD8 *)compData;

            // handle the subrectangles of the tile
            if (subencoding &rfbHextileSubrectsColoured)
            {
                for (i = 0; i < nSubrects; i++)
                {
                    GET_PIXEL16(fg, ptr);
                    sx = rfbHextileExtractX(*ptr);
                    sy = rfbHextileExtractY(*ptr);
                    ptr++;
                    sw = rfbHextileExtractW(*ptr);
                    sh = rfbHextileExtractH(*ptr);
                    ptr++;

                    decompData = dst_data + (y + sy) * bytesPerLine + (x + sx) * bytesPerPixel;
                    for (j = 0; j < sh; j++)
                    {
                        for (k = 0; k < sw; k++)
                        {
                            if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                            {
                                #ifdef TEST
                                *logofs << "UnpackHextile16: PANIC! Can't write at pixel "
                                        << (decompData - dst_data) + bytesPerPixel
                                        << " with size " << dst_size << " in context "
                                        << "[C].\n" << logofs_flush;
                                #endif

                                continue;
                            }

                            *(CARD16 *)decompData = fg;
                            decompData += bytesPerPixel;
                        }
                        decompData += bytesPerLine - sw * bytesPerPixel;
                    }
                }
            }
            else
            {
                for (i = 0; i < nSubrects; i++)
                {
                    sx = rfbHextileExtractX(*ptr);
                    sy = rfbHextileExtractY(*ptr);
                    ptr++;
                    sw = rfbHextileExtractW(*ptr);
                    sh = rfbHextileExtractH(*ptr);
                    ptr++;

                    decompData = dst_data + (y + sy) * bytesPerLine + (x + sx) * bytesPerPixel;
                    for (j = 0; j < sh; j++)
                    {
                        for (k = 0; k < sw; k++)
                        {
                            if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                            {
                                #ifdef TEST
                                *logofs << "UnpackHextile16: PANIC! Can't write at pixel "
                                        << (decompData - dst_data) + bytesPerPixel
                                        << " with size " << dst_size << " in context "
                                        << "[D].\n" << logofs_flush;
                                #endif

                                continue;
                            }

                            *(CARD16 *)decompData = fg;
                            decompData += bytesPerPixel;
                        }
                        decompData += bytesPerLine - sw * bytesPerPixel;
                    }
                }
                
            }
            compData = ptr;
        }
    }

    bg16 = bg;
    fg16 = fg;

    #ifdef DEBUG
    *logofs << "UnpackHextileTo16. Incoming " << src_size << " run "
            << compData - src_data << " outgoing " << dst_size
            << " next offset " << decompData - dst_data << ".\n"
            << logofs_flush;
    #endif

    return True;
}

int UnpackHextileTo32(int src_width, int src_height, unsigned char *src_data,
                          int src_size, int dst_depth, int dst_width, int dst_height,
                              unsigned char *dst_data, int dst_size)
{
    int i, j, k, x, y, w, h, sx, sy, sw, sh;
    CARD8 *ptr;
    CARD8 subencoding;
    CARD8 nSubrects;
    CARD32 bg, fg;
    unsigned int bytesPerPixel, bytesPerLine;

    unsigned char *compData = src_data;
    unsigned char *decompData = dst_data;

    if (dst_height == 0)
    {
      #ifdef TEST
      *logofs << "UnpackHextile32: WARNING! Spurious hextile "
              << "packet with dst_height " << dst_height
              << " and dst_size " << dst_size << ".\n"
              << logofs_flush;
      #endif

      bytesPerLine = 0;
    }
    else
    {
      bytesPerLine = dst_size / dst_height;
    }

    bytesPerPixel = dst_depth / 8;

    bg = bg32;
    fg = fg32;

    for (y = 0; y < dst_height; y += 16)
    {
        for (x = 0; x < dst_width; x += 16)
        {
            w = h = 16;
            if (dst_width - x < 16)
                w = dst_width - x;
            if (dst_height - y < 16)
                h = dst_height - y;

            // read the subencoding
            subencoding = *(CARD8 *)compData;
            compData ++;

            // raw Hextile, just copy everything in the right place
            if (subencoding & rfbHextileRaw)
            {
                decompData = dst_data + y * bytesPerLine + x * bytesPerPixel;
                for (j = 0; j < h; j++)
                {
                    if ((int) ((decompData - dst_data) + w * bytesPerPixel) > dst_size)
                    {
                        #ifdef TEST
                        *logofs << "UnpackHextile32: PANIC! Can't write at pixel "
                                << (decompData - dst_data) + bytesPerPixel
                                << " with size " << dst_size << " in context "
                                << "[A].\n" << logofs_flush;
                        #endif

                        continue;
                    }

                    memcpy(decompData, compData, w * bytesPerPixel);
                    decompData += bytesPerLine;
                    compData += w * bytesPerPixel;
                }

                continue;
            }

            // backgroundSpecified Hextile, read the new background
            if (subencoding & rfbHextileBackgroundSpecified)
            {
                    bg = *(CARD32 *)compData;
                    compData += bytesPerPixel;
            }

            // fill the tile with the background color
            decompData = dst_data + y * bytesPerLine + x * bytesPerPixel;
            for (j = 0; j < h; j++)
            {
                for (k = 0; k < w; k++)
                {
                    if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                    {
                        #ifdef TEST
                        *logofs << "UnpackHextile32: PANIC! Can't write at pixel "
                                << (decompData - dst_data) + bytesPerPixel
                                << " with size " << dst_size << " in context "
                                << "[B].\n" << logofs_flush;
                        #endif

                        continue;
                    }

                    *(CARD32 *)decompData = bg;
                    decompData += bytesPerPixel;
                }
                decompData += bytesPerLine - w * bytesPerPixel;
            }

            // foregroundSpecified Hextile, read the new foreground
            if (subencoding & rfbHextileForegroundSpecified)
            {
                    fg = *(CARD32 *)compData;
                    compData += bytesPerPixel;
            }

            if (!(subencoding & rfbHextileAnySubrects))
            {
                continue;
            }

            nSubrects = *(CARD8 *)compData;
            compData++;
            ptr = (CARD8 *)compData;

            // handle the subrectangles of the tile
            if (subencoding &rfbHextileSubrectsColoured)
            {
                for (i = 0; i < nSubrects; i++)
                {
                    GET_PIXEL32(fg, ptr);
                    sx = rfbHextileExtractX(*ptr);
                    sy = rfbHextileExtractY(*ptr);
                    ptr++;
                    sw = rfbHextileExtractW(*ptr);
                    sh = rfbHextileExtractH(*ptr);
                    ptr++;

                    decompData = dst_data + (y + sy) * bytesPerLine + (x + sx) * bytesPerPixel;
                    for (j = 0; j < sh; j++)
                    {
                        for (k = 0; k < sw; k++)
                        {
                            if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                            {
                                #ifdef TEST
                                *logofs << "UnpackHextile32: PANIC! Can't write at pixel "
                                        << (decompData - dst_data) + bytesPerPixel
                                        << " with size " << dst_size << " in context "
                                        << "[C].\n" << logofs_flush;
                                #endif

                                continue;
                            }

                            *(CARD32 *)decompData = fg;
                            decompData += bytesPerPixel;
                        }
                        decompData += bytesPerLine - sw * bytesPerPixel;
                    }
                }
            }
            else
            {
                for (i = 0; i < nSubrects; i++)
                {
                    sx = rfbHextileExtractX(*ptr);
                    sy = rfbHextileExtractY(*ptr);
                    ptr++;
                    sw = rfbHextileExtractW(*ptr);
                    sh = rfbHextileExtractH(*ptr);
                    ptr++;

                    decompData = dst_data + (y + sy) * bytesPerLine + (x + sx) * bytesPerPixel;
                    for (j = 0; j < sh; j++)
                    {
                        for (k = 0; k < sw; k++)
                        {
                            if ((int) ((decompData - dst_data) + bytesPerPixel) > dst_size)
                            {
                                #ifdef TEST
                                *logofs << "UnpackHextile32: PANIC! Can't write at pixel "
                                        << (decompData - dst_data) + bytesPerPixel
                                        << " with size " << dst_size << " in context "
                                        << "[D].\n" << logofs_flush;
                                #endif

                                continue;
                            }

                            *(CARD32 *)decompData = fg;
                            decompData += bytesPerPixel;
                        }
                        decompData += bytesPerLine - sw * bytesPerPixel;
                    }
                }
                
            }
            compData = ptr;
        }
    }

    bg32 = bg;
    fg32 = fg;

    #ifdef DEBUG
    *logofs << "UnpackHextileTo32. Incoming " << src_size << " run "
            << compData - src_data << " outgoing " << dst_size
            << " next offset " << decompData - dst_data << ".\n"
            << logofs_flush;
    #endif

    return True;
}

int UnpackHextile(int src_width, int src_height, unsigned char *src_data,
                      int src_size, int dst_depth, int dst_width, int dst_height,
                          unsigned char *dst_data, int dst_size)
{
  switch (dst_depth)
  {
    case 8:
    {
      return UnpackHextileTo8(src_width, src_height, src_data, src_size, dst_depth,
                                  dst_width, dst_height, dst_data, dst_size);
    }
    case 16:
    {
      return UnpackHextileTo16(src_width, src_height, src_data, src_size, dst_depth,
                                   dst_width, dst_height, dst_data, dst_size);
    }
    case 32:
    {
      return UnpackHextileTo32(src_width, src_height, src_data, src_size, dst_depth,
                                   dst_width, dst_height, dst_data, dst_size);
    }
    default:
    {
      #ifdef PANIC
      *logofs << "UnpackHextile: Unsupported depth " << dst_depth
              << " for Hextile encoding\n" << logofs_flush;
      #endif

      return -1;
    }
  }
}
