/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rigths reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <stdio.h>
#include <signal.h>

#include "os.h"

#include "NXlib.h"

#include "Clean.h"

/* 
 * The value that is returned by sigsetjmp
 * in case of SIGSEGV signal reception.
 */

#define SIGSEGV_JMP  2

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

static int ClientOrder(void)
{
  int whichByte = 1;

  if (*((char *) &whichByte))
  {
    return LSBFirst;
  }

  return MSBFirst;
}

/*
 * We install a SIGSEGV handler. It is used in case of
 * static data (for example when the image data is hard-
 * coded in the libraries). Be sure you restore the old
 * signal handler, stored in old_sig_act, before leaving
 * the function.
 */

static sigjmp_buf JumpBuffer; 

static void SignalHandler(int signal)
{
  #ifdef TEST
  fprintf(stderr,"NXCleanInPlaceImage: WARNING: Signal handler called during cleanup.\n");
  #endif

  siglongjmp(JumpBuffer, SIGSEGV_JMP); 
}
 
/*
 * Copy a clean version of src_image into dst_image.
 */

int CleanImage(XImage *src_image, XImage *dst_image)
{
  register long data_size;
  register int i;

  data_size = (src_image -> bytes_per_line * src_image -> height) >> 2;

  #ifdef TEST
  fprintf(stderr, "******CleanImage: Going to clean image of [%d] bits per pixel.\n",
              src_image -> bits_per_pixel);
  #endif

  switch(src_image -> bits_per_pixel)
  {
    case 32:
    {
      unsigned long mask;

      if (src_image -> byte_order == MSBFirst)
      {
        mask = 0xffffff00;
      }
      else
      {
        mask = 0x00ffffff;
      }
      for (i = 0; i < data_size; i++)
      {
        ((unsigned long *)dst_image -> data)[i] = ((unsigned long *)src_image -> data)[i] & mask;
      }

      break;
    }

    case 24:
    {
      unsigned int bytes_to_clean;

      for (i = 0; i < data_size; i++)
      {
        ((unsigned long *)dst_image -> data)[i] = ((unsigned long *)src_image -> data)[i];
      }

      bytes_to_clean = dst_image -> bytes_per_line - ((dst_image -> width *
                           dst_image -> bits_per_pixel) >> 3);

      if (bytes_to_clean)
      {
        register unsigned long mask = 0xffffffff;
        register int line_size;
        register int i;
                
        line_size = dst_image -> bytes_per_line >> 2;

        if (dst_image -> byte_order == MSBFirst)
        {
          mask = mask << (bytes_to_clean << 3);
        }
        else
        {
          mask = mask >> (bytes_to_clean << 3);
        }

        for (i = 0; i < dst_image -> height;)
        {
          ((unsigned char *)dst_image -> data)[(++i * line_size) -1] &= mask;
        }
      }

      break;
    }

    case 15:
    case 16:
    {
      for (i = 0; i < data_size; i++)
      {
        ((unsigned long *) dst_image -> data)[i] = ((unsigned long *) src_image -> data)[i];
      }

      if (src_image -> width & 0x00000001)
      {
        int card32_per_line = dst_image -> bytes_per_line >> 2;

        for (i = 0; i < dst_image -> height;)
        {
          ((unsigned long *) dst_image -> data)[(++i * card32_per_line) -1] &= 0x0000ffff;
        }
      }

      break;
    }

    case 8:
    {
      unsigned long mask = 0x00000000;

      switch (dst_image -> width % 4)
      {
        case 3:
        {
          mask = 0x00ffffff;

          break;
        }
        case 2:
        {
          mask = 0x0000ffff;

          break;
        }
        case 1:
        {
          mask = 0x000000ff;

          break;
        }
        default:
        {
          /*
           * Nothing to clean.
           */

          break;
        }
      }

      for (i = 0; i < data_size; i++)
      {
        ((unsigned long *) dst_image -> data)[i] = ((unsigned long *) src_image -> data)[i];
      }

      if (mask)
      {
        int card32_per_line;
        int i;

        card32_per_line = dst_image -> bytes_per_line >> 2;

        for (i = 0; i < dst_image -> height; i++)
        {
          ((unsigned long *) dst_image -> data)[(++i * card32_per_line) -1] &= mask;
        }
      }

      break;
    }

    default:
    {
      #ifdef PANIC
      fprintf(stderr, "******CleanImage: PANIC! Cannot clean image of [%d] bits per pixel.\n",
                  src_image -> bits_per_pixel);
      #endif

      return 0;
    }
  }

  return 1;
}

static void CleanXYImg(XImage *image)
{
  int bits_to_clean = 0;

  bits_to_clean = (image -> bytes_per_line << 3) - image -> width - image -> xoffset;

  if (bits_to_clean == 0)
  {
    return;
  }
  
  switch(image -> format)
  {
    case XYBitmap:
    {
      int j;
      
      switch(image -> bitmap_unit)
      {
        case 32:
        {
          CARD32 mask = 0xffffffff;
          CARD8 *mask_ptr8 = (CARD8*) &mask;

          if (image -> byte_order == LSBFirst)
          {
            if (image -> bitmap_bit_order == LSBFirst)
            {
              if (ClientOrder() == LSBFirst)
              {
                mask = mask >> bits_to_clean;
              }
              else
              {
                mask = mask << bits_to_clean;
              }
            }
            else
            {
              int bytes_to_clean;
              int i;
              CARD8 byte_mask = 0xff;
              bytes_to_clean = bits_to_clean >> 3;
              bits_to_clean &= 7;

              for (i = 0; i < bytes_to_clean; i++)
              {
                mask_ptr8[i] = 0x00;
              }

              mask_ptr8[i] = byte_mask >> bits_to_clean;
            }
          }
          else
          {
            if (image -> bitmap_bit_order == LSBFirst)
            {
              int bytes_to_clean;
              int i;

              CARD8 byte_mask = 0xff;
              bytes_to_clean = bits_to_clean >> 3;
              bits_to_clean &= 7;

              for (i = bytes_to_clean; i > 0; i--)
              {
                mask_ptr8[i] = 0x00;
              }

              mask_ptr8[i] = byte_mask << bits_to_clean;
            }
            else
            {
              if (ClientOrder() == LSBFirst)
              {
                mask = mask >> bits_to_clean;
              }
              else
              {
                mask = mask << bits_to_clean;
              }
            }
          }

          for (j = 1; j <= image -> height; j++)
          {
            ((CARD32 *) image -> data)[(j * (image -> bytes_per_line >> 2)) - 1] &= (CARD32) mask;
          }

          break;
        }

        case 15:
        case 16:
        {
          CARD16 mask = 0xffff;
          CARD8 *mask_ptr8 = (CARD8 *) &mask;

          /*
           * FIXME: Need to better test this.
           */

          if (image -> byte_order != LSBFirst)
          {
            if (image -> bitmap_bit_order == LSBFirst)
            {
              mask = mask >> bits_to_clean;
            }
            else
            {
              int bytes_to_clean;
              int i;

              CARD8 byte_mask = 0xff;

              bytes_to_clean = bits_to_clean >> 3;
              bits_to_clean &= 7;

              for (i = 0; i < bytes_to_clean; i++)
              {
                mask_ptr8[i] = 0x00;
              }

              mask_ptr8[i] = byte_mask >> bits_to_clean;
            }
          }
          else
          {
            if (image -> bitmap_bit_order == LSBFirst)
            {
              int bytes_to_clean;
              int i;

              CARD8 byte_mask = 0xff;

              bytes_to_clean = bits_to_clean >> 3;
              bits_to_clean &= 7;

              for (i = bytes_to_clean; i > 0; i--)
              {
                mask_ptr8[i] = 0x00;
              }

              mask_ptr8[i] = byte_mask << bits_to_clean;
            }
            else
            {
              if (ClientOrder() == LSBFirst)
              {
                mask = mask >> bits_to_clean;
              }
              else
              {
                mask = mask << bits_to_clean;
              }
            }
          }

          for (j = 1; j <= image -> height; j++)
          {
            ((CARD16 *) image -> data)[(j * (image -> bytes_per_line >> 1)) - 1] &= (CARD16) mask;
          }
        }
        break;

        case 8:
        {
          CARD8 mask = 0xff;

          if (image -> bitmap_bit_order == LSBFirst)
          {
            mask = mask >> bits_to_clean;
          }
          else
          {
            if (ClientOrder() == LSBFirst)
            {
              mask = mask >> bits_to_clean;
            }
            else
            {
              mask = mask << bits_to_clean;
            }
          }

          for (j = 1; j <= image -> height; j++)
          {
            ((CARD8 *) image -> data)[(j * image -> bytes_per_line) - 1] &= (CARD8)mask;
          }
        }
        break;

        default:
        {
          /*
           * FIXME: What to do in case of error?
           */
        }
        break;
      }

      break;
    }
    case XYPixmap:
    {
      int plane = 0;
      int j, k, card_per_line;

      switch(image -> bitmap_unit)
      {
        case 32:
        {
          CARD32 mask = (CARD32)(0xffffffff);
          CARD8 *mask_ptr8 = (CARD8*) &mask;

          if (image -> byte_order == LSBFirst)
          {
            if (image -> bitmap_bit_order == LSBFirst)
            {
              if (ClientOrder() == LSBFirst)
              {
                mask = mask >> bits_to_clean;
              }
              else
              {
                mask = mask << bits_to_clean;
              }
            }
            else
            {
              int bytes_to_clean;
              int i;

              CARD8 byte_mask = 0xff;

              bytes_to_clean = bits_to_clean >> 3;
              bits_to_clean &= 7;

              for (i = 0; i < bytes_to_clean; i++)
              {
                mask_ptr8[i] = 0x00;
              }

              mask_ptr8[i] = byte_mask >> bits_to_clean;
            }
          }
          else
          {
            if (image -> bitmap_bit_order == LSBFirst)
            {
              int bytes_to_clean;
              int i;

              CARD8 byte_mask = 0xff;

              bytes_to_clean = bits_to_clean >> 3;
              bits_to_clean &= 7;

              for (i = bytes_to_clean; i > 0; i--)
              {
                mask_ptr8[i] = 0x00;
              }

              mask_ptr8[i] = byte_mask << bits_to_clean;
            }
            else
            {
              char temp_char=0;

              if (ClientOrder() == LSBFirst)
              { 
                mask = mask << bits_to_clean;
                temp_char=mask_ptr8[0];
                mask_ptr8[0]=mask_ptr8[3];
                mask_ptr8[3]=temp_char;

                temp_char=mask_ptr8[1];
                mask_ptr8[1]=mask_ptr8[2];
                mask_ptr8[2]=temp_char;
              }
              else
              {
                mask = mask << bits_to_clean;
              }
            }
          }

          card_per_line = image -> bytes_per_line >> 2;

          for (k = image -> depth; --k >= 0; )
          {
            for (j = 1; j <= image -> height; j++)
            {
              ((CARD32 *) image -> data)[plane + (j * card_per_line) -1] &= (CARD32) mask;
            }

            plane = plane + (image -> bytes_per_line * image -> height);
          }

          break;
        }
        case 15:
        case 16:
        {
          CARD16 mask = 0xffff;
          CARD8 *mask_ptr8 = (CARD8 *) &mask;
             
          if (image -> byte_order == LSBFirst)
          {
            if (image -> bitmap_bit_order == LSBFirst)
            {
              mask = mask >> bits_to_clean;
            }
            else
            {
              int bytes_to_clean;
              int i;
              CARD8 byte_mask = 0xff;
              bytes_to_clean = bits_to_clean >> 3;
              bits_to_clean &= 7;

              for (i = 0; i < bytes_to_clean; i++)
              {
                mask_ptr8[i] = 0x00;
              }

              mask_ptr8[i] = byte_mask >> bits_to_clean;
            }
          }
          else
          {
            if (image -> bitmap_bit_order == LSBFirst)
            {
              int bytes_to_clean;
              int i;
              CARD8 byte_mask = 0xff;
              bytes_to_clean = bits_to_clean >> 3;
              bits_to_clean &= 7;

              for (i = bytes_to_clean; i > 0; i--)
              {
                mask_ptr8[i] = 0x00;
              }

              mask_ptr8[i] = byte_mask << bits_to_clean;
            }
            else
            {
              if (ClientOrder() == LSBFirst)
              {
                mask = mask << bits_to_clean;
              }
              else
              {
                mask = mask >> bits_to_clean;
              }
            }
          }

          card_per_line = image -> bytes_per_line >> 1;

          for (k = image -> depth; --k >= 0; )
          {
            for (j = 1; j <= image -> height; j++)
            {
              ((CARD16 *) image -> data)[plane + (j * card_per_line) - 1] &= (CARD16) mask;
            }
            plane = plane + (image -> bytes_per_line * image -> height);
          }
        }
        break;

        case 8:
        {
          CARD8 mask = 0xff;

          if (image -> bitmap_bit_order == LSBFirst)
          {
            mask = mask >> bits_to_clean;
          }
          else
          {
            if (ClientOrder() == LSBFirst)
            {
              mask = mask >> bits_to_clean;
            }
            else
            {
              mask = mask << bits_to_clean;
            }
          }

          for (k = image -> depth; --k >= 0; )
          {
            for (j = 1; j <= image -> height; j++)
            {
              ((CARD8 *) image -> data)[plane + (j * image -> bytes_per_line) - 1] &= (CARD8)mask;
            }

            plane = plane + (image -> bytes_per_line * image -> height);
          }
        }
        break;

        default:
        {
          /*
           * FIXME: What to do in case of error?
           */
        }

        break;
      }

      break;
    }
  }
}

/*
 * Clean the image data directly in the current buffer.
 */

int NXCleanInPlaceImage(XImage *image)
{
  struct sigaction sig_act;
  struct sigaction old_sig_act;
  int retsig = 0;
    
  #ifdef TEST
  fprintf(stderr, "NXCleanInPlaceImage: Going to clean image of [%d] bits per pixel.\n",
              image -> bits_per_pixel);
  #endif
      
  bzero(&JumpBuffer, sizeof(JumpBuffer));
  bzero(&sig_act, sizeof(sig_act));
  bzero(&old_sig_act, sizeof(old_sig_act));

  sig_act.sa_handler = &SignalHandler;

  #ifdef __sun

  sig_act.sa_flags = SA_SIGINFO | SA_NODEFER;

  #else

  sig_act.sa_flags = SA_SIGINFO | SA_NOMASK;

  #endif
  
  sigaddset(&sig_act.sa_mask,SIGSEGV);

  sigaction(SIGSEGV,&sig_act,&old_sig_act);
        
  retsig = sigsetjmp(JumpBuffer, 1);

  if (retsig != SIGSEGV_JMP) 
  {
    if (image -> depth == 1)
    {
      CleanXYImg(image);

      return 1;
    }

    switch (image -> bits_per_pixel)
    {
      case 32:
      {  
        /*
         * We should be able to detect when images
         * are carrying the alpha channel in the
         * fourth byte. Cleaning it will result in
         * images being fully transparent :-).
         */

        /*

	register unsigned long mask;
        register unsigned long data_size;
        register unsigned int i;

        data_size = (image -> bytes_per_line * image -> height) >> 2;

        if (image -> byte_order == MSBFirst)
        {
          mask = (ClientOrder() == LSBFirst ? 0xffffff00 : 0x00ffffff);
        }
        else
        {
          mask = (ClientOrder() == LSBFirst ? 0x00ffffff : 0xffffff00);
        }

        for (i = 0; i < data_size; i++)
        {
          ((unsigned long *) image -> data)[i] &= mask;
        }

	*/

        break;
      }
      case 24:
      {
        unsigned int bytes_to_clean;

        bytes_to_clean = image -> bytes_per_line - ((image -> width *
                                                     image -> bits_per_pixel) >> 3);
        if (bytes_to_clean)
        {
          register unsigned long mask = 0xffffffff;
          register int line_size;
          register int i;

          line_size = image -> bytes_per_line >> 2;

          if (image -> byte_order == MSBFirst)
          {
            mask = mask << (bytes_to_clean << 3);
          }
          else
          {
            mask = mask >> (bytes_to_clean << 3);
          }

          for (i = 0; i < image -> height;)
          {
            ((unsigned char *) image -> data)[(++i * line_size) -1] &= mask;
          }
        }

        break;
      }
      case 15:
      case 16:
      {
        if (image -> width & 0x00000001)
        {
          int card32_per_line;
          int i;
          unsigned long mask;

          /*
           * FIXME: Need to clean the pad bit in 15 bpp.
           */

          mask = 0x0000ffff;

          mask = (ClientOrder() == LSBFirst ? mask : ~mask);

          card32_per_line = image -> bytes_per_line >> 2;

          for (i = 0; i < image -> height;)
          {
            ((unsigned long *) image -> data)[(++i * card32_per_line) - 1] &= mask;
          }
        }

        break;
      }
      case 8:
      {
        unsigned long mask = 0x00000000;
        int card32_per_line;
        int i;

        switch (image -> width % 4)
        {
          case 3:
          {
            mask = 0x00ffffff;

            break;
          }
          case 2:
          {
            mask = 0x0000ffff;

            break;
          }
          case 1:
          {
            mask = 0x000000ff;

            break;
          }
          default:
          {
            /*
             * Nothing to clean.
             */

            return 1;
          }
        }

        mask = (ClientOrder() == LSBFirst ? mask : ~mask);

        mask = (image -> byte_order == LSBFirst ? mask : ~mask);

        card32_per_line = image -> bytes_per_line >> 2;

        for (i = 0; i < image -> height;)
        {
          ((unsigned long *) image -> data)[(++i * card32_per_line) - 1] &= mask;
        }

        break;
      }
      default:
      {
        #ifdef PANIC
         fprintf(stderr, "*****NXCleanInPlaceImage: PANIC! Cannot clean image of [%d] bits per pixel.\n",
                  image -> bits_per_pixel);
        #endif
      }
    }

    /* 
     * Let's clean also the bytes padding at the end
     * of the data chunk. We cleaned everything for
     * each line but probably there is some trash at
     * the end of data.
     */    

    if ((image -> bytes_per_line * image -> height) % 4 > 0)
    {
      int num_card32 = 0;
      unsigned long mask = 0x00000000;

      #ifdef TEST 
      fprintf(stderr,"NXCleanInPlaceImage: Cleaning %d padding bytes at the end of data chunk!\n",
                  (image -> bytes_per_line * image -> height) % 4);
      #endif

      switch((image -> bytes_per_line * image -> height) % 4)
      {
        case 3:
        {
          mask = 0x00ffffff;
          break;
        }
        case 2:
        {
          mask = 0x0000ffff;
          break;
        }
        case 1:
        {  
          mask = 0x000000ff;
          break;
        }
        default:
        {
          mask = 0x00000000;
        }
      }
            
      mask = (ClientOrder() == LSBFirst ? mask : ~mask); 
      
      mask = (image -> byte_order == LSBFirst ? mask : ~mask);

      num_card32 = (image -> bytes_per_line * image -> height) >> 2;

      ((unsigned long *) image -> data)[num_card32 - 1] &= mask;       
    }
  }

  /*
   * Restore the old SIGSEGV handler.
   */

  sigaction(SIGSEGV, &old_sig_act, (struct sigaction *)0);

  return 1;
}
