/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rigths reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "NXlib.h"

#include "Mask.h"
#include "Jpeg.h"

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

#define JPEG_BUF_SIZE 262144

/*
 * Local function prototypes.
 */

static void PrepareRowForJpeg(CARD8 *dst, int y, int count);
static void PrepareRowForJpeg24(CARD8 *dst, int y, int count);
static void PrepareRowForJpeg16(CARD8 *dst, int y, int count);
static void PrepareRowForJpeg32(CARD8 *dst, int y, int count);

static int JpegEmptyOutputBuffer(j_compress_ptr cinfo);

static void JpegInitDestination(j_compress_ptr cinfo);
static void JpegTermDestination(j_compress_ptr cinfo);
static void JpegSetDstManager(j_compress_ptr cinfo);

/*
 * Quality levels.
 */

static int jpegQuality[10] = {20, 30, 40, 50, 55, 60, 65, 70, 75, 80};

/*
 * Image characteristics.
 */

static int bytesPerLine;

static CARD8 bitsPerPixel;
static CARD16 redMax, greenMax, blueMax;
static CARD8  redShift, greenShift, blueShift;

/*
 * Other variables used for the Jpeg
 * encoding.
 */

static char *jpegBeforeBuf = NULL;
static char jpegCompBuf[JPEG_BUF_SIZE];
static int  jpegCompBufSize = JPEG_BUF_SIZE;
static int  jpegError;
static int  jpegDstDataLen;

static struct jpeg_destination_mgr jpegDstManager;

/*
 * Just for debugging purpose.
 */

#ifdef DEBUG

static int  jpegId;
static char jpegName[10];
static FILE *jpegFile;

#endif

/*
 * Function declarations
 */

char* JpegCompressData(XImage *image, int level, int *compressed_size)
{
  struct jpeg_compress_struct cinfo;
  struct jpeg_error_mgr jerr;

  CARD8 *srcBuf;
  JSAMPROW rowPointer[1];

  int dy, w, h;

  *compressed_size = 0;

  /*
   * Initialize the image stuff
   */

  bitsPerPixel = image -> bits_per_pixel;
  bytesPerLine = image -> bytes_per_line;

  #ifdef TEST
  fprintf(stderr, "******JpegCompressData: Image byte order [%d] bitmap bit order [%d].\n",
              image -> byte_order, image -> bitmap_bit_order);

  fprintf(stderr, "******JpegCompressData: Bits per pixel [%d] bytes per line [%d].\n",
              bitsPerPixel, bytesPerLine);
  #endif

  redShift   = FindLSB(image -> red_mask)   - 1;
  greenShift = FindLSB(image -> green_mask) - 1;
  blueShift  = FindLSB(image -> blue_mask)  - 1;

  #ifdef TEST
  fprintf(stderr, "******JpegCompressData: Red mask [0x%lx] green mask [0x%lx] blue mask [0x%lx].\n",
              image -> red_mask, image -> green_mask, image -> blue_mask);

  fprintf(stderr, "******JpegCompressData: Red shift [%d] green shift [%d] blue shift [%d].\n",
              redShift, greenShift, blueShift);
  #endif

  redMax   = image -> red_mask   >> redShift;
  greenMax = image -> green_mask >> greenShift;
  blueMax  = image -> blue_mask  >> blueShift;

  #ifdef TEST
  fprintf(stderr, "******JpegCompressData: Red max [0x%x] green max [0x%x] blue max [0x%x].\n",
              redMax, greenMax, blueMax);
  #endif

  w = image -> width;
  h = image -> height;

  jpegBeforeBuf = image -> data;

  #ifdef DEBUG
  fprintf(stderr, "******JpegCompressData: Width [%d] height [%d] level [%d].\n",
              w, h, level);
  #endif

  if (bitsPerPixel == 1 ||
          bitsPerPixel == 8)
  {
    #ifdef PANIC
    fprintf(stderr, "******JpegCompressData: PANIC! Invalid bits per pixel [%d].\n",
                bitsPerPixel);
    #endif

    return NULL;
  }

  /*
   * Allocate space for one line of the
   * resulting image, 3 bytes per pixel.
   */

  srcBuf = (CARD8 *) malloc(w * 3);

  if (srcBuf == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******JpegCompressData: PANIC! Cannot allocate [%d] bytes.\n",
                w * 3);
    #endif

    return NULL;
  }

  rowPointer[0] = srcBuf;

  cinfo.err = jpeg_std_error(&jerr);

  jpeg_create_compress(&cinfo);

  cinfo.image_width = w;
  cinfo.image_height = h;
  cinfo.input_components = 3;
  cinfo.in_color_space = JCS_RGB;

  jpeg_set_defaults(&cinfo);
  jpeg_set_quality(&cinfo, jpegQuality[level], 1);

  JpegSetDstManager(&cinfo);

  jpeg_start_compress(&cinfo, 1);

  #ifdef DEBUG
  fprintf(stderr, "******JpegCompressedData: Initialization finished\n");
  #endif

  for (dy = 0; dy < h; dy++)
  {
    PrepareRowForJpeg(srcBuf, dy, w);

    jpeg_write_scanlines(&cinfo, rowPointer, 1);

    if (jpegError)
    {
        break;
    }
  }

  #ifdef DEBUG
  fprintf(stderr, "******JpegCompressedData: Compression finished. Lines handled[%d / %d]. Error [%d]\n",
              dy, h, jpegError);
  #endif

  if (!jpegError)
  {
    jpeg_finish_compress(&cinfo);
  }

  jpeg_destroy_compress(&cinfo);

  free((char *)srcBuf);

  if (jpegError)
  {
    return NULL;
  }

  /*
   * Return with the check of the
   * size of the reulting data.
   */

  if (jpegDstDataLen > 0)
  {
    #ifdef DEBUG

    int i = 0;

    fprintf(stderr, "******JpegCompressedData: Compressed size [%d]\n",
                jpegDstDataLen);

    jpegId++;

    sprintf(jpegName, "jpeg%d", jpegId);

    jpegFile = fopen(jpegName, "w");

    for (i = 0; i < jpegDstDataLen; i++)
    {
      fprintf(jpegFile, "%c", *(jpegCompBuf + i));
    }

    fclose(jpegFile);

    #endif

    *compressed_size = jpegDstDataLen;

    return jpegCompBuf;
  }
  else
  {
    #ifdef DEBUG
    fprintf(stderr, "******JpegCompressedData: PANIC! Invalid size of the compressed data [%d]\n",
                jpegDstDataLen);
    #endif

    return NULL;
  }
}

void PrepareRowForJpeg(CARD8 *dst, int y, int count)
{
  if (bitsPerPixel == 32)
  {
    if (redMax == 0xff &&
            greenMax == 0xff &&
                blueMax == 0xff)
    {
      PrepareRowForJpeg24(dst, y, count);
    }
    else
    {
      PrepareRowForJpeg32(dst, y, count);
    }
  }
  else if (bitsPerPixel == 24)
  {
    memcpy(dst, jpegBeforeBuf + y * bytesPerLine, count * 3);
  }
  else
  {
    /*
     * 16 bpp assumed.
     */

    PrepareRowForJpeg16(dst, y, count);
  }
}

void PrepareRowForJpeg24(CARD8 *dst, int y, int count)
{
  CARD32 *fbptr;
  CARD32 pix;

  fbptr = (CARD32 *)(jpegBeforeBuf + y * bytesPerLine);

  while (count--)
  {
    pix = *fbptr++;
    *dst++ = (CARD8)(pix >> redShift);
    *dst++ = (CARD8)(pix >> greenShift);
    *dst++ = (CARD8)(pix >> blueShift);
  }
}

#define DEFINE_JPEG_GET_ROW_FUNCTION(bpp)                                   \
                                                                            \
void PrepareRowForJpeg##bpp(CARD8 *dst, int y, int count)                   \
{                                                                           \
  CARD##bpp *fbptr;                                                         \
  CARD##bpp pix;                                                            \
                                                                            \
  int inRed, inGreen, inBlue;                                               \
                                                                            \
  fbptr = (CARD##bpp *)(jpegBeforeBuf + y * bytesPerLine);                  \
                                                                            \
  while (count--)                                                           \
  {                                                                         \
    pix = *fbptr++;                                                         \
                                                                            \
    inRed = (int)                                                           \
            (pix >> redShift   & redMax);                                   \
    inGreen = (int)                                                         \
            (pix >> greenShift & greenMax);                                 \
    inBlue  = (int)                                                         \
            (pix >> blueShift  & blueMax);                                  \
                                                                            \
    *dst++ = (CARD8)((inRed   * 255 + redMax / 2) /                         \
                         redMax);                                           \
    *dst++ = (CARD8)((inGreen * 255 + greenMax / 2) /                       \
                         greenMax);                                         \
    *dst++ = (CARD8)((inBlue  * 255 + blueMax / 2) /                        \
                         blueMax);                                          \
  }                                                                         \
}

DEFINE_JPEG_GET_ROW_FUNCTION(16)
DEFINE_JPEG_GET_ROW_FUNCTION(32)

/*
 * Destination manager implementation for JPEG library.
 */

void JpegInitDestination(j_compress_ptr cinfo)
{
  jpegError = 0;
  jpegDstManager.next_output_byte = (JOCTET *)jpegCompBuf;
  jpegDstManager.free_in_buffer = (size_t)jpegCompBufSize;
}

int JpegEmptyOutputBuffer(j_compress_ptr cinfo)
{
  jpegError = 1;

  jpegDstManager.next_output_byte = (JOCTET *)jpegCompBuf;
  jpegDstManager.free_in_buffer = (size_t)jpegCompBufSize;

  return 1;
}

void JpegTermDestination(j_compress_ptr cinfo)
{
  jpegDstDataLen = jpegCompBufSize - jpegDstManager.free_in_buffer;
}

void JpegSetDstManager(j_compress_ptr cinfo)
{
  jpegDstManager.init_destination = JpegInitDestination;
  jpegDstManager.empty_output_buffer = JpegEmptyOutputBuffer;
  jpegDstManager.term_destination = JpegTermDestination;
  cinfo -> dest = &jpegDstManager;
}

