/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rigths reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <sys/socket.h>

#ifndef __sun
#include <strings.h>
#endif

#include "dix.h"
#include "os.h"

#define NEED_REPLIES

/*
 * These are needed to enable definition of _NXEnable* and
 * _NX*Params variables in nx-X11 Xlib headers.
 */

#define NX_CLEAN_ALLOC
#define NX_CLEAN_GET
#define NX_CLEAN_FLUSH
#define NX_CLEAN_SEND
#define NX_CLEAN_IMAGES

/*
 * Needed to enable definition of _NXLostSequenceFunction.
 */

#define NX_TRANS_SOCKET

#include "Xlib.h"
#include "Xutil.h"
#include "Xlibint.h"

#include "NXlib.h"
#include "NXproto.h"
#include "NXpack.h"

#include "Clean.h"
#include "Mask.h"
#include "Tight.h"
#include "Jpeg.h"
#include "Png.h"

#include "MD5.h"

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

/*
 * Maximum number of colors allowed in
 * PNG encoding.
 */

#define NB_COLOR_MAX          256

/*
 * Used in asynchronous handling of
 * GetImage replies.
 */

typedef struct
{
  unsigned long  sequence;
  unsigned int   resource;
  unsigned long  mask;
  int            format;
  int            width;
  int            height;
  _XAsyncHandler *handler;
  XImage         *image;
} _NXCollectImageState;

static _NXCollectImageState *_NXCollectedImages[NXNumberOfConnections];

/*
 * Used in asynchronous handling of
 * GetProperty replies.
 */

typedef struct
{
  unsigned long  sequence;
  unsigned int   resource;
  Window         window;
  Atom           property;
  Atom           type;
  int            format;
  unsigned long  items;
  unsigned long  after;
  _XAsyncHandler *handler;
  unsigned char  *data;
} _NXCollectPropertyState;

static _NXCollectPropertyState *_NXCollectedProperties[NXNumberOfConnections];

/*
 * Used in asynchronous handling of
 * GrabPointer replies.
 */

typedef struct
{
  unsigned long  sequence;
  unsigned int   resource;
  int            status;
  _XAsyncHandler *handler;
} _NXCollectGrabPointerState;

static _NXCollectGrabPointerState *_NXCollectedGrabPointers[NXNumberOfConnections];

/*
 * Used in asynchronous handling of
 * GetInputFocus replies.
 */

typedef struct
{
  unsigned long  sequence;
  unsigned int   resource;
  Window         focus;
  int            revert_to;
  _XAsyncHandler *handler;
} _NXCollectInputFocusState;

static _NXCollectInputFocusState *_NXCollectedInputFocuses[NXNumberOfConnections];

/*
 * Used by functions handling cache of
 * packed images.
 */

#define MD5_LENGTH            16

typedef struct
{
  md5_byte_t   md5[MD5_LENGTH];
  XImage       *ximage;
  unsigned int method;
} _NXImageCacheEntry;

unsigned int NXImageCacheSize = 0;
unsigned int NXImageCacheHits = 0;
unsigned int NXImageCacheOps  = 0;
int          NXImageCacheStat = -1;

_NXImageCacheEntry *NXImageCache = NULL;

/*
 * From X11/PutImage.c.
 *
 * Cancel a GetReq operation, before doing
 * _XSend or Data.
 */

#if (defined(__STDC__) && !defined(UNIXCPP)) || defined(ANSICPP)
#define UnGetReq(name)\
    dpy->bufptr -= SIZEOF(x##name##Req);\
    dpy->request--
#else
#define UnGetReq(name)\
    dpy->bufptr -= SIZEOF(x/**/name/**/Req);\
    dpy->request--
#endif

#if (defined(__STDC__) && !defined(UNIXCPP)) || defined(ANSICPP)
#define UnGetEmptyReq()\
    dpy->bufptr -= 4;\
    dpy->request--
#else
#define UnGetEmptyReq(name)\
    dpy->bufptr -= 4;\
    dpy->request--
#endif


/*
 * From X11/ImUtil.c.
 */

extern int _XGetBitsPerPixel(Display *dpy, int depth);
extern int _XGetScanlinePad(Display *dpy, int depth);

/*
 * This is defined in XlibInt.c.
 */

extern int _NXContinueOnDisplayError;

/*
 * Redefine the default error handler.
 */

int _NXDefaultErrorHandler(Display *dpy, XErrorEvent *error);

/*
 * Redefine the default error message printed
 * in the case of out-of-order sequence numbers.
 */

void _NXLostSequenceHandler(Display *dpy, unsigned long newseq,
                                unsigned long lastseq, unsigned int type);

#define ROUNDUP(nbytes, pad) (((nbytes) + ((pad) - 1)) & \
                                  ~(long)((pad) - 1))

static unsigned int DepthOnes(unsigned long mask)
{
  register unsigned long y;

  y = (mask >> 1) &033333333333;
  y = mask - y - ((y >>1) & 033333333333);
  return ((unsigned int) (((y + (y >> 3)) &
              030707070707) % 077));
}

#define CanMaskImage(image, mask) \
\
(image -> format == ZPixmap && mask != NULL && \
    (image -> depth == 32 || image -> depth == 24 || \
        (image -> depth == 16 && (image -> red_mask == 0xf800 && \
            image -> green_mask == 0x7e0 && image -> blue_mask == 0x1f))))

#define ShouldMaskImage(image, mask) (mask -> color_mask != 0xff)

/*
 * If set, the Xlib I/O error handler will simply
 * return, instead of quitting the program. This
 * leaves to the application the responsibility
 * of checking the state of the XlibDisplayIOEr-
 * ror flag.
 */

void NXContinueOnDisplayError(int value)
{
  #ifdef TEST
  fprintf(stderr, "******NXContinueOnDisplayError: Setting the flag to %d with old value %d.\n",
              value, _NXContinueOnDisplayError);
  #endif

  _NXContinueOnDisplayError = value;
}

/*
 * Check if the I/O error flag was set. We need a
 * function here because the flags in the display
 * structure are not visible to the applications.
 */

Bool NXDisplayError(Display *dpy)
{
  return (dpy == NULL || (dpy -> flags & XlibDisplayIOError));
}

/*
 * Shutdown the display descriptor and force Xlib
 * to set the I/O error flag.
 */

Bool NXForceDisplayError(Display *dpy)
{
  if (dpy != NULL && !(dpy -> flags & XlibDisplayIOError))
  {
    shutdown(dpy -> fd, SHUT_RDWR);

    _XIOError(dpy);

    return 1;
  }

  return 0;
}

/*
 * Dummy error handler used internally to catch
 * Xlib failures in replies.
 */

int _NXDefaultErrorHandler(Display *dpy, XErrorEvent *error)
{
  #ifdef TEST
  fprintf(stderr, "******_NXDefaultErrorHandler: Default error handler called.\n");
  #endif

  return 0;
}

void _NXLostSequenceHandler(Display *dpy, unsigned long newseq,
                                unsigned long lastseq, unsigned int type)
{
  #ifdef TEST

  fprintf(stderr, "******_NXLostSequenceHandler: WARNING! Sequence lost with new sequence %ld last request %ld.\n",
              newseq, dpy -> request);

  /*
   * FIXME: Reply or event info must be implemented.
   *
   * fprintf(stderr, "******_NXLostSequenceHandler: WARNING! Expected event or reply was %ld with sequence %ld.\n",
   *             (long) rep -> type, (long) rep -> sequenceNumber);
   */

  fprintf(stderr, "******_NXLostSequenceHandler: WARNING! Last sequence read was %ld display request is %ld.\n",
              lastseq & 0xffff, dpy -> request & 0xffff);

  #endif
}

void NXFreeCache(void)
{
  unsigned int i;

  for (i = 0 ; i < NXImageCacheSize; i++)
  {
    if (NXImageCache[i].ximage && NXImageCache[i].ximage -> data)
    {
      XFree(NXImageCache[i].ximage -> data);
      XFree(NXImageCache[i].ximage);

      NXImageCache[i].ximage = NULL;
    }
  }

  NXImageCacheHits = 0;
  NXImageCacheOps  = 0;
}

/*
 * Agent should get NX parameters at startup, just after having
 * opened the display. If agent is told to use a pack method
 * other than NO_PACK, it should only use NX_PutPackedImage to
 * send ZPixmaps. Pack method returned by this function must be
 * considered a hint. Agents should get the pack method to use
 * by calling SelectPackMethod() until they find a method that
 * is supported by both agent and the remote proxy. The reply
 * has 4 new boolean fields in the last protocol version. They
 * can be used to provide more detailed info to the user.
 */

Status NXGetControlParameters(Display *dpy, unsigned int *link_type, unsigned int *protocol_major,
                                  unsigned int *protocol_minor, unsigned int *protocol_patch,
                                      int *karma_size, int *split_size, unsigned int *pack_method,
                                          unsigned int *pack_quality, int *data_level, int *stream_level,
                                              int *delta_level, unsigned int *load_cache,
                                                  unsigned int *save_cache, unsigned int *startup_cache)
{
  xNXGetControlParametersReply rep;

  register xReq *req;

  LockDisplay(dpy);

  GetEmptyReq(NXGetControlParameters, req);

  #ifdef TEST
  fprintf(stderr, "******NXGetControlParameters: Sending message opcode [%d].\n",
              X_NXGetControlParameters);
  #endif

  if (_XReply(dpy, (xReply *) &rep, 0, xTrue) == xFalse)
  {
    #ifdef TEST
    fprintf(stderr, "******NXGetControlParameters: Error receiving reply.\n");
    #endif

    UnlockDisplay(dpy);

    SyncHandle();

    return 0;
  }

  #ifdef TEST
  fprintf(stderr, "******NXGetControlParameters: Got reply with link type [%u].\n", rep.linkType);

  fprintf(stderr, "******NXGetControlParameters: Protocol major [%u] minor [%u] patch [%u].\n",
              rep.protocolMajor, rep.protocolMinor, rep.protocolPatch);

  fprintf(stderr, "******NXGetControlParameters: Karma size [%d] split size [%d] pack method [%u].\n",
              (int) rep.karmaSize, (int) rep.splitSize, (unsigned int) rep.packMethod);

  fprintf(stderr, "******NXGetControlParameters: Data level [%d] stream level [%d].\n",
              rep.dataLevel, rep.dataLevel);
  #endif

  *link_type = rep.linkType;

  *protocol_major = rep.protocolMajor;
  *protocol_minor = rep.protocolMinor;
  *protocol_patch = rep.protocolPatch;

  *karma_size   = rep.karmaSize;
  *split_size   = rep.splitSize;
  *pack_method  = rep.packMethod;
  *pack_quality = rep.packQuality;

  *data_level   = rep.dataLevel;
  *stream_level = rep.streamLevel;
  *delta_level  = rep.deltaLevel;

  *load_cache    = rep.loadCache;
  *save_cache    = rep.saveCache;
  *startup_cache = rep.startupCache;

  UnlockDisplay(dpy);

  SyncHandle();

  /*
   * Install the out-of-sync handler.
   */

  _NXLostSequenceFunction = _NXLostSequenceHandler;

  return 1;
}

/*
 * Based on reply, agent set flags in Xlib to force
 * its allocation functions to clean padding bytes.
 * This function should always be called once at
 * agent's startup.
 */

Status NXGetCleanupParameters(Display *dpy, int *clean_get, int *clean_alloc,
                                  int *clean_flush, int *clean_send,
                                      int *clean_images)
{
  xNXGetCleanupParametersReply rep;

  register xReq *req;

  LockDisplay(dpy);

  GetEmptyReq(NXGetCleanupParameters, req);

  #ifdef TEST
  fprintf(stderr, "******NXGetCleanupParameters: Sending message opcode [%d].\n",
              X_NXGetCleanupParameters);
  #endif

  if (_XReply(dpy, (xReply *) &rep, 0, xTrue) == xFalse)
  {
    #ifdef TEST
    fprintf(stderr, "******NXGetCleanupParameters: Error receiving reply.\n");
    #endif

    UnlockDisplay(dpy);

    SyncHandle();

    return 0;
  }

  #ifdef TEST
  fprintf(stderr, "******NXGetCleanupParameters: Got reply with get [%d] alloc [%d] flush [%d] send [%d] images [%d].\n",
              rep.cleanGet, rep.cleanAlloc, rep.cleanFlush, rep.cleanSend, rep.cleanImages);
  #endif

  /*
   * Copy parameters from NX proxy to Xlib.
   */

  _NXEnableCleanGet    = rep.cleanGet;
  _NXEnableCleanAlloc  = rep.cleanAlloc;
  _NXEnableCleanFlush  = rep.cleanFlush;
  _NXEnableCleanSend   = rep.cleanSend;
  _NXEnableCleanImages = rep.cleanImages;

  /*
   * User could have set defaults in the environment.
   * Avoid Xlib would override parameters set by proxy.
   */
 
  _NXCleanupEnvironmentCached = 1;

  /*
   * Return current settings to agent.
   */

  *clean_get    = rep.cleanGet;
  *clean_alloc  = rep.cleanAlloc;
  *clean_flush  = rep.cleanFlush;
  *clean_send   = rep.cleanSend;
  *clean_images = rep.cleanImages;

  UnlockDisplay(dpy);

  SyncHandle();

  return 1;
}

/*
 * Get from NX proxy user's settings about image handling in Xlib.
 * This function should always be called once at agent's startup.
 * User's settings are applied to the corresponding variables that
 * govern handling of X_PutImage request, so user doesn't have to
 * set up the corresponding "NX_IMAGE_*" environment variables.
 *
 * There is not a function to set explicitly image parameters into
 * Xlib. Agents can directly manipulate _NX* variables if they wish,
 * though there shouldn't be any reason to do so. This can be con-
 * sidered a design error and will probably be fixed in the future.
 */

Status NXGetImageParameters(Display *dpy, int *image_split, int *image_mask, int *image_frame,
                                unsigned int *image_split_method, unsigned int *image_mask_method)
{
  xNXGetImageParametersReply rep;

  register xReq *req;

  const ColorMask     *pColorMask;
  const SplitGeometry *pSplitGeometry;

  LockDisplay(dpy);

  GetEmptyReq(NXGetImageParameters, req);

  #ifdef TEST
  fprintf(stderr, "******NXGetImageParameters: Sending message opcode [%d].\n",
              X_NXGetImageParameters);
  #endif

  if (_XReply(dpy, (xReply *) &rep, 0, xTrue) == xFalse)
  {
    #ifdef TEST
    fprintf(stderr, "******NXGetImageParameters: Error receiving reply.\n");
    #endif

    UnlockDisplay(dpy);

    SyncHandle();

    return 0;
  }

  #ifdef TEST
  fprintf(stderr, "******NXGetImageParameters: Got reply with image split [%d] mask [%d] frame [%d].\n",
              rep.imageSplit, rep.imageMask, rep.imageFrame);

  fprintf(stderr, "******NXGetImageParameters: Split method [%u] mask method [%u].\n",
              (unsigned int) rep.imageSplitMethod, (unsigned int) rep.imageMaskMethod);
  #endif

  /*
   * Copy parameters from NX proxy to Xlib.
   */

  _NXEnableImageSplit  = rep.imageSplit;
  _NXEnableImageMask   = rep.imageMask;
  _NXEnableImageFrame  = rep.imageFrame;

  pColorMask = GetColorMask(rep.imageMaskMethod);

  if (pColorMask)
  {
    _NXColorParams.color_mask      = pColorMask -> color_mask;
    _NXColorParams.correction_mask = pColorMask -> correction_mask;
    _NXColorParams.white_threshold = pColorMask -> white_threshold;
    _NXColorParams.black_threshold = pColorMask -> black_threshold;
  }

  pSplitGeometry = GetSplitGeometry(rep.imageSplitMethod);

  if (pSplitGeometry)
  {
    _NXSplitParams.bitmap_rect_width  = pSplitGeometry -> bitmap_width;
    _NXSplitParams.bitmap_rect_height = pSplitGeometry -> bitmap_height;
    _NXSplitParams.pixmap_rect_width  = pSplitGeometry -> pixmap_width;
    _NXSplitParams.pixmap_rect_height = pSplitGeometry -> pixmap_height;
  }

  /*
   * User could have set defaults in the environment.
   * Avoid Xlib would override parameters set by proxy.
   */
 
  _NXImageEnvironmentCached = 1;

  /*
   * Return current settings to agent.
   */

  *image_split = rep.imageSplit;
  *image_mask  = rep.imageMask;
  *image_frame = rep.imageFrame;

  *image_split_method = rep.imageSplitMethod;
  *image_mask_method  = rep.imageMaskMethod;

  UnlockDisplay(dpy);

  SyncHandle();

  return 1;
}

/*
 * Which unpack methods are supported by proxy?
 */

Status NXGetUnpackParameters(Display *dpy, unsigned int *entries, unsigned char supported_methods[])
{
  register xNXGetUnpackParametersReq *req;

  xNXGetUnpackParametersReply rep;

  register unsigned n;

  #ifdef TEST
  register unsigned i;
  #endif

  if (*entries < NXNumberOfPackMethods)
  {
    #ifdef TEST
    fprintf(stderr, "******NXGetUnpackParameters: Requested only [%d] entries while they should be [%d].\n",
                *entries, NXNumberOfPackMethods);
    #endif

    return 0;
  }

  LockDisplay(dpy);

  GetReq(NXGetUnpackParameters, req);

  req -> entries = *entries;

  #ifdef TEST
  fprintf(stderr, "******NXGetUnpackParameters: Sending message opcode [%d] with [%d] requested entries.\n",
              X_NXGetUnpackParameters, *entries);
  #endif

  if (_XReply(dpy, (xReply *) &rep, 0, xFalse) == xFalse || rep.length == 0)
  {
    #ifdef TEST
    fprintf(stderr, "******NXGetUnpackParameters: Error receiving reply.\n");
    #endif

    UnlockDisplay(dpy);

    SyncHandle();

    return 0;
  }

  if ((n = rep.length << 2) > *entries)
  {
    #ifdef TEST
    fprintf(stderr, "******NXGetUnpackParameters: Got [%d] bytes of reply data while they should be [%d].\n",
                n, *entries);
    #endif

    _XEatData(dpy, (unsigned long) n);

    UnlockDisplay(dpy);

    SyncHandle();

    return 0;
  }

  *entries = n;

  #ifdef TEST
  fprintf(stderr, "******NXGetUnpackParameters: Reading [%d] bytes of reply data.\n", n);
  #endif

  _XReadPad(dpy, supported_methods, n);

  #ifdef TEST

  fprintf(stderr, "******NXGetUnpackParameters: Got reply with methods: ");

  for (i = 0; i < n; i++)
  {
    if (supported_methods[i] != 0)
    {
      fprintf(stderr, "[%d]", i);
    }
  }
  
  fprintf(stderr, ".\n");

  #endif

  UnlockDisplay(dpy);

  SyncHandle();

  return 1;
}

/*
 * Query and enable MIT-SHM support between agent and
 * proxy and proxy and X server. Enable flags must be
 * true if shared memory PutImages and PutPackedImages
 * are desired. On return the flags will say if support
 * has been successfully enabled.
 */

Status NXGetShmemParameters(Display *dpy, unsigned int *enableClient,
                                unsigned int *enableServer, unsigned int *clientSegment,
                                    unsigned int *serverSegment)
{
  register xNXGetShmemParametersReq *req;

  register int stage;

  xNXGetShmemParametersReply rep;

  /*
   * Save the previous handler.
   */

  int (*handler)(Display *, XErrorEvent *) = _XErrorFunction;

  *clientSegment = 0;
  *serverSegment = 0;

  if (*enableClient)
  {
    *clientSegment = XAllocID(dpy);
  }

  if (*enableServer)
  {
    *serverSegment = XAllocID(dpy);
  }

  LockDisplay(dpy);

  _XErrorFunction = _NXDefaultErrorHandler;

  for (stage = 0; stage < 3; stage++)
  {
    GetReq(NXGetShmemParameters, req);

    req -> stage   = stage;

    req -> enableClient = (*enableClient != 0 ? 1 : 0);
    req -> enableServer = (*enableServer != 0 ? 1 : 0);

    req -> clientSegment = *clientSegment;
    req -> serverSegment = *serverSegment;

    #ifdef TEST
    fprintf(stderr, "******NXGetShmemParameters: Sending message opcode [%d] at stage [%d].\n",
                X_NXGetShmemParameters, stage);
    #endif

    #ifdef TEST

    if (stage == 0)
    {
      fprintf(stderr, "******NXGetShmemParameters: Enable client is [%u] enable server is [%u].\n",
                  *enableClient, *enableServer);

      fprintf(stderr, "******NXGetShmemParameters: Client segment is [%u] server segment is [%u].\n",
                  *clientSegment, *serverSegment);
    }

    #endif

    /*
     * There isn't X server reply in the second stage.
     * The procedure followed at X server side is:
     *
     * Stage 0: Send X_QueryExtension and masquerade
     *          the reply.
     *
     * Stage 1: Allocate the shared memory and send
     *          X_ShmAttach to the X server.
     *
     * Stage 2: Send X_GetInputFocus and masquerade
     *          the reply.
     *
     * The last message is used to force a reply and
     * collect any X error caused by a failure in the
     * shared memory initialization.
     */

    if (stage != 1)
    {
      /*
       * We are only interested in the final reply.
       */

      _XReply(dpy, (xReply *) &rep, 0, xTrue);
    }
  }

  /*
   * Return settings to client.
   */

  *enableClient = rep.clientEnabled;
  *enableServer = rep.serverEnabled;

  #ifdef TEST
  fprintf(stderr, "******NXGetShmemParameters: Got final reply with enabled client [%u] and server [%u].\n",
              *enableClient, *enableServer);
  #endif

  _XErrorFunction = handler;

  UnlockDisplay(dpy);

  SyncHandle();

  return 1;
}

/*
 * Tell the proxy to split the next messages.
 */

int NXStartSplit(Display *dpy, unsigned int resource, unsigned int mode)
{
  register xNXStartSplitReq *req;

  LockDisplay(dpy);

  GetReq(NXStartSplit, req);

  req -> resource = resource;
  req -> mode     = mode;

  #ifdef TEST
  fprintf(stderr, "******NXStartSplit: Sending opcode [%d] with resource [%d] mode [%d].\n",
              X_NXStartSplit, resource, mode);
  #endif

  UnlockDisplay(dpy);

  SyncHandle();

  return 1;
}

/*
 * Send the closure of the split sequence and
 * tell the proxy to send the results.
 */

int NXEndSplit(Display *dpy, unsigned int resource)
{
  register xNXEndSplitReq *req;

  LockDisplay(dpy);

  GetReq(NXEndSplit, req);

  req -> resource = resource;

  #ifdef TEST
  fprintf(stderr, "******NXEndSplit: Sending opcode [%d] with resource [%d].\n",
              X_NXStartSplit, resource);
  #endif

  UnlockDisplay(dpy);

  SyncHandle();

  return 1;
}

/*
 * This message must be sent whenever the proxy notifies
 * the client of the completion of a split. If the 'pro-
 * pagate' field is 0, the proxy will not send the ori-
 * ginal request to the X server, but will only free the
 * internal state.
 */

int NXCommitSplit(Display *dpy, unsigned int resource, unsigned int propagate,
                      unsigned char request, unsigned int position)
{
  register xNXCommitSplitReq *req;

  LockDisplay(dpy);

  GetReq(NXCommitSplit, req);

  req -> resource  = resource;
  req -> propagate = propagate;
  req -> request   = request;
  req -> position  = position;

  #ifdef TEST
  fprintf(stderr, "******NXCommitSplit: Sending opcode [%d] with resource [%d] propagate [%d] "
              "request [%d] position [%d].\n", X_NXCommitSplit, resource,
                  propagate, request, position);
  #endif

  UnlockDisplay(dpy);

  SyncHandle();

  return 1;
}

/*
 * Tell to remote proxy to discard expose events
 * of one or more types.
 */

int NXSetExposeEvents(Display *dpy, int expose, int graphics_expose, int no_expose)
{
  register xNXSetExposeEventsReq *req;

  LockDisplay(dpy);

  GetReq(NXSetExposeEvents, req);

  req -> expose         = expose;
  req -> graphicsExpose = graphics_expose;
  req -> noExpose       = no_expose;

  #ifdef TEST
  fprintf(stderr, "******NXSetExposeEvents: Sending message opcode [%d] with flags [%d][%d][%d].\n",
              X_NXSetExposeEvents, req -> expose, req -> graphicsExpose, req -> noExpose);
  #endif

  UnlockDisplay(dpy);

  SyncHandle();

  return 1;
}

int NXSetUnpackGeometry(Display *dpy, unsigned int resource,
                            Screen *screen, Visual *visual)
{
  register xNXSetUnpackGeometryReq *req;

  LockDisplay(dpy);

  GetReq(NXSetUnpackGeometry, req);

  req -> resource = resource;

  req -> depth1Bpp  = _XGetBitsPerPixel(dpy, 1);
  req -> depth4Bpp  = _XGetBitsPerPixel(dpy, 4);
  req -> depth8Bpp  = _XGetBitsPerPixel(dpy, 8);
  req -> depth16Bpp = _XGetBitsPerPixel(dpy, 16);
  req -> depth24Bpp = _XGetBitsPerPixel(dpy, 24);
  req -> depth32Bpp = _XGetBitsPerPixel(dpy, 32);

  if (visual != NULL)
  {
    req -> redMask   = visual -> red_mask;
    req -> greenMask = visual -> green_mask;
    req -> blueMask  = visual -> blue_mask;
  }
  else
  {
    #ifdef TEST
    fprintf(stderr, "******NXSetUnpackGeometry: Got a NULL visual. Assuming default color masks.\n");
    #endif
    req -> redMask = 0x0;
    req -> greenMask = 0x0;
    req -> blueMask = 0x0;
  }

  #ifdef TEST
  fprintf(stderr, "******NXSetUnpackGeometry: Resource [%u] Depth/Bpp [1/%d][4/%d][8/%d][16/%d][24/%d][32/%d].\n",
              resource, req -> depth1Bpp, req -> depth4Bpp, req -> depth8Bpp,
                  req -> depth16Bpp, req -> depth24Bpp, req -> depth32Bpp);

  fprintf(stderr, "******NXSetUnpackGeometry: red [0x%x] green [0x%x] blue [0x%x].\n",
               (unsigned) req -> redMask, (unsigned) req -> greenMask, (unsigned) req -> blueMask);
  #endif

  UnlockDisplay(dpy);

  SyncHandle();

  return 1;
}

/*
 * Store a colormap table in proxy so it
 * can later be used to unpack images.
 */

int NXSetUnpackColormap(Display *dpy, unsigned int resource,
                            unsigned int entries, unsigned int *data)
{
  register xNXSetUnpackColormapReq *req;

  register unsigned char *dst_data;

  register int dst_data_length;

  #ifdef DEBUG

  int i;

  #endif

  LockDisplay(dpy);

  GetReq(NXSetUnpackColormap, req);

  req -> resource = resource;
  req -> entries  = entries;

  dst_data_length = entries << 2;

  req -> length += (dst_data_length >> 2);

  #ifdef TEST
  fprintf(stderr, "******NXSetUnpackColormap: Resource [%u] number of entries [%u] destination data size [%u].\n",
              resource, entries, dst_data_length);
  #endif

  if (entries > 0)
  {
    if ((dpy -> bufptr + dst_data_length) <= dpy -> bufmax)
    {
      dst_data = dpy -> bufptr;
    }
    else
    {
      if ((dst_data = _XAllocScratch(dpy, dst_data_length)) == NULL)
      {
        #ifdef PANIC
        fprintf(stderr, "******NXSetUnpackColormap: PANIC! Cannot allocate memory.\n");
        #endif

        UnGetReq(NXSetUnpackColormap);

        return 0;
      }
    }

    memcpy(dst_data, data, entries << 2);

    #ifdef DEBUG

    fprintf(stderr, "******NXSetUnpackColormap: Dumping colormap entries:\n");

    for (i = 0; i < entries; i++)
    {
      fprintf(stderr, "******NXSetUnpackColormap: [%d] -> [%p].\n",
                  i, (void *) *((int *) (dst_data + (i * 4))));
    }

    #endif

    if (dst_data == (unsigned char *) dpy -> bufptr)
    {
      dpy -> bufptr += dst_data_length;
    }
    else
    {
      _XSend(dpy, dst_data, dst_data_length);
    }
  }

  UnlockDisplay(dpy);

  SyncHandle();

  return 1;
}

/*
 * Store data of the alpha blending channel
 * that will be combined with the next image
 * to be unpacked.
 */

int NXSetUnpackAlpha(Display *dpy, unsigned int resource,
                         unsigned int entries, unsigned char *data)
{
  register xNXSetUnpackAlphaReq *req;

  register unsigned char *dst_data;

  register unsigned int dst_data_length;

  #ifdef DEBUG

  int i;

  #endif

  LockDisplay(dpy);

  GetReq(NXSetUnpackAlpha, req);

  req -> resource = resource;
  req -> entries  = entries;

  dst_data_length = ROUNDUP(entries, 4);

  req -> length += (dst_data_length >> 2);

  #ifdef TEST
  fprintf(stderr, "******NXSetUnpackAlpha: Resource [%u] number of entries [%u] destination data size [%u].\n",
              resource, entries, dst_data_length);
  #endif

  if (entries > 0)
  {
    if ((dpy -> bufptr + dst_data_length) <= dpy -> bufmax)
    {
      dst_data = dpy -> bufptr;
    }
    else
    {
      if ((dst_data = _XAllocScratch(dpy, dst_data_length)) == NULL)
      {
        #ifdef PANIC
        fprintf(stderr, "******NXSetUnpackAlpha: PANIC! Cannot allocate memory.\n");
        #endif

        UnGetReq(NXSetUnpackAlpha);

        return 0;
      }
    }

    memcpy(dst_data, data, entries);

    if (dst_data_length != entries)
    {
      memset(dst_data + entries, 0x00, dst_data_length - entries);
    }

    #ifdef DEBUG

    fprintf(stderr, "******NXSetUnpackAlpha: Dumping alpha channel data:\n");

    for (i = 0; i < dst_data_length; i++)
    {
      fprintf(stderr, "******NXSetUnpackAlpha: [%d] -> [%02x].\n",
                  i, (unsigned int) *(dst_data + i));
    }

    #endif

    if (dst_data == (unsigned char *) dpy -> bufptr)
    {
      dpy -> bufptr += dst_data_length;
    }
    else
    {
      _XSend(dpy, dst_data, dst_data_length);
    }
  }

  UnlockDisplay(dpy);

  SyncHandle();

  return 1;
}

/*
 * Free any geometry, colormap and alpha
 * channel data stored by the remote proxy
 * to unpack an image. Resource, as usual,
 * must be a value between 0 and 255.
 */

int NXFreeUnpack(Display *dpy, unsigned int resource)
{
  register xNXFreeUnpackReq *req;

  LockDisplay(dpy);

  GetReq(NXFreeUnpack, req);

  #ifdef TEST
  fprintf(stderr, "******NXFreeUnpack: Sending message opcode [%d] with resource [%u].\n",
              X_NXFreeUnpack, resource);
  #endif

  req -> resource = resource;

  UnlockDisplay(dpy);

  SyncHandle();

  return 1;
}

/*
 * Wrapper of XCreateImage(). Note that we use offset
 * field of XImage to store size of source image in
 * packed format. Note also that method is currently
 * not stored in the NXignored.
 */

NXPackedImage *NXCreatePackedImage(Display *dpy, Visual *visual, unsigned int method,
                                       unsigned int depth, int format, char *data,
                                           int data_length, unsigned int width,
                                               unsigned int height, int bitmap_pad,
                                                   int bytes_per_line)
{
  XImage* image;

  image = XCreateImage(dpy, visual, depth, format, 0, data,
                           width, height, bitmap_pad, bytes_per_line);

  if (image != NULL)
  {
    image -> xoffset = data_length;
  }

  return (NXPackedImage *) image;
}

/*
 * Wrapper of XDestroyImage().
 */

int NXDestroyPackedImage(NXPackedImage *image)
{
  return XDestroyImage((XImage *) image);
}


NXPackedImage *NXPackImage(Display *dpy, XImage *src_image, unsigned int method)
{
  XImage *dst_image;

  const ColorMask *mask;

  unsigned int dst_data_size;
  unsigned int dst_packed_data_size;

  unsigned int dst_bits_per_pixel;
  unsigned int dst_packed_bits_per_pixel;

  #ifdef TEST
  fprintf(stderr, "******NXPackImage: Going to pack a new image with method [%d].\n",
              method);
  #endif

  /*
   * Get mask out of method and check if
   * visual is supported by current color
   * reduction algorithm.
   */

  mask = GetColorMask(method);

  if (mask == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXPackImage: WARNING! No mask to apply for pack method [%d].\n",
                method);
    #endif

    return NULL;
  }
  else if (CanMaskImage(src_image, mask) == 0)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXPackImage: PANIC! Invalid source with format [%d] depth [%d] bits per pixel [%d].\n",
                src_image -> format, src_image -> depth, src_image -> bits_per_pixel);

    fprintf(stderr, "******NXPackImage: PANIC! Visual colormask is red 0x%lx green 0x%lx blue 0x%lx.\n",
                src_image -> red_mask, src_image -> green_mask, src_image -> blue_mask);
    #endif

    return NULL;
  }

  /*
   * Create a destination image from
   * source and apply the color mask.
   */

  if ((dst_image = (XImage *) Xcalloc(1, sizeof(XImage))) == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXPackImage: PANIC! Cannot allocate [%d] bytes for masked image.\n",
                sizeof(XImage));
    #endif

    return NULL;
  }

  *dst_image = *src_image;

  #ifdef TEST
  fprintf(stderr, "******NXPackImage: Source width [%d], bytes per line [%d] with depth [%d].\n",
              src_image -> width, src_image -> bytes_per_line, src_image -> depth);
  #endif

  dst_data_size = src_image -> bytes_per_line * src_image -> height;

  dst_image -> data = Xmalloc(dst_data_size);

  if (dst_image -> data == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXPackImage: PANIC! Cannot allocate [%d] bytes for masked image data.\n",
                dst_data_size);
    #endif

    XFree(dst_image);

    return NULL;
  }

  /*
   * If pixel resulting from mask needs
   * more bits than available, then just
   * clean the pad bits in image.
   */

  dst_bits_per_pixel = dst_image -> bits_per_pixel;
  dst_packed_bits_per_pixel = GetBitsPerPixel(method);

  #ifdef TEST
  fprintf(stderr, "******NXPackImage: Destination depth [%d], bits per pixel [%d], packed bits per pixel [%d].\n",
             dst_image -> depth, dst_bits_per_pixel, dst_packed_bits_per_pixel);
  #endif

  if (dst_packed_bits_per_pixel > dst_bits_per_pixel ||
          ShouldMaskImage(src_image, mask) == 0)
  {
    /*
     * Should use the same data for src and dst to avoid memcpy.
     */

    if (CleanImage(src_image, dst_image) <= 0)
    {
      #ifdef PANIC
      fprintf(stderr, "******NXPackImage: PANIC! Failed to clean the image.\n");
      #endif

      XFree(dst_image -> data);

      XFree(dst_image);

      return NULL;
    }
  }
  else if (MaskImage(mask, src_image, dst_image) <= 0)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXPackImage: PANIC! Failed to apply the color mask.\n");
    #endif

    XFree(dst_image -> data);

    XFree(dst_image);

    return NULL;
  }

  /*
   * Let's pack the same pixels in fewer bytes.
   * Note that we save a new memory allocation
   * by using the same image as source and des-
   * tination. This means that PackImage() must
   * be able to handle ovelapping areas.
   */

  #ifdef TEST
  fprintf(stderr, "******NXPackImage: Plain bits per pixel [%d], data size [%d].\n",
              dst_bits_per_pixel, dst_data_size);
  #endif

  dst_packed_data_size = dst_data_size * dst_packed_bits_per_pixel /
                              dst_bits_per_pixel;

  #ifdef TEST
  fprintf(stderr, "******NXPackImage: Packed bits per pixel [%d], data size [%d].\n",
              dst_packed_bits_per_pixel, dst_packed_data_size);
  #endif

  if (PackImage(method, dst_data_size, dst_image,
                    dst_packed_data_size, dst_image) <= 0)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXPackImage: PANIC! Failed to pack image from [%d] to [%d] bits per pixel.\n",
                dst_bits_per_pixel, dst_packed_bits_per_pixel);
    #endif

    XFree(dst_image -> data);

    XFree(dst_image);

    return NULL;
  }

  /*
   * Save data size in xoffset field
   * to comply with NX packed images.
   */

  dst_image -> xoffset = dst_packed_data_size;

  return dst_image;
}





/*
 * NXInPlacePackImage creates a NXPackedImage from
 * an XImage, sharing the same data buffer. Do not
 * call NXDestroyPackedImage to free it, but call
 * XFree(NXPackedImage *) instead.
 */

XImage *NXInPlacePackImage(Display *dpy, XImage *src_image, unsigned int method)
{
  XImage *dst_image;

  const ColorMask *mask;

  unsigned int dst_data_size;
  unsigned int dst_packed_data_size;

  unsigned int dst_bits_per_pixel;
  unsigned int dst_packed_bits_per_pixel;

  #ifdef TEST
  fprintf(stderr, "******NXInPlacePackImage: Going to pack a new image with method [%d].\n",
              method);
  #endif

  /*
   * Get mask out of method and check if
   * visual is supported by current color
   * reduction algorithm.
   */

  mask = GetColorMask(method);

  if (mask == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXInPlacePackImage: WARNING! No mask to apply for pack method [%d].\n",
                method);
    #endif

    return NULL;
  }
  else if (CanMaskImage(src_image, mask) == 0)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXInPlacePackImage: PANIC! Invalid source with format [%d] depth [%d] bits per pixel [%d].\n",
                src_image -> format, src_image -> depth, src_image -> bits_per_pixel);

    fprintf(stderr, "******NXInPlacePackImage: PANIC! Visual colormask is red 0x%lx green 0x%lx blue 0x%lx.\n",
                src_image -> red_mask, src_image -> green_mask, src_image -> blue_mask);
    #endif
    return NULL;
  }

  /*
   * Create a destination image from
   * source and apply the color mask.
   */

  if ((dst_image = (XImage *) Xcalloc(1, sizeof(XImage))) == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXInPlacePackImage: PANIC! Cannot allocate [%d] bytes for masked image.\n",
                sizeof(XImage));
    #endif

    return NULL;
  }

  *dst_image = *src_image;

  #ifdef TEST
  fprintf(stderr, "******NXInPlacePackImage: Source width [%d], bytes per line [%d] with depth [%d].\n",
              src_image -> width, src_image -> bytes_per_line, src_image -> depth);
  #endif

  dst_data_size = src_image -> bytes_per_line * src_image -> height;

  dst_image -> data = src_image -> data;

  /*
   * If pixel resulting from mask needs
   * more bits than available, then just
   * clean the pad bits in image.
   */

  dst_bits_per_pixel = dst_image -> bits_per_pixel;
  dst_packed_bits_per_pixel = GetBitsPerPixel(method);

  #ifdef TEST
  fprintf(stderr, "******NXInPlacePackImage: Destination depth [%d], bits per pixel [%d], packed bits per pixel [%d].\n",
              dst_image -> depth, dst_bits_per_pixel, dst_packed_bits_per_pixel);
  #endif

  if (dst_packed_bits_per_pixel > dst_bits_per_pixel ||
          ShouldMaskImage(src_image, mask) == 0)
  {
    #ifdef TEST
    fprintf(stderr, "******NXInPlacePackImage: Just clean image packed_bits_per_pixel[%d], bits_per_pixel[%d].\n",
                dst_packed_bits_per_pixel, dst_bits_per_pixel);
    #endif

    if (NXCleanInPlaceImage(dst_image) <= 0)
    {
      #ifdef PANIC
      fprintf(stderr, "******NXInPlacePackImage: PANIC! Failed to clean the image.\n");
      #endif

      XFree(dst_image);

      return NULL;
    }
  }
  else if (MaskInPlaceImage(mask, dst_image) <= 0)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXInPlacePackImage: PANIC! Failed to apply the color mask.\n");
    #endif

    XFree(dst_image);

    return NULL;
  }

  /*
   * Let's pack the same pixels in fewer bytes.
   * Note that we save a new memory allocation
   * by using the same image as source and des-
   * tination. This means that PackImage() must
   * be able to handle ovelapping areas.
   */

  #ifdef TEST
  fprintf(stderr, "******NXInPlacePackImage: Plain bits per pixel [%d], data size [%d].\n",
             dst_bits_per_pixel, dst_data_size);
  #endif

  dst_packed_data_size = dst_data_size * dst_packed_bits_per_pixel /
    dst_bits_per_pixel;

  #ifdef TEST
  fprintf(stderr, "******NXInPlacePackImage: Packed bits per pixel [%d], data size [%d].\n",
             dst_packed_bits_per_pixel, dst_packed_data_size);
  #endif

  /*
   * Save data size in xoffset field
   * to comply with NX packed images.
   */

  dst_image -> xoffset = dst_packed_data_size;

  return dst_image;
}


int NXPutPackedImage(Display *dpy, unsigned int resource, Drawable drawable,
                         void *gc, NXPackedImage *image, unsigned int method,
                             unsigned int depth, int src_x, int src_y, int dst_x,
                                 int dst_y, unsigned int width, unsigned int height)
{
  register xNXPutPackedImageReq *req;

  register unsigned char *dst_data;

  register unsigned int src_data_length;
  register unsigned int dst_data_length;

  LockDisplay(dpy);

  FlushGC(dpy, (GC) gc);

  GetReq(NXPutPackedImage, req);

  req -> resource = resource;
  req -> drawable = drawable;
  req -> gc = ((GC) gc) -> gid;

  /*
   * There is no leftPad field in request. We only
   * support a leftPad of 0. Anyway, X imposes a
   * leftPad of 0 in case of ZPixmap format.
   */

  req -> format = image -> format;

  /*
   * Source depth, as well as width
   * and height, are taken from the
   * image structure.
   */

  req -> srcDepth = image -> depth;

  req -> srcX = src_x;
  req -> srcY = src_y;

  req -> srcWidth  = image -> width;
  req -> srcHeight = image -> height;

  /*
   * Destination depth is provided
   * by caller.
   */

  req -> dstDepth = depth;

  req -> dstX = dst_x;
  req -> dstY = dst_y;

  req -> dstWidth  = width;
  req -> dstHeight = height;

  /*
   * Source data length is the size of image in packed format,
   * as stored in xoffset field of XImage. Destination data
   * size is calculated according to bytes per line of target
   * image, so user must provide the right depth at the time
   * XImage structure is created.
   */

  req -> srcLength = image -> xoffset;

  if (method == PACK_RDP_TEXT)
  {
    req -> dstLength = _NXRDPTextUnpackSize(width);

    #ifdef TEST
    fprintf(stderr, "******NXPutPackedImage: Using RDP text width [%d] packed length [%ld] unpacked length [%ld].\n",
                req -> dstWidth, req -> srcLength, req -> dstLength);
    #endif
  }
  else if (image -> width == (int)width &&
               image -> height == (int)height)
  {
    req -> dstLength = image -> bytes_per_line * image -> height;
  }
  else if (image -> format == ZPixmap)
  {
    req -> dstLength = ROUNDUP((image -> bits_per_pixel * width),
                                    image -> bitmap_pad) * height >> 3;
  }
  else
  {
    req -> dstLength = ROUNDUP(width, image -> bitmap_pad) * height >> 3;
  }

  req -> method = method;

  #ifdef TEST
  fprintf(stderr, "******NXPutPackedImage: Source image depth [%d], destination depth [%d], method [%d].\n",
              req -> srcDepth, req -> dstDepth, req -> method);
  #endif

  src_data_length = image -> xoffset;

  dst_data_length = ROUNDUP(src_data_length, 4);

  #ifdef TEST
  fprintf(stderr, "******NXPutPackedImage: Source data length [%d], request data length [%d].\n",
              src_data_length, dst_data_length);
  #endif

  req -> length += (dst_data_length >> 2);

  if ((dpy -> bufptr + dst_data_length) <= dpy -> bufmax)
  {
    dst_data = dpy -> bufptr;
  }
  else
  {
    if ((dst_data = _XAllocScratch(dpy, dst_data_length)) == NULL)
    {
      #ifdef PANIC
      fprintf(stderr, "******NXPutPackedImage: PANIC! Cannot allocate memory.\n");
      #endif

      UnGetReq(NXPutPackedImage);

      return 0;
    }
  }

  memcpy(dst_data, image -> data, src_data_length);

  if (dst_data_length > src_data_length)
  {
    memset(dst_data + src_data_length, 0, dst_data_length - src_data_length);
  }

  if (method == PACK_RDP_TEXT)
  {
    #ifdef TEST
    fprintf(stderr, "******NXPutPackedImage: Updating current sequence counter from [%ld]",
                dpy -> request & 0xffff);
    #endif

    dpy -> request += (width * 2);

    #ifdef TEST
    fprintf(stderr, " to [%ld].\n",
                dpy -> request & 0xffff);
    #endif
  }

  if (dst_data == (unsigned char *) dpy -> bufptr)
  {
    dpy -> bufptr += dst_data_length;
  }
  else
  {
    _XSend(dpy, dst_data, dst_data_length);
  }

  UnlockDisplay(dpy);

  SyncHandle();

  return 1;
}

int NXAllocColors(Display *dpy, Colormap colormap, unsigned int entries,
                      XColor screens_in_out[], Bool results_in_out[])
{
  Status result = 0;
  xAllocColorReply rep;
  register xAllocColorReq *req;

  Bool alloc_error = False;

  register unsigned int i;

  LockDisplay(dpy);

  for (i = 0; i < entries; i++)
  {
    GetReq(AllocColor, req);

    req -> cmap  = colormap;

    req -> red   = screens_in_out[i].red;
    req -> green = screens_in_out[i].green;
    req -> blue  = screens_in_out[i].blue;
  }

  for (i = 0; i < entries; i++)
  {
    result = _XReply(dpy, (xReply *) &rep, 0, xTrue);

    if (result)
    {
      screens_in_out[i].pixel = rep.pixel;

      screens_in_out[i].red   = rep.red;
      screens_in_out[i].green = rep.green;
      screens_in_out[i].blue  = rep.blue;
      
      results_in_out[i] = True;
    } 
    else 
    {
      results_in_out[i] = False;

      alloc_error = True;
    }
  }

  UnlockDisplay(dpy);

  SyncHandle();

  return (alloc_error == False);
}

int NXEncodeTight(Display *dpy, Drawable drawable, Visual *visual, void *gc,
                      XImage *image, unsigned int method, int src_x, int src_y,
                          int dst_x, int dst_y, unsigned int w, unsigned int h)
{
  unsigned int nMaxRows;
  int result;
  CARD32  colorValue;
  unsigned int     dx, dy, dw, dh;
  int     x, y, x_best, y_best, w_best, h_best;
  XImage* dstImage = NULL;
  int     maskMethod;

  const ColorMask *mask;

  /*
   * First increase the depth of recursivity
   */

  IncreaseRecDepth();

  #ifdef DEBUG
  fprintf(stderr, "******NXEncodeTight: src_x[%d] src_y[%d] dst_x[%d] dst_y[%d] w[%d] h[%d] imgDepth[%d] imgBpp[%d]\n",
              src_x, src_y, dst_x, dst_y, w, h, image -> depth, image -> bits_per_pixel);
  #endif

  /*
   * Apply the required mask to the image
   */

  if (GetEncodeTightInitStatus() == 0)
  {
    /*
     * If the image has 1 bitPerPixel or
     * the format is not ZPixmap then send
     * it as a simple XPutImage
     */

    if (image -> bits_per_pixel != 8 &&
            image -> bits_per_pixel != 16 &&
                image -> bits_per_pixel != 24 &&
                    image -> bits_per_pixel != 32)
    {
      #ifdef TEST
      fprintf(stderr, "******NXEncodeTight: bitsPerPixel[%d] not supported. Sending as XPutImage\n",
                  image -> bits_per_pixel);
      #endif

      XPutImage(dpy, drawable, gc, image, src_x, src_y, dst_x, dst_y, w, h);

      NXEncodeTightUninit();

      return 1;
    }

    if (image -> format != ZPixmap)
    {
      #ifdef TEST
      fprintf(stderr, "******NXEncodeTight: Format is not ZPixmap. Sending as XPutImage\n");
      #endif

      XPutImage(dpy, drawable, gc, image, src_x, src_y, dst_x, dst_y, w, h);

      NXEncodeTightUninit();

      return 1;
    }

    /*
     * Check if the method is supported
     */

    if (method < PACK_TIGHT_8_COLORS ||
            method > PACK_TIGHT_16M_COLORS)
    {
      #ifdef PANIC
      fprintf(stderr, "******NXEncodeTight: PANIC! Unsuported method [%d] for tight encoding.\n",
                  method);
      #endif

      NXEncodeTightUninit();

      return 0;
    }

    /*
     * Choose the correct mask method
     */

    switch(method)
    {
      case PACK_TIGHT_8_COLORS:
      {
        maskMethod = MASK_8_COLORS;
        break;
      }
      case PACK_TIGHT_64_COLORS:
      {
        maskMethod = MASK_64_COLORS;
        break;
      }
      case PACK_TIGHT_256_COLORS:
      {
        maskMethod = MASK_256_COLORS;
        break;
      }
      case PACK_TIGHT_512_COLORS:
      {
        maskMethod = MASK_512_COLORS;
        break;
      }
      case PACK_TIGHT_4K_COLORS:
      {
        maskMethod = MASK_4K_COLORS;
        break;
      }
      case PACK_TIGHT_32K_COLORS:
      {
        maskMethod = MASK_32K_COLORS;
        break;
      }
      case PACK_TIGHT_64K_COLORS:
      {
        maskMethod = MASK_64K_COLORS;
        break;
      }
      case PACK_TIGHT_256K_COLORS:
      {
        maskMethod = MASK_256K_COLORS;
        break;
      }
      case PACK_TIGHT_2M_COLORS:
      {
        maskMethod = MASK_2M_COLORS;
        break;
      }
      case PACK_TIGHT_16M_COLORS:
      {
        maskMethod = MASK_16M_COLORS;
        break;
      }
      default:
      {
        #ifdef PANIC
        fprintf(stderr, "******NXEncodeTight: PANIC! Cannot find mask method for pack method [%d]\n",
                   method);
        #endif

        NXEncodeTightUninit();

        return 0;
      }
    }

    #ifdef TEST
    fprintf(stderr, "******NXEncodeTight: packMethod[%d] => maskMethod[%d]\n",
                method, maskMethod);
    #endif

    /*
     * Get mask out of method and check if
     * visual is supported by current color
     * reduction algorithm.
     */

    mask = GetColorMask(maskMethod);

    if (mask == NULL)
    {
      #ifdef PANIC
      fprintf(stderr, "******NXEncodeTight: PANIC! No mask to apply for pack method [%d].\n",
                  method);
      #endif

      NXEncodeTightUninit();

      return 0;
    }
    else if (CanMaskImage(image, mask) == 0)
    {
      #ifdef PANIC
      fprintf(stderr, "******NXEncodeTight: PANIC! Invalid source with format [%d] depth [%d] bits per pixel [%d].\n",
                  image -> format, image -> depth, image -> bits_per_pixel);

      fprintf(stderr, "******NXEncodeTight: PANIC! Visual colormask is red 0x%lx green 0x%lx blue 0x%lx.\n",
                  image -> red_mask, image -> green_mask, image -> blue_mask);
      #endif

      NXEncodeTightUninit();

      return 0;
    }

    /*
     * Create a destination image from
     * source and apply the color mask.
     */

    if ((dstImage = (XImage *) Xcalloc(1, sizeof(XImage))) == NULL)
    {
      #ifdef PANIC
      fprintf(stderr, "******NXEncodeTight: PANIC! Cannot allocate [%d] bytes for masked image.\n",
                sizeof(XImage));
      #endif

      NXEncodeTightUninit();

      return 0;
    }

    *dstImage = *image;
    dstImage -> data = Xmalloc(image -> bytes_per_line * image -> height);

    if (dstImage -> data == NULL)
    {
      #ifdef PANIC
      fprintf(stderr, "******NXEncodeTight: PANIC! Cannot allocate [%d] bytes for masked image data.\n",
                  image -> bytes_per_line * image -> height);
      #endif

      XFree(dstImage);

      NXEncodeTightUninit();

      return 0;
    }

    /*
     * Apply the mask. Note that call to ShouldMaskImage returns 0 in
     * case of MASK_256_COLORS and MASK_64K_COLORS, resulting in image
     * not being masked. Otherwise, before applying tight compression,
     * colors are reduced by the packing mechanism.
     */

    if (ShouldMaskImage(image, mask) == 0)
    {
      #ifdef TEST
      fprintf(stderr, "******NXEncodeTight: the image will not be masked\n");
      #endif

      memcpy(dstImage -> data, image -> data, image -> bytes_per_line * image -> height);
    }
    else
    {
      #ifdef TEST
      fprintf(stderr, "******NXEncodeTight: maskMethod Bpp[%d] image Bpp[%d]\n",
                  GetBitsPerPixel(maskMethod), image -> bits_per_pixel);
      #endif

      if (GetBitsPerPixel(maskMethod) < image -> bits_per_pixel)
      {
        if (MaskImage(mask, image, dstImage) <= 0)
        {
          #ifdef PANIC
          fprintf(stderr, "******NXEncodeTight: PANIC! Failed to apply the color mask.\n");
          #endif

          XFree(dstImage -> data);

          XFree(dstImage);

          NXEncodeTightUninit();

          return 0;
        }
      }
      else if (GetBitsPerPixel(maskMethod) == image -> bits_per_pixel)
      {
        #ifdef TEST
        fprintf(stderr, "******NXEncodeTight: Going to MaskInPlace the image\n");
        #endif

        memcpy(dstImage -> data, image -> data, image -> bytes_per_line * image -> height);

        if (MaskInPlaceImage(mask, dstImage) <= 0)
        {
          #ifdef PANIC
          fprintf(stderr, "******NXEncodeTight: PANIC! Failed to apply the color mask in place.\n");
          #endif

          XFree(dstImage -> data);

          XFree(dstImage);

          NXEncodeTightUninit();

          return 0;
        }
      }
    }

    /*
     * Initialise the encoding (only if needed).
     */

    NXEncodeTightInit(dpy, drawable, visual, gc, dstImage, method, maskMethod);
  }

  /*
   * Start encoding.
   */

  /*
   * Place ourselves at the right top corner
   * of the subimage inside the data part
   * of the image.
   */

  x = src_x;
  y = src_y;

  /*
   * Calculate maximum number of rows in one
   * non-solid rectangle.
   */

  nMaxRows = GetEncodeTightMaxRows(w);

  for (dy = y; dy < y + h; dy += MAX_SPLIT_TILE_SIZE)
  {
    /*
     * If a rectangle becomes too large, send
     * its upper part now.
     */

    if (dy - y >= nMaxRows)
    {
      if (!SendRectSimple(x, y, dst_x, dst_y, w, nMaxRows))
      {
        DecreaseRecDepth();

        return 0;
      }
    }

    dh = (dy + MAX_SPLIT_TILE_SIZE <= y + h) ? MAX_SPLIT_TILE_SIZE : (y + h - dy);

    for (dx = x; dx < x + w; dx += MAX_SPLIT_TILE_SIZE)
    {
      dw = (dx + MAX_SPLIT_TILE_SIZE <= x + w) ? MAX_SPLIT_TILE_SIZE : (x + w - dx);

      #ifdef DEBUG
      fprintf(stderr, "******NXEncodeTight: About to check solid tile x[%d] y[%d] w[%d] h[%d]\n",
                  dx, dy, dw, dh);
      #endif

      if (CheckSolidTile(dx, dy, dw, dh, &colorValue, 0))
      {
        /*
         * Get dimensions of solid-color area.
         */

        FindBestSolidArea(dx, dy, w - (dx - x), h - (dy - y), colorValue, &w_best, &h_best);

        /*
         * Make sure a solid rectangle is large enough
         * (or the whole rectangle is of the same color).
         */

        if (w_best * h_best != (int)(w * h) && w_best * h_best < MIN_SOLID_SUBRECT_SIZE)
        {
          continue;
        }

        /*
         * Try to extend solid rectangle to maximum size.
         */

        x_best = dx; y_best = dy;

        ExtendSolidArea(x, y, w, h, colorValue, &x_best, &y_best, &w_best, &h_best);

        /*
         * Send rectangles at top and left to solid-color
         * area.
         */

        if ( y_best != y && !SendRectSimple(x, y, dst_x, dst_y, w, y_best-y) )
        {
          DecreaseRecDepth();

          return 0;
        }

        if ( x_best != x && !NXEncodeTight(dpy, drawable, visual, gc, image, method, x, y_best,
                                               dst_x, dst_y + (y_best - y), x_best-x, h_best) )
        {
          DecreaseRecDepth();

          return 0;
        }

        /*
         * Send solid-color rectangle.
         */

        if (!SendSolidRect(x_best, y_best, dst_x + (x_best - x), dst_y + (y_best - y), w_best, h_best))
        {
          DecreaseRecDepth();

          return 0;
        }

        /*
         * Send remaining rectangles (at right and bottom).
         */

        if (x_best + w_best != x + (int)w &&
                 !NXEncodeTight(dpy, drawable, visual, gc, image, method, x_best + w_best, y_best,
                                    dst_x + (x_best - x) + w_best, dst_y + (y_best - y),
                                        w - (x_best-x) - w_best, h_best) )
        {
          DecreaseRecDepth();

          return 0;
        }

        if (y_best + h_best != y + (int)h &&
                !NXEncodeTight(dpy, drawable, visual, gc, image, method, x, y_best + h_best,
                                   dst_x, dst_y + (y_best + h_best - y), w, h - (y_best-y) - h_best) )
        {
          DecreaseRecDepth();

          return 0;
        }

        /*
         * Return after all recursive calls are done.
         */

        DecreaseRecDepth();

        return 1;
      }

    }

  }

  /*
   * No suitable solid-color rectangles found.
   */

  #ifdef DEBUG
  fprintf(stderr, "******NXEncodeTight: no suitable solid color area found\n");
  #endif

  result = SendRectSimple(x, y, dst_x, dst_y, w, h);

  DecreaseRecDepth();

  return result;
}

/*
 * Interface should be the same for all 'encode'
 * functions, but it is not the case yet...
 */

XImage *NXEncodeJpeg(XImage *src_image, unsigned int method, unsigned int quality)
{
  XImage*          dst_image =NULL;
  int              jpg_size;
  char*            jpg_data;

  /*
   * Check if the bpp of the image is valid
   * for jpeg compression.
   */

  if (src_image -> bits_per_pixel == 1 ||
          src_image -> bits_per_pixel == 8)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXEncodeJpeg: PANIC! Invalid bpp for Jpeg compression [%d]\n.",
                src_image -> bits_per_pixel);
    #endif

    return NULL;
  }

  /*
   * Create a destination image from
   * source and call the compressor.
   */

  if ((dst_image = (XImage *) Xcalloc(1, sizeof(XImage))) == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXEncodeJpeg: PANIC! Cannot allocate [%d] bytes for masked image.\n",
              sizeof(XImage));
    #endif

    return NULL;
  }

  *dst_image = *src_image;

  jpg_data = JpegCompressData(src_image, quality, &jpg_size);

  if (jpg_data == NULL || jpg_size == 0)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXEncodeJpeg: PANIC! Jpeg compression failed, bailing out!\n");
    #endif

    XFree(dst_image);

    return NULL;
  }

  /*
   * Update data of dst_image to contain the
   * compressed JPEG image.
   */

  dst_image -> data = jpg_data;
  dst_image -> xoffset = jpg_size;

  return dst_image;
}

XImage *NXEncodePng(XImage *src_image, unsigned int method, NXColorTable* color_table)
{
  XImage*          dst_image = NULL;
  int              png_size;
  char*            png_data;

  /*
   * Check if the bpp of the image is valid
   * for png compression.
   */

  if (src_image -> bits_per_pixel == 1 ||
          src_image -> bits_per_pixel == 8)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXEncodePng: PANIC! Invalid bpp for Png compression [%d].\n",
                src_image -> bits_per_pixel);
    #endif

    return NULL;
  }

  if ((dst_image = (XImage *) Xcalloc(1, sizeof(XImage))) == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXEncodePng: PANIC! Cannot allocate [%d] bytes for masked image.\n",
                sizeof(XImage));
    #endif

    return NULL;
  }

  *dst_image = *src_image;

  png_data = PngCompressData(dst_image, &png_size, color_table);

  if (png_data == NULL || png_size == 0)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXEncodePng: PANIC! Png compression failed, bailing out!\n");
    #endif

    return NULL;
  }

  /*
   * Update data of dst_image to contain the
   * compressed PNG image.
   */

  dst_image -> data = png_data;
  dst_image -> xoffset = png_size;

  return dst_image;
}

int NXCreateColorTable(XImage *src_image, NXColorTable *color_table, int nb_max)
{
  int x,y,t,p;
  long pixel;

  /*
   * TODO: Think at a more intelligent way to
   *       estimate the number of colors.
   */

  memset (color_table, 0, nb_max * sizeof(NXColorTable));

  for (x = 0, p = 0; x < src_image -> width; x++)
  {
    for (y = 0; y < src_image -> height; y++)
    {
      pixel  = XGetPixel(src_image, x, y);

      for (t = 0; t < nb_max; t++)
      {
        if ( color_table[t].found == 0)
        {
          color_table[t].pixel =  pixel;
          color_table[t].found =  1;
          p++;
          break;
        }
        else if ((color_table[t].pixel) == pixel)
        {
          break;
        }
      }

      if (p == nb_max)
      {
        return nb_max + 1;
      }
    }
  }

  return p;
}

/*
 * TODO: Create one generic function for
 *       NXDynamicSelectPackMethod to handle
 *       all of PNG, JPEG and simple X bitmap
 *       PutImages.
 */

XImage *NXDynamicPngPack(XImage *src_image, unsigned int *method, unsigned int quality)
{
  NXColorTable color_table[NB_COLOR_MAX];
  int nb_colors = 0;

  if (NXCreateColorTable(src_image, color_table, NB_COLOR_MAX) > NB_COLOR_MAX)
  {
    #ifdef DEBUG
    fprintf(stderr, "******NXDynamicPngPack: Too many colors choosing XPutImage.\n");
    #endif

    NXMaskImage(src_image, (*method));

    return NULL;
  }
  else
  {
    NXMaskImage(src_image, (*method));

    if ( (nb_colors = NXCreateColorTable(src_image, color_table, NB_COLOR_MAX)) <= NB_COLOR_MAX)
    {
      #ifdef DEBUG
      fprintf(stderr, "******NXDynamicPngPack: nb_colors=%d encoding PNG after mask=%d\n",
              nb_colors, (*method));
      #endif

      qsort(&color_table, nb_colors, sizeof(NXColorTable),
                (int (*)(const void *, const void *)) PngCompareColorTable);

      return NXEncodePng(src_image, (*method), color_table);
    }
    else
    {
      #ifdef PANIC
      fprintf(stderr, "******NXDynamicPngPack: PANIC! Inconsistency found in color_table in context [A].\n");
      #endif

      NXMaskImage(src_image, (*method));
      return NULL;
    }
  }
}

XImage *NXDynamicSelectPackMethod(XImage *src_image, unsigned int *method, unsigned int quality)
{
  NXColorTable color_table[NB_COLOR_MAX];
  int nb_colors = 0;

  if (NXCreateColorTable(src_image, color_table, NB_COLOR_MAX) > NB_COLOR_MAX)
  {
    #ifdef DEBUG
    fprintf(stderr, "******NXDynamicSelectPackMethod: Too many colors encoding JPEG.\n");
    #endif

    (*method) = PACK_JPEG_16M_COLORS;

    return NXEncodeJpeg(src_image, (*method), quality);
  }
  else
  {

    if ((nb_colors = NXCreateColorTable(src_image, color_table, NB_COLOR_MAX)) <= NB_COLOR_MAX)
    {
      #ifdef DEBUG
      fprintf(stderr, "******NXDynamicSelectPackMethod: nb_colors is %d encoding PNG after mask = %d.\n",
              nb_colors, (*method));
      #endif

      qsort(&color_table, nb_colors, sizeof(NXColorTable),
                (int (*)(const void *, const void *)) PngCompareColorTable);

      /*
       * FIXME: Try to use the corresponding method
       * in the PNG range. This is dirty and would
       * require a specific translation function. 
       */

      (*method) -= 10;

      NXMaskImage(src_image, (*method));

      return NXEncodePng(src_image, (*method), color_table);
    }
    else
    {
      #ifdef PANIC
      fprintf(stderr, "******NXDynamicSelectPackMethod: PANIC! Inconsistency found in color table.\n");
      #endif

      (*method) -= 10;

      return NULL;
    }
  }
}

NXPackedImage *NXEncodeRDPText(Display *dpy, unsigned int bgcolor, unsigned int fgcolor,
                                   int fillStyle, xNXRDPGlyph *glyphs, int elements)
{
  NXPackedImage *image = NULL;

  xNXRDPText *rdpTextData;

  rdpTextData = Xmalloc(sizeof(xNXRDPText) + (sizeof(xNXRDPGlyph) * elements));

  if (rdpTextData)
  {
    rdpTextData -> bgColor   = bgcolor;
    rdpTextData -> fgColor   = fgcolor;
    rdpTextData -> fillStyle = fillStyle;
    rdpTextData -> elements  = elements;

    memcpy(((unsigned char *) rdpTextData + sizeof(xNXRDPText)),
               glyphs, (sizeof(xNXRDPGlyph) * elements));

    image = NXCreatePackedImage(dpy, NULL, PACK_RDP_TEXT, 1, ZPixmap,
                                    (unsigned char *) rdpTextData,
                                        (sizeof(xNXRDPText) + (sizeof(xNXRDPGlyph) * elements)),
                                             elements, 1, BitmapPad(dpy), 0);
    if (image == NULL)
    {
      #ifdef PANIC
      fprintf(stderr, "******NXEncodeRDPText: PANIC! Failed to create NX image for  RDP packed text.\n");
      #endif

      XFree(rdpTextData);
    }
  }
  else
  {
    #ifdef PANIC
    fprintf(stderr, "******NXEncodeRDPText: PANIC! Failed to allocate memory for RDP packed text.\n");
    #endif
  }

  return image;
}


void NXMaskImage(XImage *image, unsigned int method)
{
  unsigned int maskMethod;

  const ColorMask *mask;

  /*
   * Choose the correct mask method
   */

  switch(method)
  {
    case PACK_JPEG_8_COLORS:
    case PACK_TIGHT_8_COLORS:
    case PACK_PNG_8_COLORS:
    case PACK_PNG_JPEG_8_COLORS:
    {
      maskMethod = MASK_8_COLORS;
      #ifdef DEBUG
      fprintf(stderr, "******NXMaskImage: Method is MASK_8_COLORS\n");
      #endif
      break;
    }
    case PACK_JPEG_64_COLORS:
    case PACK_TIGHT_64_COLORS:
    case PACK_PNG_64_COLORS:
    case PACK_PNG_JPEG_64_COLORS:
    {
      maskMethod = MASK_64_COLORS;
      #ifdef DEBUG
      fprintf(stderr, "******NXMaskImage: Method is MASK_64K_COLORS\n");
      #endif
      break;
    }
    case PACK_JPEG_256_COLORS:
    case PACK_TIGHT_256_COLORS:
    case PACK_PNG_256_COLORS:
    case PACK_PNG_JPEG_256_COLORS:
    {
      maskMethod = MASK_256_COLORS;
      #ifdef DEBUG
      fprintf(stderr, "******NXMaskImage: Method is MASK_256_COLORS\n");
      #endif
      break;
    }
    case PACK_JPEG_512_COLORS:
    case PACK_TIGHT_512_COLORS:
    case PACK_PNG_512_COLORS:
    case PACK_PNG_JPEG_512_COLORS:
    {
      maskMethod = MASK_512_COLORS;
      #ifdef DEBUG
      fprintf(stderr, "******NXMaskImage: Method is MASK_512K_COLORS\n");
      #endif
      break;
    }
    case PACK_JPEG_4K_COLORS:
    case PACK_TIGHT_4K_COLORS:
    case PACK_PNG_4K_COLORS:
    case PACK_PNG_JPEG_4K_COLORS:
    {
      maskMethod = MASK_4K_COLORS;
      #ifdef DEBUG
      fprintf(stderr, "******NXMaskImage: Method is MASK_4K_COLORS\n");
      #endif
      break;
    }
    case PACK_JPEG_32K_COLORS:
    case PACK_TIGHT_32K_COLORS:
    case PACK_PNG_32K_COLORS:
    case PACK_PNG_JPEG_32K_COLORS:
    {
      maskMethod = MASK_32K_COLORS;
      #ifdef DEBUG
      fprintf(stderr, "******NXMaskImage: Method is MASK_32K_COLORS\n");
      #endif
      break;
    }
    case PACK_JPEG_64K_COLORS:
    case PACK_TIGHT_64K_COLORS:
    case PACK_PNG_64K_COLORS:
    case PACK_PNG_JPEG_64K_COLORS:
    {
      maskMethod = MASK_64K_COLORS;
      #ifdef DEBUG
      fprintf(stderr, "******NXMaskImage: Method is MASK_64K_COLORS\n");
      #endif
      break;
    }
    case PACK_JPEG_256K_COLORS:
    case PACK_TIGHT_256K_COLORS:
    case PACK_PNG_256K_COLORS:
    case PACK_PNG_JPEG_256K_COLORS:
    {
      maskMethod = MASK_256K_COLORS;
      #ifdef DEBUG
      fprintf(stderr, "******NXMaskImage: Method is MASK_256K_COLORS\n");
      #endif
      break;
    }
    case PACK_JPEG_2M_COLORS:
    case PACK_TIGHT_2M_COLORS:
    case PACK_PNG_2M_COLORS:
    case PACK_PNG_JPEG_2M_COLORS:
    {
      maskMethod = MASK_2M_COLORS;
      #ifdef DEBUG
      fprintf(stderr, "******NXMaskImage: Method is MASK_2M_COLORS\n");
      #endif
      break;
    }
    case PACK_JPEG_16M_COLORS:
    case PACK_TIGHT_16M_COLORS:
    case PACK_PNG_16M_COLORS:
    case PACK_PNG_JPEG_16M_COLORS:
    {
      maskMethod = MASK_16M_COLORS;
      #ifdef DEBUG
      fprintf(stderr, "******NXMaskImage: Method is MASK_16M_COLORS\n");
      #endif
      break;
    }
    default:
    {
      #ifdef PANIC
      fprintf(stderr, "******NXMaskImage: PANIC! Cannot find mask method for pack method [%d]\n",
                  method);
      #endif
      return;
    }
  }

  #ifdef TEST
  fprintf(stderr, "******NXMaskImage: packMethod[%d] => maskMethod[%d]\n",
              method, maskMethod);
  #endif

  /*
   * Get mask out of method and check if
   * visual is supported by current color
   * reduction algorithm.
   */

  mask = GetColorMask(maskMethod);

  if (mask == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXMaskImage: PANIC! No mask to apply for pack method [%d].\n",
                method);
    #endif

    return;
  }
  else if (CanMaskImage(image, mask) == 0)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXMaskImage: PANIC! Invalid source with format [%d] depth [%d] bits per pixel [%d].\n",
                image -> format, image -> depth, image -> bits_per_pixel);

    fprintf(stderr, "******NXMaskImage: PANIC! Visual colormask is red 0x%lx green 0x%lx blue 0x%lx.\n",
                image -> red_mask, image -> green_mask, image -> blue_mask);
    #endif

    return;
  }

  /*
   * Calling ShouldMaskImage you get 0 in the case
   * of MASK_256_COLORS and MASK_64K_COLORS, which
   * means that the image would not be masked.
   *
   * When not using tight or jpeg, the colors are
   * reduced by the packing mechanism.
   */

  if (ShouldMaskImage(image, mask) == 0)
  {
    #ifdef TEST
    fprintf(stderr, "******NXMaskImage: the image will not be masked\n");
    #endif
  }
  else
  {
    if (MaskInPlaceImage(mask, image) <= 0)
    {
      #ifdef PANIC
      fprintf(stderr, "******NXMaskImage: PANIC! Failed to apply the color mask in place.\n");
      #endif
    }
  }
}

void NXCacheInit(int size)
{
  NXImageCacheStat = 0;
  NXImageCache = malloc(size * sizeof(_NXImageCacheEntry));

  if (NXImageCache)
  {
    memset(NXImageCache, 0, size * sizeof(_NXImageCacheEntry));

    NXImageCacheSize = size;
    NXImageCacheStat = 1;

    #ifdef DEBUG
    fprintf(stderr, "******NXCacheInit: packed images cache initialized with [%d] entries.\n", size);
    #endif
  }
}

XImage *NXCacheFindImage(XImage *src_image, unsigned int *method, unsigned char **nxMD5)
{
  md5_state_t  md5StateImage;
  md5_byte_t   *md5;
  unsigned int data_size, i;

  *nxMD5 = NULL;

  if ((md5 = Xmalloc(MD5_LENGTH)) == NULL)
  {
    return NULL;
  }

  data_size = (src_image -> bytes_per_line * src_image -> height);

  md5_init(&md5StateImage);
  md5_append(&md5StateImage,(unsigned char *) &src_image -> width, sizeof(int));
  md5_append(&md5StateImage, (unsigned char *) &src_image -> height, sizeof(int));
  md5_append(&md5StateImage, src_image -> data, data_size);
  md5_finish(&md5StateImage, md5);

  *nxMD5 = md5;

  for (i = 0; i < NXImageCacheSize; i++)
  {
    if (NXImageCache[i].ximage)
    {
      if (!memcmp(NXImageCache[i].md5, md5, MD5_LENGTH))
      {
        *method = NXImageCache[i].method;

        NXImageCacheHits++;

        #ifdef DEBUG
        fprintf(stderr, "******NXCacheFindImage: found image at pos. [%d] in cache, hits are [%d] on [%d] packs.\n",
                i, NXImageCacheHits, NXImageCacheOps);
        #endif

        return NXImageCache[i].ximage;
      }
    }
    else
    {
      break;
    }
  }

  return NULL;
}

void NXCacheAddImage(XImage *ximage, unsigned int method, unsigned char *nxMD5)
{
  unsigned int i;

  if (ximage)
  {
    i = (NXImageCacheOps < NXImageCacheSize) ? NXImageCacheOps : NXImageCacheSize;

    if (NXImageCacheOps >= NXImageCacheSize )
    {
      #ifdef DEBUG
      fprintf(stderr, "******NXCacheAddImage: Image cache is full, freeing oldest entry.\n");
      #endif

      i--;

      XFree(NXImageCache[NXImageCacheSize - 1].ximage -> data);
      XFree(NXImageCache[NXImageCacheSize - 1].ximage);
    }

    if (i > 0)
    {
      memmove(&NXImageCache[1], NXImageCache, i * sizeof(_NXImageCacheEntry));
    }

    NXImageCacheOps++;

    #ifdef DEBUG
    fprintf(stderr, "******NXCacheAddImage: Going to add new image [%d] bytes big.\n", ximage -> xoffset);
    #endif

    if ((NXImageCache[0].ximage = (XImage *) Xcalloc(1, sizeof(XImage))) != NULL)
    {
      *NXImageCache[0].ximage = *ximage;

      if ((NXImageCache[0].ximage -> data = Xmalloc(ximage -> xoffset)) != NULL)
      {
        memcpy(NXImageCache[0].ximage -> data, ximage -> data, ximage -> xoffset);

        #ifdef DEBUG
        fprintf(stderr, "******NXCacheAddImage: new image added!\n");
        #endif
      }
      else
      {
        XFree(NXImageCache[0].ximage);

        NXImageCache[0].ximage = NULL;
      }
    }

    NXImageCache[0].method = method;

    memcpy(NXImageCache[0].md5, nxMD5, MD5_LENGTH);
  }
  else
  {
    #ifdef DEBUG
    fprintf(stderr, "******NXCacheAddImage: Invalid image.\n");
    #endif
  }

  XFree(nxMD5);
}

static void _NXNotifyImage(Display *dpy, int resource, Bool success)
{
  XEvent async_event;

  /*
   * Enqueue an event to tell client
   * the result of GetImage.
   */

  async_event.type = ClientMessage;

  async_event.xclient.window       = 0;
  async_event.xclient.message_type = 0;
  async_event.xclient.format       = 32;

  async_event.xclient.data.l[0] = NXCollectImageNotify;
  async_event.xclient.data.l[1] = resource;
  async_event.xclient.data.l[2] = success;

  XPutBackEvent(dpy, &async_event);
}

static Bool _NXCollectImageHandler(Display *dpy, xReply *rep, char *buf,
                                       int len, XPointer data)
{
  register _NXCollectImageState *state;

  register xGetImageReply *async_rep;

  char *async_head;
  char *async_data;

  int async_size;

  state = (_NXCollectImageState *) data;

  if ((rep -> generic.sequenceNumber % 65536) !=
          ((int)(state -> sequence) % 65536))
  {
    #ifdef TEST
    fprintf(stderr, "******_NXCollectImageHandler: Unmatched sequence [%d] for opcode [%d] with length [%ld].\n",
                rep -> generic.sequenceNumber, rep -> generic.type, rep -> generic.length << 2);
    #endif

    return False;
  }

  #ifdef TEST
  fprintf(stderr, "******_NXCollectImageHandler: Going to handle asynchronous GetImage reply.\n");
  #endif

  /*
   * As even reply data is managed asynchronously,
   * we can use state to get to vector and vector
   * to get to handler. In this way, we can safely
   * dequeue and free the handler itself.
   */

  DeqAsyncHandler(dpy, state -> handler);

  XFree(state -> handler);

  state -> handler = NULL;

  if (rep -> generic.type == X_Error)
  {
    #ifdef TEST
    fprintf(stderr, "******_NXCollectImageHandler: Error received from X server for resource [%d].\n",
                state -> resource);
    #endif

    _NXNotifyImage(dpy, state -> resource, False);

    _NXCollectedImages[state -> resource] = NULL;

    XFree(state);

    return False;
  }

  #ifdef TEST
  fprintf(stderr, "******_NXCollectImageHandler: Matched request with sequence [%ld].\n",
              state -> sequence);
  #endif

  async_size = SIZEOF(xGetImageReply);

  async_head = Xmalloc(async_size);

  if (async_head == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******_NXCollectImageHandler: PANIC! Failed to allocate memory with resource [%d].\n",
                state -> resource);
    #endif

    _NXNotifyImage(dpy, state -> resource, False);

    _NXCollectedImages[state -> resource] = NULL;

    XFree(state);

    return False;
  }

  #ifdef TEST
  fprintf(stderr, "******_NXCollectImageHandler: Going to get reply with size [%ld].\n",
              rep -> generic.length << 2);
  #endif

  async_rep = (xGetImageReply *) _XGetAsyncReply(dpy, async_head, rep, buf, len, 0, False);

  if (async_rep == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******_NXCollectImageHandler: PANIC! Failed to get reply with resource [%d].\n",
                state -> resource);
    #endif

    _NXNotifyImage(dpy, state -> resource, False);

    _NXCollectedImages[state -> resource] = NULL;

    XFree(state);

    XFree(async_head);

    return False;
  }

  #ifdef TEST
  fprintf(stderr, "******_NXCollectImageHandler: Got reply with depth [%d] visual [%ld] size [%ld].\n",
              async_rep -> depth, async_rep -> visual, async_rep -> length << 2);
  #endif

  async_size = async_rep -> length << 2;

  if (async_size > 0)
  {
    async_data = Xmalloc(async_size);

    if (async_data == NULL)
    {
      #ifdef PANIC
      fprintf(stderr, "******_NXCollectImageHandler: PANIC! Failed to allocate memory with resource [%d].\n",
                  state -> resource);
      #endif

      _NXNotifyImage(dpy, state -> resource, False);

      _NXCollectedImages[state -> resource] = NULL;

      XFree(state);

      XFree(async_head);

      return False;
    }

    #ifdef TEST
    fprintf(stderr, "******_NXCollectImageHandler: Going to get data with size [%d].\n",
                async_size);
    #endif

    _XGetAsyncData(dpy, async_data, buf, len, SIZEOF(xGetImageReply), async_size, async_size);

    /*
     * From now on we can return True, as all
     * data has been consumed from buffer.
     */

    if (state -> format == XYPixmap)
    {
      unsigned long depth = DepthOnes(state -> mask & (((unsigned long)0xFFFFFFFF) >>
                                          (32 - async_rep -> depth)));

      state -> image = XCreateImage(dpy, _XVIDtoVisual(dpy, async_rep -> visual),
                                        depth, XYPixmap, 0, async_data, state -> width,
                                            state -> height, dpy -> bitmap_pad, 0);
    }
    else
    {
      state -> image = XCreateImage(dpy, _XVIDtoVisual(dpy, async_rep -> visual),
                                        async_rep -> depth, ZPixmap, 0, async_data, state -> width,
                                            state -> height, _XGetScanlinePad(dpy, async_rep -> depth), 0);
    }

    if (state -> image == NULL)
    {
      #ifdef PANIC
      fprintf(stderr, "******_NXCollectImageHandler: PANIC! Failed to create image for resource [%d].\n",
                  state -> resource);
      #endif

      _NXNotifyImage(dpy, state -> resource, False);

      _NXCollectedImages[state -> resource] = NULL;

      XFree(state);

      XFree(async_head);
      XFree(async_data);

      return True;
    }

    #ifdef TEST
    fprintf(stderr, "******_NXCollectImageHandler: Successfully stored image data for resource [%d].\n",
                state -> resource);
    #endif
  }
  #ifdef WARNING
  else
  {
    fprintf(stderr, "******_NXCollectImageHandler: WARNING! Null image data stored for resource [%d].\n",
                state -> resource);
  }
  #endif

  _NXNotifyImage(dpy, state -> resource, True);

  XFree(async_head);

  return True;
}

int NXGetCollectImageResource(Display *dpy)
{
  int i;

  for (i = 0; i < NXNumberOfConnections; i++)
  {
    if (_NXCollectedImages[i] == NULL)
    {
      return i;
    }
  }

  return -1;
}

int NXCollectImage(Display *dpy, unsigned int resource, Drawable drawable,
                       int src_x, int src_y, unsigned int width, unsigned int height,
                           unsigned long plane_mask, int format)
{
  register xGetImageReq *req;

  _NXCollectImageState *state;
  _XAsyncHandler *handler;

  if (resource >= NXNumberOfConnections)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXCollectImage: PANIC! Provided resource [%u] is out of range.\n",
                resource);
    #endif

    return -1;
  }

  state = _NXCollectedImages[resource];

  if (state != NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXCollectImage: PANIC! Having to remove previous state for resource [%u].\n",
                resource);
    #endif

    if (state -> handler != NULL)
    {
      DeqAsyncHandler(dpy, state -> handler);

      XFree(state -> handler);
    }

    if (state -> image != NULL)
    {
      XDestroyImage(state -> image);
    }

    XFree(state);

    _NXCollectedImages[resource] = NULL;
  }

  LockDisplay(dpy);

  GetReq(GetImage, req);

  req -> format    = format;
  req -> drawable  = drawable;
  req -> x         = src_x;
  req -> y         = src_y;
  req -> width     = width;
  req -> height    = height;
  req -> planeMask = plane_mask;

  #ifdef TEST
  fprintf(stderr, "******NXCollectImage: Sending message opcode [%d] sequence [%ld] for resource [%d].\n",
              X_GetImage, dpy -> request, resource);

  fprintf(stderr, "******NXCollectImage: Format [%d] drawable [%ld] src_x [%d] src_y [%d].\n",
              req -> format, req -> drawable, req -> x, req -> y);

  fprintf(stderr, "******NXCollectImage: Width [%d] height [%d] plane_mask [%lx].\n",
              req -> width, req -> height, req -> planeMask);
  #endif

  state   = Xmalloc(sizeof(_NXCollectImageState));
  handler = Xmalloc(sizeof(_XAsyncHandler));

  if (state == NULL || handler == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXCollectImage: PANIC! Failed to allocate memory with resource [%d].\n",
                resource);
    #endif

    UnGetReq(GetImage);

    if (state != NULL)
    {
      XFree(state);
    }

    if (handler != NULL)
    {
      XFree(handler);
    }

    UnlockDisplay(dpy);
    SyncHandle();

    return -1;
  }

  state -> sequence = dpy -> request;
  state -> resource = resource;
  state -> mask     = plane_mask;
  state -> format   = format;
  state -> width    = width;
  state -> height   = height;
  state -> image    = NULL;

  state -> handler = handler;

  handler -> next = dpy -> async_handlers;
  handler -> handler = _NXCollectImageHandler;
  handler -> data = (XPointer) state;
  dpy -> async_handlers = handler;

  _NXCollectedImages[resource] = state;

  UnlockDisplay(dpy);

  SyncHandle();

  return 1;
}

int NXGetCollectedImage(Display *dpy, unsigned int resource, XImage **image)
{
  register _NXCollectImageState *state;

  state = _NXCollectedImages[resource];

  if (state == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXGetCollectedImage: PANIC! No image collected for resource [%u].\n",
                resource);
    #endif

    return 0;
  }

  _NXCollectedImages[resource] = NULL;

  *image = state -> image;

  XFree(state);

  #ifdef TEST
  fprintf(stderr, "******NXGetCollectedImage: Returning GetImage data for resource [%u].\n",
              resource);
  #endif

  return 1;
}

static void _NXNotifyProperty(Display *dpy, int resource, Bool success)
{
  XEvent async_event;

  /*
   * Enqueue an event to tell client
   * the result of GetProperty.
   */

  async_event.type = ClientMessage;

  async_event.xclient.window       = 0;
  async_event.xclient.message_type = 0;
  async_event.xclient.format       = 32;

  async_event.xclient.data.l[0] = NXCollectPropertyNotify;
  async_event.xclient.data.l[1] = resource;
  async_event.xclient.data.l[2] = success;

  XPutBackEvent(dpy, &async_event);
}

static Bool _NXCollectPropertyHandler(Display *dpy, xReply *rep, char *buf,
                                          int len, XPointer data)
{
  register _NXCollectPropertyState *state;

  register xGetPropertyReply *async_rep;

  char *async_head;
  char *async_data;

  int async_size;

  state = (_NXCollectPropertyState *) data;

  if ((rep -> generic.sequenceNumber % 65536) !=
          ((int)(state -> sequence) % 65536))
  {
    #ifdef TEST
    fprintf(stderr, "******_NXCollectPropertyHandler: Unmatched sequence [%d] for opcode [%d] with length [%ld].\n",
                rep -> generic.sequenceNumber, rep -> generic.type, rep -> generic.length << 2);
    #endif

    return False;
  }

  #ifdef TEST
  fprintf(stderr, "******_NXCollectPropertyHandler: Going to handle asynchronous GetProperty reply.\n");
  #endif

  /*
   * Reply data is managed asynchronously. We can
   * use state to get to vector and vector to get
   * to handler. In this way, we can dequeue and
   * free the handler itself.
   */

  DeqAsyncHandler(dpy, state -> handler);

  XFree(state -> handler);

  state -> handler = NULL;

  if (rep -> generic.type == X_Error)
  {
    #ifdef TEST
    fprintf(stderr, "******_NXCollectPropertyHandler: Error received from X server for resource [%d].\n",
                state -> resource);
    #endif

    _NXNotifyProperty(dpy, state -> resource, False);

    _NXCollectedProperties[state -> resource] = NULL;

    XFree(state);

    return False;
  }

  #ifdef TEST
  fprintf(stderr, "******_NXCollectPropertyHandler: Matched request with sequence [%ld].\n",
              state -> sequence);
  #endif

  async_size = SIZEOF(xGetPropertyReply);

  async_head = Xmalloc(async_size);

  if (async_head == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******_NXCollectPropertyHandler: PANIC! Failed to allocate memory with resource [%d].\n",
                state -> resource);
    #endif

    _NXNotifyProperty(dpy, state -> resource, False);

    _NXCollectedProperties[state -> resource] = NULL;

    XFree(state);

    return False;
  }

  #ifdef TEST
  fprintf(stderr, "******_NXCollectPropertyHandler: Going to get reply with size [%ld].\n",
              rep -> generic.length << 2);
  #endif

  async_rep = (xGetPropertyReply *) _XGetAsyncReply(dpy, async_head, rep, buf, len, 0, False);

  if (async_rep == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******_NXCollectPropertyHandler: PANIC! Failed to get reply with resource [%d].\n",
                state -> resource);
    #endif

    _NXNotifyProperty(dpy, state -> resource, False);

    _NXCollectedProperties[state -> resource] = NULL;

    XFree(state);

    XFree(async_head);

    return False;
  }

  #ifdef TEST
  fprintf(stderr, "******_NXCollectPropertyHandler: Got reply with format [%d] type [%ld] size [%ld].\n",
              async_rep -> format, async_rep -> propertyType, async_rep -> length << 2);

  fprintf(stderr, "******_NXCollectPropertyHandler: Bytes after [%ld] number of items [%ld].\n",
              async_rep -> bytesAfter, async_rep -> nItems);
  #endif

  state -> format = async_rep -> format;
  state -> type   = async_rep -> propertyType;
  state -> items  = async_rep -> nItems;
  state -> after  = async_rep -> bytesAfter;

  async_size = async_rep -> length << 2;

  if (async_size > 0)
  {
    async_data = Xmalloc(async_size);

    if (async_data == NULL)
    {
      #ifdef PANIC
      fprintf(stderr, "******_NXCollectPropertyHandler: PANIC! Failed to allocate memory with resource [%d].\n",
                  state -> resource);
      #endif

      _NXNotifyProperty(dpy, state -> resource, False);

      _NXCollectedProperties[state -> resource] = NULL;

      XFree(state);

      XFree(async_head);

      return False;
    }

    #ifdef TEST
    fprintf(stderr, "******_NXCollectPropertyHandler: Going to get data with size [%d].\n",
                async_size);
    #endif

    _XGetAsyncData(dpy, async_data, buf, len, SIZEOF(xGetPropertyReply), async_size, async_size);

    /*
     * From now on we can return True, as all
     * data has been consumed from buffer.
     */

    state -> data = async_data;

    #ifdef TEST
    fprintf(stderr, "******_NXCollectPropertyHandler: Successfully stored property data for resource [%d].\n",
                state -> resource);
    #endif
  }
  #ifdef TEST
  else
  {
    fprintf(stderr, "******_NXCollectPropertyHandler: WARNING! Null property data stored for resource [%d].\n",
                state -> resource);
  }
  #endif

  _NXNotifyProperty(dpy, state -> resource, True);

  XFree(async_head);

  return True;
}

int NXGetCollectPropertyResource(Display *dpy)
{
  int i;

  for (i = 0; i < NXNumberOfConnections; i++)
  {
    if (_NXCollectedProperties[i] == NULL)
    {
      return i;
    }
  }

  return -1;
}

int NXCollectProperty(Display *dpy, unsigned int resource, Window window, Atom property,
                          long long_offset, long long_length, Bool delete, Atom req_type)
{
  register xGetPropertyReq *req;

  _NXCollectPropertyState *state;
  _XAsyncHandler *handler;

  if (resource >= NXNumberOfConnections)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXCollectProperty: PANIC! Provided resource [%u] is out of range.\n",
                resource);
    #endif

    return -1;
  }

  state = _NXCollectedProperties[resource];

  if (state != NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXCollectProperty: PANIC! Having to remove previous state for resource [%u].\n",
                resource);
    #endif

    if (state -> handler != NULL)
    {
      DeqAsyncHandler(dpy, state -> handler);

      XFree(state -> handler);
    }

    if (state -> data != NULL)
    {
      XFree(state -> data);
    }

    XFree(state);

    _NXCollectedProperties[resource] = NULL;
  }

  LockDisplay(dpy);

  GetReq(GetProperty, req);

  req -> delete     = delete;
  req -> window     = window;
  req -> property   = property;
  req -> type       = req_type;
  req -> longOffset = long_offset;
  req -> longLength = long_length;

  #ifdef TEST
  fprintf(stderr, "******NXCollectProperty: Sending message opcode [%d] sequence [%ld] for resource [%d].\n",
              X_GetProperty, dpy -> request, resource);

  fprintf(stderr, "******NXCollectProperty: Delete [%u] window [%ld] property [%ld] type [%ld].\n",
              req -> delete, req -> window, req -> property, req -> type);

  fprintf(stderr, "******NXCollectProperty: Long offset [%ld] long length [%ld].\n",
              req -> longOffset, req -> longLength);
  #endif

  state   = Xmalloc(sizeof(_NXCollectPropertyState));
  handler = Xmalloc(sizeof(_XAsyncHandler));

  if (state == NULL || handler == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXCollectProperty: Failed to allocate memory with resource [%d].\n",
                resource);
    #endif

    if (state != NULL)
    {
      XFree(state);
    }

    if (handler != NULL)
    {
      XFree(handler);
    }

    UnGetReq(GetProperty);

    UnlockDisplay(dpy);
    SyncHandle();

    return -1;
  }

  state -> sequence = dpy -> request;
  state -> resource = resource;
  state -> window   = window;
  state -> property = property;
  state -> type     = 0;
  state -> format   = 0;
  state -> items    = 0;
  state -> after    = 0;
  state -> data     = NULL;

  state -> handler = handler;

  handler -> next = dpy -> async_handlers;
  handler -> handler = _NXCollectPropertyHandler;
  handler -> data = (XPointer) state;
  dpy -> async_handlers = handler;

  _NXCollectedProperties[resource] = state;

  UnlockDisplay(dpy);

  SyncHandle();

  return True;
}

int NXGetCollectedProperty(Display *dpy, unsigned int resource, Atom *actual_type_return,
                               int *actual_format_return, unsigned long *nitems_return,
                                   unsigned long *bytes_after_return, unsigned char **data)
{
  register _NXCollectPropertyState *state;

  state = _NXCollectedProperties[resource];

  if (state == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXGetCollectedProperty: PANIC! No data collected for resource [%u].\n",
                resource);
    #endif

    return 0;
  }

  *actual_type_return   = state -> type;
  *actual_format_return = state -> format;
  *nitems_return        = state -> items;
  *bytes_after_return   = state -> after;

  *data = _NXCollectedProperties[resource] -> data;

  XFree(state);

  _NXCollectedProperties[resource] = NULL;

  #ifdef TEST
  fprintf(stderr, "******NXGetCollectedProperty: Returning GetProperty data for resource [%u].\n",
              resource);
  #endif

  return True;
}

static void _NXNotifyGrabPointer(Display *dpy, int resource, Bool success)
{
  XEvent async_event;

  async_event.type = ClientMessage;

  async_event.xclient.window       = 0;
  async_event.xclient.message_type = 0;
  async_event.xclient.format       = 32;

  async_event.xclient.data.l[0] = NXCollectGrabPointerNotify;
  async_event.xclient.data.l[1] = resource;
  async_event.xclient.data.l[2] = success;

  XPutBackEvent(dpy, &async_event);
}

static Bool _NXCollectGrabPointerHandler(Display *dpy, xReply *rep, char *buf,
                                             int len, XPointer data)
{
  register _NXCollectGrabPointerState *state;

  register xGrabPointerReply *async_rep;

  char *async_head;

  int async_size;

  state = (_NXCollectGrabPointerState *) data;

  if ((rep -> generic.sequenceNumber % 65536) !=
          ((int)(state -> sequence) % 65536))
  {
    #ifdef TEST
    fprintf(stderr, "******_NXCollectGrabPointerHandler: Unmatched sequence [%d] for opcode [%d] with length [%ld].\n",
                rep -> generic.sequenceNumber, rep -> generic.type, rep -> generic.length << 2);
    #endif

    return False;
  }

  #ifdef TEST
  fprintf(stderr, "******_NXCollectGrabPointerHandler: Going to handle asynchronous GrabPointer reply.\n");
  #endif

  DeqAsyncHandler(dpy, state -> handler);

  XFree(state -> handler);

  state -> handler = NULL;

  if (rep -> generic.type == X_Error)
  {
    #ifdef TEST
    fprintf(stderr, "******_NXCollectGrabPointerHandler: Error received from X server for resource [%d].\n",
                state -> resource);
    #endif

    _NXNotifyGrabPointer(dpy, state -> resource, False);

    _NXCollectedGrabPointers[state -> resource] = NULL;

    XFree(state);

    return False;
  }

  #ifdef TEST
  fprintf(stderr, "******_NXCollectGrabPointerHandler: Matched request with sequence [%ld].\n",
              state -> sequence);
  #endif

  async_size = SIZEOF(xGrabPointerReply);

  async_head = Xmalloc(async_size);

  if (async_head == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******_NXCollectGrabPointerHandler: PANIC! Failed to allocate memory with resource [%d].\n",
                state -> resource);
    #endif

    _NXNotifyGrabPointer(dpy, state -> resource, False);

    _NXCollectedGrabPointers[state -> resource] = NULL;

    XFree(state);

    return False;
  }

  #ifdef TEST
  fprintf(stderr, "******_NXCollectGrabPointerHandler: Going to get reply with size [%ld].\n",
              rep -> generic.length << 2);
  #endif

  async_rep = (xGrabPointerReply *) _XGetAsyncReply(dpy, async_head, rep, buf, len, 0, False);

  if (async_rep == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******_NXCollectGrabPointerHandler: PANIC! Failed to get reply with resource [%d].\n",
                state -> resource);
    #endif

    _NXNotifyGrabPointer(dpy, state -> resource, False);

    _NXCollectedGrabPointers[state -> resource] = NULL;

    XFree(state);

    XFree(async_head);

    return False;
  }

  #ifdef TEST
  fprintf(stderr, "******_NXCollectGrabPointerHandler: Got reply with status [%d] size [%ld].\n",
              async_rep -> status, async_rep -> length << 2);
  #endif

  state -> status = async_rep -> status;

  _NXNotifyGrabPointer(dpy, state -> resource, True);

  XFree(async_head);

  return True;
}

int NXGetCollectGrabPointerResource(Display *dpy)
{
  int i;

  for (i = 0; i < NXNumberOfConnections; i++)
  {
    if (_NXCollectedGrabPointers[i] == NULL)
    {
      return i;
    }
  }

  return -1;
}

int NXCollectGrabPointer(Display *dpy, unsigned int resource, Window grab_window, Bool owner_events,
                             unsigned int event_mask, int pointer_mode, int keyboard_mode,
                                 Window confine_to, Cursor cursor, Time time)
{
  register xGrabPointerReq *req;

  _NXCollectGrabPointerState *state;
  _XAsyncHandler *handler;

  if (resource >= NXNumberOfConnections)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXCollectGrabPointer: PANIC! Provided resource [%u] is out of range.\n",
                resource);
    #endif

    return -1;
  }

  state = _NXCollectedGrabPointers[resource];

  if (state != NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXCollectGrabPointer: PANIC! Having to remove previous state for resource [%u].\n",
                resource);
    #endif

    if (state -> handler != NULL)
    {
      DeqAsyncHandler(dpy, state -> handler);

      XFree(state -> handler);
    }

    XFree(state);

    _NXCollectedGrabPointers[resource] = NULL;
  }

  LockDisplay(dpy);

  GetReq(GrabPointer, req);

  req -> grabWindow   = grab_window;
  req -> ownerEvents  = owner_events;
  req -> eventMask    = event_mask;
  req -> pointerMode  = pointer_mode;
  req -> keyboardMode = keyboard_mode;
  req -> confineTo    = confine_to;
  req -> cursor       = cursor;
  req -> time         = time;

  #ifdef TEST
  fprintf(stderr, "******NXCollectGrabPointer: Sending message opcode [%d] sequence [%ld] for resource [%d].\n",
              X_GrabPointer, dpy -> request, resource);
  #endif

  state   = Xmalloc(sizeof(_NXCollectGrabPointerState));
  handler = Xmalloc(sizeof(_XAsyncHandler));

  if (state == NULL || handler == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXCollectGrabPointer: Failed to allocate memory with resource [%d].\n",
                resource);
    #endif

    if (state != NULL)
    {
      XFree(state);
    }

    if (handler != NULL)
    {
      XFree(handler);
    }

    UnGetReq(GrabPointer);

    UnlockDisplay(dpy);
    SyncHandle();

    return -1;
  }

  state -> sequence = dpy -> request;
  state -> resource = resource;
  state -> status   = 0;

  state -> handler = handler;

  handler -> next = dpy -> async_handlers;
  handler -> handler = _NXCollectGrabPointerHandler;
  handler -> data = (XPointer) state;
  dpy -> async_handlers = handler;

  _NXCollectedGrabPointers[resource] = state;

  UnlockDisplay(dpy);

  SyncHandle();

  return True;
}

int NXGetCollectedGrabPointer(Display *dpy, unsigned int resource, int *status)
{
  register _NXCollectGrabPointerState *state;

  state = _NXCollectedGrabPointers[resource];

  if (state == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXGetCollectedGrabPointer: PANIC! No data collected for resource [%u].\n",
                resource);
    #endif

    return 0;
  }

  *status = state -> status;

  XFree(state);

  _NXCollectedGrabPointers[resource] = NULL;

  #ifdef TEST
  fprintf(stderr, "******NXGetCollectedGrabPointer: Returning GrabPointer data for resource [%u].\n",
              resource);
  #endif

  return True;
}

static void _NXNotifyInputFocus(Display *dpy, int resource, Bool success)
{
  XEvent async_event;

  async_event.type = ClientMessage;

  async_event.xclient.window       = 0;
  async_event.xclient.message_type = 0;
  async_event.xclient.format       = 32;

  async_event.xclient.data.l[0] = NXCollectInputFocusNotify;
  async_event.xclient.data.l[1] = resource;
  async_event.xclient.data.l[2] = success;

  XPutBackEvent(dpy, &async_event);
}

static Bool _NXCollectInputFocusHandler(Display *dpy, xReply *rep, char *buf,
                                            int len, XPointer data)
{
  register _NXCollectInputFocusState *state;

  register xGetInputFocusReply *async_rep;

  char *async_head;

  int async_size;

  state = (_NXCollectInputFocusState *) data;

  if ((rep -> generic.sequenceNumber % 65536) !=
          ((int)(state -> sequence) % 65536))
  {
    #ifdef TEST
    fprintf(stderr, "******_NXCollectInputFocusHandler: Unmatched sequence [%d] for opcode [%d] with length [%ld].\n",
                rep -> generic.sequenceNumber, rep -> generic.type, rep -> generic.length << 2);
    #endif

    return False;
  }

  #ifdef TEST
  fprintf(stderr, "******_NXCollectInputFocusHandler: Going to handle asynchronous GetInputFocus reply.\n");
  #endif

  DeqAsyncHandler(dpy, state -> handler);

  XFree(state -> handler);

  state -> handler = NULL;

  if (rep -> generic.type == X_Error)
  {
    #ifdef TEST
    fprintf(stderr, "******_NXCollectInputFocusHandler: Error received from X server for resource [%d].\n",
                state -> resource);
    #endif

    _NXNotifyInputFocus(dpy, state -> resource, False);

    _NXCollectedInputFocuses[state -> resource] = NULL;

    XFree(state);

    return False;
  }

  #ifdef TEST
  fprintf(stderr, "******_NXCollectInputFocusHandler: Matched request with sequence [%ld].\n",
              state -> sequence);
  #endif

  async_size = SIZEOF(xGetInputFocusReply);

  async_head = Xmalloc(async_size);

  if (async_head == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******_NXCollectInputFocusHandler: PANIC! Failed to allocate memory with resource [%d].\n",
                state -> resource);
    #endif

    _NXNotifyInputFocus(dpy, state -> resource, False);

    _NXCollectedInputFocuses[state -> resource] = NULL;

    XFree(state);

    return False;
  }

  #ifdef TEST
  fprintf(stderr, "******_NXCollectInputFocusHandler: Going to get reply with size [%ld].\n",
              rep -> generic.length << 2);
  #endif

  async_rep = (xGetInputFocusReply *) _XGetAsyncReply(dpy, async_head, rep, buf, len, 0, False);

  if (async_rep == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******_NXCollectInputFocusHandler: PANIC! Failed to get reply with resource [%d].\n",
                state -> resource);
    #endif

    _NXNotifyInputFocus(dpy, state -> resource, False);

    _NXCollectedInputFocuses[state -> resource] = NULL;

    XFree(state);

    XFree(async_head);

    return False;
  }

  #ifdef TEST
  fprintf(stderr, "******_NXCollectInputFocusHandler: Got reply with focus [%d] revert to [%d] "
              "size [%ld].\n", (int) async_rep -> focus, (int) async_rep -> revertTo,
                  async_rep -> length << 2);
  #endif

  state -> focus     = async_rep -> focus;
  state -> revert_to = async_rep -> revertTo;

  _NXNotifyInputFocus(dpy, state -> resource, True);

  XFree(async_head);

  return True;
}

int NXGetCollectInputFocusResource(Display *dpy)
{
  int i;

  for (i = 0; i < NXNumberOfConnections; i++)
  {
    if (_NXCollectedInputFocuses[i] == NULL)
    {
      return i;
    }
  }

  return -1;
}

int NXCollectInputFocus(Display *dpy, unsigned int resource)
{
  register xReq *req;

  _NXCollectInputFocusState *state;
  _XAsyncHandler *handler;

  if (resource >= NXNumberOfConnections)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXCollectInputFocus: PANIC! Provided resource [%u] is out of range.\n",
                resource);
    #endif

    return -1;
  }

  state = _NXCollectedInputFocuses[resource];

  if (state != NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXCollectInputFocus: PANIC! Having to remove previous state for resource [%u].\n",
                resource);
    #endif

    if (state -> handler != NULL)
    {
      DeqAsyncHandler(dpy, state -> handler);

      XFree(state -> handler);
    }

    XFree(state);

    _NXCollectedInputFocuses[resource] = NULL;
  }

  LockDisplay(dpy);

  GetEmptyReq(GetInputFocus, req);

  #ifdef TEST
  fprintf(stderr, "******NXCollectInputFocus: Sending message opcode [%d] sequence [%ld] for resource [%d].\n",
              X_GetInputFocus, dpy -> request, resource);
  #endif

  state   = Xmalloc(sizeof(_NXCollectInputFocusState));
  handler = Xmalloc(sizeof(_XAsyncHandler));

  if (state == NULL || handler == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXCollectInputFocus: Failed to allocate memory with resource [%d].\n",
                resource);
    #endif

    if (state != NULL)
    {
      XFree(state);
    }

    if (handler != NULL)
    {
      XFree(handler);
    }

    UnGetEmptyReq();

    UnlockDisplay(dpy);
    SyncHandle();

    return -1;
  }

  state -> sequence  = dpy -> request;
  state -> resource  = resource;
  state -> focus     = 0;
  state -> revert_to = 0;

  state -> handler = handler;

  handler -> next = dpy -> async_handlers;
  handler -> handler = _NXCollectInputFocusHandler;
  handler -> data = (XPointer) state;
  dpy -> async_handlers = handler;

  _NXCollectedInputFocuses[resource] = state;

  UnlockDisplay(dpy);

  SyncHandle();

  return True;
}

int NXGetCollectedInputFocus(Display *dpy, unsigned int resource,
                                 Window *focus_return, int *revert_to_return)
{
  register _NXCollectInputFocusState *state;

  state = _NXCollectedInputFocuses[resource];

  if (state == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "******NXGetCollectedInputFocus: PANIC! No data collected for resource [%u].\n",
                resource);
    #endif

    return 0;
  }

  *focus_return     = state -> focus;
  *revert_to_return = state -> revert_to;

  XFree(state);

  _NXCollectedInputFocuses[resource] = NULL;

  #ifdef TEST
  fprintf(stderr, "******NXGetCollectedInputFocus: Returning GetInputFocus data for resource [%u].\n",
              resource);
  #endif

  return True;
}
