/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rigths reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#ifndef NXlib_H
#define NXlib_H

#ifdef __cplusplus
extern "C" {
#endif

#include <X11/X.h>
#include <X11/Xlib.h>

#include "NXpack.h"
#include "NXproto.h"

#include "Png.h"

/*
 * Maximum number of supported pack methods.
 */

#define NXNumberOfPackMethods              128

/*
 * Assume this as limit of 'resource' parameter in
 * requests where the agent's client ID needs to
 * be provided to proxy.
 */

#define NXNumberOfConnections              256

/*
 * If the flag is set to true, the Xlib I/O error
 * handler will simply return, instead of quitting
 * the program. This leaves to the application the
 * responsibility of checking the state of the
 * XlibDisplayIOError flag.
 */

void NXContinueOnDisplayError(
#if NeedFunctionPrototypes
    int value           /* flag */
#endif
);

/*
 * Check that the display is not NULL and that the
 * descriptor was not shut down after an IOError.
 */

extern Bool NXDisplayError(
#if NeedFunctionPrototypes
    Display*            /* display */
#endif
);

/*
 * Shutdown the display descriptor and force Xlib
 * to set the I/O error flag.
 */

extern Bool NXForceDisplayError(
#if NeedFunctionPrototypes
    Display*            /* display */
#endif
);

/*
 * Agent should get NX parameters at startup, just after having
 * opened the display. When told to pack images (i.e. packMethod
 * is not NO_PACK), agent should only use PutPackedImage requests
 * to send ZPixmaps over the network. This is important as proxy
 * optimizes its compression algorythms in order to reserve more
 * space in cache for packed images and only receive XYBitmaps as
 * PutImage.
 *
 * If agent is not able to satisfy the pack method set by user
 * (because a method is not applicable, is not supported or simply
 * requires a screen depth greater then available), it should fall-
 * back to the most suitable method af the same type (f.e. with
 * less colors).
 *
 * After having got the unpack methods supported by proxy (see 
 * below), agents should call SelectPackMethod() until they find a
 * method that is supported by both agent and the remote proxy.
 *
 * The reply has 4 new boolean fields in the last protocol version.
 */

extern Status NXGetControlParameters(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int*       /* link_type */,
    unsigned int*       /* protocol_major */,
    unsigned int*       /* protocol_minor */,
    unsigned int*       /* protocol_patch */,
    int*                /* karma_size */,
    int*                /* split_size */,
    unsigned int*       /* pack_method */,
    unsigned int*       /* pack_quality */,
    int*                /* data_level */,
    int*                /* stream_level */,
    int*                /* delta_level */,
    unsigned int*       /* load_cache */,
    unsigned int*       /* save_cache */,
    unsigned int*       /* startup_cache */
#endif
);

/*
 * Get from NX proxy user's settings about cleanup policies of
 * buffers allocated by Xlib. By cleaning pad bytes in requests,
 * infact, user can achieve better compression results. Cleanup
 * of pad bytes is progressively migrating in proxy, so this
 * request could be removed in future.
 *
 * Cleanup of images should always be enabled. Note that image
 * cleanup is only applied to PutImage requests. Packed images
 * are clean by definition.
 *
 * Based on reply, this function set flags in Xlib. There is no
 * function to set these flags explicitly. This can be consider-
 * ed a design error and will be fixed in future.
 */

extern Status NXGetCleanupParameters(
#if NeedFunctionPrototypes
    Display*            /* display */,
    int*                /* clean_get */,
    int*                /* clean_alloc */,
    int*                /* clean_flush */,
    int*                /* clean_send */,
    int*                /* clean_images */
#endif
);

/*
 * Get from NX proxy user's settings about X_PutImage handling
 * and apply them to Xlib. User can let image functions to split
 * big images in sub-images or apply a tiny frame around bitmaps
 * for test purposes.
 */

extern Status NXGetImageParameters(
#if NeedFunctionPrototypes
    Display*            /* display */,
    int*                /* image_split */,
    int*                /* image_mask */,
    int*                /* image_frame */,
    unsigned int*       /* image_split_method */,
    unsigned int*       /* image_mask_method */
#endif
);

/*
 * Which unpack methods are supported by proxy?
 */

extern Status NXGetUnpackParameters(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int*       /* entries */,
    unsigned char[]     /* supported_methods */
#endif
);

/*
 * Query and enable shared memory support on path agent to X
 * client proxy and X server proxy to real X server. At the
 * moment only the path proxy to real X server is implemented.
 * On return flags will say if support has been successfully
 * activated. Segments will contain the XID associated to the
 * shared memory blocks. A MIT-SHM compliant protocol is used
 * between proxy and the real server, while a simplified
 * version is used between the agent and the client proxy to
 * accomodate both packed images and plain X bitmaps.
 */

extern Status NXGetShmemParameters(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int*       /* enableClient  */,
    unsigned int*       /* enableServer  */,
    unsigned int*       /* clientSegment */,
    unsigned int*       /* serverSegment */
#endif
);

/*
 * This set of functions is used to leverage the image stream-
 * ing capabilities built in nxcomp. An image can be streamed
 * by sending a start-split message, followed by the X messa-
 * ges that will have to be split by the proxy, followed by
 * an end-split closure. Usually, in the middle of a start-
 * split/end-split sequence there will be a single PutImage()
 * or PutPackedImage(), that, in turn, can generate multiple
 * partial requests, like a SetUnpackColormap() and SetUnpack-
 * Alpha() that will be later used to decompress the image to
 * its original form. Multiple requests may be also generated
 * because of the maximum size of a X request being exceeded,
 * so that Xlib has to divide the single image in multiple
 * sub-image requests. The agent doesn't need to take care of
 * these details but will rather have to track the result of
 * the split operation. By monitoring the notify events sent
 * by the proxy, the agent will have be able to implement its
 * strategy to deal with its own the clients. For example, it
 * will be able to:
 *
 * - Suspend the client, if the image was not sent synchro-
 *   nously, in the main X oputput stream.
 *
 * - Choose to commit or discard the message at the time it
 *   will be recomposed at the remote side, including all
 *   messages that were part of the split (the colormap, the
 *   alpha channel, etc.)
 *
 * - Restart the client, in the case it had been suspended,
 *   at the end of the transferral.
 *
 * A more useful strategy would be to replace the original im-
 * age with a tiny 'placeholder' and 'synchronize' the content
 * of the drawable at later time. This is collectively referred
 * to as 'lazy encoding'.
 *
 * The agent will be able to identify the original split ope-
 * ration (the one referenced in the start-split/end-split seq-
 * uence) by the small integer number (0-255) named 'resource'.
 *
 * At the time the proxy receives the end-split, it reports the
 * result of the operation to the agent. One of the following
 * cases may be encountered:
 *
 * NXNoSplitNotify      All messages were sent in the main out-
 *                      put stream, so that no split actually
 *                      took place.
 *
 * NXStartSplitNotify   One or more messages were split, so,
 *                      at discrection of the agent, the client
 *                      may be suspended until the transferral
 *                      is completed.
 *
 * NXCommitSplitNotify  One of the requests that made up the
 *                      split was recomposed. The agent should
 *                      either commit the given request or tell
 *                      the proxy to discard it.
 *
 * NXEndSplitNotify     The split was duly completed. The agent
 *                      can restart the client.
 *
 * NXEmptySplitNotify   No more split operation are pending.
 *                      The agent can use this information to
 *                      implement specific strategies requiring
 *                      that all messages have been recomposed
 *                      at the remote end, like updating the
 *                      drawables that were not synchronized
 *                      because of the lazy encoding.
 *
 * The 'mode' field that is sent by the agent in the start-split
 * request, determines the strategy that the proxy will adopt to
 * deal with the image. If set to 'eager', the proxy will only
 * split the messages whose size exceeds the split threshold (the
 * current threshold can be found in the NXGetControlParameters()
 * reply). If the mode is set to lazy, the proxy will split any
 * image that would have generated an actual transfer of the data
 * part (in practice all images that are not found in the cache).
 * This second strategy can be leveraged by an agent to further
 * reduce the bandwidth requirements. For example, by setting the
 * mode to lazy and by monitoring the result, an agent can easi-
 * ly verify if the drawable was successfully updated, mark the
 * drawable if it was not, and synchronize it at later time.
 *
 * See NXproto.h for the definition of the available modes.
 */

extern int NXStartSplit(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int        /* resource */,
    unsigned int        /* mode */
#endif
);

extern int NXEndSplit(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int        /* resource */
#endif
);

extern int NXCommitSplit(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int        /* resource */,
    unsigned int        /* propagate */,
    unsigned char       /* request */,
    unsigned int        /* position */
#endif
);

extern int NXSetExposeEvents(
#if NeedFunctionPrototypes
    Display*            /* display */,
    int                 /* expose */,
    int                 /* graphics_expose */,
    int                 /* no_expose */
#endif
);

extern int NXSetUnpackGeometry(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int        /* resource */,
    Screen*             /* screen */,
    Visual*             /* visual */
#endif
);

extern int NXSetUnpackColormap(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int        /* resource */,
    unsigned int        /* entries */,
    unsigned int*       /* data */
#endif
);

extern int NXSetUnpackAlpha(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int        /* resource */,
    unsigned int        /* entries */,
    unsigned char*      /* data */
#endif
);

extern int NXFreeUnpack(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int        /* resource */
#endif
);

/*
 * A packed image is a XImage but with
 * offset field containing total amount
 * of packed image data.
 */

typedef XImage NXPackedImage;

NXPackedImage *NXCreatePackedImage(
#if NeedFunctionPrototypes
    Display*            /* display */,
    Visual*             /* visual */,
    unsigned int        /* method */,
    unsigned int        /* depth */,
    int                 /* format */,
    char*               /* data */,
    int                 /* data_length */,
    unsigned int        /* width */,
    unsigned int        /* height */,
    int                 /* bitmap_pad */,
    int                 /* bytes_per_line */
#endif
);

extern int NXDestroyPackedImage(
#if NeedFunctionPrototypes
    NXPackedImage*      /* image */
#endif
);

NXPackedImage *NXPackImage(
#if NeedFunctionPrototypes
    Display*            /* display */,
    XImage*             /* src_image */,
    unsigned int        /* method */
#endif
);

NXPackedImage *NXInPlacePackImage(
#if NeedFunctionPrototypes
    Display*            /* display */,
    XImage*             /* src_image */,
    unsigned int        /* method */
#endif
);

/*
 * GC is declared void * to get rid of mess
 * with different GC definitions in some X
 * server code (like in nxagent).
 */

extern int NXPutPackedImage(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int        /* resource */,
    Drawable            /* drawable */,
    void*               /* gc */,
    NXPackedImage*      /* image */,
    unsigned int        /* method */,
    unsigned int        /* depth */,
    int                 /* src_x */,
    int                 /* src_y */,
    int                 /* dst_x */,
    int                 /* dst_y */,
    unsigned int        /* width */,
    unsigned int        /* height */
#endif
);

/*
 * Get multiple colors with a single call by
 * pipelining X_AllocColor requests/replies.
 */

extern int NXAllocColors(
#if NeedFunctionPrototypes
    Display*            /* display */,
    Colormap            /* colormap */,
    unsigned int        /* entries */,
    XColor[]            /* screens_in_out */,
    Bool []             /* flags allocation errors */
#endif
);

extern int NXEncodeTight(
#if NeedFunctionPrototypes
    Display*            /* display */,
    Drawable            /* drawable */,
    Visual*             /* visual */,
    void*               /* gc */ ,
    XImage*             /* image */,
    unsigned int        /* method */,
    int                 /* src_x */,
    int                 /* src_y */,
    int                 /* dst_x */,
    int                 /* dst_y */,
    unsigned int        /* width */,
    unsigned int        /* height */
#endif
);

/*
 * NXEncodeJpeg and NXEncodeTight should
 * have the same interface.
 */

extern XImage *NXEncodeJpeg(
#if NeedFunctionPrototypes
    XImage*             /* src_image */,
    unsigned int        /* method */,
    unsigned int        /* quality */
#endif
);

extern XImage *NXEncodePng(
#if NeedFunctionPrototypes
    XImage*             /* src_image */,
    unsigned int        /* method */,
    NXColorTable*       /* color palette */
#endif
);

extern XImage *NXDynamicSelectPackMethod(
#if NeedFunctionPrototypes
    XImage*             /* src_image */,
    unsigned int*       /* method */,
    unsigned int        /* quality */
#endif
);

extern XImage* NXDynamicPngPack(
#if NeedFunctionPrototypes
  XImage*               /* *src_image */,
  unsigned int*         /* *method    */,
  unsigned int          /* quality */
#endif
);

extern int NXCreateColorTable(
#if NeedFunctionPrototypes
  XImage*                 /* src_image */,
  NXColorTable*           /* color_table */,
  int                     /* nb_max */
#endif
);

typedef struct _NXRDPGlyph {
    Pixmap      pixmap;
    int         x;      /* x offset */
    int         y;      /* y baseline */
    int         width;  /* glyph width */
    int         height; /* glpyph height */
} xNXRDPGlyph;

typedef struct _NXRDPText {
    unsigned int  fgColor;   /* foreground color */
    unsigned int  bgColor;   /* background color */
    int           fillStyle; /* fill style */
    unsigned int  elements;  /* number of _NXRDPGlyph structs that follow this header,
                                by the moment this information is repeated in the
                                width field of the NXPackedImage structure. */
} xNXRDPText;

#define _NXRDPTextUnpackSize(elements) ((sz_xChangeGCReq + (4 * 3)) + \
                                             ((sz_xChangeGCReq + (4 * 3)) + \
                                                   (sz_xPolyFillRectangleReq + (1 * 8))) * elements)
extern NXPackedImage *NXEncodeRDPText(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int        /* background color */,
    unsigned int        /* foreground color */,
    int                 /* fill style */,
    xNXRDPGlyph*        /* array of RDP Glyph structs */,
    int                 /* number of elements in the RDPGlyph array */
#endif
);

extern int NXCleanInPlaceImage(
#if NeedFunctionPrototypes
    XImage*
#endif
);

extern void NXMaskImage(
#if NeedFunctionPrototypes
    XImage*              /* pointer to image to mask */ ,
    unsigned int         /* method */
#endif
);

extern int NXImageCacheStat;

extern void NXCacheInit(
#if NeedFunctionPrototypes
    int                  /* entries in image cache */
#endif
);

extern void NXFreeCache(
#if NeedFunctionPrototypes
    void
#endif
);

extern XImage *NXCacheFindImage(
#if NeedFunctionPrototypes
    XImage*              /* image to find */,
    unsigned int*        /* pointer to the pack method if found */,
    unsigned char**      /* pointer to the calculated MD5 if found */
#endif
);

extern void NXCacheAddImage(
#if NeedFunctionPrototypes
    XImage*              /* image to add to cache */,
    unsigned int         /* pack method of the image to add */,
    unsigned char*       /* pointer to MD5 of the original unpacked image */
#endif
);


extern int NXGetCollectImageResource(
#if NeedFunctionPrototypes
    Display*            /* display */
#endif
);

extern int NXCollectImage(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int        /* resource */,
    Drawable            /* drawable */,
    int                 /* src_x */,
    int                 /* src_y */,
    unsigned int        /* width */,
    unsigned int        /* height */,
    unsigned long       /* plane_mask */,
    int                 /* format */
#endif
);

extern int NXGetCollectedImage(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int        /* resource */,
    XImage**            /* image */
#endif
);

extern int NXGetCollectPropertyResource(
#if NeedFunctionPrototypes
    Display*            /* display */
#endif
);

extern int NXCollectProperty(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int        /* resource */,
    Window              /* window */,
    Atom                /* property */,
    long                /* long_offset */,
    long                /* long_length */,
    Bool                /* delete */,
    Atom                /* req_type */
#endif
);

extern int NXGetCollectedProperty(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int        /* resource */,
    Atom*               /* actual_type_return */,
    int*                /* actual_format_return */,
    unsigned long*      /* nitems_return */,
    unsigned long*      /* bytes_after_return */,
    unsigned char**     /* data */
#endif
);

extern int NXGetCollectGrabPointerResource(
#if NeedFunctionPrototypes
    Display*            /* display */
#endif
);

extern int NXCollectGrabPointer(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int        /* resource */,
    Window              /* grab_window */,
    Bool                /* owner_events */,
    unsigned int        /* event_mask */,
    int                 /* pointer_mode */,
    int                 /* keyboard_mode */,
    Window              /* confine_to */,
    Cursor              /* cursor */,
    Time                /* time */
#endif
);

extern int NXGetCollectedGrabPointer(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int        /* resource */,
    int*                /* status */
#endif
);

extern int NXGetCollectInputFocusResource(
#if NeedFunctionPrototypes
    Display*            /* display */
#endif
);

extern int NXCollectInputFocus(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int        /* resource */
#endif
);

extern int NXGetCollectedInputFocus(
#if NeedFunctionPrototypes
    Display*            /* display */,
    unsigned int        /* resource */,
    Window*             /* focus_return */,
    int*                /* revert_to_return */
#endif
);

#ifdef __cplusplus
}
#endif

#endif /* NXlib_H */
