/*
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2005 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXVIEWER, NX protocol compression and NX extensions to this software   */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

/*
 * desktop.c - functions to deal with "desktop" window.
 */

#include <vncviewer.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xmu/Converters.h>
#ifdef MITSHM
#include <X11/extensions/XShm.h>
#endif

#include "NXpack.h"
#include "NXlib.h"
#include "Xproto.h"
#include "NXproto.h"

#define  NXVIEWER_DESKTOP_USES_PACKED_IMAGES

#undef  NXVIEWER_DESKTOP_TEST
#undef  NXVIEWER_DESKTOP_DEBUG

#include <X11/xpm.h>
#include "version.h"
#include "icon.h"

#define icon_width 16
#define icon_height 16

#ifdef NXVIEWER_DESKTOP_USES_PACKED_IMAGES

#define TIGHT_PACK_HEADER                       14

#include "NXpack.h"
#include "NXlib.h"
#include "Xproto.h"

static int karma, stopSize, protoMajor, protoMinor, protoPatch,
       dataLevel, streamLevel, deltaLevel;

static unsigned int loadCache, saveCache, startupCache;

unsigned int linkType;

/* nxproxy Xlib and image parameters */
int nxviewerCleanGet    = 0;
int nxviewerCleanAlloc  = 0;
int nxviewerCleanFlush  = 0;
int nxviewerCleanSend   = 0;
int nxviewerCleanImages = 0;

int nxviewerImageSplit  = 0;
int nxviewerImageMask   = 0;
int nxviewerImageFrame  = 0;

unsigned int nxviewerImageSplitMethod = 0;
unsigned int nxviewerImageMaskMethod  = 0;

unsigned int nxpackMethod, nxpackQuality;

static int NXVIEWER_MAX_IMGSZ  = 262144 - sz_xPutImageReq;
#endif

static unsigned int new_colormap[256];
static unsigned int r, b, g, or, ob, og, off;

GC gc;
GC srcGC, dstGC; /* used for debugging copyrect */
Window desktopWin;
Cursor dotCursor;
Widget form, viewport, desktop;

static Bool modifierPressed[256];

static XImage *image = NULL;

Atom nxviewer_NX;
Atom nxviewer_WM_START;

static int nxviewer_depth;
static int nx_white, nx_red, nx_black, nx_test;
static Pixmap nxviewerPixmapLogo;
static Pixmap nxIconPixmap;
static Pixmap nxIconShape;
Bool useXpmIcon = False;

#define NX_DEFAULT_ICON "nxviewer.xpm"

Atom nxviewer_WM;

static BOOL host_be;
static BOOL xserver_be;


static Cursor CreateDotCursor();
static void CopyBGR233ToScreen(CARD8 *buf, int x, int y, int width,int height);
static void HandleBasicDesktopEvent(Widget w, XtPointer ptr, XEvent *ev,
                                    Boolean *cont);
unsigned int NXLogoColor(unsigned int colorin);
void nxviewerLogo(Window win, GC gc, int scale, int width, int height);

void nxviewerSetIdentity(Window setWindow, int level);

Bool getNXIcon(Display *display, Pixmap *nxIcon, Pixmap *nxMask);
extern nxviewerUseNXTrans;
Bool nxviewerEnableSharedMemory = True;
Bool protocolTest = False;

static XtResource desktopBackingStoreResources[] = {
  {
    XtNbackingStore, XtCBackingStore, XtRBackingStore, sizeof(int), 0,
    XtRImmediate, (XtPointer) Always,
  },
};


/*
 * DesktopInitBeforeRealization creates the "desktop" widget and the viewport
 * which controls it.
 */

void
DesktopInitBeforeRealization()
{
  int i;

  /*---------------------------*/
  /* create the NX logo pixmap */
  XVisualInfo *nxVisuals;
  XVisualInfo vi;
  int nxNumVisuals;
  long mask;
  XVisualInfo pV;

  nxviewer_depth = DefaultDepth(dpy, DefaultScreen(dpy));
  mask = VisualScreenMask;
  vi.screen = DefaultScreen(dpy);
  nxVisuals = XGetVisualInfo(dpy, mask, &vi, &nxNumVisuals);
  pV = nxVisuals[0];
  r = pV.red_mask;
  g = pV.green_mask;
  b = pV.blue_mask;
  if(!pV.red_mask || !pV.green_mask || !pV.blue_mask)
  {
      nx_red   = 0xff0000;
      nx_white = 0xffffff;
      nx_test  = 0x0000ff;
  }
  else
  {
      for(or=0,off=0x800000;(r&(off>>or)) == 0; or++);
      for(og=0,off=0x800000;(g&(off>>og)) == 0; og++);
      for(ob=0,off=0x800000;(b&(off>>ob)) == 0; ob++);
      nx_red   = NXLogoColor(0xff0000);
      nx_test  = NXLogoColor(0x0000ff);
      /* nx_white = NXLogoColor(0xffffff); */
      nx_white = 0xffffff;
  }
  
  useXpmIcon = getNXIcon(dpy, &nxIconPixmap, &nxIconShape);

  if (useXpmIcon) XtVaSetValues(toplevel, XtNiconMask, nxIconShape, NULL);
  XtVaSetValues(toplevel, XtNiconPixmap, nxIconPixmap, NULL);
  if (appData.nxViewerName)
    XtVaSetValues(toplevel, XtNtitle, appData.nxViewerName, XtNiconName, appData.nxViewerName, NULL);
  /*---------------------------*/

  form = XtVaCreateManagedWidget("form", formWidgetClass, toplevel,
				 XtNborderWidth, 0,
				 XtNdefaultDistance, 0, NULL);

  viewport = XtVaCreateManagedWidget("viewport", viewportWidgetClass, form,
				     XtNborderWidth, 0,
				     NULL);

  desktop = XtVaCreateManagedWidget("desktop", coreWidgetClass, viewport,
				    XtNborderWidth, 0,
				    NULL);

  XtVaSetValues(desktop, XtNwidth, si.framebufferWidth,
		XtNheight, si.framebufferHeight, NULL);

  XtAddEventHandler(desktop, LeaveWindowMask|ExposureMask,
		    True, HandleBasicDesktopEvent, NULL);

  for (i = 0; i < 256; i++)
    modifierPressed[i] = False;

  image = NULL;

#ifdef MITSHM
  if (appData.useShm) {
    image = CreateShmImage();
    if (!image)
      appData.useShm = False;
  }
#endif

  if (!image) {
    image = XCreateImage(dpy, vis, visdepth, ZPixmap, 0, NULL,
			 si.framebufferWidth, si.framebufferHeight,
			 BitmapPad(dpy), 0);

    image->data = malloc(image->bytes_per_line * image->height);
    if (!image->data) {
      fprintf(stderr,"malloc failed\n");
      nxviewerExit(1);
    }
  }
}


/*
 * DesktopInitAfterRealization does things which require the X windows to
 * exist.  It creates some GCs and sets the dot cursor.
 */

void
DesktopInitAfterRealization()
{
  XGCValues gcv;
  XSetWindowAttributes attr;
  XWindowAttributes get_attributes;
  XSizeHints *sizehints;

  unsigned short test;

  desktopWin = XtWindow(desktop);

  if (nxviewerUseNXTrans)
  {
    nxviewerSetIdentity(desktopWin, 3);
#ifdef NXVIEWER_DESKTOP_DEBUG
    fprintf (stderr,"- using NXTrans -\n");
#endif
  }


  test = 1;
  host_be = !(BOOL)(*(unsigned char *)(&test));
  xserver_be = (ImageByteOrder(dpy) == MSBFirst);

  if ((nxviewer_WM = XInternAtom (dpy, "WM_PROTOCOLS", True)) != None)
  {
    Atom deleteWMatom = XInternAtom (dpy, "WM_DELETE_WINDOW", False);
    XSetWMProtocols(dpy, desktopWin, &deleteWMatom, 1);
  }

  gc = XCreateGC(dpy,desktopWin,0,NULL);

  XGetWindowAttributes(dpy, desktopWin, &get_attributes);

  sizehints = XAllocSizeHints();

  if (sizehints)
  {
    sizehints->flags = PPosition | PMinSize | PMaxSize;
    sizehints->min_width = sizehints->max_width = get_attributes.width;
    sizehints->min_height = sizehints->max_height = get_attributes.height;
    XSetStandardProperties(dpy,
                             desktopWin,
                               appData.nxViewerName ? appData.nxViewerName : "nxviewer",
                                  appData.nxViewerName ? appData.nxViewerName : "nxviewer",
                                     nxIconShape,
                                     0, 0, sizehints);
  }

  nxviewerLogo(desktopWin, gc, 1, get_attributes.width, get_attributes.height);
  sleep (3);
  XSetWindowBackgroundPixmap(dpy, desktopWin, None);
  
  gcv.function = GXxor;
  gcv.foreground = 0x0f0f0f0f;
  srcGC = XCreateGC(dpy,desktopWin,GCFunction|GCForeground,&gcv);
  gcv.foreground = 0xf0f0f0f0;
  dstGC = XCreateGC(dpy,desktopWin,GCFunction|GCForeground,&gcv);

  XtAddConverter(XtRString, XtRBackingStore, XmuCvtStringToBackingStore,
		 NULL, 0);

  XtVaGetApplicationResources(desktop, (XtPointer)&attr.backing_store,
			      desktopBackingStoreResources, 1, NULL);

  dotCursor = CreateDotCursor();
  attr.cursor = dotCursor;

  XChangeWindowAttributes(dpy, desktopWin, CWBackingStore|CWCursor, &attr);

#ifdef NXVIEWER_DESKTOP_USES_PACKED_IMAGES

  if (nxviewerUseNXTrans)
  {
    NXGetControlParameters(dpy, &linkType, &protoMajor, &protoMinor, &protoPatch,
                               &karma, &stopSize, &nxpackMethod, &nxpackQuality,
                                   &dataLevel, &streamLevel, &deltaLevel, &loadCache,
                                       &saveCache, &startupCache);

    if ((protoMajor < 1) || ((protoMajor == 1) && (protoMinor < 3)))
    {
      nxviewerEnableSharedMemory = False;
    }

    NXGetCleanupParameters(dpy, &nxviewerCleanGet, &nxviewerCleanAlloc,
                         &nxviewerCleanFlush, &nxviewerCleanSend,
                         &nxviewerCleanImages);
    NXGetImageParameters(dpy, &nxviewerImageSplit, &nxviewerImageMask,
                       &nxviewerImageFrame, &nxviewerImageSplitMethod,
                       &nxviewerImageMaskMethod);

    NXSetUnpackGeometry(dpy, 0, DefaultScreenOfDisplay(dpy),
                            DefaultVisual(dpy,DefaultScreen(dpy)));

    if (nxviewerEnableSharedMemory)
    {
      /*
       * Shared memory on path from viewer to 
       * X client proxy is not implemented.
       */
      
      unsigned int enableClient = 0;
      unsigned int enableServer = 1;
      unsigned int clientSegment, serverSegment;

      NXGetShmemParameters(dpy, &enableClient, &enableServer,
		           &clientSegment, &serverSegment);

      if (enableServer == False)
      {
         fprintf(stderr, "Info: Not using shared memory support in X server.\n"); 
      }
      else
      {
         fprintf(stderr, "Info: Using shared memory support in X server.\n"); 
      }
    }
  }

#endif
}

void nxviewerLogo(Window win, GC gc, int scale, int width, int height)
{
    XPoint    rect[4];
    XPoint    m[12];
    int w, h, c, w2, h2;
    extern Pixmap nxviewerPixmapLogo;

#ifdef NXVIEWER_LOGO_DEBUG
    fprintf(stderr,"nomachineLogo: begin\n");
    fprintf(stderr,"nomachineLogo: gen params are: %d %x %x %x\n",
            nxviewer_depth, nx_red,
            nx_white, nx_black);
#endif

    w = width/scale;
    h = height/scale;
    w2 = w/2;
    h2 = h/2;

    if (height > width)
    {	    
      c = w/30;
    } 
    else
    {
      c = w/48;
    }	    

    rect[0].x = 0;	       rect[0].y = 0;
    rect[1].x = 0;	       rect[1].y = h;
    rect[2].x = w;	       rect[2].y = h;
    rect[3].x = w;	       rect[3].y = 0;

    XSetFunction(dpy, gc, GXcopy);
    XSetFillStyle(dpy, gc, FillSolid);
    XSetForeground(dpy, gc, nx_black);
    XSetBackground(dpy, gc, nx_red);

    nxviewerPixmapLogo = XCreatePixmap(dpy, win, width, height, nxviewer_depth);
    if (!nxviewerPixmapLogo)
    {
      return;
    }
    
    XSetWindowBackgroundPixmap(dpy, win, nxviewerPixmapLogo);

    XFillPolygon(dpy, nxviewerPixmapLogo, gc, rect, 4, Convex, CoordModeOrigin);

#ifdef NXVIEWER_LOGO_DEBUG
    fprintf(stderr,"filled first poly\n");
#endif

    XSetForeground(dpy, gc, nx_red);
    XSetBackground(dpy, gc, nx_white);

    rect[0].x = w2-10*c;	       rect[0].y = h2-8*c;
    rect[1].x = w2-10*c;	       rect[1].y = h2+8*c;
    rect[2].x = w2+10*c;	       rect[2].y = h2+8*c;
    rect[3].x = w2+10*c;	       rect[3].y = h2-8*c;

    XFillPolygon(dpy, nxviewerPixmapLogo, gc, rect, 4, Convex, CoordModeOrigin);

#ifdef NXVIEWER_LOGO_DEBUG
    fprintf(stderr,"filled red rect\n");
#endif

    rect[0].x = w2-9*c;	       rect[0].y = h2-7*c;
    rect[1].x = w2-9*c;	       rect[1].y = h2+7*c;
    rect[2].x = w2+9*c;	       rect[2].y = h2+7*c;
    rect[3].x = w2+9*c;	       rect[3].y = h2-7*c;

    XSetForeground(dpy, gc, nx_white);
    XSetBackground(dpy, gc, nx_red);

    XFillPolygon(dpy, nxviewerPixmapLogo, gc, rect, 4, Convex, CoordModeOrigin);

    /* begin M */
    m[0].x = w2-3*c;  m[0].y = h2-5*c;
    m[1].x = w2+7*c;  m[1].y = h2-5*c;
    m[2].x = w2+7*c;  m[2].y = h2+5*c;
    m[3].x = w2+5*c;  m[3].y = h2+5*c;
    m[4].x = w2+5*c;  m[4].y = h2-3*c;
    m[5].x = w2+3*c;  m[5].y = h2-3*c;
    m[6].x = w2+3*c;  m[6].y = h2+5*c;
    m[7].x = w2+1*c;  m[7].y = h2+5*c;
    m[8].x = w2+1*c;  m[8].y = h2-3*c;
    m[9].x = w2-1*c;  m[9].y = h2-3*c;
    m[10].x = w2-1*c; m[10].y = h2+5*c;
    m[11].x = w2-3*c; m[11].y = h2+5*c;

    XSetForeground(dpy, gc, nx_red);
    XSetBackground(dpy, gc, nx_white);

    XFillPolygon(dpy, nxviewerPixmapLogo, gc, m, 12, Nonconvex, CoordModeOrigin);

    /* end M */

    /* begin ! */
    rect[0].x = w2-7*c;	       rect[0].y = h2-5*c;
    rect[1].x = w2-5*c;	       rect[1].y = h2-5*c;
    rect[2].x = w2-5*c;	       rect[2].y = h2+2*c;
    rect[3].x = w2-7*c;	       rect[3].y = h2+2*c;

    XFillPolygon(dpy, nxviewerPixmapLogo, gc, rect, 4, Convex, CoordModeOrigin);

    rect[0].x = w2-7*c;	       rect[0].y = h2+3*c;
    rect[1].x = w2-5*c;	       rect[1].y = h2+3*c;
    rect[2].x = w2-5*c;	       rect[2].y = h2+5*c;
    rect[3].x = w2-7*c;	       rect[3].y = h2+5*c;

    XFillPolygon(dpy, nxviewerPixmapLogo, gc, rect, 4, Convex, CoordModeOrigin);
    XClearWindow(dpy, win);
    
    
    //XFlush(dpy);
    XSync(dpy, False);
    
    XFreePixmap(dpy, nxviewerPixmapLogo);
    
#ifdef NXVIEWER_LOGO_DEBUG
    fprintf(stderr,"nomachineLogo: end\n");
#endif
}


/*
 * HandleBasicDesktopEvent - deal with expose and leave events.
 */

static void
HandleBasicDesktopEvent(Widget w, XtPointer ptr, XEvent *ev, Boolean *cont)
{
  int i;

  switch (ev->type) {

  case Expose:
  case GraphicsExpose:
    /* sometimes due to scrollbars being added/removed we get an expose outside
       the actual desktop area.  Make sure we don't pass it on to the RFB
       server. */

    if (ev->xexpose.x + ev->xexpose.width > si.framebufferWidth) {
      ev->xexpose.width = si.framebufferWidth - ev->xexpose.x;
      if (ev->xexpose.width <= 0) break;
    }

    if (ev->xexpose.y + ev->xexpose.height > si.framebufferHeight) {
      ev->xexpose.height = si.framebufferHeight - ev->xexpose.y;
      if (ev->xexpose.height <= 0) break;
    }

    SendFramebufferUpdateRequest(ev->xexpose.x, ev->xexpose.y,
				 ev->xexpose.width, ev->xexpose.height, False);
    break;

  case LeaveNotify:
    for (i = 0; i < 256; i++) {
      if (modifierPressed[i]) {
	SendKeyEvent(XKeycodeToKeysym(dpy, i, 0), False);
	modifierPressed[i] = False;
      }
    }
    break;
  }
}


/*
 * SendRFBEvent is an action which sends an RFB event.  It can be used in two
 * ways.  Without any parameters it simply sends an RFB event corresponding to
 * the X event which caused it to be called.  With parameters, it generates a
 * "fake" RFB event based on those parameters.  The first parameter is the
 * event type, either "ptr", "keydown", "keyup" or "key" (down&up).  For a
 * "key" event the second parameter is simply a keysym string as understood by
 * XStringToKeysym().  For a "ptr" event, the following three parameters are
 * just X, Y and the button mask (0 for all up, 1 for button1 down, 2 for
 * button2 down, 3 for both, etc).
 */

void
SendRFBEvent(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
  KeySym ks;
  char keyname[256];
  int buttonMask, x, y;

  if (appData.fullScreen && ev->type == MotionNotify) {
    if (BumpScroll(ev))
      return;
  }

  if (appData.viewOnly) return;

  if (*num_params != 0) {
    if (strncasecmp(params[0],"key",3) == 0) {
      if (*num_params != 2) {
	fprintf(stderr,
		"Invalid params: SendRFBEvent(key|keydown|keyup,<keysym>)\n");
	return;
      }
      ks = XStringToKeysym(params[1]);
      if (ks == NoSymbol) {
	fprintf(stderr,"Invalid keysym '%s' passed to SendRFBEvent\n",
		params[1]);
	return;
      }
      if (strcasecmp(params[0],"keydown") == 0) {
	SendKeyEvent(ks, 1);
      } else if (strcasecmp(params[0],"keyup") == 0) {
	SendKeyEvent(ks, 0);
      } else if (strcasecmp(params[0],"key") == 0) {
	SendKeyEvent(ks, 1);
	SendKeyEvent(ks, 0);
      } else {
	fprintf(stderr,"Invalid event '%s' passed to SendRFBEvent\n",
		params[0]);
	return;
      }
    } else if (strcasecmp(params[0],"ptr") == 0) {
      if (*num_params == 4) {
	x = atoi(params[1]);
	y = atoi(params[2]);
	buttonMask = atoi(params[3]);
	SendPointerEvent(x, y, buttonMask);
      } else if (*num_params == 2) {
	switch (ev->type) {
	case ButtonPress:
	case ButtonRelease:
	  x = ev->xbutton.x;
	  y = ev->xbutton.y;
	  break;
	case KeyPress:
	case KeyRelease:
	  x = ev->xkey.x;
	  y = ev->xkey.y;
	  break;
	default:
	  fprintf(stderr,
		  "Invalid event caused SendRFBEvent(ptr,<buttonMask>)\n");
	  return;
	}
	buttonMask = atoi(params[1]);
	SendPointerEvent(x, y, buttonMask);
      } else {
	fprintf(stderr,
		"Invalid params: SendRFBEvent(ptr,<x>,<y>,<buttonMask>)\n"
		"             or SendRFBEvent(ptr,<buttonMask>)\n");
	return;
      }

    } else {
      fprintf(stderr,"Invalid event '%s' passed to SendRFBEvent\n", params[0]);
    }
    return;
  }

  switch (ev->type) {

  case MotionNotify:
    while (XCheckTypedWindowEvent(dpy, desktopWin, MotionNotify, ev))
      ;	/* discard all queued motion notify events */

    SendPointerEvent(ev->xmotion.x, ev->xmotion.y,
		     (ev->xmotion.state & 0x1f00) >> 8);
    return;

  case ButtonPress:
    SendPointerEvent(ev->xbutton.x, ev->xbutton.y,
		     (((ev->xbutton.state & 0x1f00) >> 8) |
		      (1 << (ev->xbutton.button - 1))));
    #ifdef NXVIEWER_DESKTOP_DEBUG
    	      
    fprintf(stderr,"Event button press: x=%d y=%d button=%d\n",ev->xbutton.x, ev->xbutton.y,
		     ev->xbutton.button);
    #endif
    
    return;

  case ButtonRelease:
    SendPointerEvent(ev->xbutton.x, ev->xbutton.y,
		     (((ev->xbutton.state & 0x1f00) >> 8) &
		      ~(1 << (ev->xbutton.button - 1))));
    return;

  case KeyPress:

  case KeyRelease:
    XLookupString(&ev->xkey, keyname, 256, &ks, NULL);

    if (IsModifierKey(ks)) {
      ks = XKeycodeToKeysym(dpy, ev->xkey.keycode, 0);
      modifierPressed[ev->xkey.keycode] = (ev->type == KeyPress);
    }

    SendKeyEvent(ks, (ev->type == KeyPress));
    return;
  default:
    fprintf(stderr,"Invalid event passed to SendRFBEvent\n");
  }
}


/*
 * CreateDotCursor.
 */

static Cursor
CreateDotCursor()
{
  Cursor cursor;
  Pixmap src, msk;
  static char srcBits[] = { 0, 14,14,14, 0 };
  static char mskBits[] = { 14,31,31,31,14 };
  XColor fg, bg;

  src = XCreateBitmapFromData(dpy, DefaultRootWindow(dpy), srcBits, 5, 5);
  msk = XCreateBitmapFromData(dpy, DefaultRootWindow(dpy), mskBits, 5, 5);
  XAllocNamedColor(dpy, DefaultColormap(dpy,DefaultScreen(dpy)), "black",
		   &fg, &fg);
  XAllocNamedColor(dpy, DefaultColormap(dpy,DefaultScreen(dpy)), "white",
		   &bg, &bg);
  cursor = XCreatePixmapCursor(dpy, src, msk, &fg, &bg, 2, 2);
  XFreePixmap(dpy, src);
  XFreePixmap(dpy, msk);

  return cursor;
}

/*
 * CopyTightDataToScreen
 */
void
TwoByteSwap(unsigned char *buf, int nbytes)
{
    unsigned char c;

    for (; nbytes > 0; nbytes -= 2, buf += 2)
    {
        c = buf[0];
        buf[0] = buf[1];
        buf[1] = c;
    }
}

void
FourByteSwap(unsigned char *buf, int nbytes)
{
    unsigned char c;

    for (; nbytes > 0; nbytes -= 4, buf += 4)
    {
        c = buf[0];
        buf[0] = buf[3];
        buf[3] = c;
        c = buf[1];
        buf[1] = buf[2];
        buf[2] = c;
    }
}


void
CopyTightDataToScreen(char *buf, int x, int y, int width, int height, int filteredBpp, char filter,
                      char zlibComp, CARD8 comp_ctl, int dataLen, int numColors, char *palette)
{

    XImage *nxImage;
    /*int packMethod = 11 + zlibComp, i, collAlloc, card;*/
    int packMethod, card, collAlloc, i;
    char *newData;

    if(zlibComp)
        packMethod = PACK_RFB_TIGHT_COMPRESSED;
    else
        packMethod = PACK_RFB_TIGHT_PLAIN;

    card = myFormat.bitsPerPixel / 8;

    nxImage = XCreateImage(dpy, vis, visdepth, ZPixmap, 0, NULL, width, height, BitmapPad(dpy), 0);

    if(myFormat.bitsPerPixel == 32 && myFormat.depth == 24 && myFormat.redMax == 0xFF &&
       myFormat.greenMax == 0xFF && myFormat.blueMax == 0xFF)
        collAlloc = numColors;
    else
        collAlloc = numColors * (myFormat.bitsPerPixel / 8);

    nxImage -> data = newData = (char *)malloc(dataLen + TIGHT_PACK_HEADER);
    nxImage -> xoffset = dataLen + TIGHT_PACK_HEADER;

    #ifdef NXVIEWER_DESKTOP_TEST
    fprintf(stdout, "******CopyTightDataToScreen: data to be allocated [%d] dataLen[%d] collAlloc[%d] numColors[%d]\n",
            dataLen + TIGHT_PACK_HEADER + collAlloc * card, dataLen, collAlloc * card, (int)numColors);
    #endif

    *newData = filter                       ;  newData++;
    *newData = filteredBpp                  ;  newData++;
    *newData = myFormat.bitsPerPixel        ;  newData++;
    *newData = myFormat.depth               ;  newData++;
    *(CARD16 *)newData = myFormat.redMax    ;  newData += 2;
    *(CARD16 *)newData = myFormat.greenMax  ;  newData += 2;
    *(CARD16 *)newData = myFormat.blueMax   ;  newData += 2;
    *newData = myFormat.redShift            ;  newData++;
    *newData = myFormat.greenShift          ;  newData++;
    *newData = myFormat.blueShift           ;  newData++;
    *newData = comp_ctl                     ;  newData++;

    if(filter == rfbTightFilterPalette)
    {
        for(i = 0; i < collAlloc ; i++)
        {
            switch(card)
            {
            case 1:
                new_colormap[i] = *((CARD8 *)palette)  & 0x000000FF;
                break;
            case 2:
                new_colormap[i] = *((CARD16 *)palette) & 0x0000FFFF;
                break;
            case 4:
                new_colormap[i] = *((CARD32 *)palette) & 0x00FFFFFF;
                break;
            default:
                fprintf(stderr, "CopyTightDataToScreen: unsupported bpp [%d] in colormap\n", card);
                if(zlibComp)
                    free(buf);
                XDestroyImage(nxImage);
                return;
            }

            palette += card;
        }

        if ( nxviewerUseNXTrans )
        {
          if (host_be != xserver_be)
          {
            unsigned int swap_colormap[256];
            memcpy(swap_colormap, new_colormap, sizeof(unsigned int) * 256);

            switch(card)
            {
              case 1:
                break;
              case 2:
                TwoByteSwap((char*)swap_colormap, sizeof(unsigned int) * 256);
                break;
              case 4:
                FourByteSwap((char*)swap_colormap, sizeof(unsigned int) * 256);
                break;
            }

            NXSetUnpackColormap(dpy, 0, collAlloc, swap_colormap);
          }
          else
          {
            NXSetUnpackColormap(dpy, 0, collAlloc, new_colormap);
          }
	}
    }

    memcpy(newData, buf, dataLen);

    if(zlibComp)
        free(buf);

    #ifdef NXVIEWER_DESKTOP_TEST
    fprintf(stdout, "******CopyTightDataToScreen : x[%d] y[%d] width[%d] height[%d] bpp[%d] filter[%d] zlibComp[%d] dataLength[%d] offset[%d]\n",
            x, y, height, width, filteredBpp, filter, zlibComp, dataLen, nxImage -> xoffset);
    fprintf(stdout, "******CopyTightDataToScreen : header filter[%d] filteredBpp[%d] srcBpp[%d]\n",
            nxImage -> data[0], nxImage -> data[1], nxImage -> data[2]);
    #endif

    if ( nxviewerUseNXTrans )
    {	  
      NXPutPackedImage(dpy, 0, desktopWin, gc, nxImage, packMethod,
                           visdepth, 0, 0, x, y, width, height);
    }  

    if(nxImage -> height * nxImage -> bytes_per_line > NXVIEWER_MAX_IMGSZ)
        XSync(dpy, 0);

    XDestroyImage(nxImage);
}



/*
 * CopyDataToScreen.
 */

void
CopyDataToScreen(char *buf, int x, int y, int width, int height)
{
#ifdef NXVIEWER_DESKTOP_USES_PACKED_IMAGES

    XImage *nxPackedImage, *nxSubImage, *nxImage;
    char *cpData, *bckData;
    int imageSize, heightPiece, maxHeight, yPiece, pieces, cpHeight, reAlloc;

#endif

  if (appData.rawDelay != 0) {
    XFillRectangle(dpy, desktopWin, gc, x, y, width, height);

    XSync(dpy,False);

    usleep(appData.rawDelay * 1000);
  }

#ifdef NXVIEWER_DESKTOP_USES_PACKED_IMAGES
  if(myFormat.bitsPerPixel == 8 && nxviewerUseNXTrans)// && (width>1 || height>1))
  {
#endif
      if (!appData.useBGR233)
      {
          int h;
          int widthInBytes = width * myFormat.bitsPerPixel / 8;
          int scrWidthInBytes = si.framebufferWidth * myFormat.bitsPerPixel / 8;

          char *scr = (image->data + y * scrWidthInBytes
                       + x * myFormat.bitsPerPixel / 8);

          for (h = 0; h < height; h++)
          {
              memcpy(scr, buf, widthInBytes);
              buf += widthInBytes;
              scr += scrWidthInBytes;
          }
      }
      else
      {
          CopyBGR233ToScreen((CARD8 *)buf, x, y, width, height);
      }
#ifdef NXVIEWER_DESKTOP_USES_PACKED_IMAGES
  }
  else 
  {
      if (!appData.useBGR233)
      {
          int h;
          int widthInBytes = width * myFormat.bitsPerPixel / 8;
          int scrWidthInBytes = si.framebufferWidth * myFormat.bitsPerPixel / 8;

          char *scr = (image->data + y * scrWidthInBytes
                       + x * myFormat.bitsPerPixel / 8);

          for (h = 0; h < height; h++)
          {
              memcpy(scr, buf, widthInBytes);
              buf += widthInBytes;
              scr += scrWidthInBytes;
          }
      }
      else
      {
          CopyBGR233ToScreen((CARD8 *)buf, x, y, width, height);
      }

  }	  

  if(myFormat.bitsPerPixel == 8)
  {
      XPutImage(dpy, desktopWin, gc, image, x, y, x, y, width, height);
  }

#endif

#ifdef MITSHM
  if (appData.useShm)
  {
    XShmPutImage(dpy, desktopWin, gc, image, x, y, x, y, width, height, False);
    return;
  }
#endif

  if(myFormat.bitsPerPixel == 8)
  {
      XPutImage(dpy, desktopWin, gc, image, x, y, x, y, width, height);
      return;
  }

#ifdef NXVIEWER_DESKTOP_USES_PACKED_IMAGES

  #ifdef NXVIEWER_DESKTOP_DEBUG
  fprintf(stderr, "******CopyDataToScreen: XCreateImage depth[%d] width [%d] height[%d]\n",
              visdepth, width, height);
  #endif

  if (nxviewerUseNXTrans && nxpackMethod != NO_PACK)// && (width>1 || height>1))
  {

  if(width % 2 == 0 || BitmapPad(dpy) == myFormat.bitsPerPixel)
  {
      nxImage = XCreateImage(dpy, vis, visdepth, ZPixmap, 0, buf, width, height, BitmapPad(dpy), 0);
      reAlloc = 0;
  }
  else
  {
      XImage *cpImage;
      int h;

      nxImage = XCreateImage(dpy, vis, visdepth, ZPixmap, 0, NULL, width, height, BitmapPad(dpy), 0);
      cpImage = XCreateImage(dpy, vis, visdepth, ZPixmap, 0, buf, width, height, myFormat.bitsPerPixel, 0);

      nxImage -> data = (char *)malloc(height * nxImage -> bytes_per_line);
      cpData = nxImage -> data;
      bckData = cpImage -> data;
      for(h = 0; h < height; h++)
      {
          memmove(cpData, cpImage -> data, cpImage -> bytes_per_line);
          cpImage -> data += cpImage -> bytes_per_line;
          cpData += nxImage -> bytes_per_line;
      }
      cpImage -> data = bckData;
      XFree(cpImage);

      reAlloc = 1;
  }

  imageSize = nxImage -> bytes_per_line * nxImage -> height;

  cpData = nxImage -> data;
  cpHeight = height;

  if ((maxHeight = NXVIEWER_MAX_IMGSZ / nxImage -> bytes_per_line) >= 1)
  {
      pieces = imageSize / NXVIEWER_MAX_IMGSZ;
      if(imageSize % NXVIEWER_MAX_IMGSZ != 0)
          pieces += 1;

      yPiece = 0;

      for(; pieces > 0 && height > 0; pieces--)
      {
          if(pieces == 1)
              heightPiece = height;
          else
              heightPiece = maxHeight;

          #ifdef NXVIEWER_DESKTOP_DEBUG
          fprintf(stderr, "******CopyDataToScreen: NXPackImage pack method[%d].\n",
                      nxpackMethod);
          #endif

          nxImage -> height = heightPiece;

          nxPackedImage = NXPackImage(dpy, nxImage, nxpackMethod);

          #ifdef NXVIEWER_DESKTOP_DEBUG
          fprintf(stderr, "******CopyDataToScreen: NXPutPackedImage depth[%d] dst_x[%d] dst_y[%d] width[%d] height[%d]\n",
                      visdepth, x, y+yPiece, width, heightPiece);
          #endif

          if (nxPackedImage != NULL)
          {
              NXPutPackedImage(dpy, 0, desktopWin, gc, nxPackedImage,
                                   /* method*/ nxpackMethod, visdepth,
                                       0, 0, x, y+yPiece, width, heightPiece);
          }
          else
          {
              fprintf(stderr, "******CopyDataToScreen: WARNING! Failed to pack image with method [%d].\n",
                          nxpackMethod);

              return;
          }

          NXDestroyPackedImage(nxPackedImage);

          height = height - heightPiece;
          yPiece += heightPiece;
          nxImage -> data += heightPiece * nxImage -> bytes_per_line;
      }
  }
  else
  {
      #ifdef NXVIEWER_DESKTOP_DEBUG
      fprintf(stderr, "******CopyDataToScreen: Using XPutImage instead of NXPutPackedImage.\n");
      #endif

      XPutImage(dpy, desktopWin, gc, nxImage, 0, 0, x, y, width, height);
  }

  nxImage -> height = cpHeight;
  nxImage -> data = cpData;
  if(reAlloc)
      XDestroyImage(nxImage);
  else
      XFree(nxImage);

  }
  else /* if (nxviewerUseNXTrans && nxpackMethod != NO_PACK) */
  {
    XPutImage(dpy, desktopWin, gc, image, x, y, x, y, width, height);
  }
  
#else

  XPutImage(dpy, desktopWin, gc, image, x, y, x, y, width, height);

#endif
}

/*
 * CopyBGR233ToScreen.
 */

static void
CopyBGR233ToScreen(CARD8 *buf, int x, int y, int width, int height)
{
  int p, q;
  int xoff = 7 - (x & 7);
  int xcur;
  int fbwb = si.framebufferWidth / 8;
  CARD8 *scr1 = ((CARD8 *)image->data) + y * fbwb + x / 8;
  CARD8 *scrt;
  CARD8 *scr8 = ((CARD8 *)image->data) + y * si.framebufferWidth + x;
  CARD16 *scr16 = ((CARD16 *)image->data) + y * si.framebufferWidth + x;
  CARD32 *scr32 = ((CARD32 *)image->data) + y * si.framebufferWidth + x;

  switch (visbpp) {

    /* thanks to Chris Hooper for single bpp support */

  case 1:
    for (q = 0; q < height; q++) {
      xcur = xoff;
      scrt = scr1;
      for (p = 0; p < width; p++) {
	*scrt = ((*scrt & ~(1 << xcur))
		 | (BGR233ToPixel[*(buf++)] << xcur));

	if (xcur-- == 0) {
	  xcur = 7;
	  scrt++;
	}
      }
      scr1 += fbwb;
    }
    break;

  case 8:
    for (q = 0; q < height; q++) {
      for (p = 0; p < width; p++) {
	*(scr8++) = BGR233ToPixel[*(buf++)];
      }
      scr8 += si.framebufferWidth - width;
    }
    break;

  case 16:
    for (q = 0; q < height; q++) {
      for (p = 0; p < width; p++) {
	*(scr16++) = BGR233ToPixel[*(buf++)];
      }
      scr16 += si.framebufferWidth - width;
    }
    break;

  case 32:
    for (q = 0; q < height; q++) {
      for (p = 0; p < width; p++) {
	*(scr32++) = BGR233ToPixel[*(buf++)];
      }
      scr32 += si.framebufferWidth - width;
    }
    break;
  }
}

/*
 * NXLogoColor
 *
 * take color values in rgb24 0xff0000 0x00ff00 0x0000ff
 * and find an equivalent for your visual.
 */

unsigned int NXLogoColor(unsigned int colorin)
{
      
    int cr=0, cg=0, cb=0;

    cr = (colorin >> (or))&r;
    cg = (colorin >> (og-8))&g;
    cb = (colorin >> (ob-16))&b;
    return cr| cg| cb;
}

void nxviewerSetIdentity(Window setWindow, int level)
{
    Atom IdentityAtom;
    int Type[4];
    Type[0] = NXVIEWER_SESSION;
    Type[1] = NXVIEWER_MAJOR_VERSION;
    Type[2] = NXVIEWER_MINOR_VERSION;
    Type[3] = NXVIEWER_RELEASE;

    IdentityAtom = XInternAtom(dpy, "NX_IDENTITY", False);

    XChangeProperty(dpy,
                    setWindow,
                    IdentityAtom,
                    XA_ATOM,
                    sizeof(int)*8,
                    PropModeReplace,
                    (unsigned char*)&Type,
                    4);

   if (level > 1)
   {
       /*
       * Create agent ready atom, first level of autentication
       *
      */
     nxviewer_NX = XInternAtom (dpy, "NX_CUT_BUFFER_SERVER", False);
      XSetSelectionOwner(dpy, nxviewer_NX, setWindow, CurrentTime);
   }

   if (level > 2)
   {
      /*
       * Create Window manager ready ato, last level of autentication
       *
      */
      nxviewer_WM_START = XInternAtom (dpy, "WM_NX_READY", False);
      XSetSelectionOwner(dpy, nxviewer_WM_START, setWindow, CurrentTime);
   }
}


void checkNXTrans()
{
  char *widget_name = XtName(toplevel);

  if ( !strcmp( widget_name, "nxviewer" ))
  {
     /* fprintf(stderr,"XtName -> %s \n", widget_name); */
     nxviewerUseNXTrans = TRUE;
     protocolTest = TRUE;
     NXGetControlParameters(dpy, &linkType, &protoMajor, &protoMinor, &protoPatch,
                                &karma, &stopSize, &nxpackMethod, &nxpackQuality,
                                    &dataLevel, &streamLevel, &deltaLevel, &loadCache,
                                        &saveCache, &startupCache);

     if (appData.noSharedMemory)
     {
       nxviewerEnableSharedMemory = False;
     }

     protocolTest = FALSE;

     return;
  }
}

Bool getNXIcon(Display *display, Pixmap *nxIcon, Pixmap *nxMask)
{
  char *env_path = getenv("PATH");
  int lenght_env_path = 0;
  char icon_filename [256];
  char default_path [256];
  char *icon_path = malloc( strlen(env_path) + sizeof(icon_filename) );
  FILE *icon_fp;
  int status;
  Bool existXpmIcon = False;

  Pixmap IconPixmap;
  Pixmap IconShape;

  if (env_path == NULL)
    lenght_env_path = 0;
  else
    lenght_env_path = strlen(env_path) + 1;
  strncpy(icon_filename, "", 255);
  strncpy(default_path, "", 255);

  strcat(icon_filename, NX_DEFAULT_ICON);
  strcat(default_path,"/usr/NX/share/images/");
  strcat(default_path,icon_filename);

  if ((icon_fp = fopen(default_path, "r")) == NULL)
  {
    char *s;
    char *temp_path = malloc(lenght_env_path + strlen(icon_filename) );
    char *temp_path1 = malloc(lenght_env_path + strlen(icon_filename) );

    strncpy(temp_path, env_path, strlen(env_path));
    strncpy(temp_path1, "", lenght_env_path + strlen(icon_filename) );

    while ( strlen(temp_path) > 0)
    {
       s = strpbrk (temp_path, ":");
       if (s == NULL) break;

       strncpy ( temp_path1, temp_path , strlen(temp_path) - strlen(s) );
       strncat ( temp_path1, "/", 1);
       strncat ( temp_path1, icon_filename, strlen(icon_filename));
       if ((icon_fp = fopen(temp_path1, "r")) != NULL)
       {
          fclose (icon_fp);
          existXpmIcon = True;
          strcpy(icon_path,temp_path1);
          break;
       }
       strncpy(temp_path1, "", lenght_env_path + strlen(icon_filename) );
       strncpy(temp_path1, s + 1, strlen(s)-1);
       strncpy(temp_path, "", lenght_env_path + strlen(icon_filename) );
       strcpy(temp_path, temp_path1 );
       strncpy(temp_path1, "", lenght_env_path + strlen(icon_filename) );
     }
     free(temp_path);
     free(temp_path1);
  }
  else
  {
     fclose (icon_fp);
     existXpmIcon = True;
     strcpy(icon_path, default_path);
  }

  if (existXpmIcon)
  {
     status = XpmReadFileToPixmap(display,
		  	       DefaultRootWindow(display),
			       icon_path,
			       &IconPixmap,
			       &IconShape,
			       NULL);
     if (status != XpmSuccess) {
        fprintf(stderr, "XpmError: %s\n", XpmGetErrorString(status));
        /* exit(1); */
	existXpmIcon = False;
     }
  }

  if (!existXpmIcon)
  {
     IconPixmap = XCreatePixmapFromBitmapData(display,
					DefaultRootWindow(display),
					(char *)icon_bits,
					icon_width,
					icon_height,
                                        nx_red,
                                        nx_white,
					DefaultDepth(display, DefaultScreen(display)));
     IconShape = 0;
  }

  free(icon_path);
  *nxIcon = IconPixmap;
  *nxMask = IconShape;
  return existXpmIcon;
}
