/*
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2005 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXVIEWER, NX protocol compression and NX extensions to this software   */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

/*
 * hextile.c - handle hextile encoding.
 *
 * This file shouldn't be compiled directly.  It is included multiple times by
 * rfbproto.c, each time with a different definition of the macro BPP.  For
 * each value of BPP, this file defines a function which handles a hextile
 * encoded rectangle with BPP bits per pixel.
 */

#define HandleHextileBPP CONCAT2E(HandleHextile,BPP)
#define nonNXProt_HandleHextileBPP CONCAT2E(nonNXProt_HandleHextile,BPP)
#define CARDBPP CONCAT2E(CARD,BPP)
#define GET_PIXEL CONCAT2E(GET_PIXEL,BPP)

#ifdef NXVIEWER_HEXTILE_USES_PACKED_IMAGES

static Bool
HandleHextileBPP (int rx, int ry, int rw, int rh)
{
  CARDBPP bg, fg;
  XGCValues gcv;
  int i;
  CARD8 *ptr;
  int x, y, w, h;
  int sx, sy, sw, sh;
  CARD8 subencoding;
  CARD8 nSubrects;

  XImage *nxImage;
  CARDBPP color;
  int hextileSize, imgSize, linesPerPiece, pieceHeight, yPiece;
  int tiles, totalTiles;
  int tilesPerWidth, tilesPerHeight;
  int tilesHPerPiece, tilesPerPiece;

  char *cpBuffer = buffer;
  char *hexBuffer = cpBuffer;

  nxImage = XCreateImage(dpy, vis, visdepth, ZPixmap, 0, NULL, rw, rh, BitmapPad(dpy), 0);
  imgSize = nxImage -> bytes_per_line * rh;

  if(imgSize > NXVIEWER_MAX_IMGSZ)
  {
      tilesPerWidth = rw >> 4;
      if(rw % 16 != 0)
          tilesPerWidth += 1;
      tilesPerHeight = rh >> 4;
      if(rh % 16 != 0)
          tilesPerHeight += 1;

      if(rh <= 16)
      {
          fprintf(stderr, "Image too large [%d] Nr lines [%d] Cannot cut!!!\n", imgSize, rh);
          return False;
      }

      tilesHPerPiece = NXVIEWER_MAX_IMGSZ / (nxImage -> bytes_per_line << 4);
      tilesPerPiece = tilesHPerPiece * tilesPerWidth;
      linesPerPiece = tilesHPerPiece << 4;
  }
  else
  {
      tilesPerPiece = tilesPerHeight * tilesPerWidth;
      linesPerPiece = rh;
  }

  hextileSize = 0;
  tiles = 0;
  yPiece = ry;

  for (y = ry; y < ry+rh; y += 16)
  {
    for (x = rx; x < rx+rw; x += 16)
    {
      tiles ++;

      w = h = 16;
      if (rx+rw - x < 16)
	w = rx+rw - x;
      if (ry+rh - y < 16)
	h = ry+rh - y;

      if (!ReadFromRFBServer((char *)&subencoding, 1))
	return False;

      subencoding &= 31;

      if (subencoding & rfbHextileRaw)
      {

        subencoding &= 1;
        *cpBuffer = subencoding;
        cpBuffer++;
        hextileSize++;

        if (!ReadFromRFBServer(cpBuffer, w * h * (BPP / 8)))
            return False;

        #if(BPP == 32)

        for (i = 0; i < w * h; i++)
        {
            GET_PIXEL(color, cpBuffer);
            *(CARDBPP *)(cpBuffer - 4) = color & 0x00FFFFFF;
        }

        #else

        cpBuffer += w * h *(BPP / 8);

        #endif

        hextileSize += w * h * (BPP/8);

	continue;
      }
      else
      {
          *cpBuffer = subencoding;
          cpBuffer++;
          hextileSize++;
      }

      if (subencoding & rfbHextileBackgroundSpecified)
      {
	if (!ReadFromRFBServer((char *)&bg, sizeof(bg)))
            return False;

        #if (BPP == 32)
        bg &= 0x00FFFFFF;
        #endif

        #if (BPP == 8)
        if (appData.useBGR233)
            bg = BGR233ToPixel[bg];
        #endif

        *(CARDBPP *)cpBuffer = (CARDBPP)bg;
        cpBuffer += BPP / 8;
        hextileSize += BPP / 8;
      }

      if (subencoding & rfbHextileForegroundSpecified)
      {
	if (!ReadFromRFBServer((char *)&fg, sizeof(fg)))
	  return False;

        #if (BPP == 32)
        fg &= 0x00FFFFFF;
        #endif

        #if (BPP == 8)
        if (appData.useBGR233)
            fg = BGR233ToPixel[fg];
        #endif

        *(CARDBPP *)cpBuffer = (CARDBPP)fg;
        cpBuffer += BPP / 8;
        hextileSize += BPP / 8;
      }

      if (!(subencoding & rfbHextileAnySubrects))
      {
          continue;
      }

      if (!ReadFromRFBServer((char *)&nSubrects, 1))
	return False;

      *cpBuffer = nSubrects;
      cpBuffer++;
      hextileSize++;

      if (subencoding & rfbHextileSubrectsColoured)
      {
        if (!ReadFromRFBServer(cpBuffer, nSubrects * (2 + (BPP / 8))))
            return False;

        #if (BPP == 32)

        for (i = 0; i < nSubrects; i++)
        {
            GET_PIXEL(fg, cpBuffer);
            *(CARDBPP *)(cpBuffer - 4) = fg & 0x00FFFFFF;
            cpBuffer += 2;
        }

        #else

        cpBuffer += nSubrects * (2 + (BPP / 8));

        #endif

        hextileSize += nSubrects * (2 + (BPP / 8));
      }
      else
      {
        if (!ReadFromRFBServer(cpBuffer, nSubrects * 2))
	  return False;
        cpBuffer    += nSubrects * 2;
        hextileSize += nSubrects * 2;
      }
    }
    if(tiles == tilesPerPiece || (x + 16 >= rx + rw && y + 16 >= ry+rh))
    {
        if((x + 16 >= rx + rw) && (y + 16 >= ry+rh))
        {
            pieceHeight = rh+ry - yPiece;
        }
        else
        {
            pieceHeight = linesPerPiece;
        }

        #ifdef NXVIEWER_HEXTILE_DEBUG
        fprintf(stderr, "New Hextile piece. Size [%d] Height[%d] TotalHeight[%d] Width [%d] Rx[%d] Ry[%d] SubEncoding[%d] TilesT[%d]\n",
                hextileSize, pieceHeight, rh, rw, rx, ry, subencoding, tiles);
        fprintf(stderr, "size fin [%d] size comp[%d]\n", nxImage -> bytes_per_line * pieceHeight, hextileSize);
        #endif

        nxImage -> height = pieceHeight;
        nxImage -> width = rw;
        nxImage -> data = hexBuffer;
        nxImage -> xoffset = hextileSize;

        NXPutPackedImage(dpy, 0, desktopWin, gc, nxImage, PACK_RFB_HEXTILE,
                             visdepth, 0, 0, rx, yPiece , rw, pieceHeight);

        hextileSize = 0;
        tiles = 0;
        cpBuffer = buffer;
        hexBuffer = cpBuffer;
        yPiece += linesPerPiece;
    }
    
  }

  
  XFree(nxImage);

  return True;
}

static Bool
nonNXProt_HandleHextileBPP (int rx, int ry, int rw, int rh)
{
  CARDBPP bg, fg;
  XGCValues gcv;

  int hextileSize;
  int i;
  CARD8 *ptr;
  int x, y, w, h;
  int sx, sy, sw, sh;
  CARD8 subencoding;
  CARD8 nSubrects;

  char *cpBuffer = buffer;
  char *hexBuffer = cpBuffer;

  for (y = ry; y < ry+rh; y += 16)
  {
    for (x = rx; x < rx+rw; x += 16)
    {
      w = h = 16;
      if (rx+rw - x < 16)
	w = rx+rw - x;
      if (ry+rh - y < 16)
	h = ry+rh - y;

      if (!ReadFromRFBServer((char *)&subencoding, 1))
	return False;


      if (subencoding & rfbHextileRaw)
      {

	if (!ReadFromRFBServer(buffer, w * h * (BPP / 8)))
	  return False;

        CopyDataToScreen(buffer, x, y, w, h);

	continue;
      }
      else
      {
          *cpBuffer = subencoding;
          cpBuffer++;
          hextileSize++;
      }

      if (subencoding & rfbHextileBackgroundSpecified)
      {
	if (!ReadFromRFBServer((char *)&bg, sizeof(bg)))
            return False;
      }

      #if (BPP == 8)
      if (appData.useBGR233)
          gcv.foreground = BGR233ToPixel[bg];
      else
      #endif
          gcv.foreground = bg;

      XChangeGC(dpy, gc, GCForeground, &gcv);
      XFillRectangle(dpy, desktopWin, gc, x, y, w, h);

      if (subencoding & rfbHextileForegroundSpecified)
      {
	if (!ReadFromRFBServer((char *)&fg, sizeof(fg)))
	  return False;
      }

      if (!(subencoding & rfbHextileAnySubrects))
      {
          continue;
      }

      if (!ReadFromRFBServer((char *)&nSubrects, 1))
	return False;

      ptr = (CARD8 *)buffer;

      if (subencoding & rfbHextileSubrectsColoured)
      {
        if (!ReadFromRFBServer(buffer, nSubrects * (2 + (BPP / 8))))
	  return False;
	for (i = 0; i < nSubrects; i++) {
	  GET_PIXEL(fg, ptr);
	  sx = rfbHextileExtractX(*ptr);
	  sy = rfbHextileExtractY(*ptr);
	  ptr++;
	  sw = rfbHextileExtractW(*ptr);
	  sh = rfbHextileExtractH(*ptr);
	  ptr++;
          #if (BPP == 8)
	  if (appData.useBGR233)
	    gcv.foreground = BGR233ToPixel[fg];
	  else
          #endif
	    gcv.foreground = fg;

	  XChangeGC(dpy, gc, GCForeground, &gcv);
          XFillRectangle(dpy, desktopWin, gc, x+sx, y+sy, sw, sh);
        }
      }
      else
      {
	if (!ReadFromRFBServer(buffer, nSubrects * 2))
	  return False;

        #if (BPP == 8)
	if (appData.useBGR233)
	  gcv.foreground = BGR233ToPixel[fg];
	else
        #endif
	  gcv.foreground = fg;

	XChangeGC(dpy, gc, GCForeground, &gcv);

	for (i = 0; i < nSubrects; i++) {
	  sx = rfbHextileExtractX(*ptr);
	  sy = rfbHextileExtractY(*ptr);
	  ptr++;
	  sw = rfbHextileExtractW(*ptr);
	  sh = rfbHextileExtractH(*ptr);
	  ptr++;
          XFillRectangle(dpy, desktopWin, gc, x+sx, y+sy, sw, sh);
        }
      }
    }
  }

  return True;
}

#else

static Bool
HandleHextileBPP (int rx, int ry, int rw, int rh)
{
  CARDBPP bg, fg;
  XGCValues gcv;
  int i;
  CARD8 *ptr;
  int x, y, w, h;
  int sx, sy, sw, sh;
  CARD8 subencoding;
  CARD8 nSubrects;

  for (y = ry; y < ry+rh; y += 16)
  {
    for (x = rx; x < rx+rw; x += 16)
    {
      w = h = 16;
      if (rx+rw - x < 16)
	w = rx+rw - x;
      if (ry+rh - y < 16)
	h = ry+rh - y;

      if (!ReadFromRFBServer((char *)&subencoding, 1))
	return False;


      if (subencoding & rfbHextileRaw)
      {

	if (!ReadFromRFBServer(buffer, w * h * (BPP / 8)))
	  return False;

        CopyDataToScreen(buffer, x, y, w, h);

	continue;
      }
      else
      {
          *cpBuffer = subencoding;
          cpBuffer++;
          hextileSize++;
      }

      if (subencoding & rfbHextileBackgroundSpecified)
      {
	if (!ReadFromRFBServer((char *)&bg, sizeof(bg)))
            return False;
      }

      #if (BPP == 8)
      if (appData.useBGR233)
          gcv.foreground = BGR233ToPixel[bg];
      else
      #endif
          gcv.foreground = bg;

      XChangeGC(dpy, gc, GCForeground, &gcv);
      XFillRectangle(dpy, desktopWin, gc, x, y, w, h);

      if (subencoding & rfbHextileForegroundSpecified)
      {
	if (!ReadFromRFBServer((char *)&fg, sizeof(fg)))
	  return False;
      }

      if (!(subencoding & rfbHextileAnySubrects))
      {
          continue;
      }

      if (!ReadFromRFBServer((char *)&nSubrects, 1))
	return False;

      ptr = (CARD8 *)buffer;

      if (subencoding & rfbHextileSubrectsColoured)
      {
        if (!ReadFromRFBServer(buffer, nSubrects * (2 + (BPP / 8))))
	  return False;
	for (i = 0; i < nSubrects; i++) {
	  GET_PIXEL(fg, ptr);
	  sx = rfbHextileExtractX(*ptr);
	  sy = rfbHextileExtractY(*ptr);
	  ptr++;
	  sw = rfbHextileExtractW(*ptr);
	  sh = rfbHextileExtractH(*ptr);
	  ptr++;
          #if (BPP == 8)
	  if (appData.useBGR233)
	    gcv.foreground = BGR233ToPixel[fg];
	  else
          #endif
	    gcv.foreground = fg;

	  XChangeGC(dpy, gc, GCForeground, &gcv);
          XFillRectangle(dpy, desktopWin, gc, x+sx, y+sy, sw, sh);
        }
      }
      else
      {
	if (!ReadFromRFBServer(buffer, nSubrects * 2))
	  return False;

        #if (BPP == 8)
	if (appData.useBGR233)
	  gcv.foreground = BGR233ToPixel[fg];
	else
        #endif
	  gcv.foreground = fg;

	XChangeGC(dpy, gc, GCForeground, &gcv);

	for (i = 0; i < nSubrects; i++) {
	  sx = rfbHextileExtractX(*ptr);
	  sy = rfbHextileExtractY(*ptr);
	  ptr++;
	  sw = rfbHextileExtractW(*ptr);
	  sh = rfbHextileExtractH(*ptr);
	  ptr++;
          XFillRectangle(dpy, desktopWin, gc, x+sx, y+sy, sw, sh);
        }
      }
    }
  }

  return True;
}

#endif

