/*
 *  Panasonic CD drive control driver
 *  (C) 2005 Hiroyuki Ikezoe <ikezoe@good-day.co.jp>
 *
 *  derived from pcc_acpi.c, Copyright (C) 2004 Hiroshi Miura <miura@da-cha.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as 
 *  publicshed by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *---------------------------------------------------------------------------
 *
 */

#define ACPI_PCC_CD_VERSION	"0.0.1"

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/proc_fs.h>
#include <linux/ctype.h>
#include <linux/seq_file.h>
#include <asm/uaccess.h>
#include <acpi/acpi_bus.h>
#include <acpi/acpi_drivers.h>
#include <linux/input.h>

#include "seq_template.h"

ACPI_MODULE_NAME("pcc_acpi_cd")

MODULE_AUTHOR("Hiroyuki Ikezoe");
MODULE_DESCRIPTION("ACPI CD Drive driver for Panasonic Lets Note laptops");
MODULE_LICENSE("GPL");

#define LOGPREFIX "pcc_acpi_cd: "

/* Lets note cd drive on/off key */
#define CDD_ON 0x01
#define CDD_OFF 0x03

/*******************************************************************
 *
 * definitions for /proc/ interface
 *
 *******************************************************************/
#define ACPI_PCC_CLASS		"pcc"
#define PROC_PCC		ACPI_PCC_CLASS

#define ACPI_PCC_INPUT_PHYS	"panasonic/hkey0"

#define PROC_STR_MAX_LEN  8

static int __devinit acpi_pcc_cd_add (struct acpi_device *device);
static int __devexit acpi_pcc_cd_remove (struct acpi_device *device, int type);
static int acpi_pcc_cd_resume (struct acpi_device *device, int state);
static int acpi_pcc_cd_suspend(struct acpi_device *device, int state);
static int acpi_pcc_cd_match (struct acpi_device *device, struct acpi_driver *driver);

#define ACPI_PCC_CD_DRIVER_NAME	"PCC CD Drive Driver"
#define ACPI_PCC_CD_DEVICE_NAME	"PCC CD Drive"
#define ACPI_PCC_CD_CLASS	"cd"

static struct acpi_driver acpi_pcc_cd_driver = {
	.name =		ACPI_PCC_CD_DRIVER_NAME,
	.class =	ACPI_PCC_CD_CLASS,
	.ops =		{
				.add =		acpi_pcc_cd_add,
				.remove =	__devexit_p(acpi_pcc_cd_remove),
				.match  =       acpi_pcc_cd_match,
				.resume =       acpi_pcc_cd_resume,
				.suspend =      acpi_pcc_cd_suspend,
			},
};

struct acpi_pcc_cd {
	acpi_handle		handle;
	unsigned long		status;
};

struct proc_dir_entry* acpi_pcc_cd_dir = NULL;

/* --------------------------------------------------------------------------
                           method access functions
   -------------------------------------------------------------------------- */
static int acpi_pcc_write_cd_sset(struct acpi_pcc_cd *pcc_cd, int state)
{
	acpi_handle sb_handle;
	acpi_handle h_dummy;
	acpi_status status = AE_OK;
	union acpi_object	arg0 = {ACPI_TYPE_INTEGER};
	struct acpi_object_list	args = {1, &arg0};
	
	ACPI_FUNCTION_TRACE("acpi_pcc_write_cd_sset");

	switch(state) {
	case 0x00: 
		status = acpi_get_handle(NULL, "\\_SB", &sb_handle);
		if (ACPI_SUCCESS(status)) {
			if (ACPI_SUCCESS(acpi_get_handle(sb_handle, "CDON", &h_dummy))) {
				status = acpi_evaluate_object(sb_handle, "CDON", NULL, NULL);
			} else {
				arg0.integer.value = 0x04;
				status = acpi_evaluate_integer(sb_handle, "SSTA", &args, NULL);
			}
			if (ACPI_FAILURE(status)) {
				ACPI_DEBUG_PRINT((ACPI_DB_ERROR, "Couldn't turn on cd drive\n"));
			}
		}
		break;
	case 0x0f:
		arg0.integer.value = 0x01; /* dummy */
		status = acpi_evaluate_object(pcc_cd->handle, "_EJ0", &args, NULL);
		if (ACPI_FAILURE(status)) {
			ACPI_DEBUG_PRINT((ACPI_DB_ERROR, "Couldn't turn off cd drive\n"));
		}
		break;
	default:
		break;
	}

	return_VALUE(status == AE_OK);
}


/* --------------------------------------------------------------------------
                       user interface functions
   -------------------------------------------------------------------------- */

static int acpi_pcc_cd_state_show(struct seq_file *seq, void *offset)
{
	struct acpi_pcc_cd *pcc_cd = (struct acpi_pcc_cd *) seq->private;
	unsigned long state;
	acpi_status status = AE_OK;
	
	ACPI_FUNCTION_TRACE("acpi_pcc_cd_state_show");

	if (!pcc_cd)
		return 0;

	status = acpi_evaluate_integer(pcc_cd->handle, "_STA", NULL, &state);
	seq_printf(seq, "0x%02lx\n", state);

	return_VALUE(0);
}


static ssize_t acpi_pcc_write_cd_state(struct file *file, const char __user *buffer,
				       size_t count, loff_t *ppos)
{
	struct seq_file		*seq    = (struct seq_file *)file->private_data;
	struct acpi_pcc_cd	*pcc_cd = (struct acpi_pcc_cd *)seq->private;
	char str[12] = {0};
	u32 state = 0;

	ACPI_FUNCTION_TRACE("acpi_pcc_write_cd_state");

	if (count + 1 > sizeof str)
		return_VALUE(-EINVAL);

	if (copy_from_user(str, buffer, count))
		return_VALUE(-EFAULT);

	str[count] = 0;
	state = simple_strtoul(str, NULL, 0);

	switch (state)
	{
	case 0:
		acpi_pcc_write_cd_sset(pcc_cd, 15);
		break;
	case 15:
		acpi_pcc_write_cd_sset(pcc_cd, 0);
		break;
	default:
		return_VALUE(-EFAULT);
		break;
	}

	return_VALUE(count);
}


/* --------------------------------------------------------------------------
                              FS Interface (/proc)
   -------------------------------------------------------------------------- */
/* oepn proc file fs*/
SEQ_OPEN_FS(acpi_pcc_cd_state_open_fs, acpi_pcc_cd_state_show);

typedef struct file_operations fops_t;
static fops_t acpi_pcc_cd_state_fops = SEQ_FILEOPS_RW (acpi_pcc_cd_state_open_fs, acpi_pcc_write_cd_state);

/* --------------------------------------------------------------------------
                             CD on/off button 
   -------------------------------------------------------------------------- */
void acpi_pcc_cdb_notify(acpi_handle handle, u32 event, void *data)
{
	struct acpi_device	*device = NULL;
	unsigned long state;
	acpi_status status = AE_OK;

	ACPI_FUNCTION_TRACE("acpi_pcc_cdb_notify");

	if (acpi_bus_get_device(handle, &device))
		return_VOID;
	
	status = acpi_evaluate_integer(handle, "_STA", NULL, &state);

	if (ACPI_SUCCESS(status)) {
		switch(event) {
		case CDD_ON:
			acpi_bus_generate_event(device, event, state);
			break;
		case CDD_OFF: /* this event is only for Y2 series. */
			acpi_bus_generate_event(device, event, state);
			break;
		default:
			acpi_bus_generate_event(device, event, state);
			break;
		}
	}

	return_VOID;
}

/* --------------------------------------------------------------------------
                         module init
   -------------------------------------------------------------------------- */
/* --------------------------------------------------------------------------
                             CD drive on/off button 
   -------------------------------------------------------------------------- */
static int acpi_pcc_cd_add (struct acpi_device *device)
{
	acpi_status		status = AE_OK;
	struct proc_dir_entry	*proc;
	struct acpi_pcc_cd	*pcc_cd = NULL;

	ACPI_FUNCTION_TRACE("acpi_pcc_cd_add");

	if (!device) {
		return_VALUE(-EINVAL);
	}

	pcc_cd = kmalloc(sizeof(struct acpi_pcc_cd), GFP_KERNEL);
	if (!pcc_cd) {
		ACPI_DEBUG_PRINT((ACPI_DB_ERROR, "Couldn't allocate mem for pcc cd driver"));
		return_VALUE(-ENOMEM);
	}

	memset(pcc_cd, 0, sizeof(struct acpi_pcc_cd));

	pcc_cd->handle = device->handle;
	strcpy(acpi_device_name(device), ACPI_PCC_CD_DEVICE_NAME);
	strcpy(acpi_device_class(device), ACPI_PCC_CD_CLASS);
	acpi_driver_data(device) = pcc_cd;

	if (!acpi_pcc_cd_dir) {
		acpi_pcc_cd_dir = proc_mkdir("pcc_cd", acpi_root_dir);
		acpi_pcc_cd_dir->owner = THIS_MODULE;
	}

	proc = create_proc_entry("cd_state", S_IFREG | S_IRUGO | S_IWUSR, acpi_pcc_cd_dir); 
	if (likely(proc)) {
		proc->proc_fops = &acpi_pcc_cd_state_fops;
		proc->data = pcc_cd;
		proc->owner = THIS_MODULE;
	} else {
		remove_proc_entry("cd_state", acpi_pcc_cd_dir);
		return -ENODEV;
	}

	if (ACPI_SUCCESS(status)) {
		status = acpi_install_notify_handler(pcc_cd->handle, ACPI_SYSTEM_NOTIFY, acpi_pcc_cdb_notify, pcc_cd);
		if (ACPI_FAILURE(status)) {
			ACPI_DEBUG_PRINT((ACPI_DB_ERROR, "Error installing notify handler\n"));
			return_VALUE(-ENODEV);
		}
	}
	
	return_VALUE(0);
}

static int acpi_pcc_cd_remove(struct acpi_device *device, int type)
{
	acpi_status		status  = AE_OK;
	struct acpi_pcc_cd	*pcc_cd = acpi_driver_data(device);

	ACPI_FUNCTION_TRACE("acpi_pcc_cd_remove");

	if (!device || !pcc_cd)
		return_VALUE(-EINVAL);

	if (acpi_pcc_cd_dir)
		remove_proc_entry("cd_state", acpi_pcc_cd_dir);

	status = acpi_remove_notify_handler(pcc_cd->handle,
		      	ACPI_SYSTEM_NOTIFY, acpi_pcc_cdb_notify);

	if (ACPI_FAILURE(status))
		ACPI_DEBUG_PRINT((ACPI_DB_ERROR, "Error removing notify handler\n"));

	kfree(pcc_cd);

	return_VALUE(status == AE_OK);
}

static int acpi_pcc_cd_resume(struct acpi_device *device, int state)
{
	acpi_status	    status = AE_OK;
	acpi_handle sb_handle;
	union acpi_object	arg0 = {ACPI_TYPE_INTEGER};
/*	struct acpi_object_list	args = {1, &arg0}; */

	ACPI_FUNCTION_TRACE("acpi_pcc_cd_resume");

	if (!device) { return_VALUE(-EINVAL); }

	status = acpi_get_handle(NULL, "\\_SB", &sb_handle);

	if (ACPI_SUCCESS(status)) {
		arg0.integer.value = 0x04;
		status = acpi_evaluate_object(sb_handle, "DWAK", NULL, NULL);
	}

	return_VALUE(status == AE_OK ? 0 : -EINVAL);
}

static int acpi_pcc_cd_suspend(struct acpi_device *device, int state)
{
	acpi_status	    status = AE_OK;
	acpi_handle sb_handle;
	union acpi_object	arg0 = {ACPI_TYPE_INTEGER};
	struct acpi_object_list	args = {1, &arg0};
	
	ACPI_FUNCTION_TRACE("acpi_pcc_cd_suspend");

	if (!device) { return_VALUE(-EINVAL); }

	status = acpi_get_handle(NULL, "\\_SB", &sb_handle);

	if (ACPI_SUCCESS(status)) {
		arg0.integer.value = 0x04;
		status = acpi_evaluate_object(sb_handle, "DPTS", &args, NULL);
	}

	return_VALUE(status == AE_OK ? 0 : -EINVAL);
}

static int acpi_pcc_cd_match (struct acpi_device *device, struct acpi_driver *driver)
{
	acpi_handle		h_dummy1;
	acpi_handle		h_dummy2;

	ACPI_FUNCTION_TRACE("acpi_pcc_cd_match");

	if (!device || !driver)
		return_VALUE(-EINVAL);

	if (ACPI_SUCCESS(acpi_get_handle(device->handle, "_EJ0", &h_dummy1)) &&
	    ACPI_SUCCESS(acpi_get_handle(device->handle, "_STA", &h_dummy2)))
		return_VALUE(0);

	return_VALUE(-ENODEV);
}

static int __init acpi_pcc_init(void)
{
	int result = 0;
 
	ACPI_FUNCTION_TRACE("acpi_pcc_init");
	
	if (acpi_disabled) {
		return_VALUE(-ENODEV);
	}

	result = acpi_bus_register_driver(&acpi_pcc_cd_driver);
	if (result < 0) {
		ACPI_DEBUG_PRINT((ACPI_DB_ERROR, "Error registering pcc cd drive driver\n"));
		return_VALUE(-ENODEV);
	}

	return_VALUE(0);
}

static void __exit acpi_pcc_exit(void)
{
	ACPI_FUNCTION_TRACE("acpi_pcc_exit");

	acpi_bus_unregister_driver(&acpi_pcc_cd_driver); 

	return_VOID;
}

module_init(acpi_pcc_init);
module_exit(acpi_pcc_exit);
