
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * Copyright (c) 2005 Hiroyuki Ikezoe <poincare@ikezoe.net>
 * Copyright (c) 2005 Takuro Ashie <ashie@homa.ne.jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: scim_tomoe.cpp,v 1.13 2005/10/05 08:23:53 daih Exp $
 *
 */

/*
 * The original code is scim_input_pad.cpp in scim-input-pad-0.1.0.
 * Copyright (C) 2005 James Su <suzhe@tsinghua.org.cn>
 */

#include <wctype.h>
#include <gtk/gtk.h>

#define Uses_SCIM_UTILITY
#define Uses_SCIM_OBJECT
#define Uses_SCIM_POINTER
#define Uses_SCIM_EVENT
#define Uses_SCIM_HELPER
#define Uses_SCIM_CONFIG_BASE
#include <vector>
#include <scim.h>

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include "intl.h"
#include "scim_tomoe_prefs.h"
#include "scim_tomoe_prefs_win.h"
#include <tomoe-gtk.h>

#define scim_module_init tomoe_LTX_scim_module_init
#define scim_module_exit tomoe_LTX_scim_module_exit
#define scim_helper_module_number_of_helpers tomoe_LTX_scim_helper_module_number_of_helpers
#define scim_helper_module_get_helper_info tomoe_LTX_scim_helper_module_get_helper_info
#define scim_helper_module_run_helper tomoe_LTX_scim_helper_module_run_helper

#define SCIM_TOMOE_ICON (SCIM_ICONDIR "/scim-tomoe.png")

#define OBJECT_DATA_PREFS_WIN "ScimTomoe::PrefsWin"

using namespace scim;

static void       slot_exit                   (const HelperAgent   *agent,
                                               int                  ic,
                                               const String        &uuid);
static void       slot_update_screen          (const HelperAgent   *agent,
                                               int                  ic,
                                               const String        &uuid,
                                               int                  screen);
static void       run                         (const String        &display,
                                               const ConfigPointer &config);

static void       on_candidates_view_selected (TomoeCandidatesView *view,
                                               gpointer             user_data);
static void       on_space_button_clicked     (GtkButton           *button,
                                               gpointer             user_data);
static void       on_backspace_button_clicked (GtkButton           *button,
                                               gpointer             user_data);
static void       on_enter_button_clicked     (GtkButton           *button,
                                               gpointer             user_data);
static void       on_prefs_button_clicked     (GtkButton           *button,
                                               gpointer             user_data);

static HelperAgent  helper_agent;
static GtkWidget   *main_window;
static gint         main_window_xpos = 0;
static gint         main_window_ypos = 0;

static ConfigPointer m_config = NULL;

static HelperInfo  helper_info (String ("b1bfe2b4-6930-41b0-8c07-d05bce8c92e2"),
                                "",
                                String (SCIM_TOMOE_ICON),
                                "",
                                SCIM_HELPER_STAND_ALONE |
                                SCIM_HELPER_NEED_SCREEN_INFO);


//Module Interface
extern "C" {
    void scim_module_init (void)
    {
        bindtextdomain (GETTEXT_PACKAGE, SCIM_TOMOE_LOCALEDIR);
        bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");

        helper_info.name = String (_("Handwriting recognition"));
        helper_info.description = String (
            _("An front-end for handwriting recognition engine"));
    }

    void scim_module_exit (void)
    {
    }

    unsigned int scim_helper_module_number_of_helpers (void)
    {
        return 1;
    }

    bool scim_helper_module_get_helper_info (unsigned int idx, HelperInfo &info)
    {
        if (idx == 0) {
            info = helper_info; 
            return true;
        }
        return false;
    }

    void scim_helper_module_run_helper (const String &uuid,
                                        const ConfigPointer &config,
                                        const String &display)
    {
        SCIM_DEBUG_MAIN(1) << "tomoe_LTX_scim_helper_module_run_helper ()\n";

        if (uuid == "b1bfe2b4-6930-41b0-8c07-d05bce8c92e2") {
            run (display, config);
        }

        SCIM_DEBUG_MAIN(1) << "exit tomoe_LTX_scim_helper_module_run_helper ()\n";
    }
}

static gboolean
helper_agent_input_handler (GIOChannel *source,
                            GIOCondition condition,
                            gpointer user_data)
{
    if (condition == G_IO_IN) {
        HelperAgent *agent = static_cast<HelperAgent *> (user_data);
        if (agent && agent->has_pending_event ())
            agent->filter_event ();
    } else if (condition == G_IO_ERR || condition == G_IO_HUP) {
        gtk_main_quit ();
    }
    return TRUE;
}

static void
slot_exit (const HelperAgent *agent, int ic, const String &uuid)
{
    gtk_main_quit ();
}

static void
slot_update_screen (const HelperAgent *agent, int ic,
                    const String &uuid, int screen)
{
    if (gdk_display_get_n_screens (gdk_display_get_default ()) > screen) {
        GdkScreen *scr = gdk_display_get_screen (gdk_display_get_default (),
                                                 screen);
        if (scr)
            gtk_window_set_screen (GTK_WINDOW (main_window), scr);
    }
}

static void
slot_trigger_property (const HelperAgent *agent, int ic,
                       const String &uuid, const String &property)
{
    if (property == "/TOMOE") {
        if (GTK_WIDGET_VISIBLE (main_window)) {
            gtk_window_get_position (GTK_WINDOW (main_window),
                                     &main_window_xpos,
                                     &main_window_ypos);
            gtk_widget_hide (main_window);
        } else {
            gtk_window_move (GTK_WINDOW (main_window),
                             main_window_xpos,
                             main_window_ypos);
            gtk_widget_show (main_window);
        }
    }
}

void run (const String &display, const ConfigPointer &config)
{
    char **argv = new char * [4];
    int    argc = 3;

    argv [0] = "tomoe";
    argv [1] = "--display";
    argv [2] = const_cast<char *> (display.c_str ());
    argv [3] = 0;
 
    setenv ("DISPLAY", display.c_str (), 1);

    gtk_init(&argc, &argv);
  
    helper_agent.signal_connect_exit (slot (slot_exit));
	helper_agent.signal_connect_update_screen (slot (slot_update_screen));
    helper_agent.signal_connect_trigger_property (slot (slot_trigger_property));

    int fd = helper_agent.open_connection (helper_info, display);
    GIOChannel *ch = g_io_channel_unix_new (fd);

    if (fd >= 0 && ch) {
        Property prop ("/TOMOE", _("TOMOE"),
                       SCIM_TOMOE_ICON,
                       _("Show/Hide TOMOE."));
	    PropertyList props;

	    props.push_back (prop);

	    helper_agent.register_properties (props);

        g_io_add_watch (ch, G_IO_IN,
                        helper_agent_input_handler,
                        (gpointer) &helper_agent);
        g_io_add_watch (ch, G_IO_ERR,
                        helper_agent_input_handler,
                        (gpointer) &helper_agent);
        g_io_add_watch (ch, G_IO_HUP,
                        helper_agent_input_handler,
                        (gpointer) &helper_agent);
    }

    GtkWidget *window = tomoe_window_new ();
    gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);
    gtk_widget_show (window);
    g_signal_connect (G_OBJECT (window), "response",
                      G_CALLBACK (gtk_main_quit), NULL);
    main_window = window;

    // add software keyboard button
    TomoeWindow *tomoe = TOMOE_WINDOW (window);
    GtkWidget *separator, *button;

    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (tomoe->button_area), separator,
                        FALSE, FALSE, 4);
    gtk_widget_show (separator);

    button = gtk_button_new_with_mnemonic (_("Space"));
    gtk_box_pack_start (GTK_BOX (tomoe->button_area), button,
                        FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "pressed",
                      G_CALLBACK (on_space_button_clicked),
                      (gpointer) tomoe);
    gtk_widget_show (button);

    button = gtk_button_new_with_mnemonic (_("BackSpace"));
    gtk_box_pack_start (GTK_BOX (tomoe->button_area), button,
                        FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "pressed",
                      G_CALLBACK (on_backspace_button_clicked),
                      (gpointer) tomoe);
    gtk_widget_show (button);

    button = gtk_button_new_with_mnemonic (_("Enter"));
    gtk_box_pack_start (GTK_BOX (tomoe->button_area), button,
                        FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "pressed",
                      G_CALLBACK (on_enter_button_clicked),
                      (gpointer) tomoe);
    gtk_widget_show (button);

    // add preference button
    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (tomoe->button_area), separator,
                        FALSE, FALSE, 4);
    gtk_widget_show (separator);

    button = gtk_button_new_from_stock (GTK_STOCK_PREFERENCES);
    gtk_box_pack_start (GTK_BOX (tomoe->button_area), button,
                        FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_prefs_button_clicked),
                      (gpointer) tomoe);
    gtk_widget_show (button);

    // add hooks
    g_signal_connect (G_OBJECT (tomoe->candidates_view), "selected",
                      G_CALLBACK (on_candidates_view_selected),
                      (gpointer) tomoe);

    // create preferences window
    GtkWidget *prefs_win = scim_tomoe_prefs_win_new (tomoe, config);
    gtk_window_set_transient_for (GTK_WINDOW (prefs_win),
                                  GTK_WINDOW (window));
    g_object_set_data (G_OBJECT (window),
                       OBJECT_DATA_PREFS_WIN, prefs_win);

    // run
    gtk_main ();

    // quit
    gtk_widget_destroy (window);
}

static void
on_candidates_view_selected (TomoeCandidatesView *view, gpointer user_data)
{
    TomoeWindow *window = TOMOE_WINDOW (user_data);
    gpointer object_data = g_object_get_data (G_OBJECT (window),
                                              OBJECT_DATA_PREFS_WIN);
    ScimTomoePrefsWin *prefs_win = SCIM_TOMOE_PREFS_WIN (object_data);

    if (helper_agent.get_connection_number () < 0) return;

    gchar *str = tomoe_candidates_view_get_selected_letter (view);
    if (str && *str) {
        helper_agent.commit_string (-1, "", scim::utf8_mbstowcs (str));

        bool value;
        value = prefs_win->config->read (
            String (SCIM_TOMOE_CONFIG_CLEAR_ON_SELECT),
            SCIM_TOMOE_CONFIG_CLEAR_ON_SELECT_DEFAULT);
        if (value) {
            TomoeCanvas *canvas;
            canvas = TOMOE_CANVAS (tomoe_window_get_canvas (window));
            tomoe_canvas_clear (TOMOE_CANVAS (canvas));
        }
    }
}

static void
on_space_button_clicked (GtkButton *button, gpointer user_data)
{
    if (helper_agent.get_connection_number () < 0) return;

    KeyEvent key (SCIM_KEY_space, 0);
    KeyEvent key_release (SCIM_KEY_space, SCIM_KEY_ReleaseMask);
    helper_agent.forward_key_event (-1, "", key);
    helper_agent.forward_key_event (-1, "", key_release);
}

static void
on_backspace_button_clicked (GtkButton *button, gpointer user_data)
{
    if (helper_agent.get_connection_number () < 0) return;

    KeyEvent key (SCIM_KEY_BackSpace, 0);
    KeyEvent key_release (SCIM_KEY_BackSpace, SCIM_KEY_ReleaseMask);
    helper_agent.forward_key_event (-1, "", key);
    helper_agent.forward_key_event (-1, "", key_release);
}

static void
on_enter_button_clicked (GtkButton *button, gpointer user_data)
{
    if (helper_agent.get_connection_number () < 0) return;

    KeyEvent key (SCIM_KEY_Return, 0);
    KeyEvent key_release (SCIM_KEY_Return, SCIM_KEY_ReleaseMask);
    helper_agent.forward_key_event (-1, "", key);
    helper_agent.forward_key_event (-1, "", key_release);
}

static void
on_prefs_button_clicked (GtkButton *button, gpointer user_data)
{
    GtkWidget *window = GTK_WIDGET (user_data);
    gpointer object_data = g_object_get_data (G_OBJECT (window),
                                              OBJECT_DATA_PREFS_WIN);
    GtkWidget *prefs_win = GTK_WIDGET (object_data);

    gtk_widget_show (prefs_win);
    gtk_dialog_run (GTK_DIALOG (prefs_win));
    gtk_widget_hide (prefs_win);
}

/*
vi:ts=4:nowrap:ai:expandtab
*/
