/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2005 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include <glib/gi18n-lib.h>

#include "scim_tomoe_prefs_widget.h"
#include "scim_tomoe_prefs.h"
#include <tomoe-gtk.h>

enum {
    HOGE_SIGNAL,
    LAST_SIGNAL,
};

#define OBJECT_DATA_CONFIG_KEY        "ScimTomoePrefsWidget::ConfigKey"
#define OBJECT_DATA_CONFIG_VALUE      "ScimTomoePrefsWidget::ConfigValue"

G_DEFINE_TYPE (ScimTomoePrefsWidget, scim_tomoe_prefs_widget, GTK_TYPE_VBOX)

static void scim_tomoe_prefs_widget_class_init  (ScimTomoePrefsWidgetClass *klass);
static void scim_tomoe_prefs_widget_init        (ScimTomoePrefsWidget      *prefs_widget);
static void scim_tomoe_prefs_widget_dispose     (GObject                *object);

static GtkWidget *create_window_page            (ScimTomoePrefsWidget *prefs_widget);
static GtkWidget *create_handwriting_page       (ScimTomoePrefsWidget *prefs_widget);
#ifdef TOMOE_GTK_HAS_GUCHARMAP
static GtkWidget *create_unicode_table_page     (ScimTomoePrefsWidget *prefs_widget);
#endif /* TOMOE_GTK_HAS_GUCHARMAP */
static void       load_config                   (ScimTomoePrefsWidget      *prefs_widget);
static void       apply_config                  (ScimTomoePrefsWidget      *prefs_widget);

static void       on_default_toggle_button_toggled
                                                (GtkToggleButton        *togglebutton,
                                                 gpointer                user_data);
static void       on_default_toggle_button_set_sensitive
                                                (GtkToggleButton        *togglebutton,
                                                 gpointer                user_data);
static void       on_default_spin_button_changed(GtkSpinButton          *spinbutton,
                                                 gpointer                user_data);
static void       on_default_font_button_changed(GtkFontButton          *button,
                                                 gpointer                user_data);

//static guint prefs_widget_signals[LAST_SIGNAL] = { 0 };

static void
scim_tomoe_prefs_widget_class_init (ScimTomoePrefsWidgetClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

    gobject_class->dispose = scim_tomoe_prefs_widget_dispose;
}

static void
scim_tomoe_prefs_widget_init (ScimTomoePrefsWidget *prefs)
{
    GtkWidget *main_vbox = GTK_WIDGET (prefs);

    GtkWidget *notebook = gtk_notebook_new ();
    gtk_container_set_border_width (GTK_CONTAINER (notebook), 8);
    gtk_box_pack_start (GTK_BOX (main_vbox), notebook, TRUE, TRUE, 0);
    gtk_widget_show (notebook);

    /* "Window" page */
    GtkWidget *label  = gtk_label_new (_("Window"));
    GtkWidget *widget = create_window_page (prefs);
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), widget, label);
    gtk_widget_show (label);
    gtk_widget_show (widget);

    /* "Handwriting recognition" page  */
    label  = gtk_label_new (_("Handwriting recognition"));
    widget = create_handwriting_page (prefs);
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), widget, label);
    gtk_widget_show (label);
    gtk_widget_show (widget);

#ifdef TOMOE_GTK_HAS_GUCHARMAP
    /* "Unicode table" page */
    label  = gtk_label_new (_("Unicode table"));
    widget = create_unicode_table_page (prefs);
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), widget, label);
    gtk_widget_show (label);
    gtk_widget_show (widget);
#endif /* TOMOE_GTK_HAS_GUCHARMAP */
}

static void
scim_tomoe_prefs_widget_dispose (GObject *object)
{
    ScimTomoePrefsWidget *prefs = SCIM_TOMOE_PREFS_WIDGET (object);

    if (prefs->config)
        prefs->config->flush ();
    prefs->config = NULL;

    if (G_OBJECT_CLASS(scim_tomoe_prefs_widget_parent_class)->dispose)
        G_OBJECT_CLASS(scim_tomoe_prefs_widget_parent_class)->dispose(object);
}

GtkWidget *
scim_tomoe_prefs_widget_new (TomoeWindow *window, const ConfigPointer &config)
{
    GtkWidget *widget = GTK_WIDGET(g_object_new (SCIM_TOMOE_TYPE_PREFS_WIDGET,
                                                 NULL));
    ScimTomoePrefsWidget *prefs = SCIM_TOMOE_PREFS_WIDGET (widget);

    prefs->config = config;
    prefs->parent = GTK_WIDGET (window);

    load_config (SCIM_TOMOE_PREFS_WIDGET(widget));

    return widget;
}

static GtkWidget *
create_check_button (ScimTomoePrefsWidget *prefs,
                     const gchar *label_text,
                     const gchar *key,
                     bool value)
{
    GtkWidget *check = gtk_check_button_new_with_mnemonic (label_text);
    g_object_set_data (G_OBJECT (check),
                       OBJECT_DATA_CONFIG_KEY, (gpointer) key);
    g_object_set_data (G_OBJECT (check),
                       OBJECT_DATA_CONFIG_VALUE, GINT_TO_POINTER (value));
    g_signal_connect (G_OBJECT (check), "toggled",
                      G_CALLBACK (on_default_toggle_button_toggled),
                      prefs);

    prefs->widget_list = g_list_append (prefs->widget_list, check);

    return check;
}

static GtkWidget *
create_spin_button (ScimTomoePrefsWidget *prefs,
                    const gchar *label_text,
                    const gchar *unit_text,
                    const gchar *key,
                    gint value,
                    gint min, gint max, gint step, gint page_size)
{
    GtkWidget *table = gtk_table_new (4, 4, FALSE);

    GtkWidget *label = gtk_label_new_with_mnemonic (label_text);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
                      (GtkAttachOptions) (GTK_FILL),
                      (GtkAttachOptions) (GTK_FILL),
                      4, 4);
    gtk_widget_show (label);

    GtkObject *adj = gtk_adjustment_new (value, min, max, step,
                                         page_size, page_size);
    GtkWidget *spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 0.0, 0);
    g_object_set_data (G_OBJECT (spin),
                       OBJECT_DATA_CONFIG_KEY, (gpointer) key);
    g_object_set_data (G_OBJECT (spin),
                       OBJECT_DATA_CONFIG_VALUE, GINT_TO_POINTER (value));
    gtk_table_attach (GTK_TABLE (table), spin, 1, 2, 1, 2,
                      (GtkAttachOptions) (GTK_FILL),
                      (GtkAttachOptions) (GTK_FILL),
                      4, 4);
    g_signal_connect (G_OBJECT (spin), "value-changed",
                      G_CALLBACK (on_default_spin_button_changed),
                      prefs);
    gtk_widget_show (spin);

    gtk_label_set_mnemonic_widget (GTK_LABEL (label), spin);

    if (unit_text) {
        label = gtk_label_new (unit_text);
        gtk_table_attach (GTK_TABLE (table), label, 2, 3, 1, 2,
                          (GtkAttachOptions) (GTK_FILL),
                          (GtkAttachOptions) (GTK_FILL),
                          4, 4);
        gtk_widget_show (label);
    }

    prefs->widget_list = g_list_append (prefs->widget_list, spin);

    return table;
}

static GtkWidget *
create_font_button (ScimTomoePrefsWidget *prefs,
                    const gchar *label_text,
                    const gchar *key,
                    const gchar *value)
{
    GtkWidget *hbox = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (hbox);

    GtkWidget *label = gtk_label_new (label_text);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    gtk_widget_show (label);

    GtkWidget *button = gtk_font_button_new ();
    gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 4);
    gtk_widget_show (button);

    g_object_set_data (G_OBJECT (button),
                       OBJECT_DATA_CONFIG_KEY, (gpointer) key);
    g_object_set_data (G_OBJECT (button),
                       OBJECT_DATA_CONFIG_VALUE, (gpointer) value);
    g_signal_connect (G_OBJECT (button), "font-set",
                      G_CALLBACK (on_default_font_button_changed),
                      prefs);

    prefs->widget_list = g_list_append (prefs->widget_list, button);

    return hbox;
}

static void
set_default_font (ScimTomoePrefsWidget *prefs,
                  const gchar *key,
                  GtkWidget *widget)
{
    GList *node;

    for (node = prefs->widget_list; node; node = g_list_next (node)) {
        GtkWidget *w = (GtkWidget *) node->data;
        const gchar *k;
        gchar *str;

        if (!GTK_IS_FONT_BUTTON (w)) continue;

        k = (const gchar *) g_object_get_data (G_OBJECT (w),
                                               OBJECT_DATA_CONFIG_KEY);
        if (strcmp (k, key)) continue;

        str = pango_font_description_to_string (widget->style->font_desc);
        gtk_font_button_set_font_name (GTK_FONT_BUTTON (w), str);
        g_free (str);

        break;
    }
}

static GtkWidget *
create_window_page (ScimTomoePrefsWidget *prefs)
{
    GtkWidget *vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), 8);

    // show on top of other windows.
    GtkWidget *widget;
    widget = create_check_button (prefs, _("Show on _top of other windows"),
                                  SCIM_TOMOE_CONFIG_WINDOW_ALWAYS_TOP,
                                  SCIM_TOMOE_CONFIG_WINDOW_ALWAYS_TOP_DEFAULT);
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 4);
    gtk_widget_show (widget);

    return vbox;
}

static GtkWidget *
create_handwriting_page (ScimTomoePrefsWidget *prefs)
{

    GtkWidget *vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), 8);

    GtkWidget *widget;

    // use auto find
    widget = create_check_button (prefs, _("Use auto _find"),
                                  SCIM_TOMOE_CONFIG_USE_AUTO_FIND,
                                  SCIM_TOMOE_CONFIG_USE_AUTO_FIND_DEFAULT);
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 4);
    gtk_widget_show (widget);

    // auto find time
    GtkWidget *alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
    gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 0, 0, 24, 0);
    gtk_box_pack_start (GTK_BOX (vbox), alignment, FALSE, FALSE, 0);
    gtk_widget_show (alignment);

    GtkWidget *table = create_spin_button (
        prefs,
        _("_Interval time until start auto find:"),
        _("[msec]"),
        SCIM_TOMOE_CONFIG_AUTO_FIND_TIME,
        SCIM_TOMOE_CONFIG_AUTO_FIND_TIME_DEFAULT,
        0, 100000, 50, 200);
    gtk_widget_set_sensitive (table, FALSE);
    gtk_container_add (GTK_CONTAINER (alignment), table);
    gtk_widget_show (table);

    g_signal_connect (G_OBJECT (widget), "toggled",
                      G_CALLBACK (on_default_toggle_button_set_sensitive),
                      table);

    // clear the canvas
    widget = create_check_button (
        prefs,
        _("_Clear the canvas when select a candidate"),
        SCIM_TOMOE_CONFIG_CLEAR_ON_SELECT,
        SCIM_TOMOE_CONFIG_CLEAR_ON_SELECT_DEFAULT);

    // font for the candidate view
    widget = create_font_button (prefs, _("Font for the candidates view:"),
                                 SCIM_TOMOE_CONFIG_CAND_FONT,
                                 SCIM_TOMOE_CONFIG_CAND_FONT_DEFAULT);
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 4);
    gtk_widget_show (widget);

    return vbox;
}

#ifdef TOMOE_GTK_HAS_GUCHARMAP
static GtkWidget *
create_unicode_table_page (ScimTomoePrefsWidget *prefs)
{
    GtkWidget *vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), 8);

    GtkWidget *widget;
    widget = create_font_button (prefs, _("Font for the Unicode table:"),
                                 SCIM_TOMOE_CONFIG_UNICODE_TABLE_FONT,
                                 SCIM_TOMOE_CONFIG_UNICODE_TABLE_FONT_DEFAULT);
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 4);
    gtk_widget_show (widget);

    return vbox;
}
#endif /* TOMOE_GTK_HAS_GUCHARMAP */

static void
load_config (ScimTomoePrefsWidget *prefs)
{
    GList *node;

    if (prefs->config.null ())
        return;

    for (node = prefs->widget_list; node; node = g_list_next (node)) {
        GtkWidget *widget = GTK_WIDGET (node->data);

        if (GTK_IS_TOGGLE_BUTTON (widget)) {
            gpointer key_p, value_p;
            const char *key;
            bool value;

            key_p   = g_object_get_data (G_OBJECT (widget),
                                         OBJECT_DATA_CONFIG_KEY);
            value_p = g_object_get_data (G_OBJECT (widget),
                                         OBJECT_DATA_CONFIG_VALUE);

            key = (const char*) key_p;
            value = GPOINTER_TO_INT (value_p);

            value = prefs->config->read (String (key), value);
            gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), value);

        } else if (GTK_IS_SPIN_BUTTON (widget)) {
            gpointer key_p, value_p;
            const char *key;
            gint value;

            key_p   = g_object_get_data (G_OBJECT (widget),
                                         OBJECT_DATA_CONFIG_KEY);
            value_p = g_object_get_data (G_OBJECT (widget),
                                         OBJECT_DATA_CONFIG_VALUE);

            key = (const char*) key_p;
            value = GPOINTER_TO_INT (value_p);

            value = prefs->config->read (String (key), value);
            gtk_spin_button_set_value (GTK_SPIN_BUTTON (widget), value);

        } else if (GTK_IS_FONT_BUTTON (widget)) {
            gpointer key_p, value_p;
            const char *key, *value;
            String str;

            key_p   = g_object_get_data (G_OBJECT (widget),
                                         OBJECT_DATA_CONFIG_KEY);
            value_p = g_object_get_data (G_OBJECT (widget),
                                         OBJECT_DATA_CONFIG_VALUE);

            key   = (const char*) key_p;
            value = (const char*) value_p;

            str = prefs->config->read (String (key),
                                              value ? value : String (""));
            if (str.length() > 0)
                gtk_font_button_set_font_name (GTK_FONT_BUTTON (widget),
                                               str.c_str());
        }
    }

    apply_config (prefs);
}

static void
apply_config (ScimTomoePrefsWidget *prefs)
{
    TomoeWindow *parent = TOMOE_WINDOW (prefs->parent);
    TomoeHandwriting *hw = TOMOE_HANDWRITING (tomoe_window_get_handwriting_page (parent));
    TomoeCanvas *canvas = TOMOE_CANVAS (tomoe_handwriting_get_canvas (hw));

    // window z-level option
    bool always_top;
    always_top = prefs->config->read (
        String (SCIM_TOMOE_CONFIG_WINDOW_ALWAYS_TOP),
        SCIM_TOMOE_CONFIG_WINDOW_ALWAYS_TOP_DEFAULT);
    gtk_window_set_keep_above (GTK_WINDOW (parent), always_top ? TRUE : FALSE);

    // auto find
    bool use_auto_find;
    gint time1, time2;

    time1 = tomoe_canvas_get_auto_find_time (canvas);

    use_auto_find = prefs->config->read (
        String (SCIM_TOMOE_CONFIG_USE_AUTO_FIND),
        SCIM_TOMOE_CONFIG_USE_AUTO_FIND_DEFAULT);

    time2 = prefs->config->read (
        String (SCIM_TOMOE_CONFIG_AUTO_FIND_TIME),
        SCIM_TOMOE_CONFIG_AUTO_FIND_TIME_DEFAULT);

    if (!use_auto_find)
        tomoe_canvas_set_auto_find_time (canvas, -1);
    else if (time1 != time2)
        tomoe_canvas_set_auto_find_time (canvas, time2);

    // candidates view font
    GtkWidget *view = tomoe_handwriting_get_char_table (hw);
    String font;
    font = prefs->config->read (
        String (SCIM_TOMOE_CONFIG_CAND_FONT),
        String (SCIM_TOMOE_CONFIG_CAND_FONT_DEFAULT));

    if (font.length() > 0) {
        PangoFontDescription *desc;

        desc = pango_font_description_from_string (font.c_str ());
        gtk_widget_modify_font (view, desc);
        pango_font_description_free (desc);
    } else {
        set_default_font (prefs, SCIM_TOMOE_CONFIG_CAND_FONT, view);
    }

#ifdef TOMOE_GTK_HAS_GUCHARMAP
    // Unicode table font
    TomoeGucharmap *gucharmap = TOMOE_GUCHARMAP (tomoe_window_get_gucharmap_page (parent));
    GucharmapCharmap *charmap = tomoe_gucharmap_get_charmap (gucharmap);
    font = prefs->config->read (
        String (SCIM_TOMOE_CONFIG_UNICODE_TABLE_FONT),
        String (SCIM_TOMOE_CONFIG_UNICODE_TABLE_FONT_DEFAULT));

    if (font.length() > 0) {
        gucharmap_charmap_set_font (charmap, font.c_str ());
    } else {
        gucharmap_charmap_set_font (charmap, NULL);
        set_default_font (prefs, SCIM_TOMOE_CONFIG_CAND_FONT,
                          GTK_WIDGET (charmap));
    }
#endif /* TOMOE_GTK_HAS_GUCHARMAP */
}

static void
on_default_toggle_button_toggled (GtkToggleButton *togglebutton,
                                  gpointer user_data)
{
    ScimTomoePrefsWidget *prefs = SCIM_TOMOE_PREFS_WIDGET (user_data);

    gboolean value = gtk_toggle_button_get_active (togglebutton);
    gpointer key_p   = g_object_get_data (G_OBJECT (togglebutton),
                                          OBJECT_DATA_CONFIG_KEY);
    const gchar *key = (const gchar *) key_p;
    if (key && *key)
        prefs->config->write (String (key), value);

    apply_config (prefs);
}

static void
on_default_toggle_button_set_sensitive (GtkToggleButton *togglebutton,
                                        gpointer user_data)
{
    GtkWidget *widget = GTK_WIDGET (user_data);
    gboolean active = gtk_toggle_button_get_active (togglebutton);
    gtk_widget_set_sensitive (widget, active);
}

static void
on_default_spin_button_changed (GtkSpinButton *spinbutton, gpointer user_data)
{
    ScimTomoePrefsWidget *prefs = SCIM_TOMOE_PREFS_WIDGET (user_data);

    gint value = (gint) gtk_spin_button_get_value (spinbutton);
    gpointer key_p   = g_object_get_data (G_OBJECT (spinbutton),
                                          OBJECT_DATA_CONFIG_KEY);
    const gchar *key = (const gchar *) key_p;
    if (key && *key)
        prefs->config->write (String (key), value);

    apply_config (prefs);
}

static void
on_default_font_button_changed (GtkFontButton *button, gpointer user_data)
{
    ScimTomoePrefsWidget *prefs = SCIM_TOMOE_PREFS_WIDGET (user_data);

    const gchar *value = gtk_font_button_get_font_name (button);
    gpointer key_p   = g_object_get_data (G_OBJECT (button),
                                          OBJECT_DATA_CONFIG_KEY);
    const gchar *key = (const gchar *) key_p;
    if (key && *key)
        prefs->config->write (String (key), value ? value : String (""));

    apply_config (prefs);
}
/*
vi:ts=4:nowrap:ai:expandtab
*/
