/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.ml.common.util;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.flink.api.common.typeinfo.TypeInfo;
import org.apache.flink.ml.common.typeinfo.QuantileSummaryTypeInfoFactory;
import org.apache.flink.util.Preconditions;

@TypeInfo(value=QuantileSummaryTypeInfoFactory.class)
public class QuantileSummary
implements Serializable {
    private static final int DEFAULT_HEAD_SIZE = 50000;
    private static final int DEFAULT_COMPRESS_THRESHOLD = 10000;
    private double relativeError;
    private int compressThreshold;
    private long count;
    private List<StatsTuple> sampled;
    private List<Double> headBuffer = new ArrayList<Double>();
    private boolean compressed;

    public QuantileSummary() {
    }

    public QuantileSummary(double relativeError) {
        this(relativeError, 10000);
    }

    public QuantileSummary(double relativeError, int compressThreshold) {
        this(relativeError, compressThreshold, Collections.EMPTY_LIST, 0L, false);
    }

    public QuantileSummary(double relativeError, int compressThreshold, List<StatsTuple> sampled, long count, boolean compressed) {
        Preconditions.checkArgument((relativeError >= 0.0 && relativeError <= 1.0 ? 1 : 0) != 0, (Object)"An appropriate relative error must be in the range [0, 1].");
        Preconditions.checkArgument((compressThreshold > 0 ? 1 : 0) != 0, (Object)"An compress threshold must greater than 0.");
        this.relativeError = relativeError;
        this.compressThreshold = compressThreshold;
        this.sampled = sampled;
        this.count = count;
        this.compressed = compressed;
    }

    public QuantileSummary insert(double item) {
        this.headBuffer.add(item);
        this.compressed = false;
        if (this.headBuffer.size() >= 50000) {
            QuantileSummary result = this.insertHeadBuffer();
            if (result.sampled.size() >= this.compressThreshold) {
                return result.compress();
            }
            return result;
        }
        return this;
    }

    public QuantileSummary compress() {
        if (this.compressed) {
            return this;
        }
        QuantileSummary inserted = this.insertHeadBuffer();
        Preconditions.checkState((boolean)inserted.headBuffer.isEmpty());
        Preconditions.checkState((inserted.count == this.count + (long)this.headBuffer.size() ? 1 : 0) != 0);
        List<StatsTuple> compressed = this.compressInternal(inserted.sampled, 2.0 * this.relativeError * (double)inserted.count);
        return new QuantileSummary(this.relativeError, this.compressThreshold, compressed, inserted.count, true);
    }

    public QuantileSummary merge(QuantileSummary other) {
        Preconditions.checkState((boolean)this.headBuffer.isEmpty(), (Object)"Current buffer needs to be compressed before merge.");
        Preconditions.checkState((boolean)other.headBuffer.isEmpty(), (Object)"Other buffer needs to be compressed before merge.");
        if (other.count == 0L) {
            return this.shallowCopy();
        }
        if (this.count == 0L) {
            return other.shallowCopy();
        }
        ArrayList<StatsTuple> mergedSampled = new ArrayList<StatsTuple>();
        double mergedRelativeError = Math.max(this.relativeError, other.relativeError);
        long mergedCount = this.count + other.count;
        long additionalSelfDelta = Double.valueOf(Math.floor(2.0 * other.relativeError * (double)other.count)).longValue();
        long additionalOtherDelta = Double.valueOf(Math.floor(2.0 * this.relativeError * (double)this.count)).longValue();
        int selfIdx = 0;
        int otherIdx = 0;
        while (selfIdx < this.sampled.size() && otherIdx < other.sampled.size()) {
            StatsTuple nextSample;
            StatsTuple selfSample = this.sampled.get(selfIdx);
            StatsTuple otherSample = other.sampled.get(otherIdx);
            long additionalDelta = 0L;
            if (selfSample.value < otherSample.value) {
                nextSample = selfSample;
                if (otherIdx > 0) {
                    additionalDelta = additionalSelfDelta;
                }
                ++selfIdx;
            } else {
                nextSample = otherSample;
                if (selfIdx > 0) {
                    additionalDelta = additionalOtherDelta;
                }
                ++otherIdx;
            }
            nextSample = nextSample.shallowCopy();
            nextSample.delta += additionalDelta;
            mergedSampled.add(nextSample);
        }
        IntStream.range(selfIdx, this.sampled.size()).forEach(i -> mergedSampled.add(this.sampled.get(i)));
        IntStream.range(otherIdx, other.sampled.size()).forEach(i -> mergedSampled.add(other.sampled.get(i)));
        List<StatsTuple> comp = this.compressInternal(mergedSampled, 2.0 * mergedRelativeError * (double)mergedCount);
        return new QuantileSummary(mergedRelativeError, this.compressThreshold, comp, mergedCount, true);
    }

    public double query(double percentile) {
        return this.query(new double[]{percentile})[0];
    }

    public double[] query(double[] percentiles) {
        Arrays.stream(percentiles).forEach(x -> Preconditions.checkState((x >= 0.0 && x <= 1.0 ? 1 : 0) != 0, (Object)"percentile should be in the range [0.0, 1.0]."));
        Preconditions.checkState((boolean)this.headBuffer.isEmpty(), (Object)"Cannot operate on an uncompressed summary, call compress() first.");
        Preconditions.checkState((this.sampled != null && !this.sampled.isEmpty() ? 1 : 0) != 0, (Object)"Cannot query percentiles without any records inserted.");
        double targetError = -9.223372036854776E18;
        for (StatsTuple tuple : this.sampled) {
            targetError = Math.max(targetError, (double)(tuple.delta + tuple.g));
        }
        targetError /= 2.0;
        HashMap zipWithIndex = new HashMap(percentiles.length);
        IntStream.range(0, percentiles.length).forEach(i -> zipWithIndex.put(percentiles[i], i));
        int index = 0;
        long minRank = this.sampled.get((int)0).g;
        double[] sorted = Arrays.stream(percentiles).sorted().toArray();
        double[] result = new double[percentiles.length];
        for (double item : sorted) {
            int percentileIndex = (Integer)zipWithIndex.get(item);
            if (item <= this.relativeError) {
                result[percentileIndex] = this.sampled.get((int)0).value;
                continue;
            }
            if (item >= 1.0 - this.relativeError) {
                result[percentileIndex] = this.sampled.get((int)(this.sampled.size() - 1)).value;
                continue;
            }
            QueryResult queryResult = this.findApproximateQuantile(index, minRank, targetError, item);
            index = queryResult.index;
            minRank = queryResult.minRankAtIndex;
            result[percentileIndex] = queryResult.percentile;
        }
        return result;
    }

    public boolean isEmpty() {
        return this.headBuffer.isEmpty() && this.sampled.isEmpty();
    }

    private QuantileSummary insertHeadBuffer() {
        int i;
        if (this.headBuffer.isEmpty()) {
            return this;
        }
        long newCount = this.count;
        ArrayList<StatsTuple> newSamples = new ArrayList<StatsTuple>();
        List sorted = this.headBuffer.stream().sorted().collect(Collectors.toList());
        int cursor = 0;
        for (i = 0; i < sorted.size(); ++i) {
            while (cursor < this.sampled.size() && this.sampled.get((int)cursor).value <= (Double)sorted.get(i)) {
                newSamples.add(this.sampled.get(cursor));
                ++cursor;
            }
            long delta = Double.valueOf(Math.floor(2.0 * this.relativeError * (double)this.count)).longValue();
            if (newSamples.isEmpty() || cursor == this.sampled.size() && i == sorted.size() - 1) {
                delta = 0L;
            }
            StatsTuple tuple = new StatsTuple((Double)sorted.get(i), 1L, delta);
            newSamples.add(tuple);
            ++newCount;
        }
        for (i = cursor; i < this.sampled.size(); ++i) {
            newSamples.add(this.sampled.get(i));
        }
        return new QuantileSummary(this.relativeError, this.compressThreshold, newSamples, newCount, false);
    }

    private List<StatsTuple> compressInternal(List<StatsTuple> currentSamples, double mergeThreshold) {
        if (currentSamples.isEmpty()) {
            return Collections.emptyList();
        }
        LinkedList<StatsTuple> result = new LinkedList<StatsTuple>();
        StatsTuple head = currentSamples.get(currentSamples.size() - 1);
        for (int i = currentSamples.size() - 2; i >= 1; --i) {
            StatsTuple tuple = currentSamples.get(i);
            if ((double)(tuple.g + head.g + head.delta) < mergeThreshold) {
                head = head.shallowCopy();
                head.g += tuple.g;
                continue;
            }
            result.addFirst(head);
            head = tuple;
        }
        result.addFirst(head);
        StatsTuple currHead = currentSamples.get(0);
        if (currHead.value <= head.value && currentSamples.size() > 1) {
            result.addFirst(currHead);
        }
        return new ArrayList<StatsTuple>(result);
    }

    private QueryResult findApproximateQuantile(int index, long minRankAtIndex, double targetError, double percentile) {
        StatsTuple curSample = this.sampled.get(index);
        long rank = Double.valueOf(Math.ceil(percentile * (double)this.count)).longValue();
        long minRank = minRankAtIndex;
        int i = index;
        while (i < this.sampled.size() - 1) {
            long maxRank = minRank + curSample.delta;
            if ((double)maxRank - targetError < (double)rank && (double)rank <= (double)minRank + targetError) {
                return new QueryResult(i, minRank, curSample.value);
            }
            curSample = this.sampled.get(++i);
            minRank += curSample.g;
        }
        return new QueryResult(this.sampled.size() - 1, 0L, this.sampled.get((int)(this.sampled.size() - 1)).value);
    }

    public double getRelativeError() {
        return this.relativeError;
    }

    public int getCompressThreshold() {
        return this.compressThreshold;
    }

    public long getCount() {
        return this.count;
    }

    public List<StatsTuple> getSampled() {
        return this.sampled;
    }

    public List<Double> getHeadBuffer() {
        return this.headBuffer;
    }

    public boolean isCompressed() {
        return this.compressed;
    }

    private QuantileSummary shallowCopy() {
        return new QuantileSummary(this.relativeError, this.compressThreshold, this.sampled, this.count, this.compressed);
    }

    public static class StatsTuple
    implements Serializable {
        private static final long serialVersionUID = 1L;
        public double value;
        public long g;
        public long delta;

        public StatsTuple() {
        }

        public StatsTuple(double value, long g, long delta) {
            this.value = value;
            this.g = g;
            this.delta = delta;
        }

        public StatsTuple shallowCopy() {
            return new StatsTuple(this.value, this.g, this.delta);
        }
    }

    private static class QueryResult {
        private final int index;
        private final long minRankAtIndex;
        private final double percentile;

        public QueryResult(int index, long minRankAtIndex, double percentile) {
            this.index = index;
            this.minRankAtIndex = minRankAtIndex;
            this.percentile = percentile;
        }
    }
}

