/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to You under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package org.apache.geode.modules.util;

import java.util.HashMap;
import java.util.Map;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;

import org.apache.geode.cache.RegionShortcut;
import org.apache.geode.cache.execute.Function;
import org.apache.geode.cache.execute.FunctionService;
import org.apache.geode.examples.SimpleSecurityManager;
import org.apache.geode.test.junit.categories.SecurityTest;
import org.apache.geode.test.junit.rules.ConnectionConfiguration;
import org.apache.geode.test.junit.rules.GfshCommandRule;
import org.apache.geode.test.junit.rules.ServerStarterRule;

@Category(SecurityTest.class)
public class ModuleFunctionsSecurityTest {
  private static final String RESULT_HEADER = "Message";

  @ClassRule
  public static ServerStarterRule server =
      new ServerStarterRule().withJMXManager().withSecurityManager(SimpleSecurityManager.class)
          .withRegion(RegionShortcut.REPLICATE, "AuthRegion").withAutoStart();

  @Rule
  public GfshCommandRule gfsh =
      new GfshCommandRule(server::getJmxPort, GfshCommandRule.PortType.jmxManager);

  private static Map<Function, String> functionStringMap = new HashMap<>();

  @BeforeClass
  public static void setupClass() {
    functionStringMap.put(new BootstrappingFunction(), "CLUSTER:MANAGE");
    functionStringMap.put(new CreateRegionFunction(), "DATA:MANAGE");
    functionStringMap.put(new RegionSizeFunction(), "DATA:READ:AuthRegion");
    functionStringMap.put(new TouchPartitionedRegionEntriesFunction(), "DATA:READ:AuthRegion");
    functionStringMap.put(new TouchReplicatedRegionEntriesFunction(), "DATA:READ");

    functionStringMap.keySet().forEach(FunctionService::registerFunction);
  }

  @Test
  @ConnectionConfiguration(user = "user", password = "user")
  public void functionRequireExpectedPermission() {
    functionStringMap.forEach((function, permission) -> gfsh
        .executeAndAssertThat("execute function --region=AuthRegion --id=" + function.getId())
        .tableHasRowCount(1)
        .statusIsError()
        .hasTableSection()
        .hasColumn(RESULT_HEADER)
        .containsExactly("Exception: user not authorized for " + permission));
  }
}
