/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Original Copyright (c) 2005 Covalent Technologies
 *
 * FTP Protocol module for Apache 2.0
 */

/*
 * A simple example of how to override a commmand.  The FTP hooks will
 * run all handlers for a command, until a module returns anything other
 * than DECLINED.  All core commands are registered as FTP_HOOK_LAST, so
 * any module can override using FTP_HOOK_MIDDLE or FTP_HOOK_FIRST.
 *
 * Implements simple SITE PING and SITE ECHO commands
 */

#include "mod_ftp.h"
#include "apr_strings.h"

extern AP_MODULE_DECLARE_DATA module ftp_example_module;

static int mod_ftp_cmd_site(request_rec *r, const char *arg)
{
    ftp_connection *fc = ftp_get_module_config(r->connection->conn_config);

    /* Only handling SITE ECHO or SITE PING, let all others fall through
     * to another registered command handler.
     */
    if ((strncasecmp(arg, "ECHO", 4) && strncasecmp(arg, "PING", 4))
            || (arg[4] && arg[4] != ' '))
    	return DECLINED;

    if (!strncasecmp(arg, "PING", 4))
    {
        for (arg += 4; *arg == ' '; ++arg) 
            /* noop */;

        if (*arg) {
            fc->response_notes = "SITE PING accepts no parameters";
            return FTP_REPLY_SYNTAX_ERROR;
        }

        fc->response_notes = "PONG";
    }
    else /* !strncasecmp(arg, "ECHO", 4) */
    {
        fc->response_notes = ftp_escape_control_text(arg + 4, r->pool);
    }
    return FTP_REPLY_COMMAND_OK;
}

static int example_pre_config(apr_pool_t *p, apr_pool_t *plog, apr_pool_t *ptemp)
{
    ftp_hook_cmd("SITE", mod_ftp_cmd_site, FTP_HOOK_MIDDLE,
                 FTP_NEED_LOGIN | FTP_TAKE1 | FTP_KEEP_WHITESPACE,
                 "<sp> [ PING | ECHO <sp> message ]");
    return OK;
}

static void register_hooks(apr_pool_t *p)
{
    ap_hook_pre_config(example_pre_config, NULL, NULL, APR_HOOK_MIDDLE);
}

AP_MODULE_DECLARE_DATA module ftp_example_module = {
    STANDARD20_MODULE_STUFF,
    NULL,                       /* create per-directory config structure */
    NULL,                       /* merge per-directory config structures */
    NULL,                       /* create per-server config structure */
    NULL,                       /* merge per-server config structures */
    NULL,                       /* command apr_table_t */
    register_hooks              /* register hooks */
};
