/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.server;

import java.io.Closeable;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Pattern;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.cache.Cache;
import org.apache.kafka.common.cache.LRUCache;
import org.apache.kafka.common.cache.SynchronizedCache;
import org.apache.kafka.common.errors.ApiException;
import org.apache.kafka.common.errors.InvalidRequestException;
import org.apache.kafka.common.errors.TelemetryTooLargeException;
import org.apache.kafka.common.errors.ThrottlingQuotaExceededException;
import org.apache.kafka.common.errors.UnknownSubscriptionIdException;
import org.apache.kafka.common.errors.UnsupportedCompressionTypeException;
import org.apache.kafka.common.message.GetTelemetrySubscriptionsResponseData;
import org.apache.kafka.common.message.PushTelemetryResponseData;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.requests.GetTelemetrySubscriptionsRequest;
import org.apache.kafka.common.requests.GetTelemetrySubscriptionsResponse;
import org.apache.kafka.common.requests.PushTelemetryRequest;
import org.apache.kafka.common.requests.PushTelemetryResponse;
import org.apache.kafka.common.requests.RequestContext;
import org.apache.kafka.common.utils.Crc32C;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.server.metrics.ClientMetricsConfigs;
import org.apache.kafka.server.metrics.ClientMetricsInstance;
import org.apache.kafka.server.metrics.ClientMetricsInstanceMetadata;
import org.apache.kafka.server.metrics.ClientMetricsReceiverPlugin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClientMetricsManager
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(ClientMetricsManager.class);
    private static final List<Byte> SUPPORTED_COMPRESSION_TYPES = Collections.unmodifiableList(Arrays.asList(CompressionType.ZSTD.id, CompressionType.LZ4.id, CompressionType.GZIP.id, CompressionType.SNAPPY.id));
    private static final int CM_CACHE_MAX_SIZE = 16384;
    private final ClientMetricsReceiverPlugin receiverPlugin;
    private final Cache<Uuid, ClientMetricsInstance> clientInstanceCache;
    private final Map<String, SubscriptionInfo> subscriptionMap;
    private final int clientTelemetryMaxBytes;
    private final Time time;
    private final AtomicInteger subscriptionUpdateVersion;

    public ClientMetricsManager(ClientMetricsReceiverPlugin receiverPlugin, int clientTelemetryMaxBytes, Time time) {
        this.receiverPlugin = receiverPlugin;
        this.subscriptionMap = new ConcurrentHashMap<String, SubscriptionInfo>();
        this.subscriptionUpdateVersion = new AtomicInteger(0);
        this.clientInstanceCache = new SynchronizedCache((Cache)new LRUCache(16384));
        this.clientTelemetryMaxBytes = clientTelemetryMaxBytes;
        this.time = time;
    }

    public Set<String> listClientMetricsResources() {
        return this.subscriptionMap.keySet();
    }

    public void updateSubscription(String subscriptionName, Properties properties) {
        ClientMetricsConfigs.validate(subscriptionName, properties);
        if (properties.isEmpty()) {
            if (this.subscriptionMap.containsKey(subscriptionName)) {
                log.info("Removing subscription [{}] from the subscription map", (Object)subscriptionName);
                this.subscriptionMap.remove(subscriptionName);
                this.subscriptionUpdateVersion.incrementAndGet();
            }
            return;
        }
        this.updateClientSubscription(subscriptionName, new ClientMetricsConfigs(properties));
        this.subscriptionUpdateVersion.incrementAndGet();
    }

    public GetTelemetrySubscriptionsResponse processGetTelemetrySubscriptionRequest(GetTelemetrySubscriptionsRequest request, RequestContext requestContext) {
        long now = this.time.milliseconds();
        Uuid clientInstanceId = Optional.ofNullable(request.data().clientInstanceId()).filter(id -> !id.equals((Object)Uuid.ZERO_UUID)).orElse(this.generateNewClientId());
        ClientMetricsInstance clientInstance = this.clientInstance(clientInstanceId, requestContext);
        try {
            this.validateGetRequest(request, clientInstance, now);
        }
        catch (ApiException exception) {
            return request.getErrorResponse(0, (Throwable)exception);
        }
        clientInstance.lastKnownError(Errors.NONE);
        return this.createGetSubscriptionResponse(clientInstanceId, clientInstance);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public PushTelemetryResponse processPushTelemetryRequest(PushTelemetryRequest request, RequestContext requestContext) {
        Uuid clientInstanceId = request.data().clientInstanceId();
        if (clientInstanceId == null || Uuid.RESERVED.contains(clientInstanceId)) {
            String msg = String.format("Invalid request from the client [%s], invalid client instance id", clientInstanceId);
            return request.getErrorResponse(0, (Throwable)new InvalidRequestException(msg));
        }
        long now = this.time.milliseconds();
        ClientMetricsInstance clientInstance = this.clientInstance(clientInstanceId, requestContext);
        try {
            this.validatePushRequest(request, clientInstance, now);
        }
        catch (ApiException exception) {
            log.debug("Error validating push telemetry request from client [{}]", (Object)clientInstanceId, (Object)exception);
            clientInstance.lastKnownError(Errors.forException((Throwable)exception));
            PushTelemetryResponse pushTelemetryResponse = request.getErrorResponse(0, (Throwable)exception);
            return pushTelemetryResponse;
        }
        finally {
            clientInstance.terminating(request.data().terminating());
        }
        byte[] metrics = request.data().metrics();
        if (metrics != null && metrics.length > 0) {
            try {
                this.receiverPlugin.exportMetrics(requestContext, request);
            }
            catch (Exception exception) {
                clientInstance.lastKnownError(Errors.INVALID_RECORD);
                return request.errorResponse(0, Errors.INVALID_RECORD);
            }
        }
        clientInstance.lastKnownError(Errors.NONE);
        return new PushTelemetryResponse(new PushTelemetryResponseData());
    }

    public boolean isTelemetryReceiverConfigured() {
        return !this.receiverPlugin.isEmpty();
    }

    @Override
    public void close() throws IOException {
        this.subscriptionMap.clear();
    }

    private void updateClientSubscription(String subscriptionName, ClientMetricsConfigs configs) {
        List metrics = configs.getList("metrics");
        int pushInterval = configs.getInt("interval.ms");
        List clientMatchPattern = configs.getList("match");
        SubscriptionInfo newSubscription = new SubscriptionInfo(subscriptionName, metrics, pushInterval, ClientMetricsConfigs.parseMatchingPatterns(clientMatchPattern));
        this.subscriptionMap.put(subscriptionName, newSubscription);
    }

    private Uuid generateNewClientId() {
        Uuid id = Uuid.randomUuid();
        while (this.clientInstanceCache.get((Object)id) != null) {
            id = Uuid.randomUuid();
        }
        return id;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ClientMetricsInstance clientInstance(Uuid clientInstanceId, RequestContext requestContext) {
        ClientMetricsInstance clientInstance = (ClientMetricsInstance)this.clientInstanceCache.get((Object)clientInstanceId);
        if (clientInstance == null) {
            ClientMetricsManager clientMetricsManager = this;
            synchronized (clientMetricsManager) {
                clientInstance = (ClientMetricsInstance)this.clientInstanceCache.get((Object)clientInstanceId);
                if (clientInstance != null) {
                    return clientInstance;
                }
                ClientMetricsInstanceMetadata instanceMetadata = new ClientMetricsInstanceMetadata(clientInstanceId, requestContext);
                clientInstance = this.createClientInstanceAndUpdateCache(clientInstanceId, instanceMetadata);
            }
        }
        if (clientInstance.subscriptionVersion() < this.subscriptionUpdateVersion.get()) {
            ClientMetricsManager clientMetricsManager = this;
            synchronized (clientMetricsManager) {
                clientInstance = (ClientMetricsInstance)this.clientInstanceCache.get((Object)clientInstanceId);
                if (clientInstance.subscriptionVersion() >= this.subscriptionUpdateVersion.get()) {
                    return clientInstance;
                }
                clientInstance = this.createClientInstanceAndUpdateCache(clientInstanceId, clientInstance.instanceMetadata());
            }
        }
        return clientInstance;
    }

    private ClientMetricsInstance createClientInstanceAndUpdateCache(Uuid clientInstanceId, ClientMetricsInstanceMetadata instanceMetadata) {
        ClientMetricsInstance clientInstance = this.createClientInstance(clientInstanceId, instanceMetadata);
        this.clientInstanceCache.put((Object)clientInstanceId, (Object)clientInstance);
        return clientInstance;
    }

    private ClientMetricsInstance createClientInstance(Uuid clientInstanceId, ClientMetricsInstanceMetadata instanceMetadata) {
        int pushIntervalMs = 300000;
        HashSet<String> subscribedMetrics = new HashSet<String>();
        boolean allMetricsSubscribed = false;
        int currentSubscriptionVersion = this.subscriptionUpdateVersion.get();
        for (SubscriptionInfo info : this.subscriptionMap.values()) {
            if (!instanceMetadata.isMatch(info.matchPattern())) continue;
            allMetricsSubscribed = allMetricsSubscribed || info.metrics().contains("*");
            subscribedMetrics.addAll(info.metrics());
            pushIntervalMs = Math.min(pushIntervalMs, info.intervalMs());
        }
        if (allMetricsSubscribed) {
            subscribedMetrics.clear();
            subscribedMetrics.add("*");
        }
        int subscriptionId = this.computeSubscriptionId(subscribedMetrics, pushIntervalMs, clientInstanceId);
        return new ClientMetricsInstance(clientInstanceId, instanceMetadata, subscriptionId, currentSubscriptionVersion, subscribedMetrics, pushIntervalMs);
    }

    private int computeSubscriptionId(Set<String> metrics, int pushIntervalMs, Uuid clientInstanceId) {
        byte[] metricsBytes = (metrics.toString() + pushIntervalMs).getBytes(StandardCharsets.UTF_8);
        long computedCrc = Crc32C.compute((byte[])metricsBytes, (int)0, (int)metricsBytes.length);
        return (int)computedCrc ^ clientInstanceId.hashCode();
    }

    private GetTelemetrySubscriptionsResponse createGetSubscriptionResponse(Uuid clientInstanceId, ClientMetricsInstance clientInstance) {
        GetTelemetrySubscriptionsResponseData data = new GetTelemetrySubscriptionsResponseData().setClientInstanceId(clientInstanceId).setSubscriptionId(clientInstance.subscriptionId()).setRequestedMetrics(new ArrayList<String>(clientInstance.metrics())).setAcceptedCompressionTypes(SUPPORTED_COMPRESSION_TYPES).setPushIntervalMs(clientInstance.pushIntervalMs()).setTelemetryMaxBytes(this.clientTelemetryMaxBytes).setDeltaTemporality(true).setErrorCode(Errors.NONE.code());
        return new GetTelemetrySubscriptionsResponse(data);
    }

    private void validateGetRequest(GetTelemetrySubscriptionsRequest request, ClientMetricsInstance clientInstance, long timestamp) {
        if (!(clientInstance.maybeUpdateGetRequestTimestamp(timestamp) || clientInstance.lastKnownError() == Errors.UNKNOWN_SUBSCRIPTION_ID && clientInstance.lastKnownError() == Errors.UNSUPPORTED_COMPRESSION_TYPE)) {
            String msg = String.format("Request from the client [%s] arrived before the next push interval time", request.data().clientInstanceId());
            throw new ThrottlingQuotaExceededException(msg);
        }
    }

    private void validatePushRequest(PushTelemetryRequest request, ClientMetricsInstance clientInstance, long timestamp) {
        if (clientInstance.terminating()) {
            String msg = String.format("Client [%s] sent the previous request with state terminating to TRUE, can not acceptany requests after that", request.data().clientInstanceId());
            throw new InvalidRequestException(msg);
        }
        if (!clientInstance.maybeUpdatePushRequestTimestamp(timestamp) && !request.data().terminating()) {
            String msg = String.format("Request from the client [%s] arrived before the next push interval time", request.data().clientInstanceId());
            throw new ThrottlingQuotaExceededException(msg);
        }
        if (request.data().subscriptionId() != clientInstance.subscriptionId()) {
            String msg = String.format("Unknown client subscription id for the client [%s]", request.data().clientInstanceId());
            throw new UnknownSubscriptionIdException(msg);
        }
        if (!ClientMetricsManager.isSupportedCompressionType(request.data().compressionType())) {
            String msg = String.format("Unknown compression type [%s] is received in telemetry request from [%s]", request.data().compressionType(), request.data().clientInstanceId());
            throw new UnsupportedCompressionTypeException(msg);
        }
        if (request.data().metrics() != null && request.data().metrics().length > this.clientTelemetryMaxBytes) {
            String msg = String.format("Telemetry request from [%s] is larger than the maximum allowed size [%s]", request.data().clientInstanceId(), this.clientTelemetryMaxBytes);
            throw new TelemetryTooLargeException(msg);
        }
    }

    private static boolean isSupportedCompressionType(int id) {
        try {
            CompressionType.forId((int)id);
            return true;
        }
        catch (IllegalArgumentException e) {
            return false;
        }
    }

    SubscriptionInfo subscriptionInfo(String subscriptionName) {
        return this.subscriptionMap.get(subscriptionName);
    }

    Collection<SubscriptionInfo> subscriptions() {
        return Collections.unmodifiableCollection(this.subscriptionMap.values());
    }

    ClientMetricsInstance clientInstance(Uuid clientInstanceId) {
        return (ClientMetricsInstance)this.clientInstanceCache.get((Object)clientInstanceId);
    }

    int subscriptionUpdateVersion() {
        return this.subscriptionUpdateVersion.get();
    }

    public static class SubscriptionInfo {
        private final String name;
        private final Set<String> metrics;
        private final int intervalMs;
        private final Map<String, Pattern> matchPattern;

        public SubscriptionInfo(String name, List<String> metrics, int intervalMs, Map<String, Pattern> matchPattern) {
            this.name = name;
            this.metrics = new HashSet<String>(metrics);
            this.intervalMs = intervalMs;
            this.matchPattern = matchPattern;
        }

        public String name() {
            return this.name;
        }

        public Set<String> metrics() {
            return this.metrics;
        }

        public int intervalMs() {
            return this.intervalMs;
        }

        public Map<String, Pattern> matchPattern() {
            return this.matchPattern;
        }
    }
}

