/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients.consumer.internals;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.stream.Collectors;
import org.apache.kafka.clients.consumer.ConsumerRebalanceListener;
import org.apache.kafka.clients.consumer.internals.CommitRequestManager;
import org.apache.kafka.clients.consumer.internals.ConsumerMetadata;
import org.apache.kafka.clients.consumer.internals.ConsumerRebalanceListenerMethodName;
import org.apache.kafka.clients.consumer.internals.MemberState;
import org.apache.kafka.clients.consumer.internals.MemberStateListener;
import org.apache.kafka.clients.consumer.internals.MembershipManager;
import org.apache.kafka.clients.consumer.internals.NetworkClientDelegate;
import org.apache.kafka.clients.consumer.internals.SubscriptionState;
import org.apache.kafka.clients.consumer.internals.Utils;
import org.apache.kafka.clients.consumer.internals.events.BackgroundEventHandler;
import org.apache.kafka.clients.consumer.internals.events.ConsumerRebalanceListenerCallbackCompletedEvent;
import org.apache.kafka.clients.consumer.internals.events.ConsumerRebalanceListenerCallbackNeededEvent;
import org.apache.kafka.clients.consumer.internals.metrics.RebalanceMetricsManager;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicIdPartition;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.message.ConsumerGroupHeartbeatResponseData;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.telemetry.internals.ClientTelemetryReporter;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;

public class MembershipManagerImpl
implements MembershipManager {
    static final Utils.TopicPartitionComparator TOPIC_PARTITION_COMPARATOR = new Utils.TopicPartitionComparator();
    static final Utils.TopicIdPartitionComparator TOPIC_ID_PARTITION_COMPARATOR = new Utils.TopicIdPartitionComparator();
    private final String groupId;
    private final Optional<String> groupInstanceId;
    private final int rebalanceTimeoutMs;
    private String memberId = "";
    private int memberEpoch = 0;
    private MemberState state;
    private final Optional<String> serverAssignor;
    private MembershipManager.LocalAssignment currentAssignment;
    private final SubscriptionState subscriptions;
    private final ConsumerMetadata metadata;
    private final Logger log;
    private final CommitRequestManager commitRequestManager;
    private final Map<Uuid, String> assignedTopicNamesCache;
    private MembershipManager.LocalAssignment currentTargetAssignment;
    private boolean reconciliationInProgress;
    private boolean rejoinedWhileReconciliationInProgress;
    private Optional<CompletableFuture<Void>> leaveGroupInProgress = Optional.empty();
    private final List<MemberStateListener> stateUpdatesListeners;
    private final Optional<ClientTelemetryReporter> clientTelemetryReporter;
    private final BackgroundEventHandler backgroundEventHandler;
    private CompletableFuture<Void> staleMemberAssignmentRelease;
    private final RebalanceMetricsManager metricsManager;
    private final Time time;
    private boolean isPollTimerExpired;

    public MembershipManagerImpl(String groupId, Optional<String> groupInstanceId, int rebalanceTimeoutMs, Optional<String> serverAssignor, SubscriptionState subscriptions, CommitRequestManager commitRequestManager, ConsumerMetadata metadata, LogContext logContext, Optional<ClientTelemetryReporter> clientTelemetryReporter, BackgroundEventHandler backgroundEventHandler, Time time, Metrics metrics) {
        this(groupId, groupInstanceId, rebalanceTimeoutMs, serverAssignor, subscriptions, commitRequestManager, metadata, logContext, clientTelemetryReporter, backgroundEventHandler, time, new RebalanceMetricsManager(metrics));
    }

    MembershipManagerImpl(String groupId, Optional<String> groupInstanceId, int rebalanceTimeoutMs, Optional<String> serverAssignor, SubscriptionState subscriptions, CommitRequestManager commitRequestManager, ConsumerMetadata metadata, LogContext logContext, Optional<ClientTelemetryReporter> clientTelemetryReporter, BackgroundEventHandler backgroundEventHandler, Time time, RebalanceMetricsManager metricsManager) {
        this.groupId = groupId;
        this.state = MemberState.UNSUBSCRIBED;
        this.serverAssignor = serverAssignor;
        this.groupInstanceId = groupInstanceId;
        this.subscriptions = subscriptions;
        this.commitRequestManager = commitRequestManager;
        this.metadata = metadata;
        this.assignedTopicNamesCache = new HashMap<Uuid, String>();
        this.currentTargetAssignment = MembershipManager.LocalAssignment.NONE;
        this.currentAssignment = MembershipManager.LocalAssignment.NONE;
        this.log = logContext.logger(MembershipManagerImpl.class);
        this.stateUpdatesListeners = new ArrayList<MemberStateListener>();
        this.clientTelemetryReporter = clientTelemetryReporter;
        this.rebalanceTimeoutMs = rebalanceTimeoutMs;
        this.backgroundEventHandler = backgroundEventHandler;
        this.time = time;
        this.metricsManager = metricsManager;
    }

    private void transitionTo(MemberState nextState) {
        if (!this.state.equals((Object)nextState) && !nextState.getPreviousValidStates().contains((Object)this.state)) {
            throw new IllegalStateException(String.format("Invalid state transition from %s to %s", new Object[]{this.state, nextState}));
        }
        if (MembershipManagerImpl.isCompletingRebalance(this.state, nextState)) {
            this.metricsManager.recordRebalanceEnded(this.time.milliseconds());
        }
        if (MembershipManagerImpl.isStartingRebalance(this.state, nextState)) {
            this.metricsManager.recordRebalanceStarted(this.time.milliseconds());
        }
        this.log.trace("Member {} with epoch {} transitioned from {} to {}.", new Object[]{this.memberId, this.memberEpoch, this.state, nextState});
        this.state = nextState;
    }

    private static boolean isCompletingRebalance(MemberState currentState, MemberState nextState) {
        return currentState == MemberState.RECONCILING && (nextState == MemberState.STABLE || nextState == MemberState.ACKNOWLEDGING);
    }

    private static boolean isStartingRebalance(MemberState currentState, MemberState nextState) {
        return currentState != MemberState.RECONCILING && nextState == MemberState.RECONCILING;
    }

    @Override
    public String groupId() {
        return this.groupId;
    }

    @Override
    public Optional<String> groupInstanceId() {
        return this.groupInstanceId;
    }

    @Override
    public String memberId() {
        return this.memberId;
    }

    @Override
    public int memberEpoch() {
        return this.memberEpoch;
    }

    @Override
    public void onHeartbeatSuccess(ConsumerGroupHeartbeatResponseData response) {
        if (response.errorCode() != Errors.NONE.code()) {
            String errorMessage = String.format("Unexpected error in Heartbeat response. Expected no error, but received: %s", new Object[]{Errors.forCode(response.errorCode())});
            throw new IllegalArgumentException(errorMessage);
        }
        MemberState state = this.state();
        if (state == MemberState.LEAVING) {
            this.log.debug("Ignoring heartbeat response received from broker. Member {} with epoch {} is already leaving the group.", (Object)this.memberId, (Object)this.memberEpoch);
            return;
        }
        if (this.isNotInGroup()) {
            this.log.debug("Ignoring heartbeat response received from broker. Member {} is in {} state so it's not a member of the group. ", (Object)this.memberId, (Object)state);
            return;
        }
        if (response.memberId() != null && !response.memberId().equals(this.memberId)) {
            this.clientTelemetryReporter.ifPresent(reporter -> reporter.updateMetricsLabels(Collections.singletonMap("group_member_id", response.memberId())));
        }
        this.memberId = response.memberId();
        this.updateMemberEpoch(response.memberEpoch());
        ConsumerGroupHeartbeatResponseData.Assignment assignment = response.assignment();
        if (assignment != null) {
            if (!state.canHandleNewAssignment()) {
                this.log.debug("Ignoring new assignment {} received from server because member is in {} state.", (Object)assignment, (Object)state);
                return;
            }
            this.processAssignmentReceived(assignment);
        }
    }

    @Override
    public void onHeartbeatFailure() {
        this.metricsManager.maybeRecordRebalanceFailed();
    }

    private boolean isNotInGroup() {
        return this.state == MemberState.UNSUBSCRIBED || this.state == MemberState.FENCED || this.state == MemberState.FATAL || this.state == MemberState.STALE;
    }

    private void processAssignmentReceived(ConsumerGroupHeartbeatResponseData.Assignment assignment) {
        this.replaceTargetAssignmentWithNewAssignment(assignment);
        if (!this.targetAssignmentReconciled()) {
            this.transitionTo(MemberState.RECONCILING);
        } else {
            this.log.debug("Target assignment {} received from the broker is equals to the member current assignment {}. Nothing to reconcile.", (Object)this.currentTargetAssignment, (Object)this.currentAssignment);
            if (this.state == MemberState.RECONCILING || this.state == MemberState.JOINING) {
                this.transitionTo(MemberState.STABLE);
            }
        }
    }

    private void replaceTargetAssignmentWithNewAssignment(ConsumerGroupHeartbeatResponseData.Assignment assignment) {
        this.currentTargetAssignment.updateWith(assignment).ifPresent(updatedAssignment -> {
            this.log.debug("Target assignment updated from {} to {}. Member will reconcile it on the next poll.", (Object)this.currentTargetAssignment, updatedAssignment);
            this.currentTargetAssignment = updatedAssignment;
        });
    }

    @Override
    public void transitionToFenced() {
        if (this.state == MemberState.PREPARE_LEAVING) {
            this.log.info("Member {} with epoch {} got fenced but it is already preparing to leave the group, so it will stop sending heartbeat and won't attempt to rejoin.", (Object)this.memberId, (Object)this.memberEpoch);
            this.transitionToSendingLeaveGroup(false);
            this.transitionTo(MemberState.UNSUBSCRIBED);
            return;
        }
        if (this.state == MemberState.LEAVING) {
            this.log.debug("Member {} with epoch {} got fenced but it is already leaving the group with state {}, so it won't attempt to rejoin.", new Object[]{this.memberId, this.memberEpoch, this.state});
            return;
        }
        if (this.state == MemberState.UNSUBSCRIBED) {
            this.log.debug("Member {} with epoch {} got fenced but it already left the group, so it won't attempt to rejoin.", (Object)this.memberId, (Object)this.memberEpoch);
            return;
        }
        this.transitionTo(MemberState.FENCED);
        this.resetEpoch();
        this.log.debug("Member {} with epoch {} transitioned to {} state. It will release its assignment and rejoin the group.", new Object[]{this.memberId, this.memberEpoch, MemberState.FENCED});
        CompletableFuture<Void> callbackResult = this.invokeOnPartitionsLostCallback(this.subscriptions.assignedPartitions());
        callbackResult.whenComplete((result, error) -> {
            if (error != null) {
                this.log.error("onPartitionsLost callback invocation failed while releasing assignment after member got fenced. Member will rejoin the group anyways.", error);
            }
            this.clearAssignment();
            if (this.state == MemberState.FENCED) {
                this.transitionToJoining();
            } else {
                this.log.debug("Fenced member onPartitionsLost callback completed but the state has already changed to {}, so the member won't rejoin the group", (Object)this.state);
            }
        });
    }

    @Override
    public void transitionToFatal() {
        MemberState previousState = this.state;
        this.transitionTo(MemberState.FATAL);
        if (this.memberId.isEmpty()) {
            this.log.error("Member {} with epoch {} transitioned to {} state", new Object[]{this.memberId, this.memberEpoch, MemberState.FATAL});
        } else {
            this.log.error("Non-member transitioned to {} state", (Object)MemberState.FATAL);
        }
        this.notifyEpochChange(Optional.empty(), Optional.empty());
        if (previousState == MemberState.UNSUBSCRIBED) {
            this.log.debug("Member {} with epoch {} got fatal error from the broker but it already left the group, so onPartitionsLost callback won't be triggered.", (Object)this.memberId, (Object)this.memberEpoch);
            return;
        }
        CompletableFuture<Void> callbackResult = this.invokeOnPartitionsLostCallback(this.subscriptions.assignedPartitions());
        callbackResult.whenComplete((result, error) -> {
            if (error != null) {
                this.log.error("onPartitionsLost callback invocation failed while releasing assignmentafter member failed with fatal error.", error);
            }
            this.clearAssignment();
        });
    }

    @Override
    public void onSubscriptionUpdated() {
        if (this.state == MemberState.UNSUBSCRIBED) {
            this.transitionToJoining();
        }
    }

    private void clearAssignment() {
        if (this.subscriptions.hasAutoAssignedPartitions()) {
            this.subscriptions.assignFromSubscribed(Collections.emptySet());
        }
        this.currentAssignment = MembershipManager.LocalAssignment.NONE;
        this.clearPendingAssignmentsAndLocalNamesCache();
    }

    private void updateSubscriptionAwaitingCallback(SortedSet<TopicIdPartition> assignedPartitions, SortedSet<TopicPartition> addedPartitions) {
        SortedSet<TopicPartition> assignedTopicPartitions = this.toTopicPartitionSet(assignedPartitions);
        this.subscriptions.assignFromSubscribedAwaitingCallback(assignedTopicPartitions, addedPartitions);
    }

    @Override
    public void transitionToJoining() {
        if (this.state == MemberState.FATAL) {
            this.log.warn("No action taken to join the group with the updated subscription because the member is in FATAL state");
            return;
        }
        if (this.reconciliationInProgress) {
            this.rejoinedWhileReconciliationInProgress = true;
        }
        this.resetEpoch();
        this.transitionTo(MemberState.JOINING);
        this.clearPendingAssignmentsAndLocalNamesCache();
    }

    @Override
    public CompletableFuture<Void> leaveGroup() {
        if (this.isNotInGroup()) {
            if (this.state == MemberState.FENCED) {
                this.clearAssignment();
                this.transitionTo(MemberState.UNSUBSCRIBED);
            }
            return CompletableFuture.completedFuture(null);
        }
        if (this.state == MemberState.PREPARE_LEAVING || this.state == MemberState.LEAVING) {
            this.log.debug("Leave group operation already in progress for member {}", (Object)this.memberId);
            return this.leaveGroupInProgress.get();
        }
        this.transitionTo(MemberState.PREPARE_LEAVING);
        CompletableFuture<Void> leaveResult = new CompletableFuture<Void>();
        this.leaveGroupInProgress = Optional.of(leaveResult);
        CompletableFuture<Void> callbackResult = this.invokeOnPartitionsRevokedOrLostToReleaseAssignment();
        callbackResult.whenComplete((result, error) -> {
            if (error != null) {
                this.log.error("Member {} callback to release assignment failed. Member will proceed to send leave group heartbeat", (Object)this.memberId, error);
            } else {
                this.log.debug("Member {} completed callback to release assignment and will send leave group heartbeat", (Object)this.memberId);
            }
            this.subscriptions.unsubscribe();
            this.clearAssignment();
            this.transitionToSendingLeaveGroup(false);
        });
        return leaveResult;
    }

    private CompletableFuture<Void> invokeOnPartitionsRevokedOrLostToReleaseAssignment() {
        TreeSet<TopicPartition> droppedPartitions = new TreeSet<TopicPartition>(TOPIC_PARTITION_COMPARATOR);
        droppedPartitions.addAll(this.subscriptions.assignedPartitions());
        this.log.debug("Member {} is triggering callbacks to release assignment {} and leave group", (Object)this.memberId, droppedPartitions);
        CompletableFuture<Object> callbackResult = droppedPartitions.isEmpty() ? CompletableFuture.completedFuture(null) : (this.memberEpoch > 0 ? this.revokePartitions(droppedPartitions) : this.invokeOnPartitionsLostCallback(droppedPartitions));
        return callbackResult;
    }

    @Override
    public void transitionToSendingLeaveGroup(boolean dueToExpiredPollTimer) {
        if (this.state == MemberState.FATAL) {
            this.log.warn("Member {} with epoch {} won't send leave group request because it is in FATAL state", (Object)this.memberId, (Object)this.memberEpoch);
            return;
        }
        if (this.state == MemberState.UNSUBSCRIBED) {
            this.log.warn("Member {} won't send leave group request because it is already out of the group.", (Object)this.memberId);
            return;
        }
        if (dueToExpiredPollTimer) {
            this.isPollTimerExpired = true;
            this.transitionTo(MemberState.PREPARE_LEAVING);
        }
        int leaveEpoch = this.groupInstanceId.isPresent() ? -2 : -1;
        this.updateMemberEpoch(leaveEpoch);
        this.currentAssignment = MembershipManager.LocalAssignment.NONE;
        this.transitionTo(MemberState.LEAVING);
    }

    void notifyEpochChange(Optional<Integer> epoch, Optional<String> memberId) {
        this.stateUpdatesListeners.forEach(stateListener -> stateListener.onMemberEpochUpdated(epoch, memberId));
    }

    @Override
    public boolean shouldHeartbeatNow() {
        MemberState state = this.state();
        return state == MemberState.ACKNOWLEDGING || state == MemberState.LEAVING || state == MemberState.JOINING;
    }

    @Override
    public void onHeartbeatRequestSent() {
        MemberState state = this.state();
        if (state == MemberState.ACKNOWLEDGING) {
            if (this.targetAssignmentReconciled()) {
                this.transitionTo(MemberState.STABLE);
            } else {
                this.log.debug("Member {} with epoch {} transitioned to {} after a heartbeat was sent to ack a previous reconciliation. New assignments are ready to be reconciled.", new Object[]{this.memberId, this.memberEpoch, MemberState.RECONCILING});
                this.transitionTo(MemberState.RECONCILING);
            }
        } else if (state == MemberState.LEAVING) {
            if (this.isPollTimerExpired) {
                this.log.debug("Member {} sent heartbeat to leave due to expired poll timer. It will remain stale (no heartbeat) until it rejoins the group on the next consumer poll.", (Object)this.memberId);
                this.transitionToStale();
            } else {
                this.log.debug("Member {} sent heartbeat to leave group.", (Object)this.memberId);
                this.transitionToUnsubscribed();
            }
        }
    }

    @Override
    public void onHeartbeatRequestSkipped() {
        if (this.state == MemberState.LEAVING) {
            this.log.debug("Heartbeat for leaving group could not be sent. Member {} with epoch {} will transition to {}.", new Object[]{this.memberId, this.memberEpoch, MemberState.UNSUBSCRIBED});
            this.transitionToUnsubscribed();
        }
    }

    private void transitionToUnsubscribed() {
        this.transitionTo(MemberState.UNSUBSCRIBED);
        this.leaveGroupInProgress.get().complete(null);
        this.leaveGroupInProgress = Optional.empty();
    }

    private boolean targetAssignmentReconciled() {
        return this.currentAssignment.equals(this.currentTargetAssignment);
    }

    @Override
    public boolean shouldSkipHeartbeat() {
        MemberState state = this.state();
        return state == MemberState.UNSUBSCRIBED || state == MemberState.FATAL || state == MemberState.STALE || state == MemberState.FENCED;
    }

    @Override
    public boolean isLeavingGroup() {
        MemberState state = this.state();
        return state == MemberState.PREPARE_LEAVING || state == MemberState.LEAVING;
    }

    @Override
    public void maybeRejoinStaleMember() {
        this.isPollTimerExpired = false;
        if (this.state == MemberState.STALE) {
            this.log.debug("Expired poll timer has been reset so stale member {} will rejoin the groupwhen it completes releasing its previous assignment.", (Object)this.memberId);
            this.staleMemberAssignmentRelease.whenComplete((__, error) -> this.transitionToJoining());
        }
    }

    private void transitionToStale() {
        this.transitionTo(MemberState.STALE);
        CompletableFuture<Void> callbackResult = this.invokeOnPartitionsLostCallback(this.subscriptions.assignedPartitions());
        this.staleMemberAssignmentRelease = callbackResult.whenComplete((result, error) -> {
            if (error != null) {
                this.log.error("onPartitionsLost callback invocation failed while releasing assignment after member left group due to expired poll timer.", error);
            }
            this.clearAssignment();
            this.log.debug("Member {} sent leave group heartbeat and released its assignment. It will remain in {} state until the poll timer is reset, and it will then rejoin the group", (Object)this.memberId, (Object)MemberState.STALE);
        });
    }

    void maybeReconcile() {
        if (this.targetAssignmentReconciled()) {
            this.log.trace("Ignoring reconciliation attempt. Target assignment is equal to the current assignment.");
            return;
        }
        if (this.reconciliationInProgress) {
            this.log.trace("Ignoring reconciliation attempt. Another reconciliation is already in progress. Assignment " + this.currentTargetAssignment + " will be handled in the next reconciliation loop.");
            return;
        }
        SortedSet<TopicIdPartition> assignedTopicIdPartitions = this.findResolvableAssignmentAndTriggerMetadataUpdate();
        MembershipManager.LocalAssignment resolvedAssignment = new MembershipManager.LocalAssignment(this.currentTargetAssignment.localEpoch, assignedTopicIdPartitions);
        if (!this.currentAssignment.isNone() && resolvedAssignment.partitions.equals(this.currentAssignment.partitions)) {
            this.log.debug("There are unresolved partitions, and the resolvable fragment of the target assignment {} is equal to the current assignment. Bumping the local epoch of the assignment and acknowledging the partially resolved assignment", resolvedAssignment.partitions);
            this.currentAssignment = resolvedAssignment;
            this.transitionTo(MemberState.ACKNOWLEDGING);
            return;
        }
        this.markReconciliationInProgress();
        SortedSet<TopicPartition> assignedTopicPartitions = this.toTopicPartitionSet(assignedTopicIdPartitions);
        TreeSet<TopicPartition> ownedPartitions = new TreeSet<TopicPartition>(TOPIC_PARTITION_COMPARATOR);
        ownedPartitions.addAll(this.subscriptions.assignedPartitions());
        TreeSet<TopicPartition> addedPartitions = new TreeSet<TopicPartition>(TOPIC_PARTITION_COMPARATOR);
        addedPartitions.addAll(assignedTopicPartitions);
        addedPartitions.removeAll(ownedPartitions);
        TreeSet<TopicPartition> revokedPartitions = new TreeSet<TopicPartition>(TOPIC_PARTITION_COMPARATOR);
        revokedPartitions.addAll(ownedPartitions);
        revokedPartitions.removeAll(assignedTopicPartitions);
        this.log.info("Reconciling assignment with local epoch {}\n\tMember:                                    {}\n\tAssigned partitions:                       {}\n\tCurrent owned partitions:                  {}\n\tAdded partitions (assigned - owned):       {}\n\tRevoked partitions (owned - assigned):     {}\n", new Object[]{this.memberId, resolvedAssignment.localEpoch, assignedTopicPartitions, ownedPartitions, addedPartitions, revokedPartitions});
        CompletableFuture<Void> commitResult = this.commitRequestManager.maybeAutoCommitSyncBeforeRevocation(this.getDeadlineMsForTimeout(this.rebalanceTimeoutMs));
        ((CompletableFuture)commitResult.whenComplete((__, commitReqError) -> {
            if (commitReqError != null) {
                this.log.error("Auto-commit request before reconciling new assignment failed. Will proceed with the reconciliation anyway.", commitReqError);
            } else {
                this.log.debug("Auto-commit before reconciling new assignment completed successfully.");
            }
            if (!this.maybeAbortReconciliation()) {
                this.revokeAndAssign(resolvedAssignment, assignedTopicIdPartitions, revokedPartitions, addedPartitions);
            }
        })).exceptionally(error -> {
            if (error != null) {
                this.log.error("Reconciliation failed.", error);
            }
            return null;
        });
    }

    long getDeadlineMsForTimeout(long timeoutMs) {
        long expiration = this.time.milliseconds() + timeoutMs;
        if (expiration < 0L) {
            return Long.MAX_VALUE;
        }
        return expiration;
    }

    private void revokeAndAssign(MembershipManager.LocalAssignment resolvedAssignment, SortedSet<TopicIdPartition> assignedTopicIdPartitions, SortedSet<TopicPartition> revokedPartitions, SortedSet<TopicPartition> addedPartitions) {
        CompletableFuture<Object> revocationResult = !revokedPartitions.isEmpty() ? this.revokePartitions(revokedPartitions) : CompletableFuture.completedFuture(null);
        CompletionStage reconciliationResult = revocationResult.thenCompose(__ -> {
            if (!this.maybeAbortReconciliation()) {
                return this.assignPartitions(assignedTopicIdPartitions, addedPartitions);
            }
            return CompletableFuture.completedFuture(null);
        });
        ((CompletableFuture)reconciliationResult).whenComplete((__, error) -> {
            if (error != null) {
                this.log.error("Reconciliation failed.", error);
                this.markReconciliationCompleted();
            } else if (this.reconciliationInProgress && !this.maybeAbortReconciliation()) {
                this.currentAssignment = resolvedAssignment;
                this.commitRequestManager.resetAutoCommitTimer();
                this.transitionTo(MemberState.ACKNOWLEDGING);
                this.markReconciliationCompleted();
            }
        });
    }

    boolean maybeAbortReconciliation() {
        boolean shouldAbort;
        boolean bl = shouldAbort = this.state != MemberState.RECONCILING || this.rejoinedWhileReconciliationInProgress;
        if (shouldAbort) {
            String reason = this.rejoinedWhileReconciliationInProgress ? "the member has re-joined the group" : "the member already transitioned out of the reconciling state into " + (Object)((Object)this.state);
            this.log.info("Interrupting reconciliation that is not relevant anymore because " + reason);
            this.markReconciliationCompleted();
        }
        return shouldAbort;
    }

    void updateAssignment(Map<Uuid, SortedSet<Integer>> partitions) {
        this.currentAssignment = new MembershipManager.LocalAssignment(0L, partitions);
    }

    private SortedSet<TopicPartition> toTopicPartitionSet(SortedSet<TopicIdPartition> topicIdPartitions) {
        TreeSet<TopicPartition> result = new TreeSet<TopicPartition>(TOPIC_PARTITION_COMPARATOR);
        topicIdPartitions.forEach(topicIdPartition -> result.add(topicIdPartition.topicPartition()));
        return result;
    }

    void markReconciliationInProgress() {
        this.reconciliationInProgress = true;
        this.rejoinedWhileReconciliationInProgress = false;
    }

    void markReconciliationCompleted() {
        this.reconciliationInProgress = false;
        this.rejoinedWhileReconciliationInProgress = false;
    }

    private SortedSet<TopicIdPartition> findResolvableAssignmentAndTriggerMetadataUpdate() {
        TreeSet<TopicIdPartition> assignmentReadyToReconcile = new TreeSet<TopicIdPartition>(TOPIC_ID_PARTITION_COMPARATOR);
        HashMap<Uuid, SortedSet<Integer>> unresolved = new HashMap<Uuid, SortedSet<Integer>>(this.currentTargetAssignment.partitions);
        Iterator<Map.Entry<Uuid, SortedSet<Integer>>> it = unresolved.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<Uuid, SortedSet<Integer>> e = it.next();
            Uuid topicId = e.getKey();
            SortedSet<Integer> topicPartitions = e.getValue();
            Optional<String> nameFromMetadata = this.findTopicNameInGlobalOrLocalCache(topicId);
            nameFromMetadata.ifPresent(resolvedTopicName -> {
                topicPartitions.forEach(tp -> assignmentReadyToReconcile.add(new TopicIdPartition(topicId, (int)tp, (String)resolvedTopicName)));
                it.remove();
            });
        }
        if (!unresolved.isEmpty()) {
            this.log.debug("Topic Ids {} received in target assignment were not found in metadata and are not currently assigned. Requesting a metadata update now to resolve topic names.", unresolved.keySet());
            this.metadata.requestUpdate(true);
        }
        return assignmentReadyToReconcile;
    }

    private Optional<String> findTopicNameInGlobalOrLocalCache(Uuid topicId) {
        String nameFromMetadataCache = this.metadata.topicNames().getOrDefault(topicId, null);
        if (nameFromMetadataCache != null) {
            this.assignedTopicNamesCache.put(topicId, nameFromMetadataCache);
            return Optional.of(nameFromMetadataCache);
        }
        String nameFromSubscriptionCache = this.assignedTopicNamesCache.getOrDefault(topicId, null);
        return Optional.ofNullable(nameFromSubscriptionCache);
    }

    CompletableFuture<Void> revokePartitions(Set<TopicPartition> partitionsToRevoke) {
        HashSet<TopicPartition> revokedPartitions = new HashSet<TopicPartition>(partitionsToRevoke);
        revokedPartitions.retainAll(this.subscriptions.assignedPartitions());
        this.log.info("Revoking previously assigned partitions {}", (Object)revokedPartitions.stream().map(TopicPartition::toString).collect(Collectors.joining(", ")));
        this.logPausedPartitionsBeingRevoked(revokedPartitions);
        this.markPendingRevocationToPauseFetching(revokedPartitions);
        CompletableFuture<Void> revocationResult = new CompletableFuture<Void>();
        if (this.state == MemberState.FATAL) {
            String errorMsg = String.format("Member %s with epoch %s received a fatal error while waiting for a revocation commit to complete. Will abort revocation without triggering user callback.", this.memberId, this.memberEpoch);
            this.log.debug(errorMsg);
            revocationResult.completeExceptionally(new KafkaException(errorMsg));
            return revocationResult;
        }
        CompletableFuture<Void> userCallbackResult = this.invokeOnPartitionsRevokedCallback(revokedPartitions);
        userCallbackResult.whenComplete((callbackResult, callbackError) -> {
            if (callbackError != null) {
                this.log.error("onPartitionsRevoked callback invocation failed for partitions {}", (Object)revokedPartitions, callbackError);
                revocationResult.completeExceptionally((Throwable)callbackError);
            } else {
                revocationResult.complete(null);
            }
        });
        return revocationResult;
    }

    private CompletableFuture<Void> assignPartitions(SortedSet<TopicIdPartition> assignedPartitions, SortedSet<TopicPartition> addedPartitions) {
        this.updateSubscriptionAwaitingCallback(assignedPartitions, addedPartitions);
        CompletableFuture<Void> result = this.invokeOnPartitionsAssignedCallback(addedPartitions);
        result.whenComplete((__, exception) -> {
            if (exception == null) {
                this.subscriptions.enablePartitionsAwaitingCallback(addedPartitions);
            } else if (!addedPartitions.isEmpty()) {
                this.log.warn("Leaving newly assigned partitions {} marked as non-fetchable and not requiring initializing positions after onPartitionsAssigned callback failed.", (Object)addedPartitions, exception);
            }
        });
        Set assignedTopics = assignedPartitions.stream().map(TopicIdPartition::topic).collect(Collectors.toSet());
        this.assignedTopicNamesCache.values().retainAll(assignedTopics);
        return result;
    }

    private void markPendingRevocationToPauseFetching(Set<TopicPartition> partitionsToRevoke) {
        this.log.debug("Marking partitions pending for revocation: {}", partitionsToRevoke);
        this.subscriptions.markPendingRevocation(partitionsToRevoke);
    }

    private CompletableFuture<Void> invokeOnPartitionsRevokedCallback(Set<TopicPartition> partitionsRevoked) {
        Optional<ConsumerRebalanceListener> listener = this.subscriptions.rebalanceListener();
        if (!partitionsRevoked.isEmpty() && listener.isPresent()) {
            return this.enqueueConsumerRebalanceListenerCallback(ConsumerRebalanceListenerMethodName.ON_PARTITIONS_REVOKED, partitionsRevoked);
        }
        return CompletableFuture.completedFuture(null);
    }

    private CompletableFuture<Void> invokeOnPartitionsAssignedCallback(Set<TopicPartition> partitionsAssigned) {
        Optional<ConsumerRebalanceListener> listener = this.subscriptions.rebalanceListener();
        if (listener.isPresent()) {
            return this.enqueueConsumerRebalanceListenerCallback(ConsumerRebalanceListenerMethodName.ON_PARTITIONS_ASSIGNED, partitionsAssigned);
        }
        return CompletableFuture.completedFuture(null);
    }

    private CompletableFuture<Void> invokeOnPartitionsLostCallback(Set<TopicPartition> partitionsLost) {
        Optional<ConsumerRebalanceListener> listener = this.subscriptions.rebalanceListener();
        if (!partitionsLost.isEmpty() && listener.isPresent()) {
            return this.enqueueConsumerRebalanceListenerCallback(ConsumerRebalanceListenerMethodName.ON_PARTITIONS_LOST, partitionsLost);
        }
        return CompletableFuture.completedFuture(null);
    }

    private CompletableFuture<Void> enqueueConsumerRebalanceListenerCallback(ConsumerRebalanceListenerMethodName methodName, Set<TopicPartition> partitions) {
        TreeSet<TopicPartition> sortedPartitions = new TreeSet<TopicPartition>(TOPIC_PARTITION_COMPARATOR);
        sortedPartitions.addAll(partitions);
        ConsumerRebalanceListenerCallbackNeededEvent event = new ConsumerRebalanceListenerCallbackNeededEvent(methodName, sortedPartitions);
        this.backgroundEventHandler.add(event);
        this.log.debug("The event to trigger the {} method execution was enqueued successfully", (Object)methodName.fullyQualifiedMethodName());
        return event.future();
    }

    @Override
    public void consumerRebalanceListenerCallbackCompleted(ConsumerRebalanceListenerCallbackCompletedEvent event) {
        ConsumerRebalanceListenerMethodName methodName = event.methodName();
        Optional<KafkaException> error = event.error();
        CompletableFuture<Void> future = event.future();
        if (error.isPresent()) {
            Exception e = error.get();
            this.log.warn("The {} method completed with an error ({}); signaling to continue to the next phase of rebalance", (Object)methodName.fullyQualifiedMethodName(), (Object)e.getMessage());
            future.completeExceptionally(e);
        } else {
            this.log.debug("The {} method completed successfully; signaling to continue to the next phase of rebalance", (Object)methodName.fullyQualifiedMethodName());
            future.complete(null);
        }
    }

    private void logPausedPartitionsBeingRevoked(Set<TopicPartition> partitionsToRevoke) {
        Set<TopicPartition> revokePausedPartitions = this.subscriptions.pausedPartitions();
        revokePausedPartitions.retainAll(partitionsToRevoke);
        if (!revokePausedPartitions.isEmpty()) {
            this.log.info("The pause flag in partitions [{}] will be removed due to revocation.", (Object)revokePausedPartitions.stream().map(TopicPartition::toString).collect(Collectors.joining(", ")));
        }
    }

    private void clearPendingAssignmentsAndLocalNamesCache() {
        this.currentTargetAssignment = MembershipManager.LocalAssignment.NONE;
        this.assignedTopicNamesCache.clear();
    }

    private void resetEpoch() {
        this.updateMemberEpoch(0);
    }

    private void updateMemberEpoch(int newEpoch) {
        boolean newEpochReceived = this.memberEpoch != newEpoch;
        this.memberEpoch = newEpoch;
        if (newEpochReceived) {
            if (this.memberEpoch > 0) {
                this.notifyEpochChange(Optional.of(this.memberEpoch), Optional.ofNullable(this.memberId));
            } else {
                this.notifyEpochChange(Optional.empty(), Optional.empty());
            }
        }
    }

    @Override
    public MemberState state() {
        return this.state;
    }

    @Override
    public Optional<String> serverAssignor() {
        return this.serverAssignor;
    }

    @Override
    public MembershipManager.LocalAssignment currentAssignment() {
        return this.currentAssignment;
    }

    Set<Uuid> topicsAwaitingReconciliation() {
        return this.topicPartitionsAwaitingReconciliation().keySet();
    }

    Map<Uuid, SortedSet<Integer>> topicPartitionsAwaitingReconciliation() {
        if (this.currentTargetAssignment == MembershipManager.LocalAssignment.NONE) {
            return Collections.emptyMap();
        }
        if (this.currentAssignment == MembershipManager.LocalAssignment.NONE) {
            return this.currentTargetAssignment.partitions;
        }
        HashMap topicPartitionMap = new HashMap();
        this.currentTargetAssignment.partitions.forEach((topicId, targetPartitions) -> {
            SortedSet<Integer> reconciledPartitions = this.currentAssignment.partitions.get(topicId);
            if (!targetPartitions.equals(reconciledPartitions)) {
                TreeSet missingPartitions = new TreeSet(targetPartitions);
                if (reconciledPartitions != null) {
                    missingPartitions.removeAll(reconciledPartitions);
                }
                topicPartitionMap.put(topicId, missingPartitions);
            }
        });
        return Collections.unmodifiableMap(topicPartitionMap);
    }

    boolean reconciliationInProgress() {
        return this.reconciliationInProgress;
    }

    @Override
    public void registerStateListener(MemberStateListener listener) {
        if (listener == null) {
            throw new IllegalArgumentException("State updates listener cannot be null");
        }
        this.stateUpdatesListeners.add(listener);
    }

    @Override
    public NetworkClientDelegate.PollResult poll(long currentTimeMs) {
        if (this.state == MemberState.RECONCILING) {
            this.maybeReconcile();
        }
        return NetworkClientDelegate.PollResult.EMPTY;
    }

    List<MemberStateListener> stateListeners() {
        return Collections.unmodifiableList(this.stateUpdatesListeners);
    }
}

