/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
'use strict';
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __asyncValues = (this && this.__asyncValues) || function (o) {
    if (!Symbol.asyncIterator) throw new TypeError("Symbol.asyncIterator is not defined.");
    var m = o[Symbol.asyncIterator], i;
    return m ? m.call(o) : (o = typeof __values === "function" ? __values(o) : o[Symbol.iterator](), i = {}, verb("next"), verb("throw"), verb("return"), i[Symbol.asyncIterator] = function () { return this; }, i);
    function verb(n) { i[n] = o[n] && function (v) { return new Promise(function (resolve, reject) { v = o[n](v), settle(resolve, reject, v.done, v.value); }); }; }
    function settle(resolve, reject, d, v) { Promise.resolve(v).then(function(v) { resolve({ value: v, done: d }); }, reject); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deactivate = exports.activate = exports.awaitClient = exports.findClusters = exports.enableConsoleLog = exports.clearDebugConsoleListeners = exports.NbLanguageClient = exports.debugConsoleListeners = exports.TEST_PROGRESS_EVENT = exports.MINIMAL_JDK_VERSION = exports.clientRuntimeJDK = exports.client = exports.listeners = exports.COMMAND_PREFIX = void 0;
const vscode_1 = require("vscode");
const node_1 = require("vscode-languageclient/node");
const vscode_languageclient_1 = require("vscode-languageclient");
const net = require("net");
const fs = require("fs");
const path = require("path");
const child_process_1 = require("child_process");
const vscode = require("vscode");
const launcher = require("./nbcode");
const testAdapter_1 = require("./testAdapter");
const protocol_1 = require("./protocol");
const launchConfigurations = require("./launchConfigurations");
const explorer_1 = require("./explorer");
const runConfiguration_1 = require("./runConfiguration");
const dbConfigurationProvider_1 = require("./dbConfigurationProvider");
const utils_1 = require("./utils");
const propertiesView_1 = require("./propertiesView/propertiesView");
const configuration = require("./jdk/configuration");
const jdk = require("./jdk/jdk");
const validation_1 = require("./jdk/validation/validation");
const sshGuide = require("./panels/SshGuidePanel");
const runImageGuide = require("./panels/RunImageGuidePanel");
const guidesUtil_1 = require("./panels/guidesUtil");
const ssh_1 = require("./ssh/ssh");
const API_VERSION = "1.0";
exports.COMMAND_PREFIX = "nbls";
const DATABASE = 'Database';
exports.listeners = new Map();
exports.clientRuntimeJDK = null;
exports.MINIMAL_JDK_VERSION = 17;
exports.TEST_PROGRESS_EVENT = "testProgress";
const TEST_ADAPTER_CREATED_EVENT = "testAdapterCreated";
let testAdapter;
let nbProcess = null;
let debugPort = -1;
let consoleLog = !!process.env['ENABLE_CONSOLE_LOG'];
let specifiedJDKWarned = [];
exports.debugConsoleListeners = [];
class NbLanguageClient extends node_1.LanguageClient {
    constructor(id, name, s, log, c) {
        super(id, name, s, c);
        this._treeViewService = (0, explorer_1.createTreeViewService)(log, this);
    }
    findTreeViewService() {
        return this._treeViewService;
    }
    stop() {
        // stop will be called even in case of external close & client restart, so OK.
        const r = super.stop();
        this._treeViewService.dispose();
        return r;
    }
}
exports.NbLanguageClient = NbLanguageClient;
function handleLog(log, msg) {
    log.appendLine(msg);
    if (consoleLog) {
        console.log(msg);
    }
}
function handleLogNoNL(log, msg) {
    log.append(msg);
    if (consoleLog) {
        process.stdout.write(msg);
    }
}
function clearDebugConsoleListeners() {
    exports.debugConsoleListeners = [];
}
exports.clearDebugConsoleListeners = clearDebugConsoleListeners;
function enableConsoleLog() {
    consoleLog = true;
    console.log("enableConsoleLog");
}
exports.enableConsoleLog = enableConsoleLog;
function findClusters(myPath) {
    let clusters = [];
    for (let e of vscode.extensions.all) {
        if (e.extensionPath === myPath) {
            continue;
        }
        const dir = path.join(e.extensionPath, 'nbcode');
        if (!fs.existsSync(dir)) {
            continue;
        }
        const exists = fs.readdirSync(dir);
        for (let clusterName of exists) {
            let clusterPath = path.join(dir, clusterName);
            let clusterModules = path.join(clusterPath, 'config', 'Modules');
            if (!fs.existsSync(clusterModules)) {
                continue;
            }
            let perm = fs.statSync(clusterModules);
            if (perm.isDirectory()) {
                clusters.push(clusterPath);
            }
        }
    }
    return clusters;
}
exports.findClusters = findClusters;
// for tests only !
function awaitClient() {
    const c = exports.client;
    if (c && !(c instanceof InitialPromise)) {
        return c;
    }
    let nbcode = vscode.extensions.getExtension('asf.apache-netbeans-java');
    if (!nbcode) {
        return Promise.reject(new Error("Extension not installed."));
    }
    const t = nbcode.activate().then(nc => {
        if (exports.client === undefined || exports.client instanceof InitialPromise) {
            throw new Error("Client not available");
        }
        else {
            return exports.client;
        }
    });
    return Promise.resolve(t);
}
exports.awaitClient = awaitClient;
function findJDK(onChange) {
    let nowDark = isDarkColorTheme();
    let nowJavaEnabled = isJavaSupportEnabled();
    function find() {
        let nbJdk = vscode_1.workspace.getConfiguration('netbeans').get('jdkhome');
        if (nbJdk) {
            return nbJdk;
        }
        let javahome = vscode_1.workspace.getConfiguration('java').get('home');
        if (javahome) {
            return javahome;
        }
        let jdkHome = process.env.JDK_HOME;
        if (jdkHome) {
            return jdkHome;
        }
        let jHome = process.env.JAVA_HOME;
        if (jHome) {
            return jHome;
        }
        return null;
    }
    let currentJdk = find();
    let projectJdk = getProjectJDKHome();
    (0, validation_1.validateJDKCompatibility)(projectJdk);
    let timeout = undefined;
    vscode_1.workspace.onDidChangeConfiguration(params => {
        if (timeout) {
            return;
        }
        let interested = false;
        if (params.affectsConfiguration('netbeans') || params.affectsConfiguration('java')) {
            interested = true;
        }
        else if (params.affectsConfiguration('workbench.colorTheme')) {
            let d = isDarkColorTheme();
            if (d != nowDark) {
                interested = true;
            }
        }
        if (!interested) {
            return;
        }
        timeout = setTimeout(() => {
            timeout = undefined;
            let newJdk = find();
            let newD = isDarkColorTheme();
            let newJavaEnabled = isJavaSupportEnabled();
            let newProjectJDK = getProjectJDKHome();
            if (newJdk !== currentJdk || newD != nowDark || newJavaEnabled != nowJavaEnabled || newProjectJDK != projectJdk) {
                nowDark = newD;
                nowJavaEnabled = newJavaEnabled;
                currentJdk = newJdk;
                projectJdk = newProjectJDK;
                onChange(currentJdk);
            }
        }, 0);
    });
    onChange(currentJdk);
}
function contextUri(ctx) {
    var _a, _b;
    if (ctx === null || ctx === void 0 ? void 0 : ctx.fsPath) {
        return ctx;
    }
    else if (ctx === null || ctx === void 0 ? void 0 : ctx.resourceUri) {
        return ctx.resourceUri;
    }
    else if (typeof ctx == 'string') {
        try {
            return vscode.Uri.parse(ctx, true);
        }
        catch (err) {
            return vscode.Uri.file(ctx);
        }
    }
    return (_b = (_a = vscode.window.activeTextEditor) === null || _a === void 0 ? void 0 : _a.document) === null || _b === void 0 ? void 0 : _b.uri;
}
/**
 * Executes a project action. It is possible to provide an explicit configuration to use (or undefined), display output from the action etc.
 * Arguments are attempted to parse as file or editor references or Nodes; otherwise they are attempted to be passed to the action as objects.
 *
 * @param action ID of the project action to run
 * @param configuration configuration to use or undefined - use default/active one.
 * @param title Title for the progress displayed in vscode
 * @param log output channel that should be revealed
 * @param showOutput if true, reveals the passed output channel
 * @param args additional arguments
 * @returns Promise for the command's result
 */
function wrapProjectActionWithProgress(action, configuration, title, log, showOutput, ...args) {
    let items = [];
    let actionParams = {
        action: action,
        configuration: configuration,
    };
    for (let item of args) {
        let u;
        if (item === null || item === void 0 ? void 0 : item.fsPath) {
            items.push(item.fsPath.toString());
        }
        else if (item === null || item === void 0 ? void 0 : item.resourceUri) {
            items.push(item.resourceUri.toString());
        }
        else {
            items.push(item);
        }
    }
    return wrapCommandWithProgress(exports.COMMAND_PREFIX + '.project.run.action', title, log, showOutput, actionParams, ...items);
}
function wrapCommandWithProgress(lsCommand, title, log, showOutput, ...args) {
    return vscode_1.window.withProgress({ location: vscode_1.ProgressLocation.Window }, p => {
        return new Promise((resolve, reject) => __awaiter(this, void 0, void 0, function* () {
            let c = yield exports.client;
            const docsTosave = vscode.workspace.textDocuments.
                filter(d => fs.existsSync(d.uri.fsPath)).
                map(d => d.save());
            yield Promise.all(docsTosave);
            const commands = yield vscode.commands.getCommands();
            if (commands.includes(lsCommand)) {
                p.report({ message: title });
                c.outputChannel.show(true);
                const start = new Date().getTime();
                try {
                    if (log) {
                        handleLog(log, `starting ${lsCommand}`);
                    }
                    const res = yield vscode.commands.executeCommand(lsCommand, ...args);
                    const elapsed = new Date().getTime() - start;
                    if (log) {
                        handleLog(log, `finished ${lsCommand} in ${elapsed} ms with result ${res}`);
                    }
                    const humanVisibleDelay = elapsed < 1000 ? 1000 : 0;
                    setTimeout(() => {
                        if (res) {
                            resolve(res);
                        }
                        else {
                            if (log) {
                                handleLog(log, `Command ${lsCommand} takes too long to start`);
                            }
                            reject(res);
                        }
                    }, humanVisibleDelay);
                }
                catch (err) {
                    if (log) {
                        handleLog(log, `command ${lsCommand} executed with error: ${JSON.stringify(err)}`);
                    }
                    reject(err && typeof err.message === 'string' ? err.message : "Error");
                }
            }
            else {
                reject(`cannot run ${lsCommand}; client is ${c}`);
            }
        }));
    });
}
/**
 * Just a simple promise subclass, so I can test for the 'initial promise' value:
 * unlike all other promises, that must be fullfilled in order to e.g. properly stop the server or otherwise communicate with it,
 * the initial one needs to be largely ignored in the launching/mgmt code, BUT should be available to normal commands / features.
 */
class InitialPromise extends Promise {
    constructor(f) {
        super(f);
    }
}
/**
 * Determines the outcome, if there's a conflict betwee RH Java and us: disable java, enable java, ask the user.
 * @returns false, if java should be disablde; true, if enabled. Undefined if no config is present, ask the user
 */
function shouldEnableConflictingJavaSupport() {
    var _a, _b;
    // backwards compatibility; remove in NBLS 19
    if (vscode.extensions.getExtension('oracle-labs-graalvm.gcn')) {
        return false;
    }
    let r = undefined;
    for (const ext of vscode.extensions.all) {
        const services = ((_a = ext.packageJSON) === null || _a === void 0 ? void 0 : _a.contributes) && ((_b = ext.packageJSON) === null || _b === void 0 ? void 0 : _b.contributes['netbeans.options']);
        if (!services) {
            continue;
        }
        if (services['javaSupport.conflict'] !== undefined) {
            const v = !!services['javaSupport.conflict'];
            if (!v) {
                // request to disable wins.
                return false;
            }
            r = v;
        }
    }
    return r;
}
function getValueAfterPrefix(input, prefix) {
    if (input === undefined) {
        return "";
    }
    const parts = input.split(' ');
    for (let i = 0; i < parts.length; i++) {
        if (parts[i].startsWith(prefix)) {
            return parts[i].substring(prefix.length);
        }
    }
    return '';
}
class LineBufferingPseudoterminal {
    constructor(name) {
        this.writeEmitter = new vscode.EventEmitter();
        this.onDidWrite = this.writeEmitter.event;
        this.closeEmitter = new vscode.EventEmitter();
        this.onDidClose = this.closeEmitter.event;
        this.buffer = '';
        this.isOpen = false;
        this.name = name;
    }
    open() {
        this.isOpen = true;
    }
    close() {
        this.isOpen = false;
        this.closeEmitter.fire();
    }
    /**
     * Accepts partial input strings and logs complete lines when they are formed.
     * Also processes carriage returns (\r) to overwrite the current line.
     * @param input The string input to the pseudoterminal.
     */
    acceptInput(input) {
        if (!this.isOpen) {
            return;
        }
        for (const char of input) {
            if (char === '\n') {
                // Process a newline: log the current buffer and reset it
                this.logLine(this.buffer.trim());
                this.buffer = '';
            }
            else if (char === '\r') {
                // Process a carriage return: log the current buffer on the same line
                this.logInline(this.buffer.trim());
                this.buffer = '';
            }
            else {
                // Append characters to the buffer
                this.buffer += char;
            }
        }
    }
    logLine(line) {
        console.log('[Gradle Debug]', line.toString());
        this.writeEmitter.fire(`${line}\r\n`);
    }
    logInline(line) {
        // Clear the current line and move the cursor to the start
        this.writeEmitter.fire(`\x1b[2K\x1b[1G${line}`);
    }
    flushBuffer() {
        if (this.buffer.trim().length > 0) {
            this.logLine(this.buffer.trim());
            this.buffer = '';
        }
    }
    clear() {
        this.writeEmitter.fire('\x1b[2J\x1b[3J\x1b[H'); // Clear screen and move cursor to top-left
    }
    show() {
        if (!this.terminal) {
            this.terminal = vscode.window.createTerminal({
                name: this.name,
                pty: this,
            });
            // Listen for terminal close events
            vscode.window.onDidCloseTerminal((closedTerminal) => {
                if (closedTerminal === this.terminal) {
                    this.terminal = undefined; // Clear the terminal reference
                }
            });
        }
        // Prevent 'stealing' of the focus when running tests in parallel 
        if (!(testAdapter === null || testAdapter === void 0 ? void 0 : testAdapter.testInParallelProfileExist())) {
            this.terminal.show(true);
        }
    }
    /**
     * Gets an existing instance or creates a new one by the terminal name.
     * The terminal is also created and managed internally.
     * @param name The name of the pseudoterminal.
     * @returns The instance of the pseudoterminal.
     */
    static getInstance(name) {
        if (!this.instances.has(name)) {
            const instance = new LineBufferingPseudoterminal(name);
            this.instances.set(name, instance);
        }
        const instance = this.instances.get(name);
        instance.show();
        return instance;
    }
}
LineBufferingPseudoterminal.instances = new Map();
function activate(context) {
    const provider = new StringContentProvider();
    const scheme = 'in-memory';
    const providerRegistration = vscode.workspace.registerTextDocumentContentProvider(scheme, provider);
    context.subscriptions.push(vscode.commands.registerCommand('cloud.assets.policy.create', function (viewItem) {
        return __awaiter(this, void 0, void 0, function* () {
            const POLICIES_PREVIEW = 'Open a preview of the OCI policies';
            const POLICIES_UPLOAD = 'Upload the OCI Policies to OCI';
            const selected = yield vscode_1.window.showQuickPick([POLICIES_PREVIEW, POLICIES_UPLOAD], { placeHolder: 'Select a target for the OCI policies' });
            if (selected == POLICIES_UPLOAD) {
                yield vscode.commands.executeCommand('nbls.cloud.assets.policy.upload');
                return;
            }
            const content = yield vscode.commands.executeCommand('nbls.cloud.assets.policy.create.local');
            const document = vscode.Uri.parse(`${scheme}:policies.txt?${encodeURIComponent(content)}`);
            vscode.workspace.openTextDocument(document).then(doc => {
                vscode.window.showTextDocument(doc, { preview: false });
            });
        });
    }));
    context.subscriptions.push(vscode.commands.registerCommand('cloud.assets.config.create', function (viewItem) {
        return __awaiter(this, void 0, void 0, function* () {
            const CONFIG_LOCAL = 'Open a preview of the config in the editor';
            const CONFIG_TO_DEVOPS_CM = 'Upload the config to a ConfigMap artifact whithin an OCI DevOps Project';
            const CONFIG_TO_OKE_CM = 'Upload the config to a ConfigMap artifact whithin OKE cluster';
            const selected = yield vscode_1.window.showQuickPick([CONFIG_LOCAL, CONFIG_TO_OKE_CM, CONFIG_TO_DEVOPS_CM], { placeHolder: 'Select a target for the config' });
            if (selected == CONFIG_TO_DEVOPS_CM) {
                yield vscode_1.commands.executeCommand('nbls.cloud.assets.configmap.devops.upload');
                return;
            }
            else if (selected == CONFIG_TO_OKE_CM) {
                yield vscode_1.commands.executeCommand('nbls.cloud.assets.configmap.upload');
                return;
            }
            const content = yield vscode.commands.executeCommand('nbls.cloud.assets.config.create.local');
            const document = vscode.Uri.parse(`${scheme}:application.properties?${encodeURIComponent(content)}`);
            vscode.workspace.openTextDocument(document).then(doc => {
                vscode.window.showTextDocument(doc, { preview: false });
            });
        });
    }));
    let log = vscode.window.createOutputChannel("Apache NetBeans Language Server");
    var clientResolve;
    var clientReject;
    // establish a waitable Promise, export the callbacks so they can be called after activation.
    exports.client = new InitialPromise((resolve, reject) => {
        clientResolve = resolve;
        clientReject = reject;
    });
    // we need to call refresh here as @OnStart in NBLS is called before the workspace projects are opened.
    exports.client.then(() => {
        vscode.commands.executeCommand('nbls.cloud.assets.refresh');
    });
    function checkConflict() {
        let conf = vscode_1.workspace.getConfiguration();
        if (conf.get("netbeans.conflict.check")) {
            if (conf.get("netbeans.javaSupport.enabled")) {
                const e = shouldEnableConflictingJavaSupport();
                if (!e && vscode.extensions.getExtension('redhat.java')) {
                    if (e === false) {
                        // do not ask, an extension wants us to disable on conflict
                        conf.update("netbeans.javaSupport.enabled", false, true);
                    }
                    else {
                        const DISABLE_EXTENSION = `Manually disable extension`;
                        const DISABLE_JAVA = `Disable Java in Apache NetBeans Language Server`;
                        vscode.window.showInformationMessage(`Another Java support extension is already installed. It is recommended to use only one Java support per workspace.`, DISABLE_EXTENSION, DISABLE_JAVA).then((selected) => {
                            if (DISABLE_EXTENSION === selected) {
                                vscode.commands.executeCommand('workbench.extensions.action.showInstalledExtensions');
                            }
                            else if (DISABLE_JAVA === selected) {
                                conf.update("netbeans.javaSupport.enabled", false, true);
                            }
                        });
                    }
                }
            }
            else if (!vscode.extensions.getExtension('redhat.java')) {
                vscode_1.workspace.findFiles(`**/*.java`, undefined, 1).then(files => {
                    if (files.length) {
                        const ENABLE_JAVA = `Enable Java in Apache NetBeans Language Server`;
                        vscode.window.showInformationMessage(`Java in Apache NetBeans Language Server is disabled and no other Java support extension is currently installed.`, ENABLE_JAVA).then((selected) => {
                            if (ENABLE_JAVA === selected) {
                                conf.update("netbeans.javaSupport.enabled", true, true);
                            }
                        });
                    }
                });
            }
        }
    }
    checkConflict();
    // find acceptable JDK and launch the Java part
    findJDK((specifiedJDK) => __awaiter(this, void 0, void 0, function* () {
        const osExeSuffix = process.platform === 'win32' ? '.exe' : '';
        let jdkOK = true;
        let javaExecPath;
        if (!specifiedJDK) {
            javaExecPath = 'java';
        }
        else {
            javaExecPath = path.resolve(specifiedJDK, 'bin', 'java');
            jdkOK = fs.existsSync(path.resolve(specifiedJDK, 'bin', `java${osExeSuffix}`)) && fs.existsSync(path.resolve(specifiedJDK, 'bin', `javac${osExeSuffix}`));
        }
        if (jdkOK) {
            log.appendLine(`Verifying java: ${javaExecPath}`);
            // let the shell interpret PATH and .exe extension
            let javaCheck = (0, child_process_1.spawnSync)(`"${javaExecPath}" -version`, { shell: true });
            if (javaCheck.error || javaCheck.status) {
                jdkOK = false;
            }
            else {
                javaCheck.stderr.toString().split('\n').find(l => {
                    // yes, versions like 1.8 (up to 9) will be interpreted as 1, which is OK for < comparison
                    let re = /.* version \"([0-9]+)\.[^"]+\".*/.exec(l);
                    if (re) {
                        let versionNumber = Number(re[1]);
                        if (versionNumber < exports.MINIMAL_JDK_VERSION) {
                            jdkOK = false;
                        }
                    }
                });
            }
        }
        let warnedJDKs = specifiedJDKWarned;
        if (!jdkOK && !warnedJDKs.includes(specifiedJDK || '')) {
            const msg = specifiedJDK ?
                `The current path to JDK "${specifiedJDK}" may be invalid. A valid JDK ${exports.MINIMAL_JDK_VERSION}+ is required by Apache NetBeans Language Server to run.
                You should configure a proper JDK for Apache NetBeans and/or other technologies. Do you want to run JDK configuration now?` :
                `A valid JDK ${exports.MINIMAL_JDK_VERSION}+ is required by Apache NetBeans Language Server to run, but none was found. You should configure a proper JDK for Apache NetBeans and/or other technologies. ` +
                    'Do you want to run JDK configuration now?';
            const Y = "Yes";
            const N = "No";
            if ((yield vscode.window.showErrorMessage(msg, Y, N)) == Y) {
                vscode.commands.executeCommand('nbls.jdk.configuration');
                return;
            }
            else {
                warnedJDKs.push(specifiedJDK || '');
            }
        }
        let currentClusters = findClusters(context.extensionPath).sort();
        const dsSorter = (a, b) => {
            return (a.language || '').localeCompare(b.language || '')
                || (a.pattern || '').localeCompare(b.pattern || '')
                || (a.scheme || '').localeCompare(b.scheme || '');
        };
        let currentDocumentSelectors = collectDocumentSelectors().sort(dsSorter);
        context.subscriptions.push(vscode.extensions.onDidChange(() => {
            checkConflict();
            const newClusters = findClusters(context.extensionPath).sort();
            const newDocumentSelectors = collectDocumentSelectors().sort(dsSorter);
            if (newClusters.length !== currentClusters.length || newDocumentSelectors.length !== currentDocumentSelectors.length
                || newClusters.find((value, index) => value !== currentClusters[index]) || newDocumentSelectors.find((value, index) => value !== currentDocumentSelectors[index])) {
                currentClusters = newClusters;
                currentDocumentSelectors = newDocumentSelectors;
                activateWithJDK(specifiedJDK, context, log, true, clientResolve, clientReject);
            }
        }));
        activateWithJDK(specifiedJDK, context, log, true, clientResolve, clientReject);
    }));
    //register debugger:
    let debugTrackerFactory = new NetBeansDebugAdapterTrackerFactory();
    context.subscriptions.push(vscode.debug.registerDebugAdapterTrackerFactory('java+', debugTrackerFactory));
    let configInitialProvider = new NetBeansConfigurationInitialProvider();
    context.subscriptions.push(vscode.debug.registerDebugConfigurationProvider('java+', configInitialProvider, vscode.DebugConfigurationProviderTriggerKind.Initial));
    let configDynamicProvider = new NetBeansConfigurationDynamicProvider(context);
    context.subscriptions.push(vscode.debug.registerDebugConfigurationProvider('java+', configDynamicProvider, vscode.DebugConfigurationProviderTriggerKind.Dynamic));
    let configResolver = new NetBeansConfigurationResolver();
    context.subscriptions.push(vscode.debug.registerDebugConfigurationProvider('java+', configResolver));
    let configNativeResolver = new NetBeansConfigurationNativeResolver();
    context.subscriptions.push(vscode.debug.registerDebugConfigurationProvider('nativeimage', configNativeResolver));
    context.subscriptions.push(vscode.debug.onDidTerminateDebugSession(((session) => (0, dbConfigurationProvider_1.onDidTerminateSession)(session))));
    let debugDescriptionFactory = new NetBeansDebugAdapterDescriptionFactory();
    context.subscriptions.push(vscode.debug.registerDebugAdapterDescriptorFactory('java+', debugDescriptionFactory));
    context.subscriptions.push(vscode.debug.registerDebugAdapterDescriptorFactory('nativeimage', debugDescriptionFactory));
    // initialize Run Configuration
    (0, runConfiguration_1.initializeRunConfiguration)().then(initialized => {
        if (initialized) {
            context.subscriptions.push(vscode.debug.registerDebugConfigurationProvider('java+', dbConfigurationProvider_1.dBConfigurationProvider));
            context.subscriptions.push(vscode.debug.registerDebugConfigurationProvider('java', dbConfigurationProvider_1.dBConfigurationProvider));
            context.subscriptions.push(vscode.debug.registerDebugConfigurationProvider('java+', runConfiguration_1.runConfigurationProvider));
            context.subscriptions.push(vscode.debug.registerDebugConfigurationProvider('java', runConfiguration_1.runConfigurationProvider));
            context.subscriptions.push(vscode.window.registerTreeDataProvider('run-config', runConfiguration_1.runConfigurationNodeProvider));
            context.subscriptions.push(vscode.commands.registerCommand(exports.COMMAND_PREFIX + '.workspace.configureRunSettings', (...params) => {
                (0, runConfiguration_1.configureRunSettings)(context, params);
            }));
            vscode.commands.executeCommand('setContext', 'runConfigurationInitialized', true);
        }
    });
    // register commands
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.workspace.new', (ctx, template) => __awaiter(this, void 0, void 0, function* () {
        var _a, _b, _c, _d;
        let c = yield exports.client;
        const commands = yield vscode.commands.getCommands();
        if (commands.includes(exports.COMMAND_PREFIX + '.new.from.template')) {
            // first give the template (if present), then the context, and then the open-file hint in the case the context is not specific enough
            const params = [];
            if (typeof template === 'string') {
                params.push(template);
            }
            params.push((_a = contextUri(ctx)) === null || _a === void 0 ? void 0 : _a.toString(), (_d = (_c = (_b = vscode.window.activeTextEditor) === null || _b === void 0 ? void 0 : _b.document) === null || _c === void 0 ? void 0 : _c.uri) === null || _d === void 0 ? void 0 : _d.toString());
            const res = yield vscode.commands.executeCommand(exports.COMMAND_PREFIX + '.new.from.template', ...params);
            if (typeof res === 'string') {
                let newFile = vscode.Uri.parse(res);
                yield vscode.window.showTextDocument(newFile, { preview: false });
            }
            else if (Array.isArray(res)) {
                for (let r of res) {
                    if (typeof r === 'string') {
                        let newFile = vscode.Uri.parse(r);
                        yield vscode.window.showTextDocument(newFile, { preview: false });
                    }
                }
            }
        }
        else {
            throw `Client ${c} doesn't support new from template`;
        }
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.workspace.newproject', (ctx) => __awaiter(this, void 0, void 0, function* () {
        var _e;
        let c = yield exports.client;
        const commands = yield vscode.commands.getCommands();
        if (commands.includes(exports.COMMAND_PREFIX + '.new.project')) {
            const res = yield vscode.commands.executeCommand(exports.COMMAND_PREFIX + '.new.project', (_e = contextUri(ctx)) === null || _e === void 0 ? void 0 : _e.toString());
            if (typeof res === 'string') {
                let newProject = vscode.Uri.parse(res);
                const OPEN_IN_NEW_WINDOW = 'Open in new window';
                const ADD_TO_CURRENT_WORKSPACE = 'Add to current workspace';
                const value = yield vscode.window.showInformationMessage('New project created', OPEN_IN_NEW_WINDOW, ADD_TO_CURRENT_WORKSPACE);
                if (value === OPEN_IN_NEW_WINDOW) {
                    yield vscode.commands.executeCommand('vscode.openFolder', newProject, true);
                }
                else if (value === ADD_TO_CURRENT_WORKSPACE) {
                    vscode.workspace.updateWorkspaceFolders(vscode.workspace.workspaceFolders ? vscode.workspace.workspaceFolders.length : 0, undefined, { uri: newProject });
                }
            }
        }
        else {
            throw `Client ${c} doesn't support new project`;
        }
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.open.test', (ctx) => __awaiter(this, void 0, void 0, function* () {
        var e_1, _f;
        var _g, _h, _j;
        let c = yield exports.client;
        const commands = yield vscode.commands.getCommands();
        if (commands.includes(exports.COMMAND_PREFIX + '.go.to.test')) {
            try {
                const res = yield vscode.commands.executeCommand(exports.COMMAND_PREFIX + '.go.to.test', (_g = contextUri(ctx)) === null || _g === void 0 ? void 0 : _g.toString());
                if ("errorMessage" in res) {
                    throw new Error(res.errorMessage);
                }
                (_h = res === null || res === void 0 ? void 0 : res.providerErrors) === null || _h === void 0 ? void 0 : _h.map((error) => {
                    if (error === null || error === void 0 ? void 0 : error.message) {
                        vscode.window.showErrorMessage(error.message);
                    }
                });
                if ((_j = res === null || res === void 0 ? void 0 : res.locations) === null || _j === void 0 ? void 0 : _j.length) {
                    if (res.locations.length === 1) {
                        const { file, offset } = res.locations[0];
                        const filePath = vscode.Uri.parse(file);
                        const editor = yield vscode.window.showTextDocument(filePath, { preview: false });
                        if (offset != -1) {
                            const pos = editor.document.positionAt(offset);
                            editor.selections = [new vscode.Selection(pos, pos)];
                            const range = new vscode.Range(pos, pos);
                            editor.revealRange(range);
                        }
                    }
                    else {
                        const namePathMapping = {};
                        res.locations.forEach((fp) => {
                            const fileName = path.basename(fp.file);
                            namePathMapping[fileName] = fp.file;
                        });
                        const selected = yield vscode_1.window.showQuickPick(Object.keys(namePathMapping), {
                            title: 'Select files to open',
                            placeHolder: 'Test files or source files associated to each other',
                            canPickMany: true
                        });
                        if (selected) {
                            try {
                                for (var selected_1 = __asyncValues(selected), selected_1_1; selected_1_1 = yield selected_1.next(), !selected_1_1.done;) {
                                    const filePath = selected_1_1.value;
                                    let file = vscode.Uri.parse(filePath);
                                    yield vscode.window.showTextDocument(file, { preview: false });
                                }
                            }
                            catch (e_1_1) { e_1 = { error: e_1_1 }; }
                            finally {
                                try {
                                    if (selected_1_1 && !selected_1_1.done && (_f = selected_1.return)) yield _f.call(selected_1);
                                }
                                finally { if (e_1) throw e_1.error; }
                            }
                        }
                        else {
                            vscode.window.showInformationMessage("No file selected");
                        }
                    }
                }
            }
            catch (err) {
                vscode.window.showInformationMessage((err === null || err === void 0 ? void 0 : err.message) || "No Test or Tested class found");
            }
        }
        else {
            throw `Client ${c} doesn't support go to test`;
        }
    })));
    const trackerFactory = {
        createDebugAdapterTracker(_session) {
            return {
                onDidSendMessage: (message) => {
                    if (message.type === "event" && message.event === "output") {
                        const output = message.body.output;
                        exports.debugConsoleListeners.forEach((listener) => {
                            listener === null || listener === void 0 ? void 0 : listener.callback(output);
                        });
                    }
                }
            };
        }
    };
    context.subscriptions.push(vscode.debug.registerDebugAdapterTrackerFactory("*", trackerFactory));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.workspace.compile', () => wrapCommandWithProgress(exports.COMMAND_PREFIX + '.build.workspace', 'Compiling workspace...', log, true)));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.workspace.clean', () => wrapCommandWithProgress(exports.COMMAND_PREFIX + '.clean.workspace', 'Cleaning workspace...', log, true)));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.project.compile', (args) => {
        wrapProjectActionWithProgress('build', undefined, 'Compiling...', log, true, args);
    }));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.project.clean', (args) => {
        wrapProjectActionWithProgress('clean', undefined, 'Cleaning...', log, true, args);
    }));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.project.buildPushImage', () => {
        wrapCommandWithProgress(exports.COMMAND_PREFIX + '.cloud.assets.buildPushImage', 'Building and pushing container image', log, true);
    }));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.open.type', () => {
        wrapCommandWithProgress(exports.COMMAND_PREFIX + '.quick.open', 'Opening type...', log, true).then(() => {
            vscode_1.commands.executeCommand('workbench.action.focusActiveEditorGroup');
        });
    }));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.java.goto.super.implementation', () => __awaiter(this, void 0, void 0, function* () {
        var _k;
        if (((_k = vscode_1.window.activeTextEditor) === null || _k === void 0 ? void 0 : _k.document.languageId) !== "java") {
            return;
        }
        const uri = vscode_1.window.activeTextEditor.document.uri;
        const position = vscode_1.window.activeTextEditor.selection.active;
        const locations = (yield vscode.commands.executeCommand(exports.COMMAND_PREFIX + '.java.super.implementation', uri.toString(), position)) || [];
        return vscode.commands.executeCommand('editor.action.goToLocations', vscode_1.window.activeTextEditor.document.uri, position, locations.map(location => new vscode.Location(vscode.Uri.parse(location.uri), new vscode.Range(location.range.start.line, location.range.start.character, location.range.end.line, location.range.end.character))), 'peek', 'No super implementation found');
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.rename.element.at', (offset) => __awaiter(this, void 0, void 0, function* () {
        const editor = vscode_1.window.activeTextEditor;
        if (editor) {
            yield vscode_1.commands.executeCommand('editor.action.rename', [
                editor.document.uri,
                editor.document.positionAt(offset),
            ]);
        }
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.surround.with', (items) => __awaiter(this, void 0, void 0, function* () {
        const selected = yield vscode_1.window.showQuickPick(items, { placeHolder: 'Surround with ...' });
        if (selected) {
            if (selected.userData.edit) {
                const edit = yield (yield exports.client).protocol2CodeConverter.asWorkspaceEdit(selected.userData.edit);
                yield vscode_1.workspace.applyEdit(edit);
                yield vscode_1.commands.executeCommand('workbench.action.focusActiveEditorGroup');
            }
            yield vscode_1.commands.executeCommand(selected.userData.command.command, ...(selected.userData.command.arguments || []));
        }
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.db.add.all.connection', () => __awaiter(this, void 0, void 0, function* () {
        const ADD_JDBC = 'Add Database Connection';
        const ADD_ADB = 'Add Oracle Autonomous DB';
        const selected = yield vscode_1.window.showQuickPick([ADD_JDBC, ADD_ADB], { placeHolder: 'Select type...' });
        if (selected == ADD_JDBC) {
            yield vscode_1.commands.executeCommand('nbls.db.add.connection');
        }
        else if (selected == ADD_ADB) {
            yield vscode_1.commands.executeCommand('nbls:Tools:org.netbeans.modules.cloud.oracle.actions.AddADBAction');
        }
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.generate.code', (command, data) => __awaiter(this, void 0, void 0, function* () {
        const edit = yield vscode_1.commands.executeCommand(command, data);
        if (edit) {
            const wsEdit = yield (yield exports.client).protocol2CodeConverter.asWorkspaceEdit(edit);
            yield vscode_1.workspace.applyEdit(wsEdit);
            for (const entry of wsEdit.entries()) {
                const file = vscode.Uri.parse(entry[0].fsPath);
                yield vscode.window.showTextDocument(file, { preview: false });
            }
            yield vscode_1.commands.executeCommand('workbench.action.focusActiveEditorGroup');
        }
    })));
    function findRunConfiguration(uri) {
        return __awaiter(this, void 0, void 0, function* () {
            // do not invoke debug start with no (java+) configurations, as it would probably create an user prompt
            let cfg = vscode.workspace.getConfiguration("launch");
            let c = cfg.get('configurations');
            if (!Array.isArray(c)) {
                return undefined;
            }
            let f = c.filter((v) => v['type'] === 'java+');
            if (!f.length) {
                return undefined;
            }
            class P {
                resolveDebugConfigurationWithSubstitutedVariables(folder, debugConfiguration, token) {
                    this.config = debugConfiguration;
                    return undefined;
                }
            }
            let provider = new P();
            let d = vscode.debug.registerDebugConfigurationProvider('java+', provider);
            // let vscode to select a debug config
            return yield vscode.commands.executeCommand('workbench.action.debug.start', { config: {
                    type: 'java+',
                    mainClass: uri.toString()
                }, noDebug: true }).then((v) => {
                d.dispose();
                return provider.config;
            }, (err) => {
                d.dispose();
                return undefined;
            });
        });
    }
    const runDebug = (noDebug, testRun, uri, methodName, nestedClass, launchConfiguration, project = false, testInParallel = false, projects = undefined) => __awaiter(this, void 0, void 0, function* () {
        const docUri = contextUri(uri);
        if (docUri) {
            // attempt to find the active configuration in the vsode launch settings; undefined if no config is there.
            let debugConfig = (yield findRunConfiguration(docUri)) || {
                type: "java+",
                name: "Java Single Debug",
                request: "launch"
            };
            if (methodName) {
                debugConfig['methodName'] = methodName;
            }
            if (nestedClass) {
                debugConfig['nestedClass'] = nestedClass;
            }
            if (launchConfiguration == '') {
                if (debugConfig['launchConfiguration']) {
                    delete debugConfig['launchConfiguration'];
                }
            }
            else {
                debugConfig['launchConfiguration'] = launchConfiguration;
            }
            debugConfig['testRun'] = testRun;
            const workspaceFolder = vscode.workspace.getWorkspaceFolder(docUri);
            if (project || testRun) {
                debugConfig['projectFile'] = docUri.toString();
                debugConfig['project'] = true;
            }
            else {
                debugConfig['mainClass'] = docUri.toString();
            }
            const debugOptions = {
                noDebug: noDebug,
            };
            if (testInParallel) {
                debugConfig['testInParallel'] = testInParallel;
            }
            if (projects === null || projects === void 0 ? void 0 : projects.length) {
                debugConfig['projects'] = projects;
            }
            const ret = yield vscode.debug.startDebugging(workspaceFolder, debugConfig, debugOptions);
            return ret ? new Promise((resolve) => {
                const listener = vscode.debug.onDidTerminateDebugSession(() => {
                    listener.dispose();
                    resolve(true);
                });
            }) : ret;
        }
    });
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.run.test.parallel', (projects) => __awaiter(this, void 0, void 0, function* () {
        testAdapter === null || testAdapter === void 0 ? void 0 : testAdapter.runTestsWithParallelProfile(projects);
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.run.test.parallel.createProfile', (projects) => __awaiter(this, void 0, void 0, function* () {
        testAdapter === null || testAdapter === void 0 ? void 0 : testAdapter.registerRunInParallelProfile(projects);
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.run.test', (uri, methodName, launchConfiguration, nestedClass, testInParallel, projects) => __awaiter(this, void 0, void 0, function* () {
        yield runDebug(true, true, uri, methodName, nestedClass, launchConfiguration, false, testInParallel, projects);
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.debug.test', (uri, methodName, launchConfiguration, nestedClass) => __awaiter(this, void 0, void 0, function* () {
        yield runDebug(false, true, uri, methodName, nestedClass, launchConfiguration);
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.run.single', (uri, methodName, launchConfiguration, nestedClass) => __awaiter(this, void 0, void 0, function* () {
        yield runDebug(true, false, uri, methodName, nestedClass, launchConfiguration);
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.debug.single', (uri, methodName, launchConfiguration, nestedClass) => __awaiter(this, void 0, void 0, function* () {
        yield runDebug(false, false, uri, methodName, nestedClass, launchConfiguration);
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.project.run', (node, launchConfiguration) => __awaiter(this, void 0, void 0, function* () {
        var _l;
        return runDebug(true, false, ((_l = contextUri(node)) === null || _l === void 0 ? void 0 : _l.toString()) || '', undefined, undefined, launchConfiguration, true);
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.project.debug', (node, launchConfiguration) => __awaiter(this, void 0, void 0, function* () {
        var _m;
        return runDebug(false, false, ((_m = contextUri(node)) === null || _m === void 0 ? void 0 : _m.toString()) || '', undefined, undefined, launchConfiguration, true);
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.project.test', (node, launchConfiguration) => __awaiter(this, void 0, void 0, function* () {
        var _o;
        return runDebug(true, true, ((_o = contextUri(node)) === null || _o === void 0 ? void 0 : _o.toString()) || '', undefined, undefined, launchConfiguration, true);
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.package.test', (uri, launchConfiguration) => __awaiter(this, void 0, void 0, function* () {
        yield runDebug(true, true, uri, undefined, undefined, launchConfiguration);
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.open.stacktrace', (uri, methodName, fileName, line) => __awaiter(this, void 0, void 0, function* () {
        const location = uri ? yield vscode_1.commands.executeCommand(exports.COMMAND_PREFIX + '.resolve.stacktrace.location', uri, methodName, fileName) : undefined;
        if (location) {
            const lNum = line - 1;
            vscode_1.window.showTextDocument(vscode.Uri.parse(location), { selection: new vscode.Range(new vscode.Position(lNum, 0), new vscode.Position(lNum, 0)) });
        }
        else {
            if (methodName) {
                const fqn = methodName.substring(0, methodName.lastIndexOf('.'));
                vscode_1.commands.executeCommand('workbench.action.quickOpen', '#' + fqn.substring(fqn.lastIndexOf('.') + 1));
            }
        }
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.workspace.symbols', (query) => __awaiter(this, void 0, void 0, function* () {
        var _p;
        const c = yield exports.client;
        return (_p = (yield c.sendRequest('workspace/symbol', { 'query': query }))) !== null && _p !== void 0 ? _p : [];
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.workspace.symbol.resolve', (symbol) => __awaiter(this, void 0, void 0, function* () {
        var _q;
        const c = yield exports.client;
        return (_q = (yield c.sendRequest('workspaceSymbol/resolve', symbol))) !== null && _q !== void 0 ? _q : null;
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.java.complete.abstract.methods', () => __awaiter(this, void 0, void 0, function* () {
        const active = vscode.window.activeTextEditor;
        if (active) {
            const position = new vscode.Position(active.selection.start.line, active.selection.start.character);
            yield vscode_1.commands.executeCommand(exports.COMMAND_PREFIX + '.java.implement.all.abstract.methods', active.document.uri.toString(), position);
        }
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.startup.condition', () => __awaiter(this, void 0, void 0, function* () {
        return exports.client;
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.addEventListener', (eventName, listener) => {
        let ls = exports.listeners.get(eventName);
        if (!ls) {
            ls = [];
            exports.listeners.set(eventName, ls);
        }
        ls.push(listener);
    }));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.node.properties.edit', (node) => __awaiter(this, void 0, void 0, function* () { return yield propertiesView_1.PropertiesView.createOrShow(context, node, (yield exports.client).findTreeViewService()); })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.cloud.ocid.copy', (node) => __awaiter(this, void 0, void 0, function* () {
        const ocid = getValueAfterPrefix(node.contextValue, 'ocid:');
        vscode.env.clipboard.writeText(ocid);
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.cloud.publicIp.copy', (node) => __awaiter(this, void 0, void 0, function* () {
        const publicIp = getValueAfterPrefix(node.contextValue, 'publicIp:');
        vscode.env.clipboard.writeText(publicIp);
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.cloud.openConsole', (node) => __awaiter(this, void 0, void 0, function* () {
        const consoleUrl = getValueAfterPrefix(node.contextValue, 'consoleUrl:');
        const url = vscode.Uri.parse(consoleUrl);
        vscode.env.openExternal(url);
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.cloud.openInBrowser', (node) => __awaiter(this, void 0, void 0, function* () {
        const portForward = getValueAfterPrefix(node.contextValue, 'portForward:');
        const url = vscode.Uri.parse(portForward);
        vscode.env.openExternal(url);
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.cloud.imageUrl.copy', (node) => __awaiter(this, void 0, void 0, function* () {
        const imageUrl = getValueAfterPrefix(node.contextValue, 'imageUrl:');
        vscode.env.clipboard.writeText("docker pull " + imageUrl);
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.cloud.computeInstance.ssh', (node) => __awaiter(this, void 0, void 0, function* () {
        const publicIp = getValueAfterPrefix(node.contextValue, 'publicIp:');
        const ocid = getValueAfterPrefix(node.contextValue, 'ocid:');
        if (!(0, guidesUtil_1.shouldHideGuideFor)(sshGuide.viewType, ocid)) {
            sshGuide.SshGuidePanel.createOrShow(context, {
                publicIp,
                ocid
            });
        }
        const sshSession = new ssh_1.SSHSession("opc", publicIp);
        sshSession.open(node.label);
    })));
    context.subscriptions.push(vscode_1.commands.registerCommand(exports.COMMAND_PREFIX + '.cloud.container.docker', (node) => __awaiter(this, void 0, void 0, function* () {
        const publicIp = getValueAfterPrefix(node.contextValue, 'publicIp:');
        const imageUrl = getValueAfterPrefix(node.contextValue, 'imageUrl:');
        const ocid = getValueAfterPrefix(node.contextValue, 'ocid:');
        const isRepositoryPrivate = "false" === getValueAfterPrefix(node.parent.contextValue, "repositoryPublic:");
        const registryUrl = imageUrl.split('/')[0];
        if (!(0, guidesUtil_1.shouldHideGuideFor)(runImageGuide.viewType, ocid)) {
            runImageGuide.RunImageGuidePanel.createOrShow(context, {
                publicIp,
                ocid,
                isRepositoryPrivate,
                registryUrl
            });
        }
        const sshSession = new ssh_1.SSHSession("opc", publicIp);
        sshSession.runDocker(context, imageUrl, isRepositoryPrivate);
    })));
    const archiveFileProvider = {
        provideTextDocumentContent: (uri, token) => __awaiter(this, void 0, void 0, function* () {
            return yield vscode_1.commands.executeCommand(exports.COMMAND_PREFIX + '.get.archive.file.content', uri.toString());
        })
    };
    context.subscriptions.push(vscode_1.workspace.registerTextDocumentContentProvider('jar', archiveFileProvider));
    context.subscriptions.push(vscode_1.workspace.registerTextDocumentContentProvider('nbjrt', archiveFileProvider));
    launchConfigurations.updateLaunchConfig();
    configuration.initialize(context);
    jdk.initialize(context);
    // register completions:
    launchConfigurations.registerCompletion(context);
    return Object.freeze({
        version: API_VERSION,
        apiVersion: API_VERSION
    });
}
exports.activate = activate;
/**
 * Pending maintenance (install) task, activations should be chained after it.
 */
let maintenance;
function activateWithJDK(specifiedJDK, context, log, notifyKill, clientResolve, clientReject) {
    let oldClient = exports.client;
    let setClient;
    exports.client = new Promise((clientOK, clientErr) => {
        setClient = [
            function (c) {
                exports.clientRuntimeJDK = specifiedJDK;
                handleLog(log, "Launch: client OK");
                clientOK(c);
                if (clientResolve) {
                    clientResolve(c);
                }
            }, function (err) {
                handleLog(log, `Launch: client failed: ${err}`);
                clientErr(err);
                if (clientReject) {
                    clientReject(err);
                }
            }
        ];
        //setClient = [ clientOK, clientErr ];
    });
    const a = maintenance;
    const clientPromise = exports.client;
    vscode_1.commands.executeCommand('setContext', 'nbJavaLSReady', false);
    vscode_1.commands.executeCommand('setContext', 'dbAddConnectionPresent', true);
    // chain the restart after termination of the former process.
    if (a != null) {
        handleLog(log, "Server activation initiated while in maintenance mode, scheduling after maintenance");
        a.then(() => stopClient(oldClient)).then(() => killNbProcess(notifyKill, log)).then(() => {
            doActivateWithJDK(clientPromise, specifiedJDK, context, log, notifyKill, setClient);
        });
    }
    else {
        handleLog(log, "Initiating server activation");
        stopClient(oldClient).catch(e => null).then(() => {
            return killNbProcess(notifyKill, log);
        }).then(() => {
            doActivateWithJDK(clientPromise, specifiedJDK, context, log, notifyKill, setClient);
        });
    }
}
function killNbProcess(notifyKill, log, specProcess) {
    const p = nbProcess;
    handleLog(log, "Request to kill LSP server.");
    if (p && (!specProcess || specProcess == p)) {
        if (notifyKill) {
            vscode.window.setStatusBarMessage("Restarting Apache NetBeans Language Server.", 2000);
        }
        return new Promise((resolve, reject) => {
            nbProcess = null;
            p.on('close', function (code) {
                handleLog(log, "LSP server closed: " + p.pid);
                resolve();
            });
            handleLog(log, "Killing LSP server " + p.pid);
            if (!p.kill()) {
                reject("Cannot kill");
            }
        });
    }
    else {
        let msg = "Cannot kill: ";
        if (specProcess) {
            msg += "Requested kill on " + specProcess.pid + ", ";
        }
        handleLog(log, msg + "current process is " + (p ? p.pid : "None"));
        return new Promise((res, rej) => { res(); });
    }
}
/**
 * Attempts to determine if the Workbench is using dark-style color theme, so that NBLS
 * starts with some dark L&F for icon resource selection.
 */
function isDarkColorTheme() {
    var _a, _b, _c;
    const themeName = (_a = vscode_1.workspace.getConfiguration('workbench')) === null || _a === void 0 ? void 0 : _a.get('colorTheme');
    if (!themeName) {
        return false;
    }
    for (const ext of vscode.extensions.all) {
        const themeList = ((_b = ext.packageJSON) === null || _b === void 0 ? void 0 : _b.contributes) && ((_c = ext.packageJSON) === null || _c === void 0 ? void 0 : _c.contributes['themes']);
        if (!themeList) {
            continue;
        }
        let t;
        for (t of themeList) {
            if (t.id !== themeName) {
                continue;
            }
            const uiTheme = t.uiTheme;
            if (typeof (uiTheme) == 'string') {
                if (uiTheme.includes('-dark') || uiTheme.includes('-black')) {
                    return true;
                }
            }
        }
    }
    return false;
}
function isJavaSupportEnabled() {
    var _a;
    return (_a = vscode_1.workspace.getConfiguration('netbeans')) === null || _a === void 0 ? void 0 : _a.get('javaSupport.enabled');
}
function getProjectJDKHome() {
    var _a;
    return (_a = vscode_1.workspace.getConfiguration('netbeans')) === null || _a === void 0 ? void 0 : _a.get('project.jdkhome');
}
function doActivateWithJDK(promise, specifiedJDK, context, log, notifyKill, setClient) {
    // Records if the server successfully started before close/error.
    let started = false;
    // Error reported by server started from this function execution.
    let startupError = '';
    maintenance = null;
    let restartWithJDKLater = function restartLater(time, n) {
        handleLog(log, `Restart of Apache Language Server requested in ${(time / 1000)} s.`);
        setTimeout(() => {
            activateWithJDK(specifiedJDK, context, log, n);
        }, time);
    };
    const netbeansConfig = vscode_1.workspace.getConfiguration('netbeans');
    const beVerbose = netbeansConfig.get('verbose', false);
    let userdir = process.env['nbcode_userdir'] || netbeansConfig.get('userdir', 'local');
    switch (userdir) {
        case 'local':
            if (context.storagePath) {
                userdir = context.storagePath;
                break;
            }
        // fallthru
        case 'global':
            userdir = context.globalStoragePath;
            break;
        default:
        // assume storage is path on disk
    }
    let info = {
        clusters: findClusters(context.extensionPath),
        extensionPath: context.extensionPath,
        storagePath: userdir,
        jdkHome: specifiedJDK,
        verbose: beVerbose
    };
    let launchMsg = `Launching Apache NetBeans Language Server with ${specifiedJDK ? specifiedJDK : 'default system JDK'} and userdir ${userdir}`;
    handleLog(log, launchMsg);
    vscode.window.setStatusBarMessage(launchMsg, 2000);
    const connection = () => new Promise((resolve, reject) => {
        const server = net.createServer(socket => {
            server.close();
            resolve({
                reader: socket,
                writer: socket
            });
        });
        server.on('error', (err) => {
            reject(err);
        });
        server.listen(() => {
            const address = server.address();
            let extras = ["--modules", "-J-XX:PerfMaxStringConstLength=10240"];
            if (isDarkColorTheme()) {
                extras.push('--laf', 'com.formdev.flatlaf.FlatDarkLaf');
            }
            if (isJavaSupportEnabled()) {
                extras.push('--direct-disable', 'org.netbeans.modules.nbcode.integration.java');
            }
            else {
                extras.push('--enable', 'org.netbeans.modules.nbcode.integration.java');
            }
            extras.push(`--start-java-language-server=connect:${address.port}`);
            extras.push(`--start-java-debug-adapter-server=listen:0`);
            const srv = launcher.launch(info, ...extras);
            var p = srv;
            if (!srv) {
                reject();
            }
            else {
                if (!srv.stdout) {
                    reject(`No stdout to parse!`);
                    srv.disconnect();
                    return;
                }
                debugPort = -1;
                srv.stdout.on("data", (chunk) => {
                    if (debugPort < 0) {
                        const info = chunk.toString().match(/Debug Server Adapter listening at port (\d*)/);
                        if (info) {
                            debugPort = info[1];
                        }
                    }
                });
                srv.once("error", (err) => {
                    reject(err);
                });
            }
            let stdOut = '';
            function logAndWaitForEnabled(text, isOut) {
                handleLogNoNL(log, text);
                if (stdOut == null) {
                    return;
                }
                if (isOut) {
                    stdOut += text;
                }
                if (stdOut.match(/org.netbeans.modules.java.lsp.server/)) {
                    stdOut = null;
                }
            }
            handleLog(log, "LSP server launching: " + p.pid);
            handleLog(log, "LSP server user directory: " + userdir);
            p.stdout.on('data', function (d) {
                logAndWaitForEnabled(d.toString(), true);
            });
            p.stderr.on('data', function (d) {
                logAndWaitForEnabled(d.toString(), false);
            });
            nbProcess = p;
            p.on('close', function (code) {
                if (p == nbProcess) {
                    nbProcess = null;
                }
                if (p == nbProcess && code != 0 && code) {
                    vscode.window.showWarningMessage("Apache NetBeans Language Server exited with " + code);
                }
                if (stdOut != null) {
                    let match = stdOut.match(/org.netbeans.modules.java.lsp.server[^\n]*/);
                    if ((match === null || match === void 0 ? void 0 : match.length) == 1) {
                        handleLog(log, match[0]);
                    }
                    else {
                        handleLog(log, "Cannot find org.netbeans.modules.java.lsp.server in the log!");
                    }
                    log.show(false);
                    killNbProcess(false, log, p).catch(() => null).then(() => reject("Apache NetBeans Language Server not enabled!"));
                }
                else {
                    handleLog(log, "LSP server " + p.pid + " terminated with " + code);
                    handleLog(log, "Exit code " + code);
                }
            });
        });
    });
    const conf = vscode_1.workspace.getConfiguration();
    let documentSelectors = [
        { language: 'java' },
        { language: 'yaml', pattern: '**/{application,bootstrap}*.{yml,yaml}' },
        { language: 'properties', pattern: '**/{application,bootstrap}*.properties' },
        { language: 'jackpot-hint' },
        { language: 'xml', pattern: '**/pom.xml' },
        { pattern: '**/build.gradle' }
    ];
    documentSelectors.push(...collectDocumentSelectors());
    const enableJava = isJavaSupportEnabled();
    const enableGroovy = conf.get("netbeans.groovySupport.enabled");
    if (enableGroovy) {
        documentSelectors.push({ language: 'groovy' });
    }
    // Options to control the language client
    let clientOptions = {
        // Register the server for java documents
        documentSelector: documentSelectors,
        synchronize: {
            configurationSection: [
                'netbeans.hints',
                'netbeans.format',
                'netbeans.java.imports',
                'netbeans.project.jdkhome',
                'java+.runConfig.vmOptions',
                'java+.runConfig.cwd'
            ],
            fileEvents: [
                vscode_1.workspace.createFileSystemWatcher('**/*.java')
            ]
        },
        outputChannel: log,
        revealOutputChannelOn: vscode_languageclient_1.RevealOutputChannelOn.Never,
        progressOnInitialization: true,
        initializationOptions: {
            'nbcodeCapabilities': {
                'statusBarMessageSupport': true,
                'testResultsSupport': true,
                'showHtmlPageSupport': true,
                'wantsJavaSupport': enableJava,
                'wantsGroovySupport': enableGroovy
            }
        },
        errorHandler: {
            error: function (error, _message, count) {
                startupError = error.message;
                return {
                    action: started ? vscode_languageclient_1.ErrorAction.Continue : vscode_languageclient_1.ErrorAction.Shutdown,
                    message: error.message
                };
            },
            closed: function () {
                handleLog(log, "Connection to Apache NetBeans Language Server closed.");
                // restart only if the _current_ client has been closed AND the server at least booted.
                if (started && exports.client === promise) {
                    restartWithJDKLater(10000, false);
                }
                else {
                    // report a final failure upwards.
                    setClient[1](startupError);
                }
                return { action: vscode_languageclient_1.CloseAction.DoNotRestart };
            }
        }
    };
    let c = new NbLanguageClient('java', 'NetBeans Java', connection, log, clientOptions);
    handleLog(log, 'Language Client: Starting');
    c.start().then(() => {
        started = true;
        if (enableJava) {
            if (testAdapter) {
                // we need to create it again anyway, so it load()s the content.
                testAdapter.dispose();
            }
            testAdapter = new testAdapter_1.NbTestAdapter();
            const testAdapterCreatedListeners = exports.listeners.get(TEST_ADAPTER_CREATED_EVENT);
            testAdapterCreatedListeners === null || testAdapterCreatedListeners === void 0 ? void 0 : testAdapterCreatedListeners.forEach(listener => {
                vscode_1.commands.executeCommand(listener);
            });
        }
        else if (testAdapter) {
            testAdapter.dispose();
            testAdapter = undefined;
        }
        c.onNotification(protocol_1.StatusMessageRequest.type, showStatusBarMessage);
        c.onRequest(protocol_1.HtmlPageRequest.type, showHtmlPage);
        c.onRequest(protocol_1.ExecInHtmlPageRequest.type, execInHtmlPage);
        c.onNotification(vscode_languageclient_1.LogMessageNotification.type, (param) => handleLog(log, param.message));
        c.onRequest(protocol_1.QuickPickRequest.type, (param) => __awaiter(this, void 0, void 0, function* () {
            const selected = yield vscode_1.window.showQuickPick(param.items, { title: param.title, placeHolder: param.placeHolder, canPickMany: param.canPickMany, ignoreFocusOut: true });
            return selected ? Array.isArray(selected) ? selected : [selected] : undefined;
        }));
        c.onRequest(protocol_1.UpdateConfigurationRequest.type, (param) => __awaiter(this, void 0, void 0, function* () {
            yield vscode.workspace.getConfiguration(param.section).update(param.key, param.value);
            (0, runConfiguration_1.runConfigurationUpdateAll)();
        }));
        c.onRequest(protocol_1.SaveDocumentsRequest.type, (request) => __awaiter(this, void 0, void 0, function* () {
            const uriList = request.documents.map(s => {
                let re = /^file:\/(?:\/\/)?([A-Za-z]):\/(.*)$/.exec(s);
                if (!re) {
                    return s;
                }
                // don't ask why vscode mangles URIs this way; in addition, it uses lowercase drive letter ???
                return `file:///${re[1].toLowerCase()}%3A/${re[2]}`;
            });
            let ok = true;
            for (let ed of vscode_1.workspace.textDocuments) {
                let uri = ed.uri.toString();
                if (uriList.includes(uri)) {
                    ed.save();
                    continue;
                }
                if (uri.startsWith("file:///")) {
                    // make file:/// just file:/
                    uri = "file:/" + uri.substring(8);
                    if (uriList.includes(uri)) {
                        ed.save();
                        continue;
                    }
                }
                ok = false;
            }
            return ok;
        }));
        c.onRequest(protocol_1.InputBoxRequest.type, (param) => __awaiter(this, void 0, void 0, function* () {
            return yield vscode_1.window.showInputBox({ title: param.title, prompt: param.prompt, value: param.value, password: param.password });
        }));
        c.onRequest(protocol_1.MutliStepInputRequest.type, (param) => __awaiter(this, void 0, void 0, function* () {
            const data = {};
            function nextStep(input, step, state) {
                return __awaiter(this, void 0, void 0, function* () {
                    const inputStep = yield c.sendRequest(protocol_1.MutliStepInputRequest.step, { inputId: param.id, step, data: state });
                    if (inputStep && inputStep.hasOwnProperty('items')) {
                        const quickPickStep = inputStep;
                        state[inputStep.stepId] = yield input.showQuickPick({
                            title: param.title,
                            step,
                            totalSteps: quickPickStep.totalSteps,
                            placeholder: quickPickStep.placeHolder,
                            items: quickPickStep.items,
                            canSelectMany: quickPickStep.canPickMany,
                            selectedItems: quickPickStep.items.filter(item => item.picked)
                        });
                        return (input) => nextStep(input, step + 1, state);
                    }
                    else if (inputStep && inputStep.hasOwnProperty('value')) {
                        const inputBoxStep = inputStep;
                        state[inputStep.stepId] = yield input.showInputBox({
                            title: param.title,
                            step,
                            totalSteps: inputBoxStep.totalSteps,
                            value: state[inputStep.stepId] || inputBoxStep.value,
                            prompt: inputBoxStep.prompt,
                            password: inputBoxStep.password,
                            validate: (val) => {
                                const d = Object.assign({}, state);
                                d[inputStep.stepId] = val;
                                return c.sendRequest(protocol_1.MutliStepInputRequest.validate, { inputId: param.id, step, data: d });
                            }
                        });
                        return (input) => nextStep(input, step + 1, state);
                    }
                });
            }
            yield utils_1.MultiStepInput.run(input => nextStep(input, 1, data));
            return data;
        }));
        c.onNotification(protocol_1.TestProgressNotification.type, param => {
            const testProgressListeners = exports.listeners.get(exports.TEST_PROGRESS_EVENT);
            testProgressListeners === null || testProgressListeners === void 0 ? void 0 : testProgressListeners.forEach(listener => {
                vscode_1.commands.executeCommand(listener, param.suite);
            });
            if (testAdapter) {
                testAdapter.testProgress(param.suite);
            }
        });
        let decorations = new Map();
        let decorationParamsByUri = new Map();
        c.onRequest(protocol_1.TextEditorDecorationCreateRequest.type, param => {
            let decorationType = vscode.window.createTextEditorDecorationType(param);
            decorations.set(decorationType.key, decorationType);
            return decorationType.key;
        });
        c.onRequest(protocol_1.WriteOutputRequest.type, param => {
            const outputTerminal = LineBufferingPseudoterminal.getInstance(param.outputName);
            outputTerminal.acceptInput(param.message);
        });
        c.onRequest(protocol_1.ShowOutputRequest.type, param => {
            const outputTerminal = LineBufferingPseudoterminal.getInstance(param);
            outputTerminal.show();
        });
        c.onRequest(protocol_1.CloseOutputRequest.type, param => {
            const outputTerminal = LineBufferingPseudoterminal.getInstance(param);
            outputTerminal.close();
        });
        c.onRequest(protocol_1.ResetOutputRequest.type, param => {
            const outputTerminal = LineBufferingPseudoterminal.getInstance(param);
            outputTerminal.clear();
        });
        c.onNotification(protocol_1.TextEditorDecorationSetNotification.type, param => {
            let decorationType = decorations.get(param.key);
            if (decorationType) {
                let editorsWithUri = vscode.window.visibleTextEditors.filter(editor => editor.document.uri.toString() == param.uri);
                if (editorsWithUri.length > 0) {
                    editorsWithUri[0].setDecorations(decorationType, (0, protocol_1.asRanges)(param.ranges));
                    decorationParamsByUri.set(editorsWithUri[0].document.uri, param);
                }
            }
        });
        let disposableListener = vscode.window.onDidChangeVisibleTextEditors(editors => {
            editors.forEach(editor => {
                let decorationParams = decorationParamsByUri.get(editor.document.uri);
                if (decorationParams) {
                    let decorationType = decorations.get(decorationParams.key);
                    if (decorationType) {
                        editor.setDecorations(decorationType, (0, protocol_1.asRanges)(decorationParams.ranges));
                    }
                }
            });
        });
        context.subscriptions.push(disposableListener);
        c.onNotification(protocol_1.TextEditorDecorationDisposeNotification.type, param => {
            let decorationType = decorations.get(param);
            if (decorationType) {
                decorations.delete(param);
                decorationType.dispose();
                decorationParamsByUri.forEach((value, key, map) => {
                    if (value.key == param) {
                        map.delete(key);
                    }
                });
            }
        });
        c.onNotification(vscode_languageclient_1.TelemetryEventNotification.type, (param) => {
            const names = param.name !== 'SCAN_START_EVT' && param.name !== 'SCAN_END_EVT' ? [param.name] : [param.name, param.properties];
            for (const name of names) {
                const ls = exports.listeners.get(name);
                if (ls) {
                    for (const listener of ls) {
                        vscode_1.commands.executeCommand(listener);
                    }
                }
            }
        });
        handleLog(log, 'Language Client: Ready');
        setClient[0](c);
        vscode_1.commands.executeCommand('setContext', 'nbJavaLSReady', true);
        if (enableJava) {
            // create project explorer:
            //c.findTreeViewService().createView('foundProjects', 'Projects', { canSelectMany : false });
            createProjectView(context, c);
        }
        createDatabaseView(c);
        if (enableJava) {
            c.findTreeViewService().createView('cloud.resources', undefined, { canSelectMany: false });
        }
        c.findTreeViewService().createView('cloud.assets', undefined, { canSelectMany: false, showCollapseAll: false, providerInitializer: (customizable) => customizable.addItemDecorator(new CloudAssetsDecorator()) });
    }).catch(err => setClient[1](err));
    class CloudAssetsDecorator {
        decorateChildren(element, children) {
            return children;
        }
        decorateTreeItem(vis, item) {
            return __awaiter(this, void 0, void 0, function* () {
                const refName = getValueAfterPrefix(item.contextValue, "cloudAssetsReferenceName:");
                if (refName !== undefined && refName !== null && refName.length > 0) {
                    item.description = refName;
                    return item;
                }
                const imageCount = getValueAfterPrefix(item.contextValue, "imageCount:");
                const repositoryPublic = "true" === getValueAfterPrefix(item.contextValue, "repositoryPublic:");
                if (imageCount !== undefined && imageCount !== null && imageCount.length > 0) {
                    if (repositoryPublic) {
                        item.description = imageCount + " (public)";
                    }
                    else {
                        item.description = imageCount + " (private)";
                    }
                    return item;
                }
                const lifecycleState = getValueAfterPrefix(item.contextValue, "lifecycleState:");
                if (lifecycleState) {
                    item.description = lifecycleState === "PENDING_DELETION" ? '(pending deletion)' : undefined;
                }
                const clusterNamespace = getValueAfterPrefix(item.contextValue, "clusterNamespace:");
                if (clusterNamespace) {
                    item.description = clusterNamespace;
                    return item;
                }
                return item;
            });
        }
        dispose() {
        }
    }
    class Decorator {
        constructor(provider, client) {
            this.initialized = false;
            this.provider = provider;
            this.setCommand = vscode.commands.registerCommand(exports.COMMAND_PREFIX + '.local.db.set.preferred.connection', (n) => this.setPreferred(n));
        }
        decorateChildren(element, children) {
            if (element.id == this.provider.getRoot().id) {
                vscode.commands.executeCommand('setContext', 'nb.database.view.active', children.length == 0);
            }
            return children;
        }
        decorateTreeItem(vis, item) {
            return __awaiter(this, void 0, void 0, function* () {
                if (!(item.contextValue && item.contextValue.match(/class:org.netbeans.api.db.explorer.DatabaseConnection/))) {
                    return item;
                }
                return vscode.commands.executeCommand(exports.COMMAND_PREFIX + '.db.preferred.connection').then((id) => {
                    if (id == vis.id) {
                        item.description = '(default)';
                    }
                    return item;
                });
            });
        }
        setPreferred(...args) {
            var _a;
            const id = ((_a = args[0]) === null || _a === void 0 ? void 0 : _a.id) || -1;
            vscode.commands.executeCommand(exports.COMMAND_PREFIX + ':Database:netbeans.db.explorer.action.makepreferred', ...args);
            // refresh all
            this.provider.fireItemChange();
        }
        dispose() {
            var _a;
            (_a = this.setCommand) === null || _a === void 0 ? void 0 : _a.dispose();
        }
    }
    function createDatabaseView(c) {
        let decoRegister;
        c.findTreeViewService().createView('database.connections', undefined, {
            canSelectMany: true,
            providerInitializer: (customizable) => customizable.addItemDecorator(new Decorator(customizable, c))
        });
    }
    function createProjectView(ctx, client) {
        return __awaiter(this, void 0, void 0, function* () {
            const ts = client.findTreeViewService();
            let tv = yield ts.createView('foundProjects', 'Projects', { canSelectMany: false });
            if (!tv) {
                return;
            }
            const view = tv;
            function revealActiveEditor(ed) {
                var _a, _b;
                return __awaiter(this, void 0, void 0, function* () {
                    const uri = (_b = (_a = vscode_1.window.activeTextEditor) === null || _a === void 0 ? void 0 : _a.document) === null || _b === void 0 ? void 0 : _b.uri;
                    if (!uri || uri.scheme.toLowerCase() !== 'file') {
                        return;
                    }
                    if (!view.visible) {
                        return;
                    }
                    let vis = yield ts.findPath(view, uri.toString());
                    if (!vis) {
                        return;
                    }
                    view.reveal(vis, { select: true, focus: false, expand: false });
                });
            }
            ctx.subscriptions.push(vscode_1.window.onDidChangeActiveTextEditor(ed => {
                const netbeansConfig = vscode_1.workspace.getConfiguration('netbeans');
                if (netbeansConfig.get("revealActiveInProjects")) {
                    revealActiveEditor(ed);
                }
            }));
            ctx.subscriptions.push(vscode.commands.registerCommand(exports.COMMAND_PREFIX + ".select.editor.projects", () => revealActiveEditor()));
            // attempt to reveal NOW:
            if (netbeansConfig.get("revealActiveInProjects")) {
                revealActiveEditor();
            }
        });
    }
    const webviews = new Map();
    function showHtmlPage(params) {
        return __awaiter(this, void 0, void 0, function* () {
            return new Promise(resolve => {
                let data = params.text;
                const match = /<title>(.*)<\/title>/i.exec(data);
                const name = match && match.length > 1 ? match[1] : '';
                const resourceDir = vscode.Uri.joinPath(context.globalStorageUri, params.id);
                vscode_1.workspace.fs.createDirectory(resourceDir);
                let view = vscode.window.createWebviewPanel('htmlView', name, vscode.ViewColumn.Beside, {
                    enableScripts: true,
                    localResourceRoots: [resourceDir, vscode.Uri.joinPath(context.extensionUri, 'node_modules', '@vscode/codicons', 'dist')]
                });
                webviews.set(params.id, view.webview);
                const resources = params.resources;
                if (resources) {
                    for (const resourceName in resources) {
                        const resourceText = resources[resourceName];
                        const resourceUri = vscode.Uri.joinPath(resourceDir, resourceName);
                        vscode_1.workspace.fs.writeFile(resourceUri, Buffer.from(resourceText, 'utf8'));
                        data = data.replace('href="' + resourceName + '"', 'href="' + view.webview.asWebviewUri(resourceUri) + '"');
                    }
                }
                const codiconsUri = view.webview.asWebviewUri(vscode.Uri.joinPath(context.extensionUri, 'node_modules', '@vscode/codicons', 'dist', 'codicon.css'));
                view.webview.html = data.replace('href="codicon.css"', 'href="' + codiconsUri + '"');
                view.webview.onDidReceiveMessage(message => {
                    switch (message.command) {
                        case 'dispose':
                            webviews.delete(params.id);
                            view.dispose();
                            break;
                        case 'command':
                            vscode.commands.executeCommand(exports.COMMAND_PREFIX + '.htmlui.process.command', message.data);
                            break;
                    }
                });
                view.onDidDispose(() => {
                    resolve();
                    vscode_1.workspace.fs.delete(resourceDir, { recursive: true });
                });
            });
        });
    }
    function execInHtmlPage(params) {
        return __awaiter(this, void 0, void 0, function* () {
            return new Promise(resolve => {
                const webview = webviews.get(params.id);
                if (webview) {
                    webview.postMessage({
                        execScript: params.text,
                        pause: params.pause
                    }).then(ret => {
                        resolve(ret);
                    });
                }
                resolve(false);
            });
        });
    }
    function showStatusBarMessage(params) {
        let decorated = params.message;
        let defTimeout;
        switch (params.type) {
            case vscode_languageclient_1.MessageType.Error:
                decorated = '$(error) ' + params.message;
                defTimeout = 0;
                checkInstallNbJavac(params.message);
                break;
            case vscode_languageclient_1.MessageType.Warning:
                decorated = '$(warning) ' + params.message;
                defTimeout = 0;
                break;
            default:
                defTimeout = 10000;
                break;
        }
        // params.timeout may be defined but 0 -> should be used
        const timeout = params.timeout != undefined ? params.timeout : defTimeout;
        if (timeout > 0) {
            vscode_1.window.setStatusBarMessage(decorated, timeout);
        }
        else {
            vscode_1.window.setStatusBarMessage(decorated);
        }
    }
    function checkInstallNbJavac(msg) {
        const NO_JAVA_SUPPORT = "Cannot initialize Java support";
        if (msg.startsWith(NO_JAVA_SUPPORT)) {
            const yes = "Install GPLv2+CPEx code";
            vscode_1.window.showErrorMessage("Additional Java Support is needed", yes).then(reply => {
                if (yes === reply) {
                    vscode.window.setStatusBarMessage("Preparing Apache NetBeans Language Server for additional installation", 2000);
                    restartWithJDKLater = function () {
                        handleLog(log, "Ignoring request for restart of Apache NetBeans Language Server");
                    };
                    maintenance = new Promise((resolve, reject) => {
                        const kill = killNbProcess(false, log);
                        kill.then(() => {
                            let installProcess = launcher.launch(info, "-J-Dnetbeans.close=true", "--modules", "--install", ".*nbjavac.*");
                            handleLog(log, "Launching installation process: " + installProcess.pid);
                            let logData = function (d) {
                                handleLogNoNL(log, d.toString());
                            };
                            installProcess.stdout.on('data', logData);
                            installProcess.stderr.on('data', logData);
                            installProcess.addListener("error", reject);
                            // MUST wait on 'close', since stdout is inherited by children. The installProcess dies but
                            // the inherited stream will be closed by the last child dying.
                            installProcess.on('close', function (code) {
                                handleLog(log, "Installation completed: " + installProcess.pid);
                                handleLog(log, "Additional Java Support installed with exit code " + code);
                                // will be actually run after maintenance is resolve()d.
                                activateWithJDK(specifiedJDK, context, log, notifyKill);
                                resolve();
                            });
                            return installProcess;
                        });
                    });
                }
            });
        }
    }
}
function stopClient(clientPromise) {
    if (testAdapter) {
        testAdapter.dispose();
        testAdapter = undefined;
    }
    return clientPromise && !(clientPromise instanceof InitialPromise) ? clientPromise.then(c => c.stop()) : Promise.resolve();
}
function deactivate() {
    if (nbProcess != null) {
        nbProcess.kill();
    }
    return stopClient(exports.client);
}
exports.deactivate = deactivate;
function collectDocumentSelectors() {
    const selectors = [];
    for (const extension of vscode.extensions.all) {
        const contributesSection = extension.packageJSON['contributes'];
        if (contributesSection) {
            const documentSelectors = contributesSection['netbeans.documentSelectors'];
            if (Array.isArray(documentSelectors) && documentSelectors.length) {
                selectors.push(...documentSelectors);
            }
        }
    }
    return selectors;
}
class NetBeansDebugAdapterTrackerFactory {
    createDebugAdapterTracker(_session) {
        return {
            onDidSendMessage(message) {
                if (testAdapter && message.type === 'event' && message.event === 'output') {
                    testAdapter.testOutput(message.body.output);
                }
            }
        };
    }
}
class NetBeansDebugAdapterDescriptionFactory {
    createDebugAdapterDescriptor(_session, _executable) {
        return new Promise((resolve, reject) => {
            let cnt = 10;
            const fnc = () => {
                if (debugPort < 0) {
                    if (cnt-- > 0) {
                        setTimeout(fnc, 1000);
                    }
                    else {
                        reject(new Error('Apache NetBeans Debug Server Adapter not yet initialized. Please wait for a while and try again.'));
                    }
                }
                else {
                    resolve(new vscode.DebugAdapterServer(debugPort));
                }
            };
            fnc();
        });
    }
}
class NetBeansConfigurationInitialProvider {
    provideDebugConfigurations(folder, token) {
        return this.doProvideDebugConfigurations(folder, token);
    }
    doProvideDebugConfigurations(folder, _token) {
        var _a, _b;
        return __awaiter(this, void 0, void 0, function* () {
            let c = yield exports.client;
            if (!folder) {
                return [];
            }
            var u;
            if (folder && folder.uri) {
                u = folder.uri;
            }
            else {
                u = (_b = (_a = vscode.window.activeTextEditor) === null || _a === void 0 ? void 0 : _a.document) === null || _b === void 0 ? void 0 : _b.uri;
            }
            let result = [];
            const configNames = yield vscode.commands.executeCommand(exports.COMMAND_PREFIX + '.project.configurations', u === null || u === void 0 ? void 0 : u.toString());
            if (configNames) {
                let first = true;
                for (let cn of configNames) {
                    let cname;
                    if (first) {
                        // ignore the default config, comes first.
                        first = false;
                        continue;
                    }
                    else {
                        cname = "Launch Java: " + cn;
                    }
                    const debugConfig = {
                        name: cname,
                        type: "java+",
                        request: "launch",
                        launchConfiguration: cn,
                    };
                    result.push(debugConfig);
                }
            }
            return result;
        });
    }
}
class NetBeansConfigurationDynamicProvider {
    constructor(context) {
        this.commandValues = new Map();
        this.context = context;
    }
    provideDebugConfigurations(folder, token) {
        return this.doProvideDebugConfigurations(folder, this.context, this.commandValues, token);
    }
    doProvideDebugConfigurations(folder, context, commandValues, _token) {
        return __awaiter(this, void 0, void 0, function* () {
            let c = yield exports.client;
            if (!folder) {
                return [];
            }
            let result = [];
            const attachConnectors = yield vscode.commands.executeCommand(exports.COMMAND_PREFIX + '.java.attachDebugger.configurations');
            if (attachConnectors) {
                for (let ac of attachConnectors) {
                    const debugConfig = {
                        name: ac.name,
                        type: ac.type,
                        request: "attach",
                    };
                    for (let i = 0; i < ac.arguments.length; i++) {
                        let defaultValue = ac.defaultValues[i];
                        if (!defaultValue.startsWith("${command:")) {
                            // Create a command that asks for the argument value:
                            let cmd = exports.COMMAND_PREFIX + ".java.attachDebugger.connector." + ac.id + "." + ac.arguments[i];
                            debugConfig[ac.arguments[i]] = "${command:" + cmd + "}";
                            if (!commandValues.has(cmd)) {
                                commandValues.set(cmd, ac.defaultValues[i]);
                                let description = ac.descriptions[i];
                                context.subscriptions.push(vscode_1.commands.registerCommand(cmd, (ctx) => __awaiter(this, void 0, void 0, function* () {
                                    return vscode.window.showInputBox({
                                        prompt: description,
                                        value: commandValues.get(cmd),
                                    }).then((value) => {
                                        if (value) {
                                            commandValues.set(cmd, value);
                                        }
                                        return value;
                                    });
                                })));
                            }
                        }
                        else {
                            debugConfig[ac.arguments[i]] = defaultValue;
                        }
                    }
                    result.push(debugConfig);
                }
            }
            return result;
        });
    }
}
class NetBeansConfigurationResolver {
    resolveDebugConfiguration(_folder, config, _token) {
        if (!config.type) {
            config.type = 'java+';
        }
        if (!config.request) {
            config.request = 'launch';
        }
        if (vscode.window.activeTextEditor) {
            config.file = '${file}';
        }
        if (!config.classPaths) {
            config.classPaths = ['any'];
        }
        if (!config.console) {
            config.console = 'internalConsole';
        }
        return config;
    }
}
class NetBeansConfigurationNativeResolver {
    resolveDebugConfiguration(_folder, config, _token) {
        if (!config.type) {
            config.type = 'nativeimage';
        }
        if (!config.request) {
            config.request = 'launch';
        }
        if ('launch' == config.request && !config.nativeImagePath) {
            config.nativeImagePath = '${workspaceFolder}/build/native-image/application';
        }
        if (!config.miDebugger) {
            config.miDebugger = 'gdb';
        }
        if (!config.console) {
            config.console = 'internalConsole';
        }
        return config;
    }
}
class StringContentProvider {
    constructor() {
        this._onDidChange = new vscode.EventEmitter(); // Properly declare and initialize
    }
    // Constructor is not necessary if only initializing _onDidChange
    // This function must return a string as the content of the document
    provideTextDocumentContent(uri) {
        // Return the content for the given URI
        // Here, using the query part of the URI to store and retrieve the content
        return uri.query;
    }
    // Allow listeners to subscribe to content changes
    get onDidChange() {
        return this._onDidChange.event;
    }
}
//# sourceMappingURL=extension.js.map