"use strict";
/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.initialize = void 0;
const vscode = require("vscode");
const os = require("os");
const path = require("path");
const utils_1 = require("../utils");
const settings = require("./settings");
const jdk = require("./jdk");
const ACTION_NAME = 'JDK Configuration';
function initialize(context) {
    const COMMAND_JDK_CONFIGURATION = 'nbls.jdk.configuration';
    context.subscriptions.push(vscode.commands.registerCommand(COMMAND_JDK_CONFIGURATION, () => {
        configure();
    }));
}
exports.initialize = initialize;
function configure() {
    return __awaiter(this, void 0, void 0, function* () {
        const state = {};
        yield utils_1.MultiStepInput.run(input => selectFeatures(input, state));
        if (state.selectedSettings && state.selectedJdk && state.selectedScope) {
            const jdk = state.selectedJdk.jdk;
            for (const selectedSetting of state.selectedSettings) {
                yield selectedSetting.setting.setJdk(jdk, state.selectedScope.scope);
            }
            const GRAALVM_EXTENSION_ID = 'oracle-labs-graalvm.graalvm';
            const GRAALVM_SETTINGS_NAME = 'GraalVM Tools for Java';
            if (vscode.extensions.getExtension(GRAALVM_EXTENSION_ID)) {
                vscode.window.showWarningMessage(`The ${GRAALVM_SETTINGS_NAME} extension may conflict with the ${ACTION_NAME} action. Please consider uninstalling it.`, 'Got It');
            }
        }
    });
}
function totalSteps(_state) {
    return 3;
}
function selectFeatures(input, state) {
    return __awaiter(this, void 0, void 0, function* () {
        if (!state.allSettings) {
            const settings = allSettings();
            state.allSettings = settings[0];
            state.selectedSettings = settings[1];
        }
        const selected = yield input.showQuickPick({
            title: `${ACTION_NAME}: Settings`,
            step: 1,
            totalSteps: totalSteps(state),
            placeholder: 'Select features and settings for which JDK will be configured',
            items: state.allSettings,
            selectedItems: state.selectedSettings,
            canSelectMany: true,
            validate: () => Promise.resolve(undefined),
            shouldResume: () => Promise.resolve(false)
        });
        if (selected === null || selected === void 0 ? void 0 : selected.length) {
            state.selectedSettings = selected;
            return (input) => selectJDK(input, state);
        }
        return;
    });
}
function allSettings() {
    const allSettings = [];
    const selectedSettings = [];
    const availableSettings = settings.getAvailable();
    for (const setting of availableSettings) {
        const current = setting.getCurrent();
        const item = { label: setting.name, description: setting.getSetting(), detail: `$(coffee) Current: ${current || 'not defined'}`, setting: setting };
        allSettings.push(item);
        if (current || setting.configureIfNotDefined()) {
            selectedSettings.push(item);
        }
    }
    return [allSettings, selectedSettings];
}
function checkJDKCompatibility(check, features) {
    return __awaiter(this, void 0, void 0, function* () {
        if (!features) {
            return null;
        }
        if (check) {
            for (let setting of features) {
                if (!(yield setting.setting.acceptJdk(check))) {
                    return setting;
                }
            }
        }
        return null;
    });
}
function selectJDK(input, state) {
    var _a, _b, _c, _d, _e, _f;
    return __awaiter(this, void 0, void 0, function* () {
        if (!state.allJdks) {
            state.allJdks = yield allJdks(state);
        }
        let filteredJDKs = [];
        let someLeftOut = false;
        if (state.selectedSettings) {
            for (let jdk of state.allJdks) {
                if (yield checkJDKCompatibility(jdk.jdk, state.selectedSettings)) {
                    // non-null feature item -> incompatible
                    someLeftOut = true;
                }
                else {
                    filteredJDKs.push(jdk);
                }
            }
        }
        else {
            filteredJDKs = [...state.allJdks];
        }
        const selectCustom = { label: '$(folder-opened) Select Custom JDK...' };
        const selected = yield input.showQuickPick({
            title: `${ACTION_NAME}: JDK`,
            step: 2,
            totalSteps: totalSteps(state),
            placeholder: someLeftOut ? 'Select JDK (filtering compatible JDKs)' : 'Select JDK',
            items: [selectCustom, ...filteredJDKs],
            validate: () => Promise.resolve(undefined),
            shouldResume: () => Promise.resolve(false)
        });
        if (selected === null || selected === void 0 ? void 0 : selected.length) {
            if (selected[0] === selectCustom) {
                const javaHome = (_c = (_b = (_a = state.allJdks) === null || _a === void 0 ? void 0 : _a[1]) === null || _b === void 0 ? void 0 : _b.jdk) === null || _c === void 0 ? void 0 : _c.javaHome;
                const javaRoot = javaHome ? path.dirname(javaHome) : undefined;
                const customJdk = yield selectCustomJdk(javaRoot);
                if (customJdk) {
                    yield jdk.registerCustom(customJdk);
                    state.allJdks = yield allJdks(state);
                    let selected = false;
                    for (const jdkItem of state.allJdks) {
                        if (((_d = jdkItem.jdk) === null || _d === void 0 ? void 0 : _d.javaHome) === customJdk.javaHome) {
                            state.selectedJdk = jdkItem;
                            selected = true;
                            break;
                        }
                    }
                    if (!selected) {
                        // if for some reason, the custom JDK cannot be found among possible selections,
                        // reject it
                        yield vscode.window.showErrorMessage(`The selected JDK ${javaRoot} could not be used. Please choose another one`, 'Retry');
                        return (input) => selectJDK(input, state);
                    }
                }
                else {
                    return (input) => selectJDK(input, state);
                }
            }
            else if (selected[0].jdk) {
                state.selectedJdk = selected[0];
            }
        }
        const f = yield checkJDKCompatibility((_e = state.selectedJdk) === null || _e === void 0 ? void 0 : _e.jdk, state.selectedSettings);
        if (f) {
            const Y = 'Proceed Anyway';
            const N = 'Choose a Different JDK';
            const q = yield vscode.window.showWarningMessage(`The selected JDK (${(_f = state.selectedJdk) === null || _f === void 0 ? void 0 : _f.jdk.name()}) is not compatible with feature ${f.label}. Do you want to proceed ?`, N, Y);
            if (q !== Y) {
                state.selectedJdk = undefined;
                return (input) => selectJDK(input, state);
            }
        }
        return (input) => selectScope(input, state);
    });
}
function allJdks(state) {
    return __awaiter(this, void 0, void 0, function* () {
        const knownJavas = [];
        for (const setting of state.allSettings || []) {
            knownJavas.push(...setting.setting.getJavas());
        }
        const jdks = yield jdk.findAll(knownJavas);
        jdks.sort((jdk1, jdk2) => jdk1 > jdk2 ? -1 : 1);
        const jdkItems = [];
        const jdkItemsNi = [];
        const jdkItemsWoNi = [];
        for (const jdk of jdks) {
            const jdkItem = { label: `$(coffee) ${jdk.name()}`, description: `${jdk.javaHome}`, jdk: jdk, kind: vscode.QuickPickItemKind.Default };
            if (jdk.hasNativeImage()) {
                jdkItemsNi.push(jdkItem);
            }
            else {
                jdkItemsWoNi.push(jdkItem);
            }
        }
        if (jdkItemsNi.length) {
            jdkItems.push({ label: 'With Native Image', kind: vscode.QuickPickItemKind.Separator });
            jdkItems.push(...jdkItemsNi);
        }
        if (jdkItemsWoNi.length) {
            jdkItems.push({ label: 'Without Native Image', kind: vscode.QuickPickItemKind.Separator });
            jdkItems.push(...jdkItemsWoNi);
        }
        return jdkItems;
    });
}
function selectCustomJdk(javaRoot) {
    return __awaiter(this, void 0, void 0, function* () {
        const selected = yield vscode.window.showOpenDialog({
            title: 'Select Custom JDK',
            canSelectFiles: false,
            canSelectFolders: true,
            canSelectMany: false,
            defaultUri: vscode.Uri.file(javaRoot || os.homedir()),
            openLabel: 'Select'
        });
        if ((selected === null || selected === void 0 ? void 0 : selected.length) === 1) {
            const selectedJavaHome = selected[0].fsPath;
            const selectedJdk = new jdk.Java(selectedJavaHome);
            if (selectedJdk.isJdk()) {
                return selectedJdk;
            }
            else {
                vscode.window.showWarningMessage(`Not a valid JDK installation: ${selectedJavaHome}`);
                return null;
            }
        }
        return undefined;
    });
}
function selectScope(input, state) {
    return __awaiter(this, void 0, void 0, function* () {
        if (!state.allScopes) {
            state.allScopes = allScopes();
        }
        const selected = yield input.showQuickPick({
            title: `${ACTION_NAME}: Scope`,
            step: 3,
            totalSteps: totalSteps(state),
            placeholder: 'Select configuration scope',
            items: state.allScopes,
            validate: () => Promise.resolve(undefined),
            shouldResume: () => Promise.resolve(false)
        });
        if ((selected === null || selected === void 0 ? void 0 : selected.length) && selected[0].scope) {
            state.selectedScope = selected[0];
        }
    });
}
function allScopes() {
    const allScopes = [];
    allScopes.push({ label: 'User', description: 'JDK will be configured for all workspaces, may be overriden by Workspace configuration', scope: vscode.ConfigurationTarget.Global });
    if (vscode.workspace.workspaceFolders) {
        allScopes.push({ label: 'Workspace', description: 'JDK will be configured for the current workspace, overrides User configuration', scope: vscode.ConfigurationTarget.Workspace });
    }
    return allScopes;
}
//# sourceMappingURL=configuration.js.map