"use strict";
/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.registerCustom = exports.findAll = exports.Java = exports.initialize = void 0;
const vscode = require("vscode");
const path = require("path");
const fs = require("fs");
const jdkUtils = require("jdk-utils");
const CUSTOM_JAVA_ROOTS_KEY = 'nbls.jdk.customJavaRoots';
let storage;
function initialize(context) {
    storage = context.globalState;
}
exports.initialize = initialize;
class Java {
    constructor(javaHome) {
        this.javaHome = javaHome;
    }
    name() {
        let name = path.basename(this.javaHome);
        if (isMac() && name === 'Home') {
            const javaHome = path.resolve(this.javaHome, '..', '..');
            name = path.basename(javaHome);
        }
        return name;
    }
    isJava() {
        const java = path.join(this.javaHome, 'bin', isWindows() ? 'java.exe' : 'java');
        return fs.existsSync(java);
    }
    isJdk() {
        const javac = path.join(this.javaHome, 'bin', isWindows() ? 'javac.exe' : 'javac');
        return this.isJava() && fs.existsSync(javac);
    }
    hasNativeImage() {
        const ni = path.join(this.javaHome, 'bin', isWindows() ? 'native-image.cmd' : 'native-image');
        return fs.existsSync(ni);
    }
    getVersion() {
        return __awaiter(this, void 0, void 0, function* () {
            const java = yield jdkUtils.getRuntime(this.javaHome, { withVersion: true });
            return java === null || java === void 0 ? void 0 : java.version;
        });
    }
}
exports.Java = Java;
function findAll(knownJavas) {
    return __awaiter(this, void 0, void 0, function* () {
        const javaRoots = [];
        function addJavaRoot(javaHome) {
            let javaRoot = path.dirname(javaHome);
            if (isMac() && path.basename(javaRoot) === 'Contents') {
                javaHome = path.resolve(javaHome, '..', '..');
                javaRoot = path.dirname(javaHome);
            }
            const normalizedJavaRoot = normalizePath(javaRoot);
            if (!javaRoots.includes(normalizedJavaRoot)) {
                javaRoots.push(normalizedJavaRoot);
            }
        }
        const systemJavas = yield jdkUtils.findRuntimes();
        for (const sytemJava of systemJavas) {
            addJavaRoot(sytemJava.homedir);
        }
        if (knownJavas) {
            for (const knownJava of knownJavas) {
                addJavaRoot(knownJava.javaHome);
            }
        }
        const customJavaRootsArr = yield customJavaRoots();
        for (const customJavaRoot of customJavaRootsArr) {
            if (!javaRoots.includes(customJavaRoot)) {
                javaRoots.push(customJavaRoot);
            }
        }
        const jdks = [];
        for (const javaRoot of javaRoots) {
            const dirents = fs.readdirSync(javaRoot, { withFileTypes: true });
            for (const dirent of dirents) {
                const javaHome = path.join(javaRoot, dirent.name);
                // statSync returns true even for symlinked dirs
                try {
                    if (fs.statSync(javaHome).isDirectory()) {
                        const java = new Java(javaHome);
                        if (java.isJdk()) {
                            jdks.push(java);
                        }
                        else if (isMac()) {
                            const macJavaHome = path.join(javaHome, 'Contents', 'Home');
                            const macJava = new Java(macJavaHome);
                            if (macJava.isJdk()) {
                                jdks.push(macJava);
                            }
                        }
                    }
                }
                catch (statErr) {
                    // just ignore
                }
            }
        }
        return jdks;
    });
}
exports.findAll = findAll;
function customJavaRoots() {
    return __awaiter(this, void 0, void 0, function* () {
        const customJavaRoots = storage.get(CUSTOM_JAVA_ROOTS_KEY) || '';
        const customJavaRootsArr = customJavaRoots.split(path.delimiter);
        const newCustomJavaRootsArr = [];
        for (const customJavaRoot of customJavaRootsArr) {
            if (fs.existsSync(customJavaRoot)) {
                newCustomJavaRootsArr.push(customJavaRoot);
            }
        }
        if (customJavaRoots.length !== newCustomJavaRootsArr.length) {
            yield storage.update(CUSTOM_JAVA_ROOTS_KEY, newCustomJavaRootsArr.join(path.delimiter));
        }
        return newCustomJavaRootsArr;
    });
}
function registerCustom(java) {
    return __awaiter(this, void 0, void 0, function* () {
        let customJavaRoot = path.dirname(java.javaHome);
        if (isMac() && path.basename(customJavaRoot) === 'Contents') {
            const javaHome = path.resolve(java.javaHome, '..', '..');
            customJavaRoot = path.dirname(javaHome);
        }
        const customJavaRootsArr = yield customJavaRoots();
        if (!customJavaRootsArr.includes(customJavaRoot)) {
            customJavaRootsArr.push(customJavaRoot);
            yield storage.update(CUSTOM_JAVA_ROOTS_KEY, customJavaRootsArr.join(path.delimiter));
        }
    });
}
exports.registerCustom = registerCustom;
function isWindows() {
    return process.platform === 'win32';
}
function isMac() {
    return process.platform === 'darwin';
}
function normalizePath(fsPath) {
    return vscode.Uri.file(path.normalize(fsPath)).fsPath;
}
//# sourceMappingURL=jdk.js.map