"use strict";
/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getAvailable = exports.projectsSettings = exports.Setting = void 0;
const vscode = require("vscode");
const path = require("path");
const process = require("process");
const jdk = require("./jdk");
const extension_1 = require("../extension");
class Setting {
    constructor(name, property) {
        this.name = name;
        this.property = property;
    }
    acceptJdk(jdk) {
        return __awaiter(this, void 0, void 0, function* () {
            return true;
        });
    }
}
exports.Setting = Setting;
class JavaSetting extends Setting {
    constructor(name, property, minVersion, supportsWorkspaceScope = true) {
        super(name, property);
        this.minVersion = minVersion;
        this.supportsWorkspaceScope = supportsWorkspaceScope;
    }
    acceptJdk(jdk) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.minVersion) {
                return true;
            }
            let v = yield jdk.getVersion();
            return !(v === null || v === void 0 ? void 0 : v.major) || v.major >= this.minVersion;
        });
    }
    getJavaHome() {
        return vscode.workspace.getConfiguration().get(this.property);
    }
    getJava() {
        if (this.java === undefined) {
            const javaHome = this.getJavaHome();
            this.java = javaHome ? new jdk.Java(javaHome) : null;
        }
        return this.java ? this.java : undefined;
    }
    getSetting() {
        if (this.minVersion) {
            return `${this.property} (requires JDK ${this.minVersion}+)`;
        }
        else {
            return this.property;
        }
    }
    getCurrent() {
        var _a;
        return (_a = this.getJava()) === null || _a === void 0 ? void 0 : _a.javaHome;
    }
    getJavas() {
        const javas = [];
        const java = this.getJava();
        if (java) {
            javas.push(java);
        }
        return javas;
    }
    configureIfNotDefined() {
        return true;
    }
    getJdkSetting(jdk) {
        return jdk.javaHome;
    }
    setJdk(jdk, scope) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.supportsWorkspaceScope && scope !== vscode.ConfigurationTarget.Global) {
                const proceedOption = 'Configure It in User Scope';
                const skipOption = 'Skip';
                const selected = yield vscode.window.showWarningMessage(`JDK for ${this.name} can only be configured in User scope.`, proceedOption, skipOption);
                if (selected === proceedOption) {
                    scope = vscode.ConfigurationTarget.Global;
                }
                else {
                    return false;
                }
            }
            try {
                yield vscode.workspace.getConfiguration().update(this.property, this.getJdkSetting(jdk), scope);
                return true;
            }
            catch (err) {
                vscode.window.showErrorMessage(`Failed to update property ${this.getSetting()}: ${err}`);
                return false;
            }
        });
    }
}
class JavaEnvSetting extends JavaSetting {
    constructor(name, property, variable) {
        super(name, property);
        this.variable = variable;
    }
    getEnv() {
        return vscode.workspace.getConfiguration().get(this.property);
    }
    getJavaHome() {
        var _a;
        return (_a = this.getEnv()) === null || _a === void 0 ? void 0 : _a[this.variable];
    }
    getSetting() {
        return `${this.property}.${this.variable}`;
    }
    getJdkSetting(jdk) {
        const env = this.getEnv() || {};
        env[this.variable] = jdk.javaHome;
        return env;
    }
}
class JavaEnvPathSetting extends JavaEnvSetting {
    constructor(name, property, variable) {
        super(name, property, variable);
    }
    getJavaHome() {
        var _a;
        const currentPath = (_a = this.getEnv()) === null || _a === void 0 ? void 0 : _a[this.variable];
        if (currentPath) {
            const expandedPath = `${currentPath}`.replace(/\$\{env\:PATH\}/g, this.expandEnvPath());
            const paths = expandedPath.split(path.delimiter);
            for (const binPath of paths) {
                if (path.basename(binPath).toLowerCase() === 'bin') {
                    const jdkPath = path.dirname(binPath);
                    const java = new jdk.Java(jdkPath);
                    if (java.isJava()) {
                        return java.javaHome;
                    }
                }
            }
        }
        return undefined;
    }
    getJdkSetting(newJdk) {
        const env = this.getEnv() || {};
        const binPath = path.join(newJdk.javaHome, 'bin');
        const currentPath = env[this.variable];
        if (!currentPath) {
            env[this.variable] = `${binPath}${path.delimiter}${JavaEnvPathSetting.ENV_PATH_MASK}`;
        }
        else {
            const newPaths = [binPath];
            const currentPaths = currentPath.split(path.delimiter);
            for (const currentPath of currentPaths) {
                const pathName = path.basename(currentPath).toLowerCase();
                const parentPath = pathName === 'bin' ? path.dirname(currentPath) : undefined;
                const java = parentPath ? new jdk.Java(parentPath) : undefined;
                if (!java || !java.isJava()) {
                    newPaths.push(currentPath);
                }
            }
            env[this.variable] = newPaths.join(path.delimiter);
        }
        return env;
    }
    expandEnvPath() {
        const expandedPath = process.env[this.variable] || JavaEnvPathSetting.ENV_PATH_MASK;
        return expandedPath.endsWith(path.delimiter) ? expandedPath.slice(0, -path.delimiter.length) : expandedPath;
    }
}
JavaEnvPathSetting.ENV_PATH_MASK = '${env:PATH}';
class JavaEnvArrSetting extends JavaEnvSetting {
    constructor(name, property, variable) {
        super(name, property, variable);
    }
    getJavaHome() {
        const env = this.getEnv();
        if (Array.isArray(env)) {
            for (const item of env) {
                if (item.environmentVariable === this.variable) {
                    return item.value;
                }
            }
        }
        return undefined;
    }
    getJdkSetting(jdk) {
        const env = this.getEnv() || [];
        let updated = false;
        for (const item of env) {
            if (item.environmentVariable === this.variable) {
                item.value = jdk.javaHome;
                updated = true;
                break;
            }
        }
        if (!updated) {
            env.push({
                environmentVariable: this.variable,
                value: jdk.javaHome
            });
        }
        return env;
    }
}
class ProjectJavaSettings extends Setting {
    constructor(name, property) {
        super(name, property);
    }
    getSetting() {
        return this.property;
    }
    getDefinitions() {
        return vscode.workspace.getConfiguration().get(this.property);
    }
    getCurrent() {
        var _a, _b;
        const current = [];
        const definitions = this.getDefinitions();
        if (Array.isArray(definitions)) {
            for (const definition of definitions) {
                if (((_a = definition.name) === null || _a === void 0 ? void 0 : _a.length) && ((_b = definition.path) === null || _b === void 0 ? void 0 : _b.length)) {
                    current.push(`${definition.name}: ${definition.path}`);
                }
            }
        }
        return current.length ? current.join(', ') : undefined;
    }
    getJavas() {
        var _a;
        const javas = [];
        const definitions = this.getDefinitions();
        if (Array.isArray(definitions)) {
            for (const definition of definitions) {
                if ((_a = definition.path) === null || _a === void 0 ? void 0 : _a.length) {
                    javas.push(new jdk.Java(definition.path));
                }
            }
        }
        return javas;
    }
    configureIfNotDefined() {
        return false;
    }
    setJdk(jdk, scope) {
        return __awaiter(this, void 0, void 0, function* () {
            const details = yield jdk.getVersion();
            if (details === null || details === void 0 ? void 0 : details.major) {
                const major = `${details.major <= 8 ? '1.' : ''}${details.major}`;
                const version = `JavaSE-${major}`;
                const definitions = this.getDefinitions() || []; // NOTE: merges User & Workspace definitions
                if (Array.isArray(definitions)) {
                    let updated = false;
                    for (const definition of definitions) {
                        if (definition.name === version) {
                            definition.path = jdk.javaHome;
                            updated = true;
                            break;
                        }
                    }
                    if (!updated) {
                        definitions.push({
                            name: version,
                            path: jdk.javaHome
                        });
                    }
                }
                try {
                    yield vscode.workspace.getConfiguration().update(this.property, definitions, scope);
                    return true;
                }
                catch (err) {
                    vscode.window.showErrorMessage(`Failed to update property ${this.getSetting()}: ${err}`);
                }
            }
            return false;
        });
    }
}
const NBLS_EXTENSION_ID = 'asf.apache-netbeans-java';
const NBLS_SETTINGS_NAME = 'Language Server by Apache NetBeans';
const NBLS_SETTINGS_PROPERTY = 'netbeans.jdkhome';
function nblsSetting() {
    return new JavaSetting(NBLS_SETTINGS_NAME, NBLS_SETTINGS_PROPERTY, extension_1.MINIMAL_JDK_VERSION, true);
}
const NBLS_SETTINGS_PROJECT_NAME = 'Language Server by Apache NetBeans - Java Runtime for Projects';
const NBLS_SETTINGS_PROJECT_PROPERTY = 'netbeans.project.jdkhome';
function nblsProjectSetting() {
    return new JavaSetting(NBLS_SETTINGS_PROJECT_NAME, NBLS_SETTINGS_PROJECT_PROPERTY, undefined, true);
}
const JDTLS_EXTENSION_ID = 'redhat.java';
const JDTLS_SETTINGS_NAME = 'Language Server by RedHat';
const JDTLS_SETTINGS_PROPERTY = 'java.jdt.ls.java.home';
function jdtlsSetting() {
    return new JavaSetting(JDTLS_SETTINGS_NAME, JDTLS_SETTINGS_PROPERTY);
}
const PROJECTS_SETTINGS_NAME = 'Language Server by RedHat - Java Runtime for Projects';
const PROJECTS_SETTINGS_PROPERTY = 'java.configuration.runtimes';
function projectsSettings() {
    return new ProjectJavaSettings(PROJECTS_SETTINGS_NAME, PROJECTS_SETTINGS_PROPERTY);
}
exports.projectsSettings = projectsSettings;
const TERMINAL_JAVAHOME_SETTINGS_NAME = 'Integrated Terminal JAVA_HOME';
const TERMINAL_JAVAHOME_SETTINGS_PROPERTY = `terminal.integrated.env.${platformProperty()}`;
function terminalJavaHomeSetting() {
    return new JavaEnvSetting(TERMINAL_JAVAHOME_SETTINGS_NAME, TERMINAL_JAVAHOME_SETTINGS_PROPERTY, 'JAVA_HOME');
}
const TERMINAL_PATH_SETTINGS_NAME = 'Integrated Terminal PATH';
const TERMINAL_PATH_SETTINGS_PROPERTY = `terminal.integrated.env.${platformProperty()}`;
function terminalPathSetting() {
    return new JavaEnvPathSetting(TERMINAL_PATH_SETTINGS_NAME, TERMINAL_PATH_SETTINGS_PROPERTY, 'PATH');
}
const MAVEN_EXTENSION_ID = 'vscjava.vscode-maven';
const MAVEN_JAVAHOME_SETTINGS_NAME = 'Maven Runtime JAVA_HOME';
const MAVEN_JAVAHOME_SETTINGS_PROPERTY = 'maven.terminal.customEnv';
function mavenJavaHomeSetting() {
    return new JavaEnvArrSetting(MAVEN_JAVAHOME_SETTINGS_NAME, MAVEN_JAVAHOME_SETTINGS_PROPERTY, 'JAVA_HOME');
}
function getAvailable() {
    const settings = [];
    if (vscode.extensions.getExtension(NBLS_EXTENSION_ID)) {
        settings.push(nblsSetting());
        settings.push(nblsProjectSetting());
    }
    if (vscode.extensions.getExtension(JDTLS_EXTENSION_ID)) {
        settings.push(jdtlsSetting());
        settings.push(projectsSettings());
    }
    settings.push(terminalJavaHomeSetting());
    settings.push(terminalPathSetting());
    if (vscode.extensions.getExtension(MAVEN_EXTENSION_ID)) {
        settings.push(mavenJavaHomeSetting());
    }
    return settings;
}
exports.getAvailable = getAvailable;
function platformProperty() {
    const platform = process.platform;
    if (platform === 'win32') {
        return 'windows';
    }
    else if (platform === 'darwin') {
        return 'osx';
    }
    return platform;
}
//# sourceMappingURL=settings.js.map