"use strict";
/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.GradleProject = exports.MavenProject = exports.Project = exports.getProjectFrom = exports.BuildSystemType = void 0;
const xml2js = require("xml2js");
const fs = require("fs");
const path = require("path");
const vscode = require("vscode");
const os = require("os");
const javaUtil_1 = require("./javaUtil");
const GET_PROJECT_INFO = 'nbls.project.info';
const GRADLE_TARGET_COMPATIBILITY_REGEX = /targetCompatibility\s*=\s*(?:JavaVersion\s*\.\s*toVersion\s*\(\s*['"](\d+(\.\d+)?)['"]\s*\)|['"](\d+(\.\d+)?)['"])/;
const GRADLE_SOURCE_COMPATIBILITY_REGEX = /sourceCompatibility\s*=\s*(?:JavaVersion\s*\.\s*toVersion\s*\(\s*['"](\d+(\.\d+)?)['"]\s*\)|['"](\d+(\.\d+)?)['"])/;
var BuildSystemType;
(function (BuildSystemType) {
    BuildSystemType["MAVEN"] = "Maven";
    BuildSystemType["GRADLE"] = "Gradle";
    BuildSystemType["UNKNOWN"] = "Unknown";
})(BuildSystemType = exports.BuildSystemType || (exports.BuildSystemType = {}));
function getProjectFrom(projectUri) {
    return __awaiter(this, void 0, void 0, function* () {
        const projectInfos = yield vscode.commands.executeCommand(GET_PROJECT_INFO, projectUri.toString(), { recursive: true, projectStructure: true });
        if ((projectInfos === null || projectInfos === void 0 ? void 0 : projectInfos.length) && projectInfos[0]) {
            const projectDirectory = projectInfos[0].projectDirectory.toString();
            const buildSystem = resolveBuildSystemType(projectUri, projectInfos[0].projectType);
            switch (buildSystem) {
                case BuildSystemType.MAVEN:
                    const mavenSubprojects = projectInfos[0].subprojects
                        .map((subproject) => new MavenProject(subproject, []));
                    return new MavenProject(projectDirectory, mavenSubprojects);
                case BuildSystemType.GRADLE:
                    const gradleSubprojects = projectInfos[0].subprojects
                        .map((subproject) => new GradleProject(subproject, []));
                    return new GradleProject(projectDirectory, gradleSubprojects);
                default:
                    break;
            }
        }
        return Promise.resolve(undefined);
    });
}
exports.getProjectFrom = getProjectFrom;
function resolveBuildSystemType(uri, projectType) {
    if (projectType === null || projectType === void 0 ? void 0 : projectType.includes('gradle')) {
        return BuildSystemType.GRADLE;
    }
    if (projectType === null || projectType === void 0 ? void 0 : projectType.includes('maven')) {
        return BuildSystemType.MAVEN;
    }
    if (fs.existsSync(path.join(uri.fsPath, 'build.gradle'))) {
        return BuildSystemType.GRADLE;
    }
    if (fs.existsSync(path.join(uri.fsPath, 'pom.xml'))) {
        return BuildSystemType.MAVEN;
    }
    return BuildSystemType.UNKNOWN;
}
class Project {
    constructor(directory, subprojects) {
        this.directory = vscode.Uri.parse(directory).fsPath;
        this.subprojects = subprojects;
    }
    // Whether the project contains subprojects
    containsSubprojects() {
        return this.subprojects.length > 0;
    }
    getJavaVersion() {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.containsSubprojects()) {
                return this.extractJavaVersion();
            }
            let maxJavaVersion;
            for (const subproject of this.subprojects) {
                const projectDirectory = vscode.Uri.file(subproject.directory).toString();
                const subInfos = yield vscode.commands.executeCommand(GET_PROJECT_INFO, projectDirectory);
                if ((subInfos === null || subInfos === void 0 ? void 0 : subInfos.length) && subInfos[0]) {
                    const javaVersion = subproject.extractJavaVersion();
                    if (!maxJavaVersion || (javaVersion && javaVersion > maxJavaVersion)) {
                        maxJavaVersion = javaVersion;
                    }
                }
            }
            return maxJavaVersion;
        });
    }
}
exports.Project = Project;
class MavenProject extends Project {
    constructor(directory, subprojects) {
        super(directory, subprojects);
    }
    extractJavaVersion() {
        var _a;
        const buildscript = path.resolve(this.directory, 'pom.xml');
        let version;
        if (fs.existsSync(buildscript)) {
            const parser = new xml2js.Parser({ async: false });
            parser.parseString(((_a = fs.readFileSync(buildscript)) === null || _a === void 0 ? void 0 : _a.toString()) || '', (err, result) => {
                var _a;
                if (!err && result) {
                    const properties = (_a = result['project']) === null || _a === void 0 ? void 0 : _a['properties'];
                    if (properties === null || properties === void 0 ? void 0 : properties[0]) {
                        const mavenCompilerTarget = properties[0]['maven.compiler.target'];
                        if (mavenCompilerTarget === null || mavenCompilerTarget === void 0 ? void 0 : mavenCompilerTarget[0]) {
                            version = mavenCompilerTarget[0];
                            return;
                        }
                        const mavenCompilerSource = properties[0]['maven.compiler.source'];
                        if (mavenCompilerSource === null || mavenCompilerSource === void 0 ? void 0 : mavenCompilerSource[0]) {
                            version = mavenCompilerSource[0];
                            return;
                        }
                        const jdkVersion = properties[0]['jdk.version'];
                        if (jdkVersion === null || jdkVersion === void 0 ? void 0 : jdkVersion[0]) {
                            version = jdkVersion[0];
                            return;
                        }
                    }
                }
            });
        }
        return version ? Number((0, javaUtil_1.normalizeJavaVersion)(version)) : undefined;
    }
}
exports.MavenProject = MavenProject;
class GradleProject extends Project {
    constructor(directory, subprojects) {
        super(directory, subprojects);
    }
    getJavaCompatibilityFrom(buildscript, from) {
        const res = from === 'target' ? GRADLE_TARGET_COMPATIBILITY_REGEX.exec(buildscript) : GRADLE_SOURCE_COMPATIBILITY_REGEX.exec(buildscript);
        if (res === null || res === void 0 ? void 0 : res[3]) {
            return res[3]; // Get the version number directly
        }
        else if (res === null || res === void 0 ? void 0 : res[1]) {
            return res[1]; // Get the version number from JavaVersion.toVersion
        }
        return undefined;
    }
    extractJavaVersion() {
        var _a;
        let version;
        const buildscript = path.resolve(this.directory, 'build.gradle');
        if (fs.existsSync(buildscript)) {
            (_a = fs.readFileSync(buildscript)) === null || _a === void 0 ? void 0 : _a.toString().split(os.EOL).find(l => {
                let tempVersion = this.getJavaCompatibilityFrom(l, 'target');
                if (!tempVersion) {
                    tempVersion = this.getJavaCompatibilityFrom(l, 'source');
                }
                if (tempVersion) {
                    version = Number((0, javaUtil_1.normalizeJavaVersion)(tempVersion));
                    return true;
                }
                return false;
            });
        }
        return version;
    }
}
exports.GradleProject = GradleProject;
//# sourceMappingURL=project.js.map