"use strict";
/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseJavaVersion = exports.validateJDKCompatibility = void 0;
const vscode = require("vscode");
const jdkUtils = require("jdk-utils");
const extensionUtils_1 = require("./extensionUtils");
const javaUtil_1 = require("./javaUtil");
const project_1 = require("./project");
const CONFIGURE_JDK_COMMAND = 'nbls.jdk.configuration';
const CONFIGURE_JDK = 'Configure JDK';
function validateJDKCompatibility(projectPath) {
    return __awaiter(this, void 0, void 0, function* () {
        // In this case RH will try it's best to validate Java versions
        if ((0, extensionUtils_1.isRHExtensionActive)())
            return;
        const projectJavaVersion = yield getProjectJavaVersion();
        // at this point, the NBLS client should be running, so clientRuntimeJDK should be set.
        const ideJavaVersion = yield parseJavaVersion(yield (0, extensionUtils_1.currentClientJDK)());
        const ideProjectJavaVersion = yield parseJavaVersion(projectPath);
        let conflictingVersion = 0;
        if (projectJavaVersion) {
            // project settings is preferred, if defined.
            if (ideProjectJavaVersion) {
                if (ideProjectJavaVersion < projectJavaVersion) {
                    conflictingVersion = ideProjectJavaVersion;
                }
            }
            else if (ideJavaVersion && ideJavaVersion < projectJavaVersion) {
                conflictingVersion = ideJavaVersion;
            }
        }
        if (conflictingVersion) {
            const value = yield vscode.window.showWarningMessage(`Source level (JDK ${projectJavaVersion}) not compatible with current JDK installation (JDK ${conflictingVersion})`, CONFIGURE_JDK);
            if (value === CONFIGURE_JDK) {
                vscode.commands.executeCommand(CONFIGURE_JDK_COMMAND);
            }
        }
    });
}
exports.validateJDKCompatibility = validateJDKCompatibility;
function parseJavaVersion(javaPath) {
    return __awaiter(this, void 0, void 0, function* () {
        if (!javaPath)
            return undefined;
        const javaRuntime = yield jdkUtils.getRuntime(javaPath, { checkJavac: true });
        if (!(javaRuntime === null || javaRuntime === void 0 ? void 0 : javaRuntime.hasJavac)) {
            return undefined;
        }
        const version = yield (0, javaUtil_1.getJavaVersion)(javaRuntime.homedir);
        return version ? Number(version) : undefined;
    });
}
exports.parseJavaVersion = parseJavaVersion;
function getProjectJavaVersion() {
    var _a;
    return __awaiter(this, void 0, void 0, function* () {
        const folder = (_a = vscode.workspace.workspaceFolders) === null || _a === void 0 ? void 0 : _a[0];
        if (!folder)
            return undefined;
        yield (0, extensionUtils_1.waitForNblsCommandToBeAvailable)();
        const project = yield (0, project_1.getProjectFrom)(folder.uri);
        const javaVersion = yield (project === null || project === void 0 ? void 0 : project.getJavaVersion());
        return javaVersion;
    });
}
//# sourceMappingURL=validation.js.map