"use strict";
/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SSHSession = void 0;
const vscode = require("vscode");
const fs = require("fs");
const terminalRunner_1 = require("./terminalRunner");
const extension_1 = require("../extension");
const path = require("path");
const util_1 = require("util");
const Handlebars = require("handlebars");
const readFile = (0, util_1.promisify)(fs.readFile);
class SSHSession {
    constructor(username, host) {
        this.username = username;
        this.host = host;
    }
    open(sessionName) {
        let terminalName = sessionName || `${this.username}@${this.host}`;
        const sshCommand = `ssh ${this.username}@${this.host}`;
        (0, terminalRunner_1.runCommandInTerminal)(sshCommand, `SSH: ${terminalName}`);
    }
    runDocker(context, dockerImage, isRepositoryPrivate) {
        return __awaiter(this, void 0, void 0, function* () {
            const configFiles = yield vscode.commands.executeCommand(extension_1.COMMAND_PREFIX + '.config.file.path');
            const { applicationProperties, bootstrapProperties } = configFiles;
            let bearerTokenFile;
            const applicationPropertiesRemotePath = `/home/${this.username}/application.properties`;
            const bootstrapPropertiesRemotePath = `/home/${this.username}/bootstrap.properties`;
            const bearerTokenRemotePath = `/home/${this.username}/token.txt`;
            const applicationPropertiesContainerPath = "/home/app/application.properties";
            const bootstrapPropertiesContainerPath = "/home/app/bootstrap.properties";
            const ocirServer = dockerImage.split('/')[0];
            const remotePathToCopyTo = `/home/${this.username}/`;
            let sshCommand = "";
            let mountVolume = "";
            let micronautConfigFilesEnv = "";
            let filesToCopy = "";
            let renameFilesCommand = "";
            const removeOldFilesCommand = `rm -f ${bootstrapPropertiesRemotePath} ${applicationPropertiesRemotePath} ${bearerTokenRemotePath}`;
            if (isRepositoryPrivate) {
                bearerTokenFile = yield vscode.commands.executeCommand(extension_1.COMMAND_PREFIX + '.cloud.assets.createBearerToken', ocirServer);
                if (bearerTokenFile) {
                    filesToCopy = bearerTokenFile;
                    renameFilesCommand = `mv ${remotePathToCopyTo}${path.basename(bearerTokenFile)} ${bearerTokenRemotePath} && `;
                }
            }
            if (bootstrapProperties) {
                filesToCopy += ` ${bootstrapProperties}`;
                renameFilesCommand += ` mv ${remotePathToCopyTo}${path.basename(bootstrapProperties)} ${bootstrapPropertiesRemotePath} && `;
                mountVolume = `-v ${bootstrapPropertiesRemotePath}:${bootstrapPropertiesContainerPath}:Z `;
                micronautConfigFilesEnv = `${bootstrapPropertiesContainerPath}`;
            }
            if (applicationProperties) {
                filesToCopy += ` ${applicationProperties}`;
                renameFilesCommand += ` mv ${remotePathToCopyTo}${path.basename(applicationProperties)} ${applicationPropertiesRemotePath} && `;
                mountVolume += ` -v ${applicationPropertiesRemotePath}:${applicationPropertiesContainerPath}:Z`;
                micronautConfigFilesEnv += `${bootstrapProperties ? "," : ""}${applicationPropertiesContainerPath}`;
            }
            let templateFilePath = path.join(context.extensionPath, "templates", "run-container.sh.handlebars");
            const template = yield this.getTemplateFromPath(templateFilePath);
            const script = template({
                username: this.username,
                isRepositoryPrivate,
                bearerTokenRemotePath,
                ocirServer,
                dockerImage,
                mountVolume,
                micronautConfigFilesEnv
            });
            const tempDir = process.env.TEMP || process.env.TMP || '/tmp';
            const scriptName = `run-container-${Date.now()}.sh`;
            const runContainerScript = path.join(tempDir, scriptName);
            fs.writeFileSync(runContainerScript, script);
            renameFilesCommand += ` mv -f ${scriptName} run-container.sh `;
            sshCommand = `scp ${filesToCopy} ${runContainerScript} ${this.username}@${this.host}:${remotePathToCopyTo} && `;
            sshCommand += `ssh ${this.username}@${this.host} "${removeOldFilesCommand} && ${renameFilesCommand} && rm -f ${scriptName} && chmod +x run-container.sh && ./run-container.sh" `;
            (0, terminalRunner_1.runCommandInTerminal)(sshCommand, `Container: ${this.username}@${this.host}`);
        });
    }
    getTemplateFromPath(path) {
        return __awaiter(this, void 0, void 0, function* () {
            const templateFile = yield readFile(path, "utf-8");
            return Handlebars.compile(templateFile);
        });
    }
}
exports.SSHSession = SSHSession;
//# sourceMappingURL=ssh.js.map