/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
'use strict';
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.NbTestAdapter = void 0;
const vscode_1 = require("vscode");
const path = require("path");
const protocol_1 = require("./protocol");
const extension_1 = require("./extension");
class NbTestAdapter {
    constructor() {
        this.disposables = [];
        this.started = false;
        this.testController = vscode_1.tests.createTestController('apacheNetBeansController', 'Apache NetBeans');
        const runHandler = (request, cancellation) => this.run(request, cancellation);
        this.testController.createRunProfile('Run Tests', vscode_1.TestRunProfileKind.Run, runHandler);
        this.testController.createRunProfile('Debug Tests', vscode_1.TestRunProfileKind.Debug, runHandler);
        this.disposables.push(this.testController);
        this.load();
        this.suiteStates = new Map();
    }
    registerRunInParallelProfile(projects) {
        if (!this.parallelRunProfile) {
            const runHandler = (request, cancellation) => this.run(request, cancellation, true, projects);
            this.parallelRunProfile = this.testController.createRunProfile("Run Tests In Parallel", vscode_1.TestRunProfileKind.Run, runHandler, true);
        }
        this.testController.items.replace([]);
        this.load();
    }
    testInParallelProfileExist() {
        return this.parallelRunProfile ? true : false;
    }
    runTestsWithParallelProfile(projects) {
        if (this.parallelRunProfile) {
            this.run(new vscode_1.TestRunRequest(undefined, undefined, this.parallelRunProfile), new vscode_1.CancellationTokenSource().token, true, projects);
        }
    }
    load() {
        return __awaiter(this, void 0, void 0, function* () {
            for (let workspaceFolder of vscode_1.workspace.workspaceFolders || []) {
                const loadedTests = yield vscode_1.commands.executeCommand(extension_1.COMMAND_PREFIX + '.load.workspace.tests', workspaceFolder.uri.toString());
                if (loadedTests) {
                    loadedTests.forEach((suite) => {
                        this.updateTests(suite);
                    });
                }
            }
        });
    }
    run(request, cancellation, testInParallel = false, projects) {
        var _a, _b, _c;
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.currentRun) {
                if (!testInParallel) {
                    vscode_1.commands.executeCommand('workbench.debug.action.focusRepl');
                }
                cancellation.onCancellationRequested(() => this.cancel());
                this.currentRun = this.testController.createTestRun(request);
                this.currentRun.token.onCancellationRequested(() => this.cancel());
                this.itemsToRun = new Set();
                this.started = false;
                if (request.include) {
                    const include = [...new Map(request.include.map(item => { var _a; return !item.uri && ((_a = item.parent) === null || _a === void 0 ? void 0 : _a.uri) ? [item.parent.id, item.parent] : [item.id, item]; })).values()];
                    for (let item of include) {
                        if (item.uri) {
                            this.set(item, 'enqueued');
                            (_a = item.parent) === null || _a === void 0 ? void 0 : _a.children.forEach(child => {
                                var _a;
                                if ((_a = child.id) === null || _a === void 0 ? void 0 : _a.includes(item.id)) {
                                    this.set(child, 'enqueued');
                                }
                            });
                            const idx = item.id.indexOf(':');
                            const isNestedClass = item.id.includes('$');
                            const topLevelClassName = item.id.lastIndexOf('.');
                            let nestedClass;
                            if (isNestedClass && topLevelClassName > 0) {
                                nestedClass = idx < 0
                                    ? item.id.slice(topLevelClassName + 1)
                                    : item.id.substring(topLevelClassName + 1, idx);
                                nestedClass = nestedClass.replace('$', '.');
                            }
                            if (!cancellation.isCancellationRequested) {
                                try {
                                    yield vscode_1.commands.executeCommand(((_b = request.profile) === null || _b === void 0 ? void 0 : _b.kind) === vscode_1.TestRunProfileKind.Debug ? extension_1.COMMAND_PREFIX + '.debug.single' : extension_1.COMMAND_PREFIX + '.run.single', item.uri.toString(), idx < 0 ? undefined : item.id.slice(idx + 1), undefined /* configuration */, nestedClass);
                                }
                                catch (err) {
                                    // test state will be handled in the code below
                                    console.log(err);
                                }
                            }
                        }
                    }
                }
                else {
                    this.testController.items.forEach(item => this.set(item, 'enqueued'));
                    for (let workspaceFolder of vscode_1.workspace.workspaceFolders || []) {
                        if (!cancellation.isCancellationRequested) {
                            try {
                                if (testInParallel) {
                                    yield vscode_1.commands.executeCommand(extension_1.COMMAND_PREFIX + '.run.test', workspaceFolder.uri.toString(), undefined, undefined, undefined, true, projects);
                                }
                                else {
                                    yield vscode_1.commands.executeCommand(((_c = request.profile) === null || _c === void 0 ? void 0 : _c.kind) === vscode_1.TestRunProfileKind.Debug ? extension_1.COMMAND_PREFIX + '.debug.test' : extension_1.COMMAND_PREFIX + '.run.test', workspaceFolder.uri.toString());
                                }
                            }
                            catch (err) {
                                // test state will be handled in the code below
                                console.log(err);
                            }
                        }
                    }
                }
                if (this.started) {
                    this.itemsToRun.forEach(item => this.set(item, 'skipped'));
                }
                // TBD - message
                else {
                    this.itemsToRun.forEach(item => this.set(item, 'failed', new vscode_1.TestMessage('Build failure'), false, true));
                }
                this.itemsToRun = undefined;
                this.currentRun.end();
                this.currentRun = undefined;
            }
        });
    }
    set(item, state, message, noPassDown, dispatchBuildFailEvent) {
        var _a, _b, _c;
        if (this.currentRun) {
            switch (state) {
                case 'enqueued':
                    this.dispatchTestEvent(state, item);
                    (_a = this.itemsToRun) === null || _a === void 0 ? void 0 : _a.add(item);
                    this.currentRun.enqueued(item);
                    break;
                case 'skipped':
                    this.dispatchTestEvent(state, item);
                case 'started':
                case 'passed':
                    (_b = this.itemsToRun) === null || _b === void 0 ? void 0 : _b.delete(item);
                    this.currentRun[state](item);
                    break;
                case 'failed':
                case 'errored':
                    if (dispatchBuildFailEvent) {
                        this.dispatchTestEvent(state, item);
                    }
                    (_c = this.itemsToRun) === null || _c === void 0 ? void 0 : _c.delete(item);
                    this.currentRun[state](item, message || new vscode_1.TestMessage(""));
                    break;
            }
            this.suiteStates.set(item, state);
            if (!noPassDown) {
                item.children.forEach(child => this.set(child, state, message, noPassDown, dispatchBuildFailEvent));
            }
        }
    }
    dispatchTestEvent(state, testItem) {
        var _a, _b, _c, _d, _e, _f;
        if (testItem.parent && testItem.children.size > 0) {
            if (testItem.id.includes(":") && testItem.parent.parent) {
                // special case when parameterized test
                const testEvent = this.getParametrizedTestEvent(state, testItem);
                if (!testEvent)
                    return;
                this.dispatchEvent(testEvent);
            }
            else {
                this.dispatchEvent({
                    name: testItem.id,
                    moduleName: testItem.parent.id,
                    modulePath: (_a = testItem.parent.uri) === null || _a === void 0 ? void 0 : _a.path,
                    state,
                });
            }
        }
        else if (testItem.children.size === 0) {
            const testSuite = testItem.parent;
            const parentState = testSuite && this.suiteStates.get(testSuite) ? this.suiteStates.get(testSuite) : state;
            if (testSuite) {
                let moduleName = (_b = testSuite.parent) === null || _b === void 0 ? void 0 : _b.id;
                let modulePath = (_d = (_c = testSuite.parent) === null || _c === void 0 ? void 0 : _c.uri) === null || _d === void 0 ? void 0 : _d.path;
                if (testSuite.id.includes(":") && ((_e = testSuite.parent) === null || _e === void 0 ? void 0 : _e.parent)) {
                    // special case when parameterized test
                    moduleName = testSuite.parent.parent.id;
                    modulePath = (_f = testSuite.parent.parent.uri) === null || _f === void 0 ? void 0 : _f.path;
                }
                const testSuiteEvent = {
                    name: testSuite.id,
                    moduleName,
                    modulePath,
                    state: parentState,
                    tests: []
                };
                testSuite === null || testSuite === void 0 ? void 0 : testSuite.children.forEach(suite => {
                    var _a;
                    if (suite.id === testItem.id) {
                        const idx = suite.id.indexOf(':');
                        if (idx >= 0) {
                            const name = suite.id.slice(idx + 1);
                            (_a = testSuiteEvent.tests) === null || _a === void 0 ? void 0 : _a.push({
                                id: suite.id,
                                name,
                                state
                            });
                        }
                    }
                });
                this.dispatchEvent(testSuiteEvent);
            }
        }
    }
    getParametrizedTestEvent(state, testItem) {
        var _a;
        if (!testItem.parent || !testItem.parent.parent) {
            return undefined;
        }
        let name = testItem.parent.id;
        const idx = name.indexOf(':');
        return {
            name,
            moduleName: testItem.parent.parent.id,
            modulePath: (_a = testItem.parent.parent.uri) === null || _a === void 0 ? void 0 : _a.path,
            state,
            tests: [
                {
                    id: name,
                    name: name.slice(idx + 1),
                    state
                }
            ]
        };
    }
    dispatchEvent(event) {
        const testProgressListeners = extension_1.listeners.get(extension_1.TEST_PROGRESS_EVENT);
        testProgressListeners === null || testProgressListeners === void 0 ? void 0 : testProgressListeners.forEach(listener => {
            vscode_1.commands.executeCommand(listener, event);
        });
    }
    cancel() {
        vscode_1.debug.stopDebugging();
    }
    dispose() {
        this.cancel();
        for (const disposable of this.disposables) {
            disposable.dispose();
        }
        this.disposables = [];
    }
    testOutput(output) {
        if (this.currentRun && output) {
            this.currentRun.appendOutput(output.replace(/\n/g, '\r\n'));
        }
    }
    testProgress(suite) {
        var _a;
        const currentModule = this.testController.items.get(this.getModuleItemId(suite.moduleName));
        const currentSuite = currentModule === null || currentModule === void 0 ? void 0 : currentModule.children.get(suite.name);
        switch (suite.state) {
            case 'loaded':
                this.updateTests(suite);
                break;
            case 'started':
                this.started = true;
                if (currentSuite) {
                    this.set(currentSuite, 'started');
                }
                break;
            case 'passed':
            case "failed":
            case 'errored':
            case 'skipped':
                if (suite.tests) {
                    this.updateTests(suite, true);
                    if (currentSuite && currentModule) {
                        const suiteMessages = [];
                        (_a = suite.tests) === null || _a === void 0 ? void 0 : _a.forEach(test => {
                            var _a, _b, _c;
                            if (this.currentRun) {
                                let currentTest = currentSuite.children.get(test.id);
                                if (!currentTest) {
                                    currentSuite.children.forEach(item => {
                                        if (!currentTest) {
                                            const subName = this.subTestName(item, test);
                                            if (subName) {
                                                currentTest = subName === '()' ? item : item.children.get(test.id);
                                            }
                                        }
                                    });
                                }
                                let message;
                                if (test.stackTrace) {
                                    message = new vscode_1.TestMessage(this.stacktrace2Message((_a = currentTest === null || currentTest === void 0 ? void 0 : currentTest.uri) === null || _a === void 0 ? void 0 : _a.toString(), test.stackTrace));
                                    if (currentTest) {
                                        const testUri = currentTest.uri || ((_b = currentTest.parent) === null || _b === void 0 ? void 0 : _b.uri);
                                        if (testUri) {
                                            const fileName = path.basename(testUri.path);
                                            const line = test.stackTrace.map(frame => {
                                                const info = frame.match(/^\s*at[^\(]*\((\S*):(\d*)\)$/);
                                                if (info && info.length >= 3 && info[1] === fileName) {
                                                    return parseInt(info[2]);
                                                }
                                                return null;
                                            }).find(l => l);
                                            const pos = line ? new vscode_1.Position(line - 1, 0) : (_c = currentTest.range) === null || _c === void 0 ? void 0 : _c.start;
                                            if (pos) {
                                                message.location = new vscode_1.Location(testUri, pos);
                                            }
                                        }
                                    }
                                    else {
                                        message.location = new vscode_1.Location(currentSuite.uri, currentSuite.range.start);
                                    }
                                }
                                if (currentTest && test.state !== 'loaded') {
                                    this.set(currentTest, test.state, message, true);
                                }
                                else if (test.state !== 'passed' && message) {
                                    suiteMessages.push(message);
                                }
                            }
                        });
                        if (suiteMessages.length > 0) {
                            this.set(currentSuite, 'errored', suiteMessages, true);
                            currentSuite.children.forEach(item => this.set(item, 'skipped'));
                        }
                        else {
                            this.set(currentSuite, suite.state, undefined, true);
                        }
                        this.set(currentModule, this.calculateStateFor(currentModule), undefined, true);
                    }
                }
                break;
        }
    }
    calculateStateFor(testItem) {
        let passed = 0;
        testItem.children.forEach(item => {
            const state = this.suiteStates.get(item);
            if (state === 'enqueued' || state === 'failed')
                return state;
            if (state === 'passed')
                passed++;
        });
        if (passed > 0)
            return 'passed';
        return 'skipped';
    }
    updateTests(suite, testExecution) {
        var _a, _b;
        const moduleName = this.getModuleItemId(suite.moduleName);
        let currentModule = this.testController.items.get(moduleName);
        if (!currentModule) {
            const parsedName = this.parseModuleName(moduleName);
            currentModule = this.testController.createTestItem(moduleName, this.getNameWithIcon(parsedName, 'module'), this.getModulePath(suite));
            this.testController.items.add(currentModule);
        }
        const suiteChildren = [];
        let currentSuite = currentModule.children.get(suite.name);
        const suiteUri = suite.file ? vscode_1.Uri.parse(suite.file) : undefined;
        if (!currentSuite || suiteUri && ((_a = currentSuite.uri) === null || _a === void 0 ? void 0 : _a.toString()) !== suiteUri.toString()) {
            currentSuite = this.testController.createTestItem(suite.name, this.getNameWithIcon(suite.name, 'class'), suiteUri);
            suiteChildren.push(currentSuite);
        }
        currentModule.children.forEach(suite => suiteChildren.push(suite));
        const suiteRange = (0, protocol_1.asRange)(suite.range);
        if (!testExecution && suiteRange && suiteRange !== currentSuite.range) {
            currentSuite.range = suiteRange;
        }
        const children = [];
        const parentTests = new Map();
        (_b = suite.tests) === null || _b === void 0 ? void 0 : _b.forEach(test => {
            var _a;
            let currentTest = currentSuite === null || currentSuite === void 0 ? void 0 : currentSuite.children.get(test.id);
            const testUri = test.file ? vscode_1.Uri.parse(test.file) : undefined;
            if (currentTest) {
                if (testUri && ((_a = currentTest.uri) === null || _a === void 0 ? void 0 : _a.toString()) !== (testUri === null || testUri === void 0 ? void 0 : testUri.toString())) {
                    currentTest = this.testController.createTestItem(test.id, this.getNameWithIcon(test.name, 'method'), testUri);
                    currentSuite === null || currentSuite === void 0 ? void 0 : currentSuite.children.add(currentTest);
                }
                const testRange = (0, protocol_1.asRange)(test.range);
                if (!testExecution && testRange && testRange !== currentTest.range) {
                    currentTest.range = testRange;
                }
                children.push(currentTest);
            }
            else {
                if (testExecution) {
                    const parents = new Map();
                    currentSuite === null || currentSuite === void 0 ? void 0 : currentSuite.children.forEach(item => {
                        const subName = this.subTestName(item, test);
                        if (subName && '()' !== subName) {
                            parents.set(item, subName);
                        }
                    });
                    const parent = this.selectParent(parents);
                    if (parent) {
                        let arr = parentTests.get(parent.test);
                        if (!arr) {
                            parentTests.set(parent.test, arr = []);
                            children.push(parent.test);
                        }
                        arr.push(this.testController.createTestItem(test.id, this.getNameWithIcon(parent.label, 'method')));
                    }
                }
                else {
                    currentTest = this.testController.createTestItem(test.id, this.getNameWithIcon(test.name, 'method'), testUri);
                    currentTest.range = (0, protocol_1.asRange)(test.range);
                    children.push(currentTest);
                    currentSuite === null || currentSuite === void 0 ? void 0 : currentSuite.children.add(currentTest);
                }
            }
        });
        if (testExecution) {
            parentTests.forEach((val, key) => {
                const item = this.testController.createTestItem(key.id, key.label, key.uri);
                item.range = key.range;
                item.children.replace(val);
                currentSuite === null || currentSuite === void 0 ? void 0 : currentSuite.children.add(item);
            });
        }
        else {
            currentSuite.children.replace(children);
            currentModule.children.replace(suiteChildren);
        }
    }
    getModuleItemId(moduleName) {
        return (moduleName === null || moduleName === void 0 ? void 0 : moduleName.replace(":", "-")) || "";
    }
    parseModuleName(moduleName) {
        if (!this.parallelRunProfile) {
            return moduleName.replace(":", "-");
        }
        const index = moduleName.indexOf(":");
        if (index !== -1) {
            return moduleName.slice(index + 1);
        }
        const parts = moduleName.split("-");
        return parts[parts.length - 1];
    }
    getModulePath(suite) {
        return vscode_1.Uri.parse(suite.modulePath || "");
    }
    getNameWithIcon(itemName, itemType) {
        switch (itemType) {
            case 'module':
                return `$(project) ${itemName}`;
            case 'class':
                return `$(symbol-class) ${itemName}`;
            case 'method':
                return `$(symbol-method) ${itemName}`;
            default:
                return itemName;
        }
    }
    getNameWithoutIcon(itemName) {
        return itemName.replace(/^\$\([^)]+\)\s*/, "");
    }
    subTestName(item, test) {
        if (test.id.startsWith(item.id)) {
            let label = test.name;
            const nameWithoutIcon = this.getNameWithoutIcon(item.label);
            if (label.startsWith(nameWithoutIcon)) {
                label = label.slice(nameWithoutIcon.length).trim();
            }
            return label;
        }
        else {
            const regexp = new RegExp(item.id.replace(/[-[\]{}()*+?.,\\^$|\s]/g, '\\$&').replace(/#\w*/g, '\\S*'));
            if (regexp.test(test.id)) {
                return test.name;
            }
        }
        return undefined;
    }
    selectParent(parents) {
        let ret = undefined;
        parents.forEach((label, parentTest) => {
            if (ret) {
                if (parentTest.id.replace(/#\w*/g, '').length > ret.test.id.replace(/#\w*/g, '').length) {
                    ret = { test: parentTest, label };
                }
            }
            else {
                ret = { test: parentTest, label };
            }
        });
        return ret;
    }
    stacktrace2Message(currentTestUri, stacktrace) {
        const regExp = /(\s*at\s+(?:[\w$\\.]+\/)?((?:[\w$]+\.)+[\w\s$<>]+))\(((.*):(\d+))\)/;
        const message = new vscode_1.MarkdownString();
        message.isTrusted = true;
        message.supportHtml = true;
        for (const line of stacktrace) {
            if (message.value.length) {
                message.appendMarkdown('<br/>');
            }
            const result = regExp.exec(line);
            if (result) {
                message.appendText(result[1]).appendText('(').appendMarkdown(`[${result[3]}](command:${extension_1.COMMAND_PREFIX}.open.stacktrace?${encodeURIComponent(JSON.stringify([currentTestUri, result[2], result[4], +result[5]]))})`).appendText(')');
            }
            else {
                message.appendText(line);
            }
        }
        return message;
    }
}
exports.NbTestAdapter = NbTestAdapter;
//# sourceMappingURL=testAdapter.js.map