/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.boot.autoconfigure.ssl;

import java.io.Closeable;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.time.Duration;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.core.log.LogMessage;
import org.springframework.util.Assert;

class FileWatcher
implements Closeable {
    private static final Log logger = LogFactory.getLog(FileWatcher.class);
    private final Duration quietPeriod;
    private final Object lock = new Object();
    private WatcherThread thread;

    FileWatcher(Duration quietPeriod) {
        Assert.notNull((Object)quietPeriod, (String)"QuietPeriod must not be null");
        this.quietPeriod = quietPeriod;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void watch(Set<Path> paths, Runnable action) {
        Assert.notNull(paths, (String)"Paths must not be null");
        Assert.notNull((Object)action, (String)"Action must not be null");
        if (paths.isEmpty()) {
            return;
        }
        Object object = this.lock;
        synchronized (object) {
            try {
                if (this.thread == null) {
                    this.thread = new WatcherThread();
                    this.thread.start();
                }
                this.thread.register(new Registration(paths, action));
            }
            catch (IOException ex) {
                throw new UncheckedIOException("Failed to register paths for watching: " + paths, ex);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        Object object = this.lock;
        synchronized (object) {
            if (this.thread != null) {
                this.thread.close();
                this.thread.interrupt();
                try {
                    this.thread.join();
                }
                catch (InterruptedException ex) {
                    Thread.currentThread().interrupt();
                }
                this.thread = null;
            }
        }
    }

    private class WatcherThread
    extends Thread
    implements Closeable {
        private final WatchService watchService = FileSystems.getDefault().newWatchService();
        private final Map<WatchKey, List<Registration>> registrations = new ConcurrentHashMap<WatchKey, List<Registration>>();
        private volatile boolean running = true;

        WatcherThread() throws IOException {
            this.setName("ssl-bundle-watcher");
            this.setDaemon(true);
            this.setUncaughtExceptionHandler(this::onThreadException);
        }

        private void onThreadException(Thread thread, Throwable throwable) {
            logger.error((Object)"Uncaught exception in file watcher thread", throwable);
        }

        void register(Registration registration) throws IOException {
            for (Path path : registration.paths()) {
                if (!Files.isRegularFile(path, new LinkOption[0]) && !Files.isDirectory(path, new LinkOption[0])) {
                    throw new IOException("'%s' is neither a file nor a directory".formatted(path));
                }
                Path directory = Files.isDirectory(path, new LinkOption[0]) ? path : path.getParent();
                WatchKey watchKey = this.register(directory);
                this.registrations.computeIfAbsent(watchKey, key -> new CopyOnWriteArrayList()).add(registration);
            }
        }

        private WatchKey register(Path directory) throws IOException {
            logger.debug((Object)LogMessage.format((String)"Registering '%s'", (Object)directory));
            return directory.register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_MODIFY, StandardWatchEventKinds.ENTRY_DELETE);
        }

        @Override
        public void run() {
            logger.debug((Object)"Watch thread started");
            HashSet<Runnable> actions = new HashSet<Runnable>();
            while (this.running) {
                try {
                    long timeout = FileWatcher.this.quietPeriod.toMillis();
                    WatchKey key = this.watchService.poll(timeout, TimeUnit.MILLISECONDS);
                    if (key == null) {
                        actions.forEach(this::runSafely);
                        actions.clear();
                        continue;
                    }
                    this.accumulate(key, actions);
                    key.reset();
                }
                catch (InterruptedException ex) {
                    Thread.currentThread().interrupt();
                }
                catch (ClosedWatchServiceException ex) {
                    logger.debug((Object)"File watcher has been closed");
                    this.running = false;
                }
            }
            logger.debug((Object)"Watch thread stopped");
        }

        private void runSafely(Runnable action) {
            try {
                action.run();
            }
            catch (Throwable ex) {
                logger.error((Object)"Unexpected SSL reload error", ex);
            }
        }

        private void accumulate(WatchKey key, Set<Runnable> actions) {
            List<Registration> registrations = this.registrations.get(key);
            Path directory = (Path)key.watchable();
            for (WatchEvent<?> event : key.pollEvents()) {
                Path file = directory.resolve((Path)event.context());
                for (Registration registration : registrations) {
                    if (!registration.manages(file)) continue;
                    actions.add(registration.action());
                }
            }
        }

        @Override
        public void close() throws IOException {
            this.running = false;
            this.watchService.close();
        }
    }

    private record Registration(Set<Path> paths, Runnable action) {
        Registration {
            paths = paths.stream().map(Path::toAbsolutePath).collect(Collectors.toSet());
        }

        boolean manages(Path file) {
            Path absolutePath = file.toAbsolutePath();
            return this.paths.contains(absolutePath) || this.isInDirectories(absolutePath);
        }

        private boolean isInDirectories(Path file) {
            return this.paths.stream().filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).anyMatch(file::startsWith);
        }
    }
}

