/*
 * Decompiled with CFR 0.152.
 */
package alluxio.client.file.cache.store;

import alluxio.client.file.cache.PageId;
import alluxio.client.file.cache.PageStore;
import alluxio.client.file.cache.store.PageStoreDir;
import alluxio.client.file.cache.store.PageStoreOptions;
import alluxio.exception.PageCorruptedException;
import alluxio.exception.PageNotFoundException;
import alluxio.exception.status.ResourceExhaustedException;
import alluxio.file.ReadTargetBuffer;
import alluxio.network.protocol.databuffer.DataFileChannel;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import javax.annotation.concurrent.NotThreadSafe;
import org.apache.commons.io.FileUtils;

@NotThreadSafe
public class LocalPageStore
implements PageStore {
    private static final String ERROR_NO_SPACE_LEFT = "No space left on device";
    public static final String TEMP_DIR = "TEMP";
    private final Path mRoot;
    private final long mPageSize;
    private final long mCapacity;
    private final int mFileBuckets;

    public LocalPageStore(PageStoreOptions options) {
        this.mRoot = options.getRootDir();
        this.mPageSize = options.getPageSize();
        this.mCapacity = (long)((double)options.getCacheSize() / (1.0 + options.getOverheadRatio()));
        this.mFileBuckets = options.getFileBuckets();
    }

    @Override
    public void put(PageId pageId, ByteBuffer page, boolean isTemporary) throws ResourceExhaustedException, IOException {
        Path pagePath = this.getPagePath(pageId, isTemporary);
        try {
            LOG.debug("Put page: {}, page's position: {}, page's limit: {}, page's capacity: {}", new Object[]{pageId, page.position(), page.limit(), page.capacity()});
            if (!Files.exists(pagePath, new LinkOption[0])) {
                Path parent = (Path)Preconditions.checkNotNull((Object)pagePath.getParent(), (Object)"parent of cache file should not be null");
                Files.createDirectories(parent, new FileAttribute[0]);
            }
            try (FileOutputStream fos = new FileOutputStream(pagePath.toFile(), false);){
                fos.getChannel().write(page);
            }
        }
        catch (Throwable t) {
            Files.deleteIfExists(pagePath);
            if (t.getMessage() != null && t.getMessage().contains(ERROR_NO_SPACE_LEFT)) {
                throw new ResourceExhaustedException(String.format("%s is full, configured with %d bytes", this.mRoot, this.mCapacity), t);
            }
            throw new IOException("Failed to write file " + pagePath + " for page " + pageId, t);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public int get(PageId pageId, int pageOffset, int bytesToRead, ReadTargetBuffer target, boolean isTemporary) throws IOException, PageNotFoundException {
        Preconditions.checkArgument((pageOffset >= 0 ? 1 : 0) != 0, (Object)"page offset should be non-negative");
        Preconditions.checkArgument((bytesToRead >= 0 ? 1 : 0) != 0, (Object)"bytes to read should be non-negative");
        if (target.remaining() == 0L) return 0;
        if (bytesToRead == 0) {
            return 0;
        }
        Path pagePath = this.getPagePath(pageId, isTemporary);
        try (RandomAccessFile localFile = new RandomAccessFile(pagePath.toString(), "r");){
            int n;
            int bytes;
            long pageLength = localFile.length();
            if ((long)(pageOffset + bytesToRead) > pageLength) {
                throw new PageCorruptedException(String.format("The page %s (%s) probably has been corrupted, page-offset %s, bytes to read %s, page file length %s", pageId, pagePath, pageOffset, bytesToRead, pageLength));
            }
            int bytesSkipped = localFile.skipBytes(pageOffset);
            if (pageOffset != bytesSkipped) {
                throw new IOException(String.format("Failed to read page %s (%s) from offset %s: %s bytes skipped", pageId, pagePath, pageOffset, bytesSkipped));
            }
            int bytesRead = 0;
            for (int bytesLeft = Math.min((int)target.remaining(), bytesToRead); bytesLeft > 0 && (bytes = target.readFromFile(localFile, bytesLeft)) > 0; bytesRead += bytes, bytesLeft -= bytes) {
            }
            if (bytesRead == 0) {
                SAMPLING_LOG.warn("Read 0 bytes from page {}, the page is probably empty", (Object)pageId);
                n = -1;
                return n;
            }
            n = bytesRead;
            return n;
        }
        catch (FileNotFoundException e) {
            throw new PageNotFoundException(pagePath.toString());
        }
    }

    @Override
    public void delete(PageId pageId, boolean isTemporary) throws IOException, PageNotFoundException {
        block16: {
            Path pagePath = this.getPagePath(pageId, isTemporary);
            if (!Files.exists(pagePath, new LinkOption[0])) {
                throw new PageNotFoundException(pagePath.toString());
            }
            Files.delete(pagePath);
            Path parent = (Path)Preconditions.checkNotNull((Object)pagePath.getParent(), (Object)"parent of cache file should not be null");
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(parent);){
                if (!stream.iterator().hasNext()) {
                    Files.delete(parent);
                }
            }
            catch (NoSuchFileException e) {
                if (!LOG.isDebugEnabled()) break block16;
                LOG.debug("Parent path is deleted by other thread, ignore and continue.", (Throwable)e);
            }
        }
    }

    @Override
    public void commit(String fileId, String newFileId) throws IOException {
        Path filePath = this.getFilePath(newFileId);
        Path bucketPath = (Path)Preconditions.checkNotNull((Object)filePath.getParent(), (String)"%s does not have a parent path", (Object)filePath);
        if (!Files.exists(bucketPath, new LinkOption[0])) {
            Files.createDirectories(bucketPath, new FileAttribute[0]);
        }
        Files.move(this.getTempFilePath(fileId), filePath, StandardCopyOption.ATOMIC_MOVE);
    }

    @Override
    public void abort(String fileId) throws IOException {
        FileUtils.deleteDirectory((File)this.getTempFilePath(fileId).toFile());
    }

    private Path getTempFilePath(String fileId) {
        return Paths.get(this.mRoot.toString(), Long.toString(this.mPageSize), TEMP_DIR, fileId);
    }

    private Path getFilePath(String fileId) {
        return Paths.get(this.mRoot.toString(), Long.toString(this.mPageSize), PageStoreDir.getFileBucket(this.mFileBuckets, fileId), fileId);
    }

    @VisibleForTesting
    public Path getPagePath(PageId pageId, boolean isTemporary) {
        Path filePath = isTemporary ? this.getTempFilePath(pageId.getFileId()) : this.getFilePath(pageId.getFileId());
        return filePath.resolve(Long.toString(pageId.getPageIndex()));
    }

    @Override
    public DataFileChannel getDataFileChannel(PageId pageId, int pageOffset, int bytesToRead, boolean isTemporary) throws PageNotFoundException {
        Preconditions.checkArgument((pageOffset >= 0 ? 1 : 0) != 0, (Object)"page offset should be non-negative");
        Preconditions.checkArgument((!isTemporary ? 1 : 0) != 0, (Object)"cannot acquire a data file channel to a temporary page");
        Path pagePath = this.getPagePath(pageId, isTemporary);
        File pageFile = pagePath.toFile();
        if (!pageFile.exists()) {
            throw new PageNotFoundException(pagePath.toString());
        }
        long fileLength = pageFile.length();
        if (fileLength == 0L && pageId.getPageIndex() > 0L) {
            SAMPLING_LOG.warn("Length of page {} is 0, removing this malformed page", (Object)pageId);
            try {
                Files.deleteIfExists(pagePath);
            }
            catch (IOException iOException) {
                // empty catch block
            }
            throw new PageNotFoundException(pagePath.toString());
        }
        if (fileLength < (long)pageOffset) {
            throw new IllegalArgumentException(String.format("offset %s exceeds length of page %s", pageOffset, fileLength));
        }
        if ((long)(pageOffset + bytesToRead) > fileLength) {
            bytesToRead = (int)(fileLength - (long)pageOffset);
        }
        DataFileChannel dataFileChannel = new DataFileChannel(pageFile, (long)pageOffset, (long)bytesToRead);
        return dataFileChannel;
    }

    @Override
    public void close() {
    }
}

