/*
 * Decompiled with CFR 0.152.
 */
package alluxio.client.file.cache.store;

import alluxio.client.file.cache.PageId;
import alluxio.client.file.cache.PageStore;
import alluxio.client.file.cache.store.PageStoreDir;
import alluxio.client.file.cache.store.PageStoreOptions;
import alluxio.exception.PageNotFoundException;
import alluxio.file.ReadTargetBuffer;
import alluxio.proto.client.Cache;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.protobuf.InvalidProtocolBufferException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import javax.annotation.Nullable;
import javax.annotation.concurrent.NotThreadSafe;
import org.rocksdb.ColumnFamilyDescriptor;
import org.rocksdb.ColumnFamilyHandle;
import org.rocksdb.ColumnFamilyOptions;
import org.rocksdb.CompressionType;
import org.rocksdb.DBOptions;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.RocksIterator;
import org.rocksdb.WriteOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class RocksPageStore
implements PageStore {
    private static final Logger LOG = LoggerFactory.getLogger(RocksPageStore.class);
    private static final String PAGE_COLUMN = "PAGE";
    private static final byte[] CONF_KEY = "CONF".getBytes();
    private static final int DEFAULT_COLUMN_INDEX = 0;
    private static final int PAGE_COLUMN_INDEX = 1;
    private static final long WRITE_BUFFER_SIZE = 0x4000000L;
    private final long mCapacity;
    private final RocksDB mDb;
    private final ColumnFamilyHandle mDefaultColumnHandle;
    private final ColumnFamilyHandle mPageColumnHandle;
    private final DBOptions mRocksOptions;
    private final WriteOptions mWriteOptions = new WriteOptions();

    public static RocksPageStore open(PageStoreOptions pageStoreOptions) {
        RocksDB.loadLibrary();
        DBOptions rocksOptions = RocksPageStore.createDbOptions();
        RocksDB db = null;
        ArrayList<Object> columnHandles = new ArrayList<ColumnFamilyHandle>();
        try {
            db = RocksPageStore.openDB(pageStoreOptions, rocksOptions, columnHandles);
        }
        catch (RocksDBException e) {
            try {
                PageStoreDir.clear(pageStoreOptions.getRootDir());
                rocksOptions = RocksPageStore.createDbOptions();
                columnHandles = new ArrayList();
                db = RocksPageStore.openDB(pageStoreOptions, rocksOptions, columnHandles);
            }
            catch (IOException | RocksDBException ex) {
                throw new RuntimeException("Couldn't open rocksDB database", e);
            }
        }
        catch (InvalidProtocolBufferException e) {
            throw new RuntimeException("Couldn't open rocksDB database", e);
        }
        return new RocksPageStore(pageStoreOptions, rocksOptions, db, (ColumnFamilyHandle)columnHandles.get(0), (ColumnFamilyHandle)columnHandles.get(1));
    }

    private static DBOptions createDbOptions() {
        DBOptions rocksOptions = new DBOptions().setCreateIfMissing(true).setCreateMissingColumnFamilies(true);
        return rocksOptions;
    }

    private static RocksDB openDB(PageStoreOptions pageStoreOptions, DBOptions rocksOptions, List<ColumnFamilyHandle> columnHandles) throws RocksDBException, InvalidProtocolBufferException {
        Cache.PRocksPageStoreOptions persistedOptions;
        ImmutableList columnDescriptors = ImmutableList.of((Object)new ColumnFamilyDescriptor(RocksDB.DEFAULT_COLUMN_FAMILY), (Object)new ColumnFamilyDescriptor(PAGE_COLUMN.getBytes(), new ColumnFamilyOptions().setWriteBufferSize(0x4000000L).setCompressionType(CompressionType.NO_COMPRESSION)));
        RocksDB db = RocksDB.open((DBOptions)rocksOptions, (String)pageStoreOptions.getRootDir().toString(), (List)columnDescriptors, columnHandles);
        byte[] confData = db.get(columnHandles.get(0), CONF_KEY);
        Cache.PRocksPageStoreOptions pOptions = RocksPageStore.toProto(pageStoreOptions.getPageSize(), pageStoreOptions.getCacheSize(), pageStoreOptions.getAlluxioVersion());
        if (confData != null && !(persistedOptions = Cache.PRocksPageStoreOptions.parseFrom((byte[])confData)).equals((Object)pOptions)) {
            db.close();
            rocksOptions.close();
            throw new RocksDBException("Inconsistent configuration for RocksPageStore");
        }
        db.put(columnHandles.get(0), CONF_KEY, pOptions.toByteArray());
        return db;
    }

    private static Cache.PRocksPageStoreOptions toProto(long pageSize, long cacheSize, String alluxioVersion) {
        return Cache.PRocksPageStoreOptions.newBuilder().setCommonOptions(Cache.PPageStoreCommonOptions.newBuilder().setPageSize(pageSize).setCacheSize(cacheSize).setAlluxioVersion(alluxioVersion)).build();
    }

    private RocksPageStore(PageStoreOptions pageStoreOptions, DBOptions rocksOptions, RocksDB rocksDB, ColumnFamilyHandle defaultColumnHandle, ColumnFamilyHandle pageColumnHandle) {
        this.mCapacity = (long)((double)pageStoreOptions.getCacheSize() / (1.0 + pageStoreOptions.getOverheadRatio()));
        this.mRocksOptions = rocksOptions;
        this.mDb = rocksDB;
        this.mDefaultColumnHandle = defaultColumnHandle;
        this.mPageColumnHandle = pageColumnHandle;
    }

    @Override
    public void put(PageId pageId, ByteBuffer page, boolean isTemporary) throws IOException {
        try {
            ByteBuffer key = RocksPageStore.getKeyFromPageId(pageId, page.isDirect());
            if (page.isDirect()) {
                this.mDb.put(this.mPageColumnHandle, this.mWriteOptions, key, page);
            } else {
                this.mDb.put(this.mPageColumnHandle, this.mWriteOptions, key.array(), page.array());
            }
        }
        catch (RocksDBException e) {
            throw new IOException("Failed to store page", e);
        }
    }

    @Override
    public int get(PageId pageId, int pageOffset, int bytesToRead, ReadTargetBuffer target, boolean isTemporary) throws IOException, PageNotFoundException {
        Preconditions.checkArgument((pageOffset >= 0 ? 1 : 0) != 0, (Object)"page offset should be non-negative");
        try {
            byte[] key = RocksPageStore.getKeyFromPageId(pageId, false).array();
            byte[] page = this.mDb.get(this.mPageColumnHandle, key);
            if (page == null) {
                throw new PageNotFoundException(new String(key));
            }
            Preconditions.checkArgument((pageOffset <= page.length ? 1 : 0) != 0, (String)"page offset %s exceeded page size %s", (int)pageOffset, (int)page.length);
            int bytesLeft = Math.min(page.length - pageOffset, Math.min((int)target.remaining(), bytesToRead));
            System.arraycopy(page, pageOffset, target.byteArray(), target.offset(), bytesLeft);
            return bytesLeft;
        }
        catch (RocksDBException e) {
            throw new IOException("Failed to retrieve page", e);
        }
    }

    @Override
    public void delete(PageId pageId) throws PageNotFoundException {
        try {
            byte[] key = RocksPageStore.getKeyFromPageId(pageId, false).array();
            this.mDb.delete(this.mPageColumnHandle, key);
        }
        catch (RocksDBException e) {
            throw new PageNotFoundException("Failed to remove page", (Throwable)e);
        }
    }

    @Override
    public void close() {
        LOG.info("Closing RocksPageStore and recycling all RocksDB JNI objects");
        this.mDb.close();
        this.mRocksOptions.close();
        this.mDefaultColumnHandle.close();
        this.mPageColumnHandle.close();
        LOG.info("RocksPageStore closed");
    }

    static ByteBuffer getKeyFromPageId(PageId pageId, boolean isDirect) {
        byte[] fileId = pageId.getFileId().getBytes();
        ByteBuffer buf = isDirect ? ByteBuffer.allocateDirect(8 + fileId.length) : ByteBuffer.allocate(8 + fileId.length);
        buf.putLong(pageId.getPageIndex());
        buf.put(fileId);
        return buf;
    }

    @Nullable
    static PageId getPageIdFromKey(byte[] key) {
        if (key.length < 8) {
            return null;
        }
        ByteBuffer buf = ByteBuffer.wrap(key);
        long pageIndex = buf.getLong();
        String fileId = Charset.defaultCharset().decode(buf).toString();
        return new PageId(fileId, pageIndex);
    }

    public RocksIterator createNewInterator() {
        return this.mDb.newIterator(this.mPageColumnHandle);
    }
}

