/*
 * Decompiled with CFR 0.152.
 */
package com.azure.storage.common.policy;

import com.azure.core.http.HttpMethod;
import com.azure.core.http.HttpPipelineCallContext;
import com.azure.core.http.HttpPipelineNextPolicy;
import com.azure.core.http.HttpPipelineNextSyncPolicy;
import com.azure.core.http.HttpRequest;
import com.azure.core.http.HttpResponse;
import com.azure.core.http.policy.HttpPipelinePolicy;
import com.azure.core.util.BinaryData;
import com.azure.core.util.Context;
import com.azure.core.util.Contexts;
import com.azure.core.util.ProgressReporter;
import com.azure.core.util.UrlBuilder;
import com.azure.core.util.logging.ClientLogger;
import com.azure.storage.common.policy.RequestRetryOptions;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeoutException;
import reactor.core.Exceptions;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public final class RequestRetryPolicy
implements HttpPipelinePolicy {
    private static final ClientLogger LOGGER = new ClientLogger(RequestRetryPolicy.class);
    private final RequestRetryOptions requestRetryOptions;

    public RequestRetryPolicy(RequestRetryOptions requestRetryOptions) {
        this.requestRetryOptions = requestRetryOptions;
    }

    public HttpResponse processSync(HttpPipelineCallContext context, HttpPipelineNextSyncPolicy next) {
        boolean considerSecondary = this.requestRetryOptions.getSecondaryHost() != null && (HttpMethod.GET.equals((Object)context.getHttpRequest().getHttpMethod()) || HttpMethod.HEAD.equals((Object)context.getHttpRequest().getHttpMethod()));
        HttpRequest originalHttpRequest = context.getHttpRequest();
        BinaryData originalRequestBody = originalHttpRequest.getBodyAsBinaryData();
        if (this.requestRetryOptions.getMaxTries() > 1 && originalRequestBody != null && !originalRequestBody.isReplayable()) {
            context.getHttpRequest().setBody(context.getHttpRequest().getBodyAsBinaryData().toReplayableBinaryData());
        }
        return this.attemptSync(context, next, originalHttpRequest, considerSecondary, 1, 1, null);
    }

    public Mono<HttpResponse> process(HttpPipelineCallContext context, HttpPipelineNextPolicy next) {
        boolean considerSecondary = this.requestRetryOptions.getSecondaryHost() != null && (HttpMethod.GET.equals((Object)context.getHttpRequest().getHttpMethod()) || HttpMethod.HEAD.equals((Object)context.getHttpRequest().getHttpMethod()));
        HttpRequest originalHttpRequest = context.getHttpRequest();
        BinaryData originalRequestBody = originalHttpRequest.getBodyAsBinaryData();
        if (this.requestRetryOptions.getMaxTries() > 1 && originalRequestBody != null && !originalRequestBody.isReplayable()) {
            Flux bufferedBody = context.getHttpRequest().getBody().map(ByteBuffer::duplicate);
            context.getHttpRequest().setBody(bufferedBody);
        }
        return this.attemptAsync(context, next, context.getHttpRequest(), considerSecondary, 1, 1, null);
    }

    private Mono<HttpResponse> attemptAsync(HttpPipelineCallContext context, HttpPipelineNextPolicy next, HttpRequest originalRequest, boolean considerSecondary, int primaryTry, int attempt, List<Throwable> suppressed) {
        boolean tryingPrimary = !considerSecondary || attempt % 2 != 0;
        long delayMs = this.getDelayMs(primaryTry, tryingPrimary);
        context.setHttpRequest(originalRequest.copy());
        try {
            RequestRetryPolicy.updateUrlToSecondaryHost(tryingPrimary, this.requestRetryOptions.getSecondaryHost(), context);
        }
        catch (IllegalArgumentException e) {
            return Mono.error((Throwable)e);
        }
        RequestRetryPolicy.updateRetryCountContext(context, attempt);
        RequestRetryPolicy.resetProgress(context);
        Mono responseMono = next.clone().process();
        if (this.requestRetryOptions.getTryTimeoutDuration().getSeconds() != Integer.MAX_VALUE) {
            responseMono = responseMono.timeout(this.requestRetryOptions.getTryTimeoutDuration());
        }
        if (delayMs > 0L) {
            responseMono = responseMono.delaySubscription(Duration.ofMillis(delayMs));
        }
        return responseMono.flatMap(response -> {
            boolean newConsiderSecondary = considerSecondary;
            int statusCode = response.getStatusCode();
            boolean retry = RequestRetryPolicy.shouldStatusCodeBeRetried(statusCode, tryingPrimary);
            if (!tryingPrimary && statusCode == 404) {
                newConsiderSecondary = false;
            }
            if (retry && attempt < this.requestRetryOptions.getMaxTries()) {
                int newPrimaryTry = RequestRetryPolicy.getNewPrimaryTry(considerSecondary, primaryTry, tryingPrimary);
                Flux responseBody = response.getBody();
                response.close();
                if (responseBody == null) {
                    return this.attemptAsync(context, next, originalRequest, newConsiderSecondary, newPrimaryTry, attempt + 1, suppressed);
                }
                return responseBody.ignoreElements().then(this.attemptAsync(context, next, originalRequest, newConsiderSecondary, newPrimaryTry, attempt + 1, suppressed));
            }
            return Mono.just((Object)response);
        }).onErrorResume(throwable -> {
            if (throwable instanceof IllegalStateException && attempt > 1) {
                return Mono.error((Throwable)new IllegalStateException("The request failed because the size of the contents of the provided Flux did not match the provided data size upon attempting to retry. This is likely caused by the Flux not being replayable. To support retries, all Fluxes must produce the same data for each subscriber. Please ensure this behavior.", (Throwable)throwable));
            }
            ExceptionRetryStatus exceptionRetryStatus = RequestRetryPolicy.shouldErrorBeRetried(throwable, attempt, this.requestRetryOptions.getMaxTries());
            if (exceptionRetryStatus.canBeRetried) {
                int newPrimaryTry = RequestRetryPolicy.getNewPrimaryTry(considerSecondary, primaryTry, tryingPrimary);
                List suppressedLocal = suppressed == null ? new LinkedList() : suppressed;
                suppressedLocal.add(exceptionRetryStatus.unwrappedThrowable);
                return this.attemptAsync(context, next, originalRequest, considerSecondary, newPrimaryTry, attempt + 1, suppressedLocal);
            }
            if (suppressed != null) {
                suppressed.forEach(throwable::addSuppressed);
            }
            return Mono.error((Throwable)throwable);
        });
    }

    private HttpResponse attemptSync(HttpPipelineCallContext context, HttpPipelineNextSyncPolicy next, HttpRequest originalRequest, boolean considerSecondary, int primaryTry, int attempt, List<Throwable> suppressed) {
        boolean tryingPrimary = !considerSecondary || attempt % 2 != 0;
        long delayMs = this.getDelayMs(primaryTry, tryingPrimary);
        context.setHttpRequest(originalRequest.copy());
        RequestRetryPolicy.updateUrlToSecondaryHost(tryingPrimary, this.requestRetryOptions.getSecondaryHost(), context);
        RequestRetryPolicy.updateRetryCountContext(context, attempt);
        RequestRetryPolicy.resetProgress(context);
        try {
            if (delayMs > 0L) {
                try {
                    Thread.sleep(delayMs);
                }
                catch (InterruptedException ie) {
                    throw LOGGER.logExceptionAsError(new RuntimeException(ie));
                }
            }
            Mono httpResponseMono = Mono.fromCallable(() -> next.clone().processSync());
            if (this.requestRetryOptions.getTryTimeoutDuration().getSeconds() != Integer.MAX_VALUE) {
                httpResponseMono = httpResponseMono.timeout(this.requestRetryOptions.getTryTimeoutDuration());
            }
            HttpResponse response = (HttpResponse)httpResponseMono.block();
            boolean newConsiderSecondary = considerSecondary;
            int statusCode = response.getStatusCode();
            boolean retry = RequestRetryPolicy.shouldStatusCodeBeRetried(statusCode, tryingPrimary);
            if (!tryingPrimary && statusCode == 404) {
                newConsiderSecondary = false;
            }
            if (retry && attempt < this.requestRetryOptions.getMaxTries()) {
                int newPrimaryTry = RequestRetryPolicy.getNewPrimaryTry(considerSecondary, primaryTry, tryingPrimary);
                if (response.getBody() != null) {
                    response.getBodyAsBinaryData().toByteBuffer();
                }
                response.close();
                return this.attemptSync(context, next, originalRequest, newConsiderSecondary, newPrimaryTry, attempt + 1, suppressed);
            }
            return response;
        }
        catch (RuntimeException throwable) {
            if (throwable instanceof IllegalStateException && attempt > 1) {
                throw LOGGER.logExceptionAsError((RuntimeException)new IllegalStateException("The request failed because the size of the contents of the provided data did not match the provided data size upon attempting to retry. This is likely caused by the data not being replayable. To support retries, all Fluxes must produce the same data for each subscriber. Please ensure this behavior.", throwable));
            }
            ExceptionRetryStatus exceptionRetryStatus = RequestRetryPolicy.shouldErrorBeRetried(throwable, attempt, this.requestRetryOptions.getMaxTries());
            if (exceptionRetryStatus.canBeRetried) {
                int newPrimaryTry = RequestRetryPolicy.getNewPrimaryTry(considerSecondary, primaryTry, tryingPrimary);
                LinkedList<Throwable> suppressedLocal = suppressed == null ? new LinkedList<Throwable>() : suppressed;
                suppressedLocal.add(exceptionRetryStatus.unwrappedThrowable);
                return this.attemptSync(context, next, originalRequest, considerSecondary, newPrimaryTry, attempt + 1, suppressedLocal);
            }
            if (suppressed != null) {
                suppressed.forEach(throwable::addSuppressed);
            }
            throw LOGGER.logExceptionAsError(throwable);
        }
    }

    private static void updateRetryCountContext(HttpPipelineCallContext context, int attempt) {
        context.setData("requestRetryCount", (Object)attempt);
    }

    private static void resetProgress(HttpPipelineCallContext context) {
        ProgressReporter progressReporter = Contexts.with((Context)context.getContext()).getHttpRequestProgressReporter();
        if (progressReporter != null) {
            progressReporter.reset();
        }
    }

    private static void updateUrlToSecondaryHost(boolean tryingPrimary, String secondaryHost, HttpPipelineCallContext context) {
        if (!tryingPrimary) {
            UrlBuilder builder = UrlBuilder.parse((URL)context.getHttpRequest().getUrl());
            builder.setHost(secondaryHost);
            try {
                context.getHttpRequest().setUrl(builder.toUrl());
            }
            catch (MalformedURLException e) {
                throw LOGGER.logExceptionAsWarning((RuntimeException)new IllegalArgumentException("'url' must be a valid URL", e));
            }
        }
    }

    static ExceptionRetryStatus shouldErrorBeRetried(Throwable error, int attempt, int maxAttempts) {
        Throwable unwrappedThrowable = Exceptions.unwrap((Throwable)error);
        if (attempt >= maxAttempts) {
            return new ExceptionRetryStatus(false, unwrappedThrowable);
        }
        if (unwrappedThrowable instanceof IOException || unwrappedThrowable instanceof TimeoutException) {
            return new ExceptionRetryStatus(true, unwrappedThrowable);
        }
        for (Throwable causalException = unwrappedThrowable.getCause(); causalException != null; causalException = causalException.getCause()) {
            if (!(causalException instanceof IOException) && !(causalException instanceof TimeoutException)) continue;
            return new ExceptionRetryStatus(true, unwrappedThrowable);
        }
        return new ExceptionRetryStatus(false, unwrappedThrowable);
    }

    static boolean shouldStatusCodeBeRetried(int statusCode, boolean isPrimary) {
        return statusCode == 429 || statusCode == 500 || statusCode == 503 || !isPrimary && statusCode == 404;
    }

    private long getDelayMs(int primaryTry, boolean tryingPrimary) {
        long delayMs = tryingPrimary ? this.requestRetryOptions.calculateDelayInMs(primaryTry) : (long)(((double)(ThreadLocalRandom.current().nextFloat() / 2.0f) + 0.8) * 1000.0);
        return delayMs;
    }

    private static int getNewPrimaryTry(boolean considerSecondary, int primaryTry, boolean tryingPrimary) {
        return !tryingPrimary || !considerSecondary ? primaryTry + 1 : primaryTry;
    }

    static final class ExceptionRetryStatus {
        final boolean canBeRetried;
        final Throwable unwrappedThrowable;

        ExceptionRetryStatus(boolean canBeRetried, Throwable unwrappedThrowable) {
            this.canBeRetried = canBeRetried;
            this.unwrappedThrowable = unwrappedThrowable;
        }
    }
}

