/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.tieredstore.container;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.RemovalCause;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.rocketmq.common.message.MessageQueue;
import org.apache.rocketmq.logging.org.slf4j.Logger;
import org.apache.rocketmq.logging.org.slf4j.LoggerFactory;
import org.apache.rocketmq.store.DispatchRequest;
import org.apache.rocketmq.tieredstore.common.AppendResult;
import org.apache.rocketmq.tieredstore.common.BoundaryType;
import org.apache.rocketmq.tieredstore.common.InflightRequestFuture;
import org.apache.rocketmq.tieredstore.common.InflightRequestKey;
import org.apache.rocketmq.tieredstore.common.TieredMessageStoreConfig;
import org.apache.rocketmq.tieredstore.container.TieredCommitLog;
import org.apache.rocketmq.tieredstore.container.TieredConsumeQueue;
import org.apache.rocketmq.tieredstore.container.TieredContainerManager;
import org.apache.rocketmq.tieredstore.container.TieredIndexFile;
import org.apache.rocketmq.tieredstore.metadata.QueueMetadata;
import org.apache.rocketmq.tieredstore.metadata.TieredMetadataStore;
import org.apache.rocketmq.tieredstore.metadata.TopicMetadata;
import org.apache.rocketmq.tieredstore.util.CQItemBufferUtil;
import org.apache.rocketmq.tieredstore.util.MessageBufferUtil;
import org.apache.rocketmq.tieredstore.util.TieredStoreUtil;

public class TieredMessageQueueContainer {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)"RocketmqTieredStore");
    private volatile boolean closed = false;
    private final MessageQueue messageQueue;
    private final int topicId;
    private final TieredMessageStoreConfig storeConfig;
    private final TieredMetadataStore metadataStore;
    private final TieredCommitLog commitLog;
    private final TieredConsumeQueue consumeQueue;
    private final TieredIndexFile indexFile;
    private QueueMetadata queueMetadata;
    private long dispatchOffset;
    private final ReentrantLock queueLock = new ReentrantLock();
    private int readAheadFactor;
    private final Cache<String, Long> groupOffsetCache;
    private final ConcurrentMap<InflightRequestKey, InflightRequestFuture> inFlightRequestMap;

    public TieredMessageQueueContainer(MessageQueue messageQueue, TieredMessageStoreConfig storeConfig) throws ClassNotFoundException, NoSuchMethodException {
        this.messageQueue = messageQueue;
        this.storeConfig = storeConfig;
        this.metadataStore = TieredStoreUtil.getMetadataStore(storeConfig);
        TopicMetadata topicMetadata = this.metadataStore.getTopic(messageQueue.getTopic());
        if (topicMetadata == null) {
            topicMetadata = this.metadataStore.addTopic(messageQueue.getTopic(), -1L);
        }
        this.topicId = topicMetadata.getTopicId();
        this.queueMetadata = this.metadataStore.getQueue(messageQueue);
        if (this.queueMetadata == null) {
            this.queueMetadata = this.metadataStore.addQueue(messageQueue, -1L);
        }
        if (this.queueMetadata.getMaxOffset() < this.queueMetadata.getMinOffset()) {
            this.queueMetadata.setMaxOffset(this.queueMetadata.getMinOffset());
        }
        this.dispatchOffset = this.queueMetadata.getMaxOffset();
        this.commitLog = new TieredCommitLog(messageQueue, storeConfig);
        this.consumeQueue = new TieredConsumeQueue(messageQueue, storeConfig);
        if (!this.consumeQueue.isInitialized() && this.dispatchOffset != -1L) {
            this.consumeQueue.setBaseOffset(this.dispatchOffset * 20L);
        }
        this.indexFile = TieredContainerManager.getIndexFile(storeConfig);
        this.readAheadFactor = storeConfig.getReadAheadMinFactor();
        this.inFlightRequestMap = new ConcurrentHashMap<InflightRequestKey, InflightRequestFuture>();
        this.groupOffsetCache = Caffeine.newBuilder().expireAfterWrite(2L, TimeUnit.MINUTES).removalListener((key, value, cause) -> {
            if (cause.equals((Object)RemovalCause.EXPIRED)) {
                this.inFlightRequestMap.remove(new InflightRequestKey((String)key));
            }
        }).build();
    }

    public boolean isClosed() {
        return this.closed;
    }

    public ReentrantLock getQueueLock() {
        return this.queueLock;
    }

    public MessageQueue getMessageQueue() {
        return this.messageQueue;
    }

    public long getCommitLogMinOffset() {
        return this.commitLog.getMinOffset();
    }

    public long getCommitLogMaxOffset() {
        return this.commitLog.getMaxOffset();
    }

    public long getCommitLogBeginTimestamp() {
        return this.commitLog.getBeginTimestamp();
    }

    public long getConsumeQueueBaseOffset() {
        return this.consumeQueue.getBaseOffset();
    }

    public long getConsumeQueueMinOffset() {
        return this.consumeQueue.getMinOffset() / 20L;
    }

    public long getConsumeQueueCommitOffset() {
        return this.consumeQueue.getCommitOffset() / 20L;
    }

    public long getConsumeQueueMaxOffset() {
        return this.consumeQueue.getMaxOffset() / 20L;
    }

    public long getConsumeQueueEndTimestamp() {
        return this.consumeQueue.getEndTimestamp();
    }

    public long getDispatchOffset() {
        return this.dispatchOffset;
    }

    public CompletableFuture<ByteBuffer> getMessageAsync(long queueOffset) {
        return this.readConsumeQueue(queueOffset).thenComposeAsync(cqBuffer -> {
            long commitLogOffset = CQItemBufferUtil.getCommitLogOffset(cqBuffer);
            int length = CQItemBufferUtil.getSize(cqBuffer);
            return this.readCommitLog(commitLogOffset, length);
        });
    }

    public long binarySearchInQueueByTime(long timestamp, BoundaryType boundaryType) {
        Pair<Long, Long> pair = this.consumeQueue.getQueueOffsetInFileByTime(timestamp, boundaryType);
        long minQueueOffset = (Long)pair.getLeft();
        long maxQueueOffset = (Long)pair.getRight();
        if (maxQueueOffset == -1L || maxQueueOffset < minQueueOffset) {
            return -1L;
        }
        long low = minQueueOffset;
        long high = maxQueueOffset;
        long offset = 0L;
        ByteBuffer message = this.getMessageAsync(maxQueueOffset).join();
        long storeTime = MessageBufferUtil.getStoreTimeStamp(message);
        if (storeTime < timestamp) {
            switch (boundaryType) {
                case LOWER: {
                    return maxQueueOffset + 1L;
                }
                case UPPER: {
                    return maxQueueOffset;
                }
            }
            LOGGER.warn("TieredMessageQueueContainer#getQueueOffsetByTime: unknown boundary boundaryType");
        }
        if ((storeTime = MessageBufferUtil.getStoreTimeStamp(message = this.getMessageAsync(minQueueOffset).join())) > timestamp) {
            switch (boundaryType) {
                case LOWER: {
                    return minQueueOffset;
                }
                case UPPER: {
                    return 0L;
                }
            }
            LOGGER.warn("TieredMessageQueueContainer#getQueueOffsetByTime: unknown boundary boundaryType");
        }
        long midOffset = -1L;
        long targetOffset = -1L;
        long leftOffset = -1L;
        long rightOffset = -1L;
        while (high >= low) {
            midOffset = (low + high) / 2L;
            message = this.getMessageAsync(midOffset).join();
            storeTime = MessageBufferUtil.getStoreTimeStamp(message);
            if (storeTime == timestamp) {
                targetOffset = midOffset;
                break;
            }
            if (storeTime > timestamp) {
                high = midOffset - 1L;
                rightOffset = midOffset;
                continue;
            }
            low = midOffset + 1L;
            leftOffset = midOffset;
        }
        if (targetOffset != -1L) {
            offset = targetOffset;
            long previousAttempt = targetOffset;
            switch (boundaryType) {
                case LOWER: {
                    long attempt;
                    while ((attempt = previousAttempt - 1L) >= minQueueOffset && (storeTime = MessageBufferUtil.getStoreTimeStamp(message = this.getMessageAsync(attempt).join())) == timestamp) {
                        previousAttempt = attempt;
                    }
                    offset = previousAttempt;
                    break;
                }
                case UPPER: {
                    long attempt;
                    while ((attempt = previousAttempt + 1L) <= maxQueueOffset && (storeTime = MessageBufferUtil.getStoreTimeStamp(message = this.getMessageAsync(attempt).join())) == timestamp) {
                        previousAttempt = attempt;
                    }
                    offset = previousAttempt;
                    break;
                }
                default: {
                    LOGGER.warn("TieredMessageQueueContainer#getQueueOffsetByTime: unknown boundary boundaryType");
                    break;
                }
            }
        } else {
            switch (boundaryType) {
                case LOWER: {
                    offset = rightOffset;
                    break;
                }
                case UPPER: {
                    offset = leftOffset;
                    break;
                }
                default: {
                    LOGGER.warn("TieredMessageQueueContainer#getQueueOffsetByTime: unknown boundary boundaryType");
                }
            }
        }
        return offset;
    }

    public void initOffset(long offset) {
        if (!this.consumeQueue.isInitialized()) {
            this.queueMetadata.setMinOffset(offset);
            this.queueMetadata.setMaxOffset(offset);
        }
        if (!this.consumeQueue.isInitialized()) {
            this.consumeQueue.setBaseOffset(offset * 20L);
        }
        this.dispatchOffset = offset;
    }

    public long getBuildCQMaxOffset() {
        return this.commitLog.getCommitMsgQueueOffset();
    }

    public AppendResult appendCommitLog(ByteBuffer message) {
        return this.appendCommitLog(message, false);
    }

    public AppendResult appendCommitLog(ByteBuffer message, boolean commit) {
        if (this.closed) {
            return AppendResult.FILE_CLOSED;
        }
        long queueOffset = MessageBufferUtil.getQueueOffset(message);
        if (queueOffset != this.dispatchOffset) {
            return AppendResult.OFFSET_INCORRECT;
        }
        AppendResult result = this.commitLog.append(message, commit);
        if (result == AppendResult.SUCCESS) {
            ++this.dispatchOffset;
        }
        return result;
    }

    public AppendResult appendConsumeQueue(DispatchRequest request) {
        return this.appendConsumeQueue(request, false);
    }

    public AppendResult appendConsumeQueue(DispatchRequest request, boolean commit) {
        if (this.closed) {
            return AppendResult.FILE_CLOSED;
        }
        if (request.getConsumeQueueOffset() != this.getConsumeQueueMaxOffset()) {
            return AppendResult.OFFSET_INCORRECT;
        }
        return this.consumeQueue.append(request.getCommitLogOffset(), request.getMsgSize(), request.getTagsCode(), request.getStoreTimestamp(), commit);
    }

    public AppendResult appendIndexFile(DispatchRequest request) {
        AppendResult result;
        if (this.closed) {
            return AppendResult.FILE_CLOSED;
        }
        if (StringUtils.isNotBlank((CharSequence)request.getUniqKey()) && (result = this.indexFile.append(this.messageQueue, this.topicId, request.getUniqKey(), request.getCommitLogOffset(), request.getMsgSize(), request.getStoreTimestamp())) != AppendResult.SUCCESS) {
            return result;
        }
        for (String key : request.getKeys().split(" ")) {
            AppendResult result2;
            if (!StringUtils.isNotBlank((CharSequence)key) || (result2 = this.indexFile.append(this.messageQueue, this.topicId, key, request.getCommitLogOffset(), request.getMsgSize(), request.getStoreTimestamp())) == AppendResult.SUCCESS) continue;
            return result2;
        }
        return AppendResult.SUCCESS;
    }

    public CompletableFuture<ByteBuffer> readCommitLog(long offset, int length) {
        return this.commitLog.readAsync(offset, length);
    }

    public CompletableFuture<ByteBuffer> readConsumeQueue(long queueOffset) {
        return this.readConsumeQueue(queueOffset, 1);
    }

    public CompletableFuture<ByteBuffer> readConsumeQueue(long queueOffset, int count) {
        return this.consumeQueue.readAsync(queueOffset * 20L, count * 20);
    }

    public void flushMetadata() {
        try {
            if (this.consumeQueue.getCommitOffset() < this.queueMetadata.getMinOffset()) {
                return;
            }
            this.queueMetadata.setMaxOffset(this.consumeQueue.getCommitOffset() / 20L);
            this.metadataStore.updateQueue(this.queueMetadata);
        }
        catch (Exception e) {
            LOGGER.error("TieredMessageQueueContainer#flushMetadata: update queue metadata failed: topic: {}, queue: {}", new Object[]{this.messageQueue.getTopic(), this.messageQueue.getQueueId(), e});
        }
    }

    public void commitCommitLog() {
        this.commitLog.commit(true);
    }

    public void commitConsumeQueue() {
        this.consumeQueue.commit(true);
    }

    public void cleanExpiredFile(long expireTimestamp) {
        this.commitLog.cleanExpiredFile(expireTimestamp);
        this.consumeQueue.cleanExpiredFile(expireTimestamp);
    }

    public void destroyExpiredFile() {
        this.commitLog.destroyExpiredFile();
        this.consumeQueue.destroyExpiredFile();
    }

    public void commit(boolean sync) {
        this.commitLog.commit(sync);
        this.consumeQueue.commit(sync);
    }

    public void increaseReadAheadFactor() {
        this.readAheadFactor = Math.min(this.readAheadFactor + 1, this.storeConfig.getReadAheadMaxFactor());
    }

    public void decreaseReadAheadFactor() {
        this.readAheadFactor = Math.max(this.readAheadFactor - 1, this.storeConfig.getReadAheadMinFactor());
    }

    public void setNotReadAhead() {
        this.readAheadFactor = 1;
    }

    public int getReadAheadFactor() {
        return this.readAheadFactor;
    }

    public void recordGroupAccess(String group, long offset) {
        this.groupOffsetCache.put((Object)group, (Object)offset);
    }

    public long getActiveGroupCount(long minOffset, long maxOffset) {
        return this.groupOffsetCache.asMap().values().stream().filter(offset -> offset >= minOffset && offset <= maxOffset).count();
    }

    public long getActiveGroupCount() {
        return this.groupOffsetCache.estimatedSize();
    }

    public InflightRequestFuture getInflightRequest(long offset, int batchSize) {
        Optional<InflightRequestFuture> optional = this.inFlightRequestMap.entrySet().stream().filter(entry -> {
            InflightRequestKey key = (InflightRequestKey)entry.getKey();
            return Math.max(key.getOffset(), offset) <= Math.min(key.getOffset() + (long)key.getBatchSize(), offset + (long)batchSize);
        }).max(Comparator.comparing(entry -> ((InflightRequestKey)entry.getKey()).getRequestTime())).map(Map.Entry::getValue);
        return optional.orElseGet(() -> new InflightRequestFuture(Long.MAX_VALUE, new ArrayList<Pair<Integer, CompletableFuture<Long>>>()));
    }

    public InflightRequestFuture getInflightRequest(String group, long offset, int batchSize) {
        InflightRequestFuture future = (InflightRequestFuture)this.inFlightRequestMap.get(new InflightRequestKey(group));
        if (future != null && !future.isAllDone()) {
            return future;
        }
        return this.getInflightRequest(offset, batchSize);
    }

    public void putInflightRequest(String group, long offset, int requestMsgCount, List<Pair<Integer, CompletableFuture<Long>>> futureList) {
        InflightRequestKey key = new InflightRequestKey(group, offset, requestMsgCount);
        this.inFlightRequestMap.remove(key);
        this.inFlightRequestMap.putIfAbsent(key, new InflightRequestFuture(offset, futureList));
    }

    public int hashCode() {
        return this.messageQueue.hashCode();
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        return this.messageQueue.equals((Object)((TieredMessageQueueContainer)obj).messageQueue);
    }

    public void shutdown() {
        this.closed = true;
        this.commitLog.commit(true);
        this.consumeQueue.commit(true);
        this.flushMetadata();
    }

    public void destroy() {
        this.closed = true;
        this.commitLog.destroy();
        this.consumeQueue.destroy();
        try {
            this.metadataStore.deleteFileSegment(this.messageQueue);
            this.metadataStore.deleteQueue(this.messageQueue);
        }
        catch (Exception e) {
            LOGGER.error("TieredMessageQueueContainer#destroy: clean metadata failed: ", (Throwable)e);
        }
    }
}

