/**
 * 
 */
package prefuse.data.util;

import java.util.Comparator;

import prefuse.data.Tuple;
import prefuse.util.collections.DefaultLiteralComparator;
import prefuse.util.collections.LiteralComparator;


/**
 * Comparator that compares Tuples based on the value of a single field.
 * 
 * @author <a href="http://jheer.org">jeffrey heer</a>
 */
public class TupleComparator implements Comparator {

    private String m_field;
    private int m_col;
    private Comparator m_cmp;
    private Class m_type;
    private int m_rev;
    
    /**
     * Creates a new TupleComparator.
     * @param field the data field to compare
     * @param type the expected type of the data field
     * @param ascend true to sort in ascending order, false for descending
     */
    public TupleComparator(String field, Class type, boolean ascend) {
        this(field, type, ascend, DefaultLiteralComparator.getInstance());
    }
    
    /**
     * Creates a new TupleComparator.
     * @param field the data field to compare
     * @param type the expected type of the data field
     * @param ascend true to sort in ascending order, false for descending
     * @param c the comparator to use. Note that for primitive types,
     * this should be an instance of LiteralComparator, otherwise
     * subequent errors will occur.
     */
    public TupleComparator(String field, Class type, 
                           boolean ascend, Comparator c)
    {
        this.m_field = field;
        this.m_col = -1;
        this.m_type = type;
        this.m_rev = ascend ? 1 : -1;
        this.m_cmp = c;
    }
    
    /**
     * Creates a new TupleComparator.
     * @param col the column number of the data field to compare
     * @param type the expected type of the data field
     * @param ascend true to sort in ascending order, false for descending
     */
    public TupleComparator(int col, Class type, boolean ascend) {
        this(col, type, ascend, DefaultLiteralComparator.getInstance());
    }
    
    /**
     * Creates a new TupleComparator.
     * @param col the column number of the data field to compare
     * @param type the expected type of the data field
     * @param ascend true to sort in ascending order, false for descending
     */
    public TupleComparator(int col, Class type, boolean ascend, Comparator c) {
        this.m_field = null;
        this.m_col = col;
        this.m_type = type;
        this.m_rev = ascend ? 1 : -1;
        this.m_cmp = c;
    }
    
    /**
     * Compares two tuples. If either input Object is not a Tuple,
     * a ClassCastException will be thrown.
     * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
     */
    public int compare(Object o1, Object o2) {
        Tuple t1 = (Tuple)o1, t2 = (Tuple)o2;
        int c = 0;
        
        if ( this.m_col == -1 ) {
            if ( this.m_type == int.class || this.m_type == byte.class ) {
                c = ((LiteralComparator)this.m_cmp).compare(
                        t1.getInt(this.m_field), t2.getInt(this.m_field));
            } else if ( this.m_type == double.class ) {
                c = ((LiteralComparator)this.m_cmp).compare(
                        t1.getDouble(this.m_field), t2.getDouble(this.m_field));
            } else if ( this.m_type == long.class ) {
                c = ((LiteralComparator)this.m_cmp).compare(
                        t1.getLong(this.m_field), t2.getLong(this.m_field));
            } else if ( this.m_type == float.class ) {
                c = ((LiteralComparator)this.m_cmp).compare(
                        t1.getFloat(this.m_field), t2.getFloat(this.m_field));
            } else if ( this.m_type == boolean.class ) {
                c = ((LiteralComparator)this.m_cmp).compare(
                        t1.getBoolean(this.m_field), t2.getBoolean(this.m_field));
            } else if ( !this.m_type.isPrimitive() ) {
                c = this.m_cmp.compare(t1.get(this.m_field), t2.get(this.m_field));
            } else {
                throw new IllegalStateException(
                        "Unsupported type: " + this.m_type.getName());
            }
        } else {
            if ( this.m_type == int.class || this.m_type == byte.class ) {
                c = ((LiteralComparator)this.m_cmp).compare(
                        t1.getInt(this.m_col), t2.getInt(this.m_col));
            } else if ( this.m_type == double.class ) {
                c = ((LiteralComparator)this.m_cmp).compare(
                        t1.getDouble(this.m_col), t2.getDouble(this.m_col));
            } else if ( this.m_type == long.class ) {
                c = ((LiteralComparator)this.m_cmp).compare(
                        t1.getLong(this.m_col), t2.getLong(this.m_col));
            } else if ( this.m_type == float.class ) {
                c = ((LiteralComparator)this.m_cmp).compare(
                        t1.getFloat(this.m_col), t2.getFloat(this.m_col));
            } else if ( this.m_type == boolean.class ) {
                c = ((LiteralComparator)this.m_cmp).compare(
                        t1.getBoolean(this.m_col), t2.getBoolean(this.m_col));
            } else if ( !this.m_type.isPrimitive() ) {
                c = this.m_cmp.compare(t1.get(this.m_col), t2.get(this.m_col));
            } else {
                throw new IllegalStateException(
                        "Unsupported type: " + this.m_type.getName());
            }
        }
        return this.m_rev * c;
    }

} // end of class TupleComparator
