/**
 * Copyright (c) 2016 CEA LIST
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Shuai Li (CEA LIST) <shuai.li@cea.fr> - Initial API and implementation
 *   Van Cam Pham (CEA LIST) <vancam.pham@cea.fr> - Reverse implementation
 */
package org.eclipse.papyrus.designer.languages.cpp.reverse.reverse;

import com.google.common.collect.Iterables;
import java.util.List;
import java.util.function.Consumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.eclipse.cdt.core.dom.ast.ExpansionOverlapsBoundaryException;
import org.eclipse.cdt.core.dom.ast.IASTArrayModifier;
import org.eclipse.cdt.core.dom.ast.IASTDeclSpecifier;
import org.eclipse.cdt.core.dom.ast.IASTDeclarator;
import org.eclipse.cdt.core.dom.ast.IASTEqualsInitializer;
import org.eclipse.cdt.core.dom.ast.IASTExpression;
import org.eclipse.cdt.core.dom.ast.IASTInitializer;
import org.eclipse.cdt.core.dom.ast.IASTInitializerClause;
import org.eclipse.cdt.core.dom.ast.IASTPointerOperator;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTArrayDeclarator;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTReferenceOperator;
import org.eclipse.cdt.core.model.CoreModel;
import org.eclipse.cdt.core.model.CoreModelUtil;
import org.eclipse.cdt.core.model.ICContainer;
import org.eclipse.cdt.core.model.ICElement;
import org.eclipse.cdt.core.model.ICProject;
import org.eclipse.cdt.core.model.IEnumeration;
import org.eclipse.cdt.core.model.IInclude;
import org.eclipse.cdt.core.model.INamespace;
import org.eclipse.cdt.core.model.IParent;
import org.eclipse.cdt.core.model.IStructure;
import org.eclipse.cdt.core.model.IStructureDeclaration;
import org.eclipse.cdt.core.model.ITranslationUnit;
import org.eclipse.cdt.core.model.ITypeDef;
import org.eclipse.cdt.core.parser.IToken;
import org.eclipse.core.resources.IFile;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Array;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Const;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.External;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Friend;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Inline;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Mutable;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Ptr;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Ref;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Variadic;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Virtual;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Volatile;
import org.eclipse.papyrus.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.OpaqueExpression;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.TypedElement;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.ValueSpecification;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Utility methods for the reverse
 */
@SuppressWarnings("all")
public class ReverseUtils {
  public enum StereotypeType {
    CONST,
    
    VOLATILE,
    
    ARRAY,
    
    POINTER,
    
    REFERENCE,
    
    EXTERNAL,
    
    FRIENDLINE,
    
    VIRTUAL,
    
    INLINE,
    
    CREATE,
    
    DESTROY,
    
    MUTABLE,
    
    VARIADIC;
  }
  
  private static ReverseUtils instance;
  
  public static ReverseUtils getInstance() {
    ReverseUtils _xblockexpression = null;
    {
      if ((ReverseUtils.instance == null)) {
        ReverseUtils _reverseUtils = new ReverseUtils();
        ReverseUtils.instance = _reverseUtils;
      }
      _xblockexpression = ReverseUtils.instance;
    }
    return _xblockexpression;
  }
  
  public void analyzeDeclaration(final List<IASTDeclarator> declarators, final Type type, final TypedElement typedElement, final String langID) {
    for (final IASTDeclarator declarator : declarators) {
      this.analyzeDeclaration(declarator, type, typedElement, langID);
    }
  }
  
  public void analyzeDeclaration(final IASTDeclarator declarator, final Type type, final TypedElement typedElement, final String langID) {
    try {
      if ((declarator instanceof ICPPASTArrayDeclarator)) {
        ICPPASTArrayDeclarator arrayDeclarator = ((ICPPASTArrayDeclarator) declarator);
        IASTArrayModifier[] arrays = arrayDeclarator.getArrayModifiers();
        final IASTArrayModifier[] _converted_arrays = (IASTArrayModifier[])arrays;
        int _size = ((List<IASTArrayModifier>)Conversions.doWrapArray(_converted_arrays)).size();
        boolean _greaterThan = (_size > 0);
        if (_greaterThan) {
          boolean _isApplied = StereotypeUtil.isApplied(typedElement, Array.class);
          if (_isApplied) {
            StereotypeUtil.unapply(typedElement, Array.class);
          }
          this.applyStereotype(typedElement, true, ReverseUtils.StereotypeType.ARRAY, "");
          final IASTArrayModifier[] _converted_arrays_1 = (IASTArrayModifier[])arrays;
          final Consumer<IASTArrayModifier> _function = new Consumer<IASTArrayModifier>() {
            @Override
            public void accept(final IASTArrayModifier it) {
              IASTExpression expr = it.getConstantExpression();
              if ((expr != null)) {
                String definition = UMLUtil.<Array>getStereotypeApplication(typedElement, Array.class).getDefinition();
                Array _stereotypeApplication = UMLUtil.<Array>getStereotypeApplication(typedElement, Array.class);
                String _string = expr.toString();
                String _plus = ((definition + "[") + _string);
                String _plus_1 = (_plus + "]");
                _stereotypeApplication.setDefinition(_plus_1);
              }
            }
          };
          ((List<IASTArrayModifier>)Conversions.doWrapArray(_converted_arrays_1)).forEach(_function);
        }
      }
      IASTPointerOperator[] _pointerOperators = declarator.getPointerOperators();
      boolean _tripleNotEquals = (_pointerOperators != null);
      if (_tripleNotEquals) {
        boolean _isApplied_1 = StereotypeUtil.isApplied(typedElement, Ptr.class);
        if (_isApplied_1) {
          Ptr _stereotypeApplication = UMLUtil.<Ptr>getStereotypeApplication(typedElement, Ptr.class);
          _stereotypeApplication.setDeclaration("");
        }
        IASTPointerOperator[] _pointerOperators_1 = declarator.getPointerOperators();
        for (final IASTPointerOperator pointerOperator : _pointerOperators_1) {
          if ((pointerOperator instanceof ICPPASTReferenceOperator)) {
            ICPPASTReferenceOperator reference = ((ICPPASTReferenceOperator) pointerOperator);
            boolean _isApplied_2 = StereotypeUtil.isApplied(typedElement, Ref.class);
            boolean _not = (!_isApplied_2);
            if (_not) {
              this.applyStereotype(typedElement, true, ReverseUtils.StereotypeType.REFERENCE, reference.getSyntax().toString());
            } else {
              String value = UMLUtil.<Ref>getStereotypeApplication(typedElement, Ref.class).getDeclaration();
              Ref _stereotypeApplication_1 = UMLUtil.<Ref>getStereotypeApplication(typedElement, Ref.class);
              String _string = reference.getSyntax().toString();
              String _plus = (value + _string);
              _stereotypeApplication_1.setDeclaration(_plus);
            }
          } else {
            if ((pointerOperator instanceof IASTPointerOperator)) {
              IASTPointerOperator pointer = ((IASTPointerOperator) pointerOperator);
              boolean _isApplied_3 = StereotypeUtil.isApplied(typedElement, Ptr.class);
              boolean _not_1 = (!_isApplied_3);
              if (_not_1) {
                this.applyStereotype(typedElement, true, ReverseUtils.StereotypeType.POINTER, pointer.getSyntax().toString());
              } else {
                String value_1 = UMLUtil.<Ptr>getStereotypeApplication(typedElement, Ptr.class).getDeclaration();
                Ptr _stereotypeApplication_2 = UMLUtil.<Ptr>getStereotypeApplication(typedElement, Ptr.class);
                String _string_1 = pointer.getSyntax().toString();
                String _plus_1 = (value_1 + _string_1);
                _stereotypeApplication_2.setDeclaration(_plus_1);
              }
            }
          }
        }
        boolean _isApplied_4 = StereotypeUtil.isApplied(typedElement, Ptr.class);
        if (_isApplied_4) {
          String value_2 = UMLUtil.<Ptr>getStereotypeApplication(typedElement, Ptr.class).getDeclaration();
          if ((value_2 == null)) {
            value_2 = "*";
          }
          Pattern pattern = Pattern.compile("(\\*)([\\s]*)(const)");
          Matcher matcher = pattern.matcher(declarator.getRawSignature());
          boolean _find = matcher.find();
          if (_find) {
            String _value = value_2;
            value_2 = (_value + " const");
            Ptr _stereotypeApplication_3 = UMLUtil.<Ptr>getStereotypeApplication(typedElement, Ptr.class);
            _stereotypeApplication_3.setDeclaration(value_2);
          }
          boolean _equals = value_2.trim().equals("*");
          if (_equals) {
            Ptr _stereotypeApplication_4 = UMLUtil.<Ptr>getStereotypeApplication(typedElement, Ptr.class);
            _stereotypeApplication_4.setDeclaration(null);
          }
        }
        boolean _isApplied_5 = StereotypeUtil.isApplied(typedElement, Ref.class);
        if (_isApplied_5) {
          String value_3 = UMLUtil.<Ref>getStereotypeApplication(typedElement, Ref.class).getDeclaration();
          if (((value_3 != null) && value_3.trim().equals("&"))) {
            Ref _stereotypeApplication_5 = UMLUtil.<Ref>getStereotypeApplication(typedElement, Ref.class);
            _stereotypeApplication_5.setDeclaration(null);
          }
        }
      }
      IASTInitializer initilizer = declarator.getInitializer();
      final Function1<ValueSpecification, Boolean> _function_1 = new Function1<ValueSpecification, Boolean>() {
        @Override
        public Boolean apply(final ValueSpecification it) {
          return Boolean.valueOf(it.getName().equals("defaultValue"));
        }
      };
      ValueSpecification valueExisting = IterableExtensions.<ValueSpecification>head(IterableExtensions.<ValueSpecification>filter(Iterables.<ValueSpecification>filter(typedElement.getOwnedElements(), ValueSpecification.class), _function_1));
      if ((valueExisting != null)) {
        valueExisting.destroy();
      }
      if ((initilizer != null)) {
        ValueSpecification vs = null;
        if ((typedElement instanceof Property)) {
          vs = ((Property) typedElement).createDefaultValue("defaultValue", ((Property)typedElement).getType(), 
            UMLPackage.Literals.OPAQUE_EXPRESSION);
        } else {
          if ((typedElement instanceof Parameter)) {
            vs = ((Parameter) typedElement).createDefaultValue("default", ((Parameter)typedElement).getType(), 
              UMLPackage.Literals.OPAQUE_EXPRESSION);
          }
        }
        if ((vs == null)) {
          return;
        }
        OpaqueExpression oe = ((OpaqueExpression) vs);
        oe.getLanguages().add(langID);
        if ((initilizer instanceof IASTEqualsInitializer)) {
          IASTEqualsInitializer equalsInitialiser = ((IASTEqualsInitializer) initilizer);
          IASTInitializerClause clause = equalsInitialiser.getInitializerClause();
          if ((clause != null)) {
            oe.getBodies().add(clause.getRawSignature());
          }
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public void applyStereotype(final Element element, final boolean isApply, final ReverseUtils.StereotypeType stType, final String additional) {
    if ((!isApply)) {
      return;
    }
    if (stType != null) {
      switch (stType) {
        case CONST:
          StereotypeUtil.apply(element, Const.class);
          break;
        case VOLATILE:
          StereotypeUtil.apply(element, Volatile.class);
          break;
        case ARRAY:
          StereotypeUtil.apply(element, Array.class);
          Array _stereotypeApplication = UMLUtil.<Array>getStereotypeApplication(element, Array.class);
          _stereotypeApplication.setDefinition(additional);
          break;
        case POINTER:
          StereotypeUtil.apply(element, Ptr.class);
          Ptr _stereotypeApplication_1 = UMLUtil.<Ptr>getStereotypeApplication(element, Ptr.class);
          _stereotypeApplication_1.setDeclaration(additional);
          break;
        case REFERENCE:
          StereotypeUtil.apply(element, Ref.class);
          Ref _stereotypeApplication_2 = UMLUtil.<Ref>getStereotypeApplication(element, Ref.class);
          _stereotypeApplication_2.setDeclaration(additional);
          break;
        case EXTERNAL:
          StereotypeUtil.apply(element, External.class);
          break;
        case FRIENDLINE:
          StereotypeUtil.apply(element, Friend.class);
          break;
        case VIRTUAL:
          StereotypeUtil.apply(element, Virtual.class);
          break;
        case INLINE:
          StereotypeUtil.apply(element, Inline.class);
          break;
        case CREATE:
          StereotypeUtil.apply(element, "StandardProfile::Create");
          break;
        case DESTROY:
          StereotypeUtil.apply(element, "StandardProfile::Destroy");
          break;
        case MUTABLE:
          StereotypeUtil.apply(element, Mutable.class);
          break;
        case VARIADIC:
          StereotypeUtil.apply(element, Variadic.class);
          break;
        default:
          break;
      }
    }
  }
  
  public String getCppTypeName(final IASTDeclSpecifier declarator) {
    String parameterTypeName = "";
    try {
      IToken token = declarator.getSyntax();
      while ((token != null)) {
        {
          String tokenStr = token.toString();
          boolean _equals = tokenStr.equals("*");
          if (_equals) {
          } else {
            boolean _equals_1 = tokenStr.equals("&");
            if (_equals_1) {
            } else {
              boolean _equals_2 = tokenStr.equals("const");
              if (_equals_2) {
              } else {
                int _length = parameterTypeName.length();
                boolean _greaterThan = (_length > 0);
                if (_greaterThan) {
                  String _parameterTypeName = parameterTypeName;
                  parameterTypeName = (_parameterTypeName + " ");
                }
                String _parameterTypeName_1 = parameterTypeName;
                parameterTypeName = (_parameterTypeName_1 + tokenStr);
              }
            }
          }
          token = token.getNext();
        }
      }
    } catch (final Throwable _t) {
      if (_t instanceof ExpansionOverlapsBoundaryException) {
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return parameterTypeName;
  }
  
  public static String getCppTypeName(final String name) {
    String trimName = name.trim();
    trimName = trimName.replace("*", "");
    trimName = trimName.replace("&", "");
    trimName = trimName.replace("[", "");
    trimName = trimName.replace("]", "");
    trimName = trimName.replace("const ", "");
    trimName = trimName.replace(" const", "");
    trimName = trimName.replace("volatile", "");
    trimName = trimName.replace(" volatile", "");
    trimName = trimName.trim();
    return trimName;
  }
  
  public List<ICElement> getAllIStructures(final IParent parent, final boolean allIncludes, final boolean lookNestedTypes, final ICProject project) {
    try {
      List<ICElement> ret = new UniqueEList<ICElement>();
      ICElement[] childrend = parent.getChildren();
      for (int i = 0; (i < childrend.length); i++) {
        {
          ICElement child = childrend[i];
          boolean _matched = false;
          if ((child instanceof IStructure)) {
            _matched=true;
            ret.add(((IStructure) child));
            if (lookNestedTypes) {
              ret.addAll(this.getAllIStructures(((IStructure) child), allIncludes, lookNestedTypes, project));
            }
          }
          if (!_matched) {
            if ((child instanceof IEnumeration)) {
              _matched=true;
              ret.add(((IEnumeration) child));
            }
          }
          if (!_matched) {
            if ((child instanceof IParent)) {
              _matched=true;
              ret.addAll(this.getAllIStructures(((IParent) child), allIncludes, lookNestedTypes, project));
            }
          }
          if (!_matched) {
            if ((child instanceof IInclude)) {
              _matched=true;
              if (allIncludes) {
                ITranslationUnit unit = this.getTranslationUnitFromInclude(((IInclude) child), project);
                if ((unit != null)) {
                  ret.addAll(this.getAllIStructures(unit, allIncludes, lookNestedTypes, project));
                }
              }
            }
          }
          if (!_matched) {
            if ((child instanceof ITypeDef)) {
              _matched=true;
              ret.add(child);
            }
          }
          if (!_matched) {
            if ((child instanceof IStructureDeclaration)) {
              _matched=true;
              ret.add(child);
            }
          }
        }
      }
      return ret;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public ITranslationUnit getTranslationUnitFromInclude(final IInclude include, final ICProject project) {
    boolean _contains = include.getFullFileName().contains(project.getElementName());
    boolean _not = (!_contains);
    if (_not) {
      return null;
    }
    ITranslationUnit ret = null;
    String fileIncludePath = IterableExtensions.<String>last(((Iterable<String>)Conversions.doWrapArray(include.getFullFileName().split(project.getElementName()))));
    IFile file = project.getProject().getFile(fileIncludePath);
    if ((file != null)) {
      ITranslationUnit unit = CoreModelUtil.findTranslationUnit(file);
      if ((unit == null)) {
        unit = CoreModel.getDefault().createTranslationUnitFrom(project, file.getLocation());
      }
      ret = unit;
    }
    return ret;
  }
  
  public ITranslationUnit getTranslationUnitFromElement(final ICElement element) {
    ICElement owner = element.getParent();
    while (((owner != null) && (!(owner instanceof ITranslationUnit)))) {
      owner = owner.getParent();
    }
    return ((ITranslationUnit) owner);
  }
  
  public boolean isSatisfyNamespace(final List<String> iUsings, final ICElement istructure) {
    if (((!(istructure.getParent() instanceof INamespace)) && (!(istructure.getParent() instanceof IStructure)))) {
      return true;
    }
    final StringBuilder namespaces = new StringBuilder();
    ICElement owner = istructure.getParent();
    namespaces.append(owner.getElementName());
    while (((owner.getParent() instanceof INamespace) || (owner.getParent() instanceof IStructure))) {
      {
        ICElement _parent = owner.getParent();
        String _elementName = ((INamespace) _parent).getElementName();
        String _plus = (_elementName + "::");
        namespaces.insert(0, _plus);
        owner = owner.getParent();
      }
    }
    final Function1<String, Boolean> _function = new Function1<String, Boolean>() {
      @Override
      public Boolean apply(final String it) {
        return Boolean.valueOf((it.equals(namespaces.toString()) || it.contains(namespaces.toString().trim())));
      }
    };
    boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<String>filter(iUsings, _function));
    return (!_isEmpty);
  }
  
  public List<String> getContextNamespaces(final ICElement element) {
    List<String> ret = new UniqueEList<String>();
    ICElement owner = element.getParent();
    while ((owner != null)) {
      {
        if ((((owner instanceof INamespace) || (owner instanceof IStructure)) || (owner instanceof ICContainer))) {
          ret.add(owner.getElementName());
          final List<String> _converted_ret = (List<String>)ret;
          int _length = ((Object[])Conversions.unwrapArray(_converted_ret, Object.class)).length;
          boolean _greaterThan = (_length > 1);
          if (_greaterThan) {
            String n = IterableExtensions.<String>head(ret);
            final List<String> _converted_ret_1 = (List<String>)ret;
            int len = ((Object[])Conversions.unwrapArray(_converted_ret_1, Object.class)).length;
            for (int i = 1; (i < len); i++) {
              boolean _contains = ret.get(i).contains("::");
              boolean _not = (!_contains);
              if (_not) {
                String _get = ret.get(i);
                String _plus = (_get + "::");
                String _plus_1 = (_plus + n);
                n = _plus_1;
              }
            }
            ret.add(n);
          }
        }
        owner = owner.getParent();
      }
    }
    return ret;
  }
  
  public org.eclipse.uml2.uml.Package getNearestPackage(final Element element) {
    return element.getNearestPackage();
  }
  
  public void unapplyAllStereotypes(final Element element) {
    final EList<EObject> stereotypeAppList = element.getStereotypeApplications();
    final Consumer<EObject> _function = new Consumer<EObject>() {
      @Override
      public void accept(final EObject it) {
        final Stereotype stereotype = UMLUtil.getStereotype(it);
        element.unapplyStereotype(stereotype);
      }
    };
    stereotypeAppList.forEach(_function);
  }
}
