/**
 * Copyright (c) 2016, 2021 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *  Van Cam Pham        <VanCam.PHAM@cea.fr>
 *  Ported from C++ code - Bug 568883
 */
package org.eclipse.papyrus.designer.transformation.languages.java.library.statemachine;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.papyrus.designer.transformation.languages.java.library.statemachine.CDefinitions;
import org.eclipse.papyrus.designer.transformation.languages.java.library.statemachine.SM2ClassesTransformationCore;
import org.eclipse.papyrus.designer.transformation.languages.java.library.statemachine.SMJavaCodeGeneratorConstants;
import org.eclipse.papyrus.designer.transformation.library.statemachine.SMCodeGeneratorConstants;
import org.eclipse.papyrus.designer.transformation.library.statemachine.TransformationUtil;
import org.eclipse.uml2.uml.Constraint;
import org.eclipse.uml2.uml.FinalState;
import org.eclipse.uml2.uml.Pseudostate;
import org.eclipse.uml2.uml.PseudostateKind;
import org.eclipse.uml2.uml.Region;
import org.eclipse.uml2.uml.State;
import org.eclipse.uml2.uml.Transition;
import org.eclipse.uml2.uml.Vertex;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Pseudo state generator for JAVA. TODO: not really adapted to JAVA yet, i.e. very likely to fail, if there are actual pseudo
 * states (besides the initial one).
 */
@SuppressWarnings("all")
public class PseudostateGenerator {
  @Extension
  protected CDefinitions cdefs;
  
  private SM2ClassesTransformationCore core;
  
  public PseudostateGenerator(final SM2ClassesTransformationCore core) {
    this.core = core;
    this.cdefs = core.cdefs;
  }
  
  public String generatePseudo(final Pseudostate p) {
    String ret = "";
    PseudostateKind _kind = p.getKind();
    if (_kind != null) {
      switch (_kind) {
        case JUNCTION_LITERAL:
          ret = this.generateJunction(p);
          break;
        case JOIN_LITERAL:
          ret = this.generateJoin(p);
          break;
        case FORK_LITERAL:
          ret = this.generateFork(p);
          break;
        case CHOICE_LITERAL:
          ret = this.generateChoice(p);
          break;
        case SHALLOW_HISTORY_LITERAL:
          ret = this.generateShallowHistory(p);
          break;
        case DEEP_HISTORY_LITERAL:
          ret = this.generateDeepHistory(p);
          break;
        case ENTRY_POINT_LITERAL:
          ret = this.generateEntryPoint(p);
          break;
        case EXIT_POINT_LITERAL:
          ret = this.generateExitPoint(p);
          break;
        case INITIAL_LITERAL:
          break;
        case TERMINATE_LITERAL:
          break;
        default:
          break;
      }
    }
    return ret;
  }
  
  public String generateJunction(final Pseudostate p) {
    StringConcatenation _builder = new StringConcatenation();
    {
      EList<Transition> _outgoings = p.getOutgoings();
      boolean _hasElements = false;
      for(final Transition out : _outgoings) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(" else ", "");
        }
        _builder.append("if (");
        String _name = p.getName();
        _builder.append(_name);
        _builder.append(" == ");
        int _indexOf = p.getOutgoings().indexOf(out);
        _builder.append(_indexOf);
        _builder.append(") {");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        String _generateBetweenVertex = this.generateBetweenVertex(p, out.getTarget(), out);
        _builder.append(_generateBetweenVertex, "\t");
        _builder.newLineIfNotEmpty();
        _builder.append("}");
        _builder.newLine();
      }
    }
    String ret = _builder.toString();
    return ret;
  }
  
  public String generateBetweenVertex(final Vertex v1, final Vertex v2, final Transition t) {
    ArrayList<Vertex> exits = new ArrayList<Vertex>();
    ArrayList<Vertex> enters = new ArrayList<Vertex>();
    this.core.eventTransform.calculateExitsEnters(t, exits, enters);
    Vertex cpEx = IterableExtensions.<Vertex>last(exits);
    StringConcatenation _builder = new StringConcatenation();
    {
      if ((cpEx instanceof State)) {
        String _generateExitingSubStates = this.core.generateExitingSubStates(((State) cpEx), true);
        _builder.append(_generateExitingSubStates);
        _builder.newLineIfNotEmpty();
      }
    }
    String _transitionEffect = TransformationUtil.getTransitionEffect(t);
    _builder.append(_transitionEffect);
    _builder.newLineIfNotEmpty();
    String ret = _builder.toString();
    Vertex cpEn = IterableExtensions.<Vertex>last(enters);
    Vertex innermost = IterableExtensions.<Vertex>head(enters);
    if ((innermost instanceof State)) {
      String _ret = ret;
      StringConcatenation _builder_1 = new StringConcatenation();
      {
        if ((innermost instanceof FinalState)) {
          {
            Region _container = ((FinalState)innermost).getContainer();
            boolean _equals = Objects.equal(_container, this.core.topRegion);
            if (_equals) {
              _builder_1.append(SMCodeGeneratorConstants.ACTIVE_ROOT_STATE_ID);
              _builder_1.append(" = ");
              _builder_1.append(SMCodeGeneratorConstants.STATE_MAX);
              _builder_1.append(";");
              _builder_1.newLineIfNotEmpty();
            } else {
              _builder_1.append(SMCodeGeneratorConstants.STATE_ARRAY_ATTRIBUTE);
              _builder_1.append("[");
              _builder_1.append(SMCodeGeneratorConstants.ACTIVE_ROOT_STATE_ID);
              _builder_1.append("].");
              _builder_1.append(SMCodeGeneratorConstants.ACTIVE_SUB_STATES);
              _builder_1.append("[");
              int _indexOf = ((FinalState)innermost).getContainer().getState().getRegions().indexOf(((FinalState)innermost).getContainer());
              _builder_1.append(_indexOf);
              _builder_1.append("] = ");
              _builder_1.append(SMCodeGeneratorConstants.STATE_MAX);
              _builder_1.append(";");
              _builder_1.newLineIfNotEmpty();
              String _generateCompletionCall = this.core.generateCompletionCall(((State)innermost));
              _builder_1.append(_generateCompletionCall);
              _builder_1.newLineIfNotEmpty();
            }
          }
        } else {
          String _generateEnteringOnSubVertex = this.core.generateEnteringOnSubVertex(((State) cpEn), innermost);
          _builder_1.append(_generateEnteringOnSubVertex);
          _builder_1.newLineIfNotEmpty();
        }
      }
      ret = (_ret + _builder_1);
    } else {
      String _ret_1 = ret;
      StringConcatenation _builder_2 = new StringConcatenation();
      {
        if ((cpEn instanceof State)) {
          String _generateEnteringOnSubVertex_1 = this.core.generateEnteringOnSubVertex(((State)cpEn), innermost);
          _builder_2.append(_generateEnteringOnSubVertex_1);
          _builder_2.newLineIfNotEmpty();
        } else {
          String _generatePseudo = this.generatePseudo(((Pseudostate) innermost));
          _builder_2.append(_generatePseudo);
          _builder_2.newLineIfNotEmpty();
        }
      }
      ret = (_ret_1 + _builder_2);
    }
    return ret;
  }
  
  public String generateDeepHistory(final Pseudostate p) {
    int regionIndex = p.getContainer().getState().getRegions().indexOf(p.getContainer());
    final Function1<State, Boolean> _function = new Function1<State, Boolean>() {
      @Override
      public Boolean apply(final State it) {
        return Boolean.valueOf(((!(it instanceof FinalState)) && it.isComposite()));
      }
    };
    Iterable<State> subCompositeStates = IterableExtensions.<State>filter(Iterables.<State>filter(p.getContainer().getSubvertices(), State.class), _function);
    String _upperCase = p.getContainer().getState().getName().toUpperCase();
    String parentId = (_upperCase + "_ID");
    final Function1<State, Boolean> _function_1 = new Function1<State, Boolean>() {
      @Override
      public Boolean apply(final State it) {
        return Boolean.valueOf(((!(it instanceof FinalState)) && (!it.isComposite())));
      }
    };
    Iterable<State> subAtomics = IterableExtensions.<State>filter(Iterables.<State>filter(p.getContainer().getSubvertices(), State.class), _function_1);
    final Function1<State, Boolean> _function_2 = new Function1<State, Boolean>() {
      @Override
      public Boolean apply(final State it) {
        return Boolean.valueOf(((PseudostateGenerator.this.core.states2TimeEvents.get(it) != null) && (!PseudostateGenerator.this.core.states2TimeEvents.get(it).isEmpty())));
      }
    };
    Iterable<State> atomicsWithTimeEvent = IterableExtensions.<State>filter(subAtomics, _function_2);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("unsigned int ");
    CharSequence _deepHistoryVariableName = this.getDeepHistoryVariableName(p.getContainer(), p);
    _builder.append(_deepHistoryVariableName);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.append("if (");
    _builder.append(SMCodeGeneratorConstants.STATE_ARRAY_ATTRIBUTE);
    _builder.append("[");
    _builder.append(parentId);
    _builder.append("].");
    _builder.append(SMCodeGeneratorConstants.PREVIOUS_STATES);
    _builder.append("[");
    _builder.append(regionIndex);
    _builder.append("] != ");
    _builder.append(SMCodeGeneratorConstants.STATE_MAX);
    _builder.append(") {");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("//restore active sub-state of ");
    String _name = p.getContainer().getState().getName();
    _builder.append(_name, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    CharSequence _deepHistoryVariableName_1 = this.getDeepHistoryVariableName(p.getContainer(), p);
    _builder.append(_deepHistoryVariableName_1, "\t");
    _builder.append(" = ");
    _builder.append(SMCodeGeneratorConstants.STATE_ARRAY_ATTRIBUTE, "\t");
    _builder.append("[");
    _builder.append(parentId, "\t");
    _builder.append("].");
    _builder.append(SMCodeGeneratorConstants.PREVIOUS_STATES, "\t");
    _builder.append("[");
    _builder.append(regionIndex, "\t");
    _builder.append("];");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("(this->*");
    _builder.append(SMCodeGeneratorConstants.STATE_ARRAY_ATTRIBUTE, "\t");
    _builder.append("[");
    CharSequence _deepHistoryVariableName_2 = this.getDeepHistoryVariableName(p.getContainer(), p);
    _builder.append(_deepHistoryVariableName_2, "\t");
    _builder.append("].");
    _builder.append(SMCodeGeneratorConstants.ENTRY_NAME, "\t");
    _builder.append(")();");
    _builder.newLineIfNotEmpty();
    {
      boolean _hasElements = false;
      for(final State a : atomicsWithTimeEvent) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(" else ", "\t");
        }
        _builder.append("\t");
        _builder.append("if (");
        String _upperCase_1 = a.getName().toUpperCase();
        _builder.append(_upperCase_1, "\t");
        _builder.append("_ID == ");
        CharSequence _deepHistoryVariableName_3 = this.getDeepHistoryVariableName(p.getContainer(), p);
        _builder.append(_deepHistoryVariableName_3, "\t");
        _builder.append(") {");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        CharSequence _generateActivateTimeEvent = this.core.generateActivateTimeEvent(a, "true");
        _builder.append(_generateActivateTimeEvent, "\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("}");
        _builder.newLine();
      }
    }
    _builder.append("\t");
    _builder.append(SMCodeGeneratorConstants.SET_FLAG, "\t");
    _builder.append("(");
    CharSequence _deepHistoryVariableName_4 = this.getDeepHistoryVariableName(p.getContainer(), p);
    _builder.append(_deepHistoryVariableName_4, "\t");
    _builder.append(", ");
    _builder.append(SMJavaCodeGeneratorConstants.THREAD_FUNC_DOACTIVITY_TYPE, "\t");
    _builder.append(", true);");
    _builder.newLineIfNotEmpty();
    {
      boolean _hasElements_1 = false;
      for(final State comp : subCompositeStates) {
        if (!_hasElements_1) {
          _hasElements_1 = true;
        } else {
          _builder.appendImmediate(" else ", "\t");
        }
        _builder.append("\t");
        _builder.append("if (");
        String _upperCase_2 = comp.getName().toUpperCase();
        _builder.append(_upperCase_2, "\t");
        _builder.append("_ID == ");
        CharSequence _deepHistoryVariableName_5 = this.getDeepHistoryVariableName(p.getContainer(), p);
        _builder.append(_deepHistoryVariableName_5, "\t");
        _builder.append(") {");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        String _generateRestoreString = this.generateRestoreString(comp, p);
        _builder.append(_generateRestoreString, "\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("}");
        _builder.newLine();
      }
    }
    _builder.append("} else {");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("//for the first time, the history should be initialized");
    _builder.newLine();
    {
      int _size = p.getOutgoings().size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        _builder.append("\t");
        String _generateBetweenVertex = this.generateBetweenVertex(p, IterableExtensions.<Transition>head(p.getOutgoings()).getTarget(), IterableExtensions.<Transition>head(p.getOutgoings()));
        _builder.append(_generateBetweenVertex, "\t");
        _builder.newLineIfNotEmpty();
      } else {
        _builder.append("\t");
        String _regionMethodName = this.core.getRegionMethodName(p.getContainer());
        _builder.append(_regionMethodName, "\t");
        _builder.append("(");
        String _initialMacroName = this.core.getInitialMacroName(p.getContainer());
        _builder.append(_initialMacroName, "\t");
        _builder.append(");");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("}");
    String ret = _builder.toString();
    return ret;
  }
  
  private String generateRestoreString(final State s, final Pseudostate p) {
    final Function1<State, Boolean> _function = new Function1<State, Boolean>() {
      @Override
      public Boolean apply(final State it) {
        return Boolean.valueOf(((!(it instanceof FinalState)) && (!it.isComposite())));
      }
    };
    Iterable<State> subAtomics = IterableExtensions.<State>filter(Iterables.<State>filter(p.getContainer().getSubvertices(), State.class), _function);
    final Function1<State, Boolean> _function_1 = new Function1<State, Boolean>() {
      @Override
      public Boolean apply(final State it) {
        return Boolean.valueOf(((PseudostateGenerator.this.core.states2TimeEvents.get(it) != null) && (!PseudostateGenerator.this.core.states2TimeEvents.get(it).isEmpty())));
      }
    };
    Iterable<State> atomicsWithTimeEvent = IterableExtensions.<State>filter(subAtomics, _function_1);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("// restore active-stub state of ");
    String _name = s.getName();
    _builder.append(_name);
    _builder.newLineIfNotEmpty();
    {
      EList<Region> _regions = s.getRegions();
      for(final Region r : _regions) {
        _builder.append("unsigned int ");
        CharSequence _deepHistoryVariableName = this.getDeepHistoryVariableName(r, p);
        _builder.append(_deepHistoryVariableName);
        _builder.append(" = ");
        _builder.append(SMCodeGeneratorConstants.STATE_ARRAY_ATTRIBUTE);
        _builder.append("[");
        String _upperCase = s.getName().toUpperCase();
        _builder.append(_upperCase);
        _builder.append("_ID].");
        _builder.append(SMCodeGeneratorConstants.PREVIOUS_STATES);
        _builder.append("[");
        int _indexOf = s.getRegions().indexOf(r);
        _builder.append(_indexOf);
        _builder.append("];");
        _builder.newLineIfNotEmpty();
        _builder.append("(this->*");
        _builder.append(SMCodeGeneratorConstants.STATE_ARRAY_ATTRIBUTE);
        _builder.append("[");
        CharSequence _deepHistoryVariableName_1 = this.getDeepHistoryVariableName(r, p);
        _builder.append(_deepHistoryVariableName_1);
        _builder.append("].");
        _builder.append(SMCodeGeneratorConstants.ENTRY_NAME);
        _builder.append(")();");
        _builder.newLineIfNotEmpty();
        {
          boolean _hasElements = false;
          for(final State a : atomicsWithTimeEvent) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(" else ", "");
            }
            _builder.append("if (");
            String _upperCase_1 = a.getName().toUpperCase();
            _builder.append(_upperCase_1);
            _builder.append("_ID == ");
            CharSequence _deepHistoryVariableName_2 = this.getDeepHistoryVariableName(r, p);
            _builder.append(_deepHistoryVariableName_2);
            _builder.append(") {");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            CharSequence _generateActivateTimeEvent = this.core.generateActivateTimeEvent(a, "true");
            _builder.append(_generateActivateTimeEvent, "\t");
            _builder.newLineIfNotEmpty();
            _builder.append("}");
            _builder.newLine();
          }
        }
        _builder.append(SMCodeGeneratorConstants.SET_FLAG);
        _builder.append("(");
        CharSequence _deepHistoryVariableName_3 = this.getDeepHistoryVariableName(r, p);
        _builder.append(_deepHistoryVariableName_3);
        _builder.append(", ");
        _builder.append(SMJavaCodeGeneratorConstants.THREAD_FUNC_DOACTIVITY_TYPE);
        _builder.append(", true);");
        _builder.newLineIfNotEmpty();
        final Function1<State, Boolean> _function_2 = new Function1<State, Boolean>() {
          @Override
          public Boolean apply(final State it) {
            return Boolean.valueOf(((!(it instanceof FinalState)) && it.isComposite()));
          }
        };
        Iterable<State> subCompositeStates = IterableExtensions.<State>filter(Iterables.<State>filter(r.getSubvertices(), State.class), _function_2);
        _builder.newLineIfNotEmpty();
        {
          boolean _hasElements_1 = false;
          for(final State comp : subCompositeStates) {
            if (!_hasElements_1) {
              _hasElements_1 = true;
            } else {
              _builder.appendImmediate(" else ", "");
            }
            _builder.append("if (");
            String _upperCase_2 = comp.getName().toUpperCase();
            _builder.append(_upperCase_2);
            _builder.append("_ID == ");
            CharSequence _deepHistoryVariableName_4 = this.getDeepHistoryVariableName(r, p);
            _builder.append(_deepHistoryVariableName_4);
            _builder.append(") {");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            String _generateRestoreString = this.generateRestoreString(comp, p);
            _builder.append(_generateRestoreString, "\t");
            _builder.newLineIfNotEmpty();
            _builder.append("}");
            _builder.newLine();
          }
        }
      }
    }
    return _builder.toString();
  }
  
  public CharSequence getDeepHistoryVariableName(final Region r, final Pseudostate p) {
    StringConcatenation _builder = new StringConcatenation();
    String _name = r.getState().getName();
    _builder.append(_name);
    _builder.append("_");
    String _name_1 = r.getName();
    _builder.append(_name_1);
    _builder.append("_");
    String _name_2 = p.getName();
    _builder.append(_name_2);
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  public String generateShallowHistory(final Pseudostate p) {
    int regionIndex = p.getContainer().getState().getRegions().indexOf(p.getContainer());
    final Function1<State, Boolean> _function = new Function1<State, Boolean>() {
      @Override
      public Boolean apply(final State it) {
        return Boolean.valueOf(((!(it instanceof FinalState)) && it.isComposite()));
      }
    };
    Iterable<State> subCompositeStates = IterableExtensions.<State>filter(Iterables.<State>filter(p.getContainer().getSubvertices(), State.class), _function);
    Pseudostate pseudoInitial = TransformationUtil.firstPseudoState(p.getContainer(), PseudostateKind.INITIAL_LITERAL);
    String _upperCase = p.getContainer().getState().getName().toUpperCase();
    String parentId = (_upperCase + "_ID");
    final Function1<State, Boolean> _function_1 = new Function1<State, Boolean>() {
      @Override
      public Boolean apply(final State it) {
        return Boolean.valueOf(((!(it instanceof FinalState)) && (!it.isComposite())));
      }
    };
    Iterable<State> subAtomics = IterableExtensions.<State>filter(Iterables.<State>filter(p.getContainer().getSubvertices(), State.class), _function_1);
    final Function1<State, Boolean> _function_2 = new Function1<State, Boolean>() {
      @Override
      public Boolean apply(final State it) {
        return Boolean.valueOf(((PseudostateGenerator.this.core.states2TimeEvents.get(it) != null) && (!PseudostateGenerator.this.core.states2TimeEvents.get(it).isEmpty())));
      }
    };
    Iterable<State> atomicsWithTimeEvent = IterableExtensions.<State>filter(subAtomics, _function_2);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("unsigned int loc_ActiveId;");
    _builder.newLine();
    _builder.append("if (");
    _builder.append(SMCodeGeneratorConstants.STATE_ARRAY_ATTRIBUTE);
    _builder.append("[");
    _builder.append(parentId);
    _builder.append("].");
    _builder.append(SMCodeGeneratorConstants.PREVIOUS_STATES);
    _builder.append("[");
    _builder.append(regionIndex);
    _builder.append("] != ");
    _builder.append(SMCodeGeneratorConstants.STATE_MAX);
    _builder.append(") {");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("loc_ActiveId = ");
    _builder.append(SMCodeGeneratorConstants.STATE_ARRAY_ATTRIBUTE, "\t");
    _builder.append("[");
    _builder.append(parentId, "\t");
    _builder.append("].");
    _builder.append(SMCodeGeneratorConstants.PREVIOUS_STATES, "\t");
    _builder.append("[");
    _builder.append(regionIndex, "\t");
    _builder.append("];");
    _builder.newLineIfNotEmpty();
    _builder.append("} else {");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("//for the first time, the history should be initialized");
    _builder.newLine();
    {
      int _size = p.getOutgoings().size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        _builder.append("\t");
        String _transitionEffect = TransformationUtil.getTransitionEffect(IterableExtensions.<Transition>head(p.getOutgoings()));
        _builder.append(_transitionEffect, "\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("loc_ActiveId = ");
        String _upperCase_1 = IterableExtensions.<Transition>head(p.getOutgoings()).getTarget().getName().toUpperCase();
        _builder.append(_upperCase_1, "\t");
        _builder.append("_ID;");
        _builder.newLineIfNotEmpty();
      } else {
        _builder.append("\t");
        String _transitionEffect_1 = TransformationUtil.getTransitionEffect(IterableExtensions.<Transition>head(pseudoInitial.getOutgoings()));
        _builder.append(_transitionEffect_1, "\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("loc_ActiveId = ");
        String _upperCase_2 = IterableExtensions.<Transition>head(pseudoInitial.getOutgoings()).getTarget().getName().toUpperCase();
        _builder.append(_upperCase_2, "\t");
        _builder.append("_ID;");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("}");
    _builder.newLine();
    _builder.append(SMCodeGeneratorConstants.STATE_ARRAY_ATTRIBUTE);
    _builder.append("[");
    _builder.append(parentId);
    _builder.append("].");
    _builder.append(SMCodeGeneratorConstants.ACTIVE_SUB_STATES);
    _builder.append("[");
    _builder.append(regionIndex);
    _builder.append("] = loc_ActiveId;");
    _builder.newLineIfNotEmpty();
    {
      boolean _hasElements = false;
      for(final State comp : subCompositeStates) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(" else ", "");
        }
        _builder.append("if (");
        String _upperCase_3 = comp.getName().toUpperCase();
        _builder.append(_upperCase_3);
        _builder.append("_ID == loc_ActiveId) {");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        String _generateEnteringOnSubVertex = this.core.generateEnteringOnSubVertex(comp, null);
        _builder.append(_generateEnteringOnSubVertex, "\t");
        _builder.newLineIfNotEmpty();
        _builder.append("}");
        _builder.newLine();
        _builder.append("\t\t\t");
      }
    }
    _builder.append(" ");
    {
      int _size_1 = IterableExtensions.size(subCompositeStates);
      boolean _greaterThan_1 = (_size_1 > 0);
      if (_greaterThan_1) {
        _builder.append(" else { ");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("(this->*");
    _builder.append(SMCodeGeneratorConstants.STATE_ARRAY_ATTRIBUTE, "\t");
    _builder.append("[loc_ActiveId].");
    _builder.append(SMCodeGeneratorConstants.ENTRY_NAME, "\t");
    _builder.append(")();");
    _builder.newLineIfNotEmpty();
    {
      boolean _hasElements_1 = false;
      for(final State a : atomicsWithTimeEvent) {
        if (!_hasElements_1) {
          _hasElements_1 = true;
        } else {
          _builder.appendImmediate(" else ", "\t");
        }
        _builder.append("\t");
        _builder.append("if (");
        String _upperCase_4 = a.getName().toUpperCase();
        _builder.append(_upperCase_4, "\t");
        _builder.append("_ID == loc_ActiveId) {");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        CharSequence _generateActivateTimeEvent = this.core.generateActivateTimeEvent(a, "true");
        _builder.append(_generateActivateTimeEvent, "\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("}");
        _builder.newLine();
      }
    }
    _builder.append("\t");
    _builder.append(SMCodeGeneratorConstants.SET_FLAG, "\t");
    _builder.append("(loc_ActiveId, ");
    _builder.append(SMJavaCodeGeneratorConstants.THREAD_FUNC_DOACTIVITY_TYPE, "\t");
    _builder.append(", true);");
    _builder.newLineIfNotEmpty();
    {
      int _size_2 = IterableExtensions.size(subCompositeStates);
      boolean _greaterThan_2 = (_size_2 > 0);
      if (_greaterThan_2) {
        _builder.append("}");
      }
    }
    _builder.newLineIfNotEmpty();
    return _builder.toString();
  }
  
  public String generateChoice(final Pseudostate p) {
    StringConcatenation _builder = new StringConcatenation();
    {
      final Function1<Transition, Boolean> _function = new Function1<Transition, Boolean>() {
        @Override
        public Boolean apply(final Transition it) {
          Constraint _guard = it.getGuard();
          return Boolean.valueOf((_guard != null));
        }
      };
      Iterable<Transition> _filter = IterableExtensions.<Transition>filter(p.getOutgoings(), _function);
      boolean _hasElements = false;
      for(final Transition out : _filter) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(" else ", "");
        }
        _builder.append("if (");
        String _guard = this.core.getGuard(out);
        _builder.append(_guard);
        _builder.append(") {");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        String _generateBetweenVertex = this.generateBetweenVertex(p, out.getTarget(), out);
        _builder.append(_generateBetweenVertex, "\t");
        _builder.newLineIfNotEmpty();
        _builder.append("}");
        _builder.newLine();
      }
    }
    _builder.append("else {");
    _builder.newLine();
    _builder.append("\t");
    final Function1<Transition, Boolean> _function_1 = new Function1<Transition, Boolean>() {
      @Override
      public Boolean apply(final Transition it) {
        Constraint _guard = it.getGuard();
        return Boolean.valueOf((_guard == null));
      }
    };
    Transition t = IterableExtensions.<Transition>head(IterableExtensions.<Transition>filter(p.getOutgoings(), _function_1));
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    String _generateBetweenVertex_1 = this.generateBetweenVertex(p, t.getTarget(), t);
    _builder.append(_generateBetweenVertex_1, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("}");
    _builder.newLine();
    return _builder.toString();
  }
  
  public List<Vertex> intersect(final List<Vertex> l1, final List<Vertex> l2) {
    final Function1<Vertex, Boolean> _function = new Function1<Vertex, Boolean>() {
      @Override
      public Boolean apply(final Vertex it) {
        return Boolean.valueOf(l2.contains(it));
      }
    };
    return IterableExtensions.<Vertex>toList(IterableExtensions.<Vertex>filter(l1, _function));
  }
  
  public String generateEntryPoint(final Pseudostate p) {
    final Function1<Transition, Vertex> _function = new Function1<Transition, Vertex>() {
      @Override
      public Vertex apply(final Transition it) {
        return it.getTarget();
      }
    };
    final List<Vertex> targets = IterableExtensions.<Vertex>toList(ListExtensions.<Transition, Vertex>map(p.getOutgoings(), _function));
    EObject _eContainer = p.eContainer();
    final Function1<Region, Boolean> _function_1 = new Function1<Region, Boolean>() {
      @Override
      public Boolean apply(final Region it) {
        int _size = PseudostateGenerator.this.intersect(IterableExtensions.<Vertex>toList(PseudostateGenerator.this.core.allSubVertexes(it)), targets).size();
        return Boolean.valueOf((_size > 0));
      }
    };
    final List<Region> enteredRegions = IterableExtensions.<Region>toList(IterableExtensions.<Region>filter(((State) _eContainer).getRegions(), _function_1));
    EObject _eContainer_1 = p.eContainer();
    final Function1<Region, Boolean> _function_2 = new Function1<Region, Boolean>() {
      @Override
      public Boolean apply(final Region it) {
        boolean _contains = enteredRegions.contains(it);
        return Boolean.valueOf((!_contains));
      }
    };
    Iterable<Region> defaultRegions = IterableExtensions.<Region>filter(((State) _eContainer_1).getRegions(), _function_2);
    StringConcatenation _builder = new StringConcatenation();
    {
      EList<Transition> _outgoings = p.getOutgoings();
      for(final Transition out : _outgoings) {
        String _transitionEffect = TransformationUtil.getTransitionEffect(out);
        _builder.append(_transitionEffect);
        _builder.newLineIfNotEmpty();
      }
    }
    {
      final Function1<Region, Boolean> _function_3 = new Function1<Region, Boolean>() {
        @Override
        public Boolean apply(final Region it) {
          State _findInitialState = TransformationUtil.findInitialState(it);
          return Boolean.valueOf((_findInitialState != null));
        }
      };
      Iterable<Region> _filter = IterableExtensions.<Region>filter(defaultRegions, _function_3);
      for(final Region r : _filter) {
        String _regionMethodName = this.core.getRegionMethodName(r);
        _builder.append(_regionMethodName);
        _builder.append("(");
        String _initialMacroName = this.core.getInitialMacroName(r);
        _builder.append(_initialMacroName);
        _builder.append(");");
        _builder.newLineIfNotEmpty();
      }
    }
    {
      for(final Region r_1 : enteredRegions) {
        String _regionMethodName_1 = this.core.getRegionMethodName(r_1);
        _builder.append(_regionMethodName_1);
        _builder.append("(");
        final Function1<Vertex, Boolean> _function_4 = new Function1<Vertex, Boolean>() {
          @Override
          public Boolean apply(final Vertex it) {
            return Boolean.valueOf(r_1.allNamespaces().contains(it));
          }
        };
        String _vertexMacroName = this.core.getVertexMacroName(IterableExtensions.<Vertex>head(IterableExtensions.<Vertex>filter(targets, _function_4)));
        _builder.append(_vertexMacroName);
        _builder.append(");");
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder.toString();
  }
  
  public String generateExitPoint(final Pseudostate p) {
    StringConcatenation _builder = new StringConcatenation();
    String _generateBetweenVertex = this.generateBetweenVertex(p, IterableExtensions.<Transition>head(p.getOutgoings()).getTarget(), IterableExtensions.<Transition>head(p.getOutgoings()));
    _builder.append(_generateBetweenVertex);
    _builder.newLineIfNotEmpty();
    return _builder.toString();
  }
  
  public String generateFork(final Pseudostate p) {
    try {
      final Function1<Transition, Vertex> _function = new Function1<Transition, Vertex>() {
        @Override
        public Vertex apply(final Transition it) {
          return it.getTarget();
        }
      };
      List<Vertex> targets = ListExtensions.<Transition, Vertex>map(p.getOutgoings(), _function);
      StringConcatenation _builder = new StringConcatenation();
      String ret = _builder.toString();
      ArrayList<Vertex> exits = new ArrayList<Vertex>();
      ArrayList<Vertex> enters = new ArrayList<Vertex>();
      this.core.eventTransform.calculateExitsEnters(IterableExtensions.<Transition>head(p.getOutgoings()), exits, enters);
      Vertex cpEx = IterableExtensions.<Vertex>last(exits);
      Vertex _last = IterableExtensions.<Vertex>last(enters);
      State parent = ((State) _last);
      boolean _isOrthogonal = parent.isOrthogonal();
      boolean _not = (!_isOrthogonal);
      if (_not) {
        throw new Exception("state machine is ill-formed");
      }
      String _ret = ret;
      StringConcatenation _builder_1 = new StringConcatenation();
      {
        if ((cpEx instanceof State)) {
          String _generateExitingSubStates = this.core.generateExitingSubStates(((State) cpEx), true);
          _builder_1.append(_generateExitingSubStates);
          _builder_1.newLineIfNotEmpty();
        }
      }
      _builder_1.append("// need to be concurrently executed");
      _builder_1.newLine();
      {
        EList<Transition> _outgoings = p.getOutgoings();
        for(final Transition out : _outgoings) {
          String _transitionEffect = TransformationUtil.getTransitionEffect(out);
          _builder_1.append(_transitionEffect);
          _builder_1.newLineIfNotEmpty();
        }
      }
      ret = (_ret + _builder_1);
      StringConcatenation _builder_2 = new StringConcatenation();
      _builder_2.append(SMCodeGeneratorConstants.STATE_ARRAY_ATTRIBUTE);
      _builder_2.append("[");
      String _upperCase = parent.getName().toUpperCase();
      _builder_2.append(_upperCase);
      _builder_2.append("_ID]");
      String pAttr = _builder_2.toString();
      String _ret_1 = ret;
      StringConcatenation _builder_3 = new StringConcatenation();
      String _generateChangeState = this.core.generateChangeState(parent);
      _builder_3.append(_generateChangeState);
      _builder_3.newLineIfNotEmpty();
      String _fptrCall = this.core.getFptrCall(pAttr, SMCodeGeneratorConstants.ENTRY_NAME);
      _builder_3.append(_fptrCall);
      _builder_3.append(";");
      _builder_3.newLineIfNotEmpty();
      _builder_3.append("// start activity of ");
      String _name = parent.getName();
      _builder_3.append(_name);
      _builder_3.append(" by calling setFlag");
      _builder_3.newLineIfNotEmpty();
      _builder_3.append(SMCodeGeneratorConstants.SET_FLAG);
      _builder_3.append("(");
      String _upperCase_1 = parent.getName().toUpperCase();
      _builder_3.append(_upperCase_1);
      _builder_3.append("_ID, ");
      _builder_3.append(SMJavaCodeGeneratorConstants.THREAD_FUNC_DOACTIVITY_TYPE);
      _builder_3.append(", true);");
      _builder_3.newLineIfNotEmpty();
      {
        EList<Region> _regions = parent.getRegions();
        for(final Region r : _regions) {
          {
            for(final Vertex target : targets) {
              {
                boolean _contains = this.core.allSubVertexes(r).contains(target);
                if (_contains) {
                  String _generateForkCall = this.core.concurrency.generateForkCall(r, true, this.core.getVertexMacroName(target));
                  _builder_3.append(_generateForkCall);
                  _builder_3.newLineIfNotEmpty();
                }
              }
            }
          }
        }
      }
      ret = (_ret_1 + _builder_3);
      return ret;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public String generateJoin(final Pseudostate p) {
    StringConcatenation _builder = new StringConcatenation();
    String _generateBetweenVertex = this.generateBetweenVertex(p, IterableExtensions.<Transition>head(p.getOutgoings()).getTarget(), IterableExtensions.<Transition>head(p.getOutgoings()));
    _builder.append(_generateBetweenVertex);
    _builder.newLineIfNotEmpty();
    return _builder.toString();
  }
}
