/**
 * Copyright (c) 2021 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *  Van Cam Pham        <VanCam.PHAM@cea.fr>
 */
package org.eclipse.papyrus.designer.transformation.library.statemachine;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.papyrus.designer.transformation.base.utils.OperationUtils;
import org.eclipse.papyrus.designer.transformation.library.statemachine.TransformationUtil;
import org.eclipse.papyrus.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.ParameterDirectionKind;
import org.eclipse.uml2.uml.Region;
import org.eclipse.uml2.uml.State;
import org.eclipse.uml2.uml.Transition;
import org.eclipse.uml2.uml.Vertex;
import org.eclipse.xtext.xbase.lib.CollectionExtensions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

@SuppressWarnings("all")
public class SMCommon {
  private static boolean firstIf;
  
  public static int findMax(final List<Integer> l) {
    Integer ret = IterableExtensions.<Integer>head(l);
    for (final Integer i : l) {
      boolean _greaterThan = (i.compareTo(ret) > 0);
      if (_greaterThan) {
        ret = i;
      }
    }
    return (ret).intValue();
  }
  
  public static int calculateDepth(final Region topRegion, final State s) {
    Region _container = s.getContainer();
    boolean _equals = Objects.equal(_container, topRegion);
    if (_equals) {
      return 1;
    } else {
      int _calculateDepth = SMCommon.calculateDepth(topRegion, s.getContainer().getState());
      return (1 + _calculateDepth);
    }
  }
  
  public static List<Map.Entry<List<State>, Integer>> calculateAcceptingStates(final Region topRegion, final State parent, final List<State> acceptingStates) {
    List<State> transitiveSubstates = TransformationUtil.transitiveSubStates(parent);
    final ArrayList<State> retTmp = new ArrayList<State>();
    final Consumer<State> _function = new Consumer<State>() {
      @Override
      public void accept(final State it) {
        boolean _contains = acceptingStates.contains(it);
        if (_contains) {
          retTmp.add(it);
        }
      }
    };
    transitiveSubstates.forEach(_function);
    final List<Map.Entry<List<State>, Integer>> ret = new ArrayList<Map.Entry<List<State>, Integer>>();
    final Map<Integer, List<State>> map = new HashMap<Integer, List<State>>();
    final Consumer<State> _function_1 = new Consumer<State>() {
      @Override
      public void accept(final State it) {
        int depth = SMCommon.calculateDepth(topRegion, it);
        boolean _containsKey = map.containsKey(Integer.valueOf(depth));
        boolean _not = (!_containsKey);
        if (_not) {
          ArrayList<State> _arrayList = new ArrayList<State>();
          map.put(Integer.valueOf(depth), _arrayList);
        }
        map.get(Integer.valueOf(depth)).add(it);
      }
    };
    retTmp.forEach(_function_1);
    int maxDepth = SMCommon.findMax(IterableExtensions.<Integer>toList(map.keySet()));
    for (int i = 0; (i <= maxDepth); i++) {
      boolean _containsKey = map.containsKey(Integer.valueOf(i));
      if (_containsKey) {
        int _size = map.get(Integer.valueOf(i)).size();
        boolean _equals = (_size == 1);
        if (_equals) {
          ArrayList<State> newList = new ArrayList<State>();
          newList.addAll(map.get(Integer.valueOf(i)));
          HashMap<List<State>, Integer> mapEntry = new HashMap<List<State>, Integer>();
          mapEntry.put(newList, Integer.valueOf(0));
          ret.addAll(mapEntry.entrySet());
        } else {
          final Map<State, List<State>> m = new HashMap<State, List<State>>();
          List<State> _get = map.get(Integer.valueOf(i));
          for (final State s : _get) {
            {
              boolean _containsKey_1 = m.containsKey(s.getContainer().getState());
              boolean _not = (!_containsKey_1);
              if (_not) {
                State _state = s.getContainer().getState();
                ArrayList<State> _arrayList = new ArrayList<State>();
                m.put(_state, _arrayList);
              }
              m.get(s.getContainer().getState()).add(s);
            }
          }
          Set<Map.Entry<State, List<State>>> _entrySet = m.entrySet();
          for (final Map.Entry<State, List<State>> en : _entrySet) {
            {
              HashMap<List<State>, Integer> mapEntry_1 = new HashMap<List<State>, Integer>();
              ArrayList<State> newList_1 = new ArrayList<State>();
              newList_1.addAll(en.getValue());
              boolean _isOrthogonal = en.getKey().isOrthogonal();
              if (_isOrthogonal) {
                mapEntry_1.put(newList_1, Integer.valueOf(1));
                ret.addAll(mapEntry_1.entrySet());
              } else {
                mapEntry_1.put(newList_1, Integer.valueOf(0));
                ret.addAll(mapEntry_1.entrySet());
              }
            }
          }
        }
      }
    }
    return ret;
  }
  
  public static List<State> getTransitiveParentStates(final Vertex s) {
    ArrayList<State> ret = new ArrayList<State>();
    State parent = s.getContainer().getState();
    while ((parent != null)) {
      {
        ret.add(parent);
        parent = parent.getContainer().getState();
      }
    }
    return ret;
  }
  
  public static boolean checkSubStatesAcceptEvent(final State parent, final List<Transition> transitions) {
    final ArrayList<State> substates = new ArrayList<State>();
    final Consumer<Region> _function = new Consumer<Region>() {
      @Override
      public void accept(final Region it) {
        Iterables.<State>addAll(substates, Iterables.<State>filter(it.getSubvertices(), State.class));
      }
    };
    parent.getRegions().forEach(_function);
    boolean ret = false;
    for (final Transition t : transitions) {
      if (((!ret) && substates.contains(t.getSource()))) {
        ret = true;
      }
    }
    return ret;
  }
  
  public static boolean checkTransitiveSubStatesAcceptEvent(final State parent, final List<Transition> transitions) {
    final List<State> substates = TransformationUtil.transitiveSubStates(parent);
    boolean ret = false;
    for (final Transition t : transitions) {
      if (((!ret) && substates.contains(t.getSource()))) {
        ret = true;
      }
    }
    return ret;
  }
  
  public static void copyParameters(final Operation source, final Operation target, final boolean isCopyReturn) {
    String name = target.getName();
    OperationUtils.syncOperation(source, target);
    if ((!isCopyReturn)) {
      final Function1<Parameter, Boolean> _function = new Function1<Parameter, Boolean>() {
        @Override
        public Boolean apply(final Parameter it) {
          ParameterDirectionKind _direction = it.getDirection();
          return Boolean.valueOf(Objects.equal(_direction, ParameterDirectionKind.RETURN_LITERAL));
        }
      };
      Iterable<Parameter> ret = IterableExtensions.<Parameter>filter(target.getOwnedParameters(), _function);
      CollectionExtensions.<Parameter>removeAll(target.getOwnedParameters(), ret);
    }
    EList<EObject> _stereotypeApplications = target.getStereotypeApplications();
    for (final EObject stt : _stereotypeApplications) {
      StereotypeUtil.apply(target, stt.getClass());
    }
    target.setName(name);
  }
  
  /**
   * init "else", see condElse
   */
  public static boolean initCondElse() {
    return SMCommon.firstIf = true;
  }
  
  /**
   * return an "else " when called more than once.
   * => helps creating "else if" instead of "isProcessed" (bug 549801O) in multiple "if"
   * statements
   */
  public static String condElse() {
    if ((!SMCommon.firstIf)) {
      return "else ";
    } else {
      SMCommon.firstIf = false;
      return "";
    }
  }
  
  /**
   * Return all outgoing transitions of a vertex, including those
   * of targeted vertices that are not states (transitive hull).
   */
  public static List<Transition> findTrans(final Vertex v) {
    UniqueEList<Transition> ret = new UniqueEList<Transition>();
    ret.addAll(v.getOutgoings());
    EList<Transition> _outgoings = v.getOutgoings();
    for (final Transition out : _outgoings) {
      Vertex _target = out.getTarget();
      boolean _not = (!(_target instanceof State));
      if (_not) {
        ret.addAll(SMCommon.findTrans(out.getTarget()));
      }
    }
    return ret;
  }
  
  public static List<State> getRootStates(final Region topRegion, final List<State> states) {
    final ArrayList<State> ret = new ArrayList<State>();
    final Function1<State, Boolean> _function = new Function1<State, Boolean>() {
      @Override
      public Boolean apply(final State it) {
        Region _container = it.getContainer();
        return Boolean.valueOf(Objects.equal(_container, topRegion));
      }
    };
    Iterables.<State>addAll(ret, IterableExtensions.<State>filter(states, _function));
    return ret;
  }
  
  public static int getRegionNumber(final Region topRegion, final State child) {
    Region _container = child.getContainer();
    boolean _equals = Objects.equal(_container, topRegion);
    if (_equals) {
      return 0;
    }
    return child.getContainer().getState().getRegions().indexOf(child.getContainer());
  }
}
