/**
 * Copyright (c) 2021 CEA LIST.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.designer.transformation.tracing.library;

import java.util.HashMap;
import java.util.Map;
import org.eclipse.papyrus.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.State;
import org.eclipse.uml2.uml.Transition;
import org.eclipse.uml2.uml.profile.standard.Create;
import org.eclipse.uml2.uml.profile.standard.Destroy;
import org.eclipse.xtend2.lib.StringConcatenation;

/**
 * The names of several trace events. Independent of a concrete trace implementation
 */
@SuppressWarnings("all")
public class EventNames {
  private static Map<Element, Integer> idMap;
  
  private static int counter;
  
  public static int init() {
    int _xblockexpression = (int) 0;
    {
      HashMap<Element, Integer> _hashMap = new HashMap<Element, Integer>();
      EventNames.idMap = _hashMap;
      _xblockexpression = EventNames.counter = 0;
    }
    return _xblockexpression;
  }
  
  /**
   * The name of an operation event start
   * If the port name is non null, it is taken into account
   */
  public static CharSequence operationStartsEventName(final Operation operation, final Port port) {
    CharSequence _xifexpression = null;
    if ((port != null)) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("port_op_starts_");
      String _name = operation.getClass_().getName();
      _builder.append(_name);
      _builder.append("_");
      String _name_1 = port.getName();
      _builder.append(_name_1);
      _builder.append("_");
      String _name_2 = operation.getName();
      _builder.append(_name_2);
      _builder.append("_");
      String _id = EventNames.getId(operation);
      _builder.append(_id);
      _xifexpression = _builder;
    } else {
      CharSequence _xifexpression_1 = null;
      boolean _isApplied = StereotypeUtil.isApplied(operation, Create.class);
      if (_isApplied) {
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("creation_starts_");
        String _name_3 = operation.getClass_().getName();
        _builder_1.append(_name_3);
        _builder_1.append("_");
        String _id_1 = EventNames.getId(operation);
        _builder_1.append(_id_1);
        _xifexpression_1 = _builder_1;
      } else {
        CharSequence _xifexpression_2 = null;
        boolean _isApplied_1 = StereotypeUtil.isApplied(operation, Destroy.class);
        if (_isApplied_1) {
          StringConcatenation _builder_2 = new StringConcatenation();
          _builder_2.append("destruction_starts_");
          String _name_4 = operation.getClass_().getName();
          _builder_2.append(_name_4);
          _builder_2.append("_");
          String _id_2 = EventNames.getId(operation);
          _builder_2.append(_id_2);
          _xifexpression_2 = _builder_2;
        } else {
          StringConcatenation _builder_3 = new StringConcatenation();
          _builder_3.append("op_starts_");
          String _name_5 = operation.getClass_().getName();
          _builder_3.append(_name_5);
          _builder_3.append("_");
          String _name_6 = operation.getName();
          _builder_3.append(_name_6);
          _builder_3.append("_");
          String _id_3 = EventNames.getId(operation);
          _builder_3.append(_id_3);
          _xifexpression_2 = _builder_3;
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }
  
  /**
   * The name of an operation event start
   * If the port name is non null, it is taken into account
   */
  public static CharSequence operationEndsEventName(final Operation operation, final Port port) {
    CharSequence _xifexpression = null;
    if ((port != null)) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("port_op_ends_");
      String _name = operation.getClass_().getName();
      _builder.append(_name);
      _builder.append("_");
      String _name_1 = port.getName();
      _builder.append(_name_1);
      _builder.append("_");
      String _name_2 = operation.getName();
      _builder.append(_name_2);
      _builder.append("_");
      String _id = EventNames.getId(operation);
      _builder.append(_id);
      _xifexpression = _builder;
    } else {
      CharSequence _xifexpression_1 = null;
      boolean _isApplied = StereotypeUtil.isApplied(operation, Create.class);
      if (_isApplied) {
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("creation_ends_");
        String _name_3 = operation.getClass_().getName();
        _builder_1.append(_name_3);
        _builder_1.append("_");
        String _id_1 = EventNames.getId(operation);
        _builder_1.append(_id_1);
        _xifexpression_1 = _builder_1;
      } else {
        CharSequence _xifexpression_2 = null;
        boolean _isApplied_1 = StereotypeUtil.isApplied(operation, Destroy.class);
        if (_isApplied_1) {
          StringConcatenation _builder_2 = new StringConcatenation();
          _builder_2.append("destruction_ends_");
          String _name_4 = operation.getClass_().getName();
          _builder_2.append(_name_4);
          _builder_2.append("_");
          String _id_2 = EventNames.getId(operation);
          _builder_2.append(_id_2);
          _xifexpression_2 = _builder_2;
        } else {
          StringConcatenation _builder_3 = new StringConcatenation();
          _builder_3.append("op_ends_");
          String _name_5 = operation.getClass_().getName();
          _builder_3.append(_name_5);
          _builder_3.append("_");
          String _name_6 = operation.getName();
          _builder_3.append(_name_6);
          _builder_3.append("_");
          String _id_3 = EventNames.getId(operation);
          _builder_3.append(_id_3);
          _xifexpression_2 = _builder_3;
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }
  
  /**
   * The name of a state enter event
   */
  public static CharSequence enterStateEventName(final State state) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("enter_state_");
    String _name = state.containingStateMachine().getContext().getName();
    _builder.append(_name);
    _builder.append("_");
    String _name_1 = state.getName();
    _builder.append(_name_1);
    _builder.append("_");
    String _id = EventNames.getId(state);
    _builder.append(_id);
    return _builder;
  }
  
  /**
   * The name of a state exit/leave event
   */
  public static CharSequence exitStateEventName(final State state) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("exit_state_");
    String _name = state.containingStateMachine().getContext().getName();
    _builder.append(_name);
    _builder.append("_");
    String _name_1 = state.getName();
    _builder.append(_name_1);
    _builder.append("_");
    String _id = EventNames.getId(state);
    _builder.append(_id);
    return _builder;
  }
  
  /**
   * The name of a transition event
   * if the transition has no name, use source & target instead
   */
  public static CharSequence transitionEventName(final Transition t) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("transition_");
    String _name = t.containingStateMachine().getContext().getName();
    _builder.append(_name);
    _builder.append("_");
    String _transitionName = EventNames.transitionName(t);
    _builder.append(_transitionName);
    _builder.append("_");
    String _id = EventNames.getId(t);
    _builder.append(_id);
    return _builder;
  }
  
  /**
   * Helper function for transitionEventName - return either transition name or
   * a concatenation of source and target name
   */
  protected static String transitionName(final Transition t) {
    if (((t.getName() != null) && (t.getName().length() > 0))) {
      return t.getName();
    } else {
      StringConcatenation _builder = new StringConcatenation();
      String _name = t.getSource().getName();
      _builder.append(_name);
      _builder.append("_");
      String _name_1 = t.getTarget().getName();
      _builder.append(_name_1);
      return _builder.toString();
    }
  }
  
  protected static String getId(final Element element) {
    Integer id = EventNames.idMap.get(element);
    if ((id == null)) {
      EventNames.idMap.put(element, Integer.valueOf(EventNames.counter));
      int _plusPlus = EventNames.counter++;
      id = Integer.valueOf(_plusPlus);
    }
    return String.format("%05d", id);
  }
}
