/*****************************************************************************
 * Copyright (c) 2022 CEA LIST, OBEO
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  OBEO - Initial API and implementation
 *****************************************************************************/
package org.eclipse.papyrus.uml.domain.services.drop.diagrams;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.ECrossReferenceAdapter;
import org.eclipse.papyrus.uml.domain.services.IEditableChecker;
import org.eclipse.papyrus.uml.domain.services.drop.IInternalSourceToRepresentationDropBehaviorProvider;
import org.eclipse.papyrus.uml.domain.services.modify.ElementFeatureModifier;
import org.eclipse.papyrus.uml.domain.services.modify.IFeatureModifier;
import org.eclipse.papyrus.uml.domain.services.status.State;
import org.eclipse.papyrus.uml.domain.services.status.Status;
import org.eclipse.uml2.uml.BehavioredClassifier;
import org.eclipse.uml2.uml.Collaboration;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.FunctionBehavior;
import org.eclipse.uml2.uml.InformationItem;
import org.eclipse.uml2.uml.Interaction;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.Package;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.ProtocolStateMachine;
import org.eclipse.uml2.uml.StateMachine;
import org.eclipse.uml2.uml.StructuredClassifier;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.util.UMLSwitch;

/**
 * Drop behavior provider of a diagram element to a Composite Structure Diagram
 * Element.
 * 
 * @author Jessy MALLET
 *
 */
public class CompositeStructureInternalSourceToRepresentationDropBehaviorProvider
        implements IInternalSourceToRepresentationDropBehaviorProvider {

    @Override
    public Status drop(EObject droppedElement, EObject oldContainer, EObject newContainer,
            ECrossReferenceAdapter crossRef, IEditableChecker editableChecker) {
        return new StructureCompositeDropOutsideRepresentationBehaviorProviderSwitch(oldContainer, newContainer,
                crossRef, editableChecker).doSwitch(droppedElement);
    }

    static class StructureCompositeDropOutsideRepresentationBehaviorProviderSwitch extends UMLSwitch<Status> {

        private static final String OWNED_ATTRIBUTE = UMLPackage.eINSTANCE.getStructuredClassifier_OwnedAttribute()
                .getName();

        private static final String NESTED_CLASSIFIER_REF = UMLPackage.eINSTANCE.getClass_NestedClassifier().getName();

        private static final String PACKAGED_ELEMENT_REF = UMLPackage.eINSTANCE.getPackage_PackagedElement().getName();

        private final EObject oldContainer;

        private final EObject newContainer;

        private final ECrossReferenceAdapter crossRef;

        private final IEditableChecker editableChecker;

        StructureCompositeDropOutsideRepresentationBehaviorProviderSwitch(EObject oldContainer, EObject newContainer,
                ECrossReferenceAdapter crossRef, IEditableChecker editableChecker) {
            super();
            this.oldContainer = oldContainer;
            this.newContainer = newContainer;
            this.crossRef = crossRef;
            this.editableChecker = editableChecker;
        }

        @Override
        public Status caseClass(org.eclipse.uml2.uml.Class droppedClass) {
            Status dropStatus;
            IFeatureModifier modifier = new ElementFeatureModifier(this.crossRef, this.editableChecker);
            if (this.oldContainer != this.newContainer) {
                String refName = droppedClass.eContainmentFeature().getName();
                dropStatus = modifier.removeValue(this.oldContainer, refName, droppedClass);
                if (State.DONE == dropStatus.getState()) {
                    if (this.newContainer instanceof org.eclipse.uml2.uml.Class) {
                        dropStatus = modifier.addValue(this.newContainer, NESTED_CLASSIFIER_REF, droppedClass);
                    } else if (this.newContainer instanceof Package) {
                        dropStatus = modifier.addValue(this.newContainer, PACKAGED_ELEMENT_REF, droppedClass);
                    }
                }
                return dropStatus;
            }
            return super.caseClass(droppedClass);
        }

        @Override
        public Status caseCollaboration(Collaboration droppedCollaboration) {
            Status dropStatus;
            IFeatureModifier modifier = new ElementFeatureModifier(this.crossRef, this.editableChecker);
            if (this.oldContainer != this.newContainer) {
                String refName = droppedCollaboration.eContainmentFeature().getName();
                dropStatus = modifier.removeValue(this.oldContainer, refName, droppedCollaboration);
                if (State.DONE == dropStatus.getState()) {
                    if (this.newContainer instanceof org.eclipse.uml2.uml.Class) {
                        dropStatus = modifier.addValue(this.newContainer, NESTED_CLASSIFIER_REF, droppedCollaboration);
                    } else if (this.newContainer instanceof Package) {
                        dropStatus = modifier.addValue(this.newContainer, PACKAGED_ELEMENT_REF, droppedCollaboration);
                    }
                }
                return dropStatus;
            }
            return super.caseCollaboration(droppedCollaboration);
        }

        /**
         * Default Behavior : UML element can be D&D by using the same reference
         * containment.
         * 
         * @see org.eclipse.uml2.uml.util.UMLSwitch#caseElement(org.eclipse.uml2.uml.Element)
         *
         * @param droppedElement
         *                       the element to drop
         * @return OK or Failing status according to the complete D&D.
         */
        @Override
        public Status caseElement(Element droppedElement) {
            Status dropStatus;
            IFeatureModifier modifier = new ElementFeatureModifier(this.crossRef, this.editableChecker);
            if (this.oldContainer != this.newContainer) {
                String refName = droppedElement.eContainmentFeature().getName();
                if (this.oldContainer.eClass().getEStructuralFeature(refName) != null
                        && this.newContainer.eClass().getEStructuralFeature(refName) != null) {
                    dropStatus = modifier.removeValue(this.oldContainer, refName, droppedElement);
                    if (State.DONE == dropStatus.getState()) {
                        dropStatus = modifier.addValue(this.newContainer, refName, droppedElement);
                    }
                    return dropStatus;
                }
            }
            return super.caseElement(droppedElement);
        }

        @Override
        public Status caseFunctionBehavior(FunctionBehavior functionBehavior) {
            IFeatureModifier modifier = new ElementFeatureModifier(this.crossRef, this.editableChecker);
            if (this.oldContainer != this.newContainer) {
                return this.handleBehavioredClassifier(functionBehavior, modifier);
            }
            return super.caseFunctionBehavior(functionBehavior);
        }

        @Override
        public Status caseInformationItem(InformationItem droppedInformationItem) {
            Status dropStatus;
            IFeatureModifier modifier = new ElementFeatureModifier(this.crossRef, this.editableChecker);
            if (this.oldContainer != this.newContainer) {
                String refName = droppedInformationItem.eContainmentFeature().getName();
                dropStatus = modifier.removeValue(this.oldContainer, refName, droppedInformationItem);
                if (State.DONE == dropStatus.getState()) {
                    if (this.newContainer instanceof org.eclipse.uml2.uml.Class) {
                        dropStatus = modifier.addValue(this.newContainer, NESTED_CLASSIFIER_REF,
                                droppedInformationItem);
                    } else if (this.newContainer instanceof Package) {
                        dropStatus = modifier.addValue(this.newContainer, PACKAGED_ELEMENT_REF, droppedInformationItem);
                    }
                }
                return dropStatus;
            }
            return super.caseInformationItem(droppedInformationItem);
        }

        @Override
        public Status caseInteraction(Interaction interaction) {
            IFeatureModifier modifier = new ElementFeatureModifier(this.crossRef, this.editableChecker);
            if (this.oldContainer != this.newContainer) {
                return this.handleBehavioredClassifier(interaction, modifier);
            }
            return super.caseInteraction(interaction);
        }

        @Override
        public Status caseOpaqueBehavior(OpaqueBehavior opaqueBehavior) {
            IFeatureModifier modifier = new ElementFeatureModifier(this.crossRef, this.editableChecker);
            if (this.oldContainer != this.newContainer) {
                return this.handleBehavioredClassifier(opaqueBehavior, modifier);
            }
            return super.caseOpaqueBehavior(opaqueBehavior);
        }

        @Override
        public Status caseParameter(Parameter parameter) {
            Status dropStatus = new Status(State.FAILED, null);
            IFeatureModifier modifier = new ElementFeatureModifier(this.crossRef, this.editableChecker);
            if (this.oldContainer != this.newContainer) {
                String refName = parameter.eContainmentFeature().getName();
                if (this.newContainer instanceof org.eclipse.uml2.uml.Behavior) {
                    dropStatus = modifier.addValue(this.newContainer,
                            UMLPackage.eINSTANCE.getBehavior_OwnedParameter().getName(), parameter);
                }
                if (State.DONE == dropStatus.getState()) {
                    dropStatus = modifier.removeValue(this.oldContainer, refName, parameter);
                }
                return dropStatus;
            }
            return super.caseParameter(parameter);
        }

        @Override
        public Status caseProperty(Property droppedProperty) {
            IFeatureModifier modifier = new ElementFeatureModifier(this.crossRef, this.editableChecker);
            if (this.oldContainer != this.newContainer) {
                EObject oldSemanticContainer = droppedProperty.eContainer();
                if (oldSemanticContainer != null) {
                    String refName = droppedProperty.eContainingFeature().getName();
                    EObject newSemanticContainer = this.getPropertyNewSemanticContainer();
                    Status dropStatus = null;
                    if (newSemanticContainer != null) {
                        dropStatus = modifier.removeValue(oldSemanticContainer, refName, droppedProperty);
                        if (State.DONE == dropStatus.getState()) {
                            dropStatus = modifier.addValue(newSemanticContainer, OWNED_ATTRIBUTE, droppedProperty);
                        }
                    } else {
                        dropStatus = Status.createFailingStatus(
                                "Container should be a Structured Classifier or a Typed Property.");
                    }
                    return dropStatus;
                }
            }
            return super.caseProperty(droppedProperty);
        }

        @Override
        public Status caseProtocolStateMachine(ProtocolStateMachine protocolStateMachine) {
            IFeatureModifier modifier = new ElementFeatureModifier(this.crossRef, this.editableChecker);
            if (this.oldContainer != this.newContainer) {
                return this.handleBehavioredClassifier(protocolStateMachine, modifier);
            }
            return super.caseProtocolStateMachine(protocolStateMachine);
        }

        @Override
        public Status caseStateMachine(StateMachine stateMachine) {
            IFeatureModifier modifier = new ElementFeatureModifier(this.crossRef, this.editableChecker);
            if (this.oldContainer != this.newContainer) {
                return this.handleBehavioredClassifier(stateMachine, modifier);
            }
            return super.caseStateMachine(stateMachine);
        }

        private Status handleBehavioredClassifier(BehavioredClassifier behavioredClassifier,
                IFeatureModifier modifier) {
            Status dropStatus = new Status(State.FAILED, null);
            String refName = behavioredClassifier.eContainmentFeature().getName();
            if (this.newContainer instanceof org.eclipse.uml2.uml.BehavioredClassifier
                    && !(this.newContainer instanceof Collaboration)) {
                dropStatus = modifier.addValue(this.newContainer,
                        UMLPackage.eINSTANCE.getBehavioredClassifier_OwnedBehavior().getName(), behavioredClassifier);
            } else if (this.newContainer instanceof Package) {
                dropStatus = modifier.addValue(this.newContainer,
                        UMLPackage.eINSTANCE.getPackage_PackagedElement().getName(), behavioredClassifier);
            }
            if (State.DONE == dropStatus.getState()) {
                dropStatus = modifier.removeValue(this.oldContainer, refName, behavioredClassifier);
            }
            return dropStatus;
        }

        private EObject getPropertyNewSemanticContainer() {
            EObject newSemanticContainer = null;
            if (this.newContainer instanceof Property) {
                Type type = ((Property) this.newContainer).getType();
                if (type instanceof StructuredClassifier) {
                    newSemanticContainer = type;
                }
            } else if (this.newContainer instanceof StructuredClassifier) {
                newSemanticContainer = this.newContainer;
            }
            return newSemanticContainer;
        }
    }

}
