################################################################################
# Copyright (c) 2024-2025 Bayerische Motoren Werke Aktiengesellschaft (BMW AG)
#
# This program and the accompanying materials are made available under the
# terms of the Eclipse Public License 2.0 which is available at
# http://www.eclipse.org/legal/epl-2.0.
#
# SPDX-License-Identifier: EPL-2.0
################################################################################

"""
CoreInformation.json to simulationOutput.xml Converter

This module provides functionality to read a JSON file containing CoreInformation
and convert it into an XML structure conforming to the SimulationOutput format.
"""

import json
import re
import sys
import logging
import xml.etree.ElementTree as ET
from pathlib import Path
from jsonschema import validate, ValidationError
from pytest_optestrunner.utils import find_matching_file

class SimulationOutput:
    """
    A class used to convert a dictionary of simulation data into an XML file format, specifically
    `simulationOutput.xml`.
    """
    def __init__(self, dictionary, results_path):
        self.root_tag = 'SimulationOutput'  # Top-level tag for XML
        self.dictionary = dictionary
        self.results_path = results_path

    def handle_root_attributes(self, elem):
        '''Function to write the attributes of the root tag'''

        root_attributes_mapping = {"simulatorVersion" : "FrameworkVersion",
                                "schemaVersion": "SchemaVersion"}
        for key, val in root_attributes_mapping.items():
            if key in self.dictionary:
                elem.set(val, str(self.dictionary[key]))

    def resolve_scenery_file(self, root_xodr_files, subfolder_xodr_files):
        if len(root_xodr_files) == 1:
            return str(root_xodr_files[0].name)
        if len(root_xodr_files) == 0 and len(subfolder_xodr_files) == 1:
            return str(subfolder_xodr_files[0])
        if len(root_xodr_files) > 1 or (len(root_xodr_files) == 0 and len(subfolder_xodr_files) > 1):
            return "FOUND MORE THAN ONE SCENERY FILE (kind regards, mergescript)"

        return "NO SCENERY FILE GIVEN OR FOUND (kind regards, mergescript)"

    def handle_scenary_file(self, elem):
        '''Function to write the SceneryFile tag'''

        scenery_file_mapping = {"sceneryFile" : "SceneryFile"}
        scenery_file_tag = ET.SubElement(elem, scenery_file_mapping["sceneryFile"])

        if "sceneryFile" in self.dictionary:
            scenery_file_tag.text = self.dictionary["sceneryFile"]
            return

        root_xodr_files = list(self.results_path.glob('*.xodr'))
        subfolder_xodr_files =  [file.relative_to(self.results_path) for file in self.results_path.rglob('*.xodr')]

        scenery_file_tag.text = self.resolve_scenery_file(root_xodr_files, subfolder_xodr_files)

    def handle_run_statistics(self, elem, runresult):
        '''Function to write the RunStatistics tag'''

        run_statistics_tag = ET.SubElement(elem, "RunStatistics")

        random_seed_tag = ET.SubElement(run_statistics_tag, "RandomSeed")
        random_seed_tag.text = str(runresult["randomSeed"]) if "randomSeed" in runresult else "0"

        visibility_distance_tag = ET.SubElement(run_statistics_tag, "VisibilityDistance")
        visibility_distance_tag.text = str(runresult["visibilityDistance"]) if "visibilityDistance" in runresult else "0"

        stop_reason_tag = ET.SubElement(run_statistics_tag, "StopReason")
        stop_reason_tag.text = str(runresult["stopReason"]) if "stopReason" in runresult else "Not available"

        stop_time_tag = ET.SubElement(run_statistics_tag, "StopTime")
        stop_time_tag.text = str(runresult["stopTime"]) if "stopTime" in runresult else "0"

        ego_accident_tag = ET.SubElement(run_statistics_tag, "EgoAccident")
        ego_accident_tag.text = str(runresult["egoAccident"]) if "egoAccident" in runresult else "False"

        total_distance_traveled_tag = ET.SubElement(run_statistics_tag, "TotalDistanceTraveled")
        total_distance_traveled_tag.text = str(runresult["totalDistanceTraveled"]) if "totalDistanceTraveled" in runresult else "0"

        ego_distance_traveled_tag = ET.SubElement(run_statistics_tag, "EgoDistanceTraveled")
        ego_distance_traveled_tag.text = str(runresult["egoDistanceTraveled"]) if "egoDistanceTraveled" in runresult else "0"

    def create_event(self, event_data):
        """
        Create a legacy XML Event tag from validated event data
        """
        event = ET.Element(
            "Event",
            Time=str(event_data['time']),
            Source="Core",
            Name=event_data['description']
        )

        triggering_entities = ET.SubElement(event, "TriggeringEntities")
        ET.SubElement(event, "AffectedEntities")
        if 'entityIds' in event_data:
            [ET.SubElement(triggering_entities, "Entity", Id=str(entity_id)) for entity_id in event_data['entityIds']]

        parameters = ET.SubElement(event, "Parameters")
        if 'properties' in event_data:
            [ET.SubElement(parameters, "Parameter", Key=key, Value=value)
             for key, value in event_data['properties'].items()]

        return event

    def handle_events(self, elem, run_id):
        """
        Search for a matching Events_Run_{run_id}.json file,
        validate it against the EventsSchema,
        and transforms it to the legacy XML Tag
        """
        events_tag = ET.SubElement(elem, "Events")
        for events_file in self.results_path.glob('*.json'):
            run_id_match = re.search(r'Events_Run_(?P<run_id>\d+).json', events_file.name)
            if run_id_match and int(run_id_match.group('run_id')) == run_id:
                json_data = read_json(events_file)
                validate_json(json_data, read_json(Path("schemas/EventsSchema.json")))
                [events_tag.append(self.create_event(event_data)) for event_data in json_data]
                return
        logging.debug(f"No Events_Run_{run_id}.json file is available at {self.results_path}")

    def handle_agents(self, elem, runresult):
        '''Function to write the Agents tag'''

        agents_tag = ET.SubElement(elem, "Agents")
        for each_agent in runresult["entities"]:
            agent_tag = ET.SubElement(agents_tag, "Agent")
            agent_tag.set("Id", str(each_agent["id"]))
            agent_tag.set("AgentName", str(each_agent["name"]))

    def handle_cyclics(self, elem, run_id):
        """
        Search for a matching Cyclics_Run_{run_id}.csv file,
        and add its name to the Cyclics tag.
        """
        cyclics_tag = ET.SubElement(elem, "Cyclics")
        cyclics_file = find_matching_file(self.results_path, rf'Cyclics_Run_0*{run_id}\.csv')
        cyclics_file_tag = ET.SubElement(cyclics_tag, "CyclicsFile")
        cyclics_file_tag.text = str(cyclics_file)
        return

    def handle_each_runresults(self, elem):
        '''Function to write each RunResult tag'''

        for runresult in self.dictionary["simulationRuns"]:
            runresult_tag = ET.SubElement(elem, 'RunResult')
            runresult_tag.set("RunId", str(runresult["id"]))
            self.handle_run_statistics(runresult_tag, runresult)
            self.handle_events(runresult_tag, runresult["id"])
            self.handle_agents(runresult_tag, runresult)
            self.handle_cyclics(runresult_tag, runresult["id"])

    def build(self):
        """Turn a dictionary into XML"""

        elem = ET.Element(self.root_tag)
        self.handle_root_attributes(elem)
        self.handle_scenary_file(elem)
        runresults_tag = ET.SubElement(elem, 'RunResults')
        self.handle_each_runresults(runresults_tag)
        return elem

def read_json(json_file: Path):
    """Function to read the json file"""

    current_dir = Path(__file__).parent
    full_path = current_dir / json_file
    with open(full_path, 'r', encoding="utf-8") as f:
        return json.load(f)

def validate_json(json_data, schema_file=None):
    """Validate Json file with the corresponding schema"""
    try:
        validate(instance=json_data, schema=schema_file)
    except ValidationError as e:
        logging.debug(f"Validation failed: {e.message}")
        sys.exit(1)

def convert_json_to_xml(json_file_path: Path):
    """Function to convert the json to xml conforming to the simulationOutput requirements"""

    json_data = read_json(json_file_path)
    validate_json(json_data, read_json(Path("schemas/CoreInformationSchema.json")))
    simulation_output = SimulationOutput(json_data, json_file_path.parent)
    return simulation_output.build()
