#!/usr/bin/env python3
################################################################################
# Copyright (c) 2024 Bayerische Motoren Werke Aktiengesellschaft (BMW AG)
#
# This program and the accompanying materials are made available under the
# terms of the Eclipse Public License 2.0 which is available at
# http://www.eclipse.org/legal/epl-2.0.
#
# SPDX-License-Identifier: EPL-2.0
################################################################################

import re
import json
import sys
import xml.etree.ElementTree as ET
import shutil
import logging
from pytest_optestrunner import chore_tasks
from pytest_optestrunner.utils import merge_multiple_CoreInformation
from pytest_optestrunner.path_manager import PathManager
from pathlib import Path
from pytest_optestrunner.convert_json2xml import convert_json_to_xml, read_json, validate_json

logger = logging.getLogger(__file__)

USER_DEFINED_CONTROLLER_NAME = "DefaultSystem"
CONTROLLER_TAG = 'entityProfile'

def camel_to_pascal(string):
    if len(string) == 0:
        return string  # Return the empty string if input is empty
    return string[0].upper() + string[1:] if len(string) > 1 else string.upper()

def adjust_sensor_data(string):
    mapping = {
        "mountingPositionLongitudinal": "MountingPosLongitudinal",
        "mountingPositionLateral": "MountingPosLateral",
        "mountingPositionsHeigh": "MountingPosHeight"
    }
    return mapping.get(string, camel_to_pascal(string))

def to_string(value):
    if isinstance(value, float):
        return f"{value:.6f}"  # Format float to six decimal places
    return str(value)  # Convert other types to string

def collect_json_data(run_id_path):
    events_files = []
    vehicle_properties_files = []

    for entity in sorted(run_id_path.iterdir()):
        if entity.is_dir() and re.match(r'entity\d+', entity.name):
            vehicle_properties = []
            for component in entity.iterdir():
                if component.is_dir() and re.match(r'(?P<component>[^/]+)', component.name):
                    for json_file in component.glob('**/*.json'):
                        if json_file.is_file() and re.search(r'Events.json', str(json_file)):
                            events_files.append(json_file)
                        if json_file.is_file() and re.search(r'EntityProperties.json', str(json_file)):
                            vehicle_properties.append(json_file)

            if len(vehicle_properties) <= 1: # default controller only
                vehicle_properties_files.extend(vehicle_properties)
            elif len(vehicle_properties) == 2: # default controller + user defined controller
                for file in vehicle_properties:
                    with open(file, 'r') as f:
                        data = json.load(f)
                        # "DefaultSystem" is historically the name for the DefaultSystem in opSimulation
                        if data.get(CONTROLLER_TAG) != USER_DEFINED_CONTROLLER_NAME:
                            vehicle_properties_files.append(file)
            elif len(vehicle_properties) >= 3:
                logger.error(f"Found {len(vehicle_properties)} EntityProperties.json files in {entity.name}. Expected one default and/or one user-defined controler (max. 2).")

    return events_files, vehicle_properties_files

def check_attribute(xml_tag, xml_attr, json_data, json_attr):
    if xml_tag.attrib.get(xml_attr) and json_data.get(json_attr) and xml_tag.attrib[xml_attr] != json_data[json_attr]:
        logger.error(
            f"Attribute value mismatch detected:\n"
            f"- simulationOutput.xml: Attribute '{xml_attr}' has value '{xml_tag.attrib[xml_attr]}'\n"
            f"- EntityProperties.json: Attribute '{json_attr}' has value '{json_data[json_attr]}'")

def update_or_create_agent(agent_data, existing_agent=None, agent_id=None):
    agent_attr = {
        'AgentTypeName': CONTROLLER_TAG,
        'VehicleModelType': 'model',
        'AgentType': 'entityType'
    }

    agent = existing_agent if existing_agent is not None else ET.Element("Agent", Id=str(agent_id))
    for attr, data_key in agent_attr.items():
        check_attribute(agent, attr, agent_data, data_key)
        agent.set(attr, agent_data.get(data_key, agent.get(attr)))

    driver_profile = agent_data.get('driverProfile', '')
    if driver_profile:
        check_attribute(agent, 'DriverProfileName', agent_data, 'driverProfile')
    agent.set('DriverProfileName', driver_profile)

    vehicle_attributes = agent.find("VehicleAttributes")
    if vehicle_attributes is None:
        vehicle_attributes = ET.SubElement(agent, "VehicleAttributes")
    for attr in ['width', 'length', 'height', 'longitudinalPivotOffset']:
        check_attribute(agent, attr, agent_data, attr)
        vehicle_attributes.set(camel_to_pascal(attr), to_string(agent_data.get(attr, vehicle_attributes.get(attr))))

    if 'components' in agent_data:
        components = ET.SubElement(agent, "Components")
        [ET.SubElement(components, "Component", Type=type, Profile=profile)
         for type, profile in agent_data['components'].items()]

    if 'sensors' in agent_data:
        sensors = ET.SubElement(agent, "Sensors")
        [ET.SubElement(sensors, "Sensor", **{adjust_sensor_data(k): to_string(v) for k, v in sensor_info.items()})
         for sensor_info in agent_data['sensors']]

    return agent

def create_event_from_file(event_data, events_file: Path):
    event = ET.Element(
        "Event",
        Time=str(event_data['time']),
        Source=events_file.parent.name,
        Name=event_data['description']
    )

    pattern = r"entity(\d{4})"
    grandparent_dir = events_file.parent.parent.name
    entity = str(int(re.match(pattern, grandparent_dir).group(1)))

    triggering_entities = ET.SubElement(event, "TriggeringEntities")
    ET.SubElement(triggering_entities, "Entity", Id=entity)

    affected_entities = ET.SubElement(event, "AffectedEntities")
    ET.SubElement(affected_entities, "Entity", Id=entity)

    parameters = ET.SubElement(event, "Parameters")
    if 'properties' in event_data:
        [ET.SubElement(parameters, "Parameter", Key=key, Value=value)
         for key, value in event_data['properties'].items()]

    return event

def get_entity_id(filename: Path):
    entity_id_match = re.search(r'entity(\d+)', str(filename))
    if not entity_id_match:
       raise ValueError(f"Entity ID not found in filename: {filename}")
    return int(entity_id_match.group(1))

def add_agents_to_xml(vehicle_properties, agents, vehicle_properties_schema):
    for file in vehicle_properties:
        with open(file, 'r') as f:
            agent_data = json.load(f)
            validate_json(agent_data, vehicle_properties_schema)
            agent_id = get_entity_id(file)
            existing_agent = agents.find(f"./Agent[@Id='{agent_id}']")
            if existing_agent is None:
                agents.append(update_or_create_agent(agent_data, agent_id=agent_id))
            else:
                update_or_create_agent(agent_data, existing_agent=existing_agent)

def add_events_to_xml(events_files, events, events_schema):
    for events_file in events_files:
        with open(events_file, 'r') as json_file:
            json_data = json.load(json_file)
            validate_json(json_data, events_schema)
            [events.append(create_event_from_file(event_data, events_file)) for event_data in json_data]

def clear_sources(json_file_paths):
    for file_path in json_file_paths:
        if file_path.exists():
            file_path.unlink()

def write_xml_tree(element: ET.Element, output_file: Path, indent=None):
    if indent is None:
        indent = '    '
    output_file.parent.mkdir(parents=True, exist_ok=True)
    xml_tree = ET.ElementTree(element)
    ET.indent(xml_tree, indent)
    xml_tree.write(output_file, encoding='UTF-8', xml_declaration=True, short_empty_elements=True)

def extract_json_data(results_path: Path, sim_output: Path, run_id_path: Path, keep_source: bool):
    sim_output = ET.parse(sim_output).getroot()
    run_id_match = re.search(r'run(?P<run_id>\d+)', run_id_path.name)
    run_result = next((rr for rr in sim_output.findall('.//RunResult') if int(rr.get("RunId")) == int(run_id_match.group('run_id'))), None)
    events = run_result.find('.//Events')
    agents = run_result.find('.//Agents')

    events_files, vehicle_properties_files = collect_json_data(run_id_path)
    add_agents_to_xml(vehicle_properties_files, agents, read_json(Path("schemas/EntityPropertiesSchema.json")))
    add_events_to_xml(events_files, events, read_json(Path("schemas/EventsSchema.json")))

    logger.debug(f'Writing merged data to "{sim_output}"')
    write_xml_tree(sim_output, results_path / 'simulationOutput.xml')

    if keep_source == False:
        clear_sources(events_files)
        clear_sources(vehicle_properties_files)

def merge(results_path: Path, backup_dst_files: bool, keep_source: bool):
    sim_output = results_path / 'simulationOutput.xml'
    core_information = results_path / 'CoreInformation.json'

    if sim_output.exists() and core_information.exists():
        logger.error('Unable to proceed: Both "simulationOutput.xml" and "CoreInformation.json" are detected')
        sys.exit(1)

    if core_information.exists():
        converted_xml_data = convert_json_to_xml(core_information)
        write_xml_tree(converted_xml_data, results_path / 'simulationOutput.xml', indent='  ')

    if sim_output.exists() and backup_dst_files:
        logger.debug(f'Creating backup of {sim_output} as {sim_output.with_suffix(".orig")}')
        shutil.copyfile(sim_output, sim_output.with_suffix('.orig'))

    for run_id_path in PathManager.get_run_ids(results_path):
        if run_id_path:
            extract_json_data(results_path, sim_output, run_id_path, keep_source)

if __name__ == "__main__":
    args = chore_tasks.parse_args()
    chore_tasks.init_logger(f'{__file__.replace(".py", ".log")}', args.debug)
    merge_multiple_CoreInformation(args.results_path)
    merge(args.results_path, args.backup, args.keep_source)
