################################################################################
# Copyright (c) 2025 Bayerische Motoren Werke Aktiengesellschaft (BMW AG)
#
# This program and the accompanying materials are made available under the
# terms of the Eclipse Public License 2.0 which is available at
# http://www.eclipse.org/legal/epl-2.0.
#
# SPDX-License-Identifier: EPL-2.0
################################################################################
import pytest
import configparser
from pathlib import Path
from tempfile import TemporaryDirectory
from pytest_optestrunner.user_settings_manipulator import CasePreservingConfigParser, IniFileParser

def test_case_preserving_config_parser():
    config = CasePreservingConfigParser()
    config.add_section("TestSection")
    config.set("TestSection", "Key", "Value")

    # Assert the case of the key is preserved
    assert "Key" in config.options("TestSection")
    assert config.get("TestSection", "Key") == "Value"

# Test IniFileParser
@pytest.fixture
def temp_ini_file():
    """Create a temporary .ini file for testing."""
    with TemporaryDirectory() as temp_dir:
        temp_file_path = Path(temp_dir) / "test.ini"
        with temp_file_path.open('w') as file:
            file.write("""[Section1]
Key1=Value1
Key2=Value2
""")
        yield temp_file_path

# Test successful initialization of IniFileParser
def test_ini_file_parser_initialization(temp_ini_file):
    parser = IniFileParser(temp_ini_file)
    assert parser.file_path == temp_ini_file
    assert isinstance(parser.config, configparser.ConfigParser)

# Test initialization with a non-existent file
def test_ini_file_parser_file_not_found():
    with pytest.raises(FileNotFoundError):
        IniFileParser(Path("non_existent.ini"))

# Test update_path with a valid key
def test_update_path_valid_key(temp_ini_file):
    parser = IniFileParser(temp_ini_file)
    parser.update_path("NewValue", "Key1")

    # Verify that the value has been updated
    parser.config.read(temp_ini_file)
    assert parser.config.get("Section1", "Key1") == "NewValue"

# Test update_path with a missing key
def test_update_path_missing_key(temp_ini_file):
    parser = IniFileParser(temp_ini_file)

    with pytest.raises(KeyError):
        parser.update_path("NewValue", "NonExistentKey")

# Test update_path when no sections are present
def test_update_path_no_sections():
    with TemporaryDirectory() as temp_dir:
        temp_file_path = Path(temp_dir) / "empty.ini"
        temp_file_path.touch()  # Create an empty file

        parser = IniFileParser(temp_file_path)

        with pytest.raises(KeyError):
            parser.update_path("NewValue", "Key1")
