################################################################################
# Copyright (c) 2025 Bayerische Motoren Werke Aktiengesellschaft (BMW AG)
#
# This program and the accompanying materials are made available under the
# terms of the Eclipse Public License 2.0 which is available at
# http://www.eclipse.org/legal/epl-2.0.
#
# SPDX-License-Identifier: EPL-2.0
################################################################################

import re
import json
from pathlib import Path
import shutil

def find_matching_file(directory: Path, pattern: str) -> str:
    """Helper function to find files matching the given regex pattern."""
    matching_files = [f.name for f in Path(directory).iterdir() if f.is_file() and re.search(pattern, f.name)]
    if len(matching_files) != 1:
        raise FileNotFoundError(
            f"Either no file matching '{pattern}' was found, \
            or multiple files were found in the '{directory}' folder."
        )
    return matching_files[0]

def merge_multiple_CoreInformation(directory: Path):
    pattern = re.compile(r"CoreInformation_\d+\.json")
    json_files = [file for file in directory.iterdir() if file.is_file() and pattern.match(file.name)]

    if len(json_files) > 1:

        merged_data = None 
        merged_simulation_runs = []

        for file in json_files:
            run_id_match =re.search(r'CoreInformation_(?P<run_id>\d+)\.json',file.name)
            run_id = int(run_id_match.group("run_id"))
            try:
                data = json.loads(file.read_text(encoding="utf-8"))

                if merged_data is None:
                    merged_data = {key: value for key, value in data.items() if key != "simulationRuns"}

                if "simulationRuns" in data:
                    data["simulationRuns"][0]["id"] = run_id
                    merged_simulation_runs.extend(data["simulationRuns"])

            except json.JSONDecodeError as e:
                print(f"Error reading {file.name}: {e}")
    
        merged_data["simulationRuns"] = merged_simulation_runs

        merged_file = directory / "CoreInformation.json"
        merged_file.write_text(json.dumps(merged_data, indent=4), encoding="utf-8")

    elif len(json_files) == 1:
        single_file = json_files[0]
        new_file = directory / "CoreInformation.json"
        shutil.copy(single_file, new_file)
