################################################################################
# Copyright (c) 2020-2021 in-tech GmbH
#               2022-2025 Bayerische Motoren Werke Aktiengesellschaft (BMW AG)
#
# This program and the accompanying materials are made available under the
# terms of the Eclipse Public License 2.0 which is available at
# http://www.eclipse.org/legal/epl-2.0.
#
# SPDX-License-Identifier: EPL-2.0
################################################################################

import logging
import sys
import os.path
from lxml import etree
from pathlib import Path
from pytest_optestrunner.xml_definitions import XmlNode, XmlSetter

def terminate_program(message):
    logging.error(message)
    sys.exit(message)


def get_file_or_default(config_under_test, file):
    """If avilable, returns the file from the current config path, else from the default path"""
    in_config_folder = os.path.join(
        config_under_test.base_path, config_under_test.name, file)
    if os.path.isfile(in_config_folder):
        return in_config_folder

    in_default_folder = os.path.join(config_under_test.default_path, file)
    if os.path.isfile(in_default_folder):
        return in_default_folder

    terminate_program(
        f'No file "{file}" in "{config_under_test.base_path}" or "{config_under_test.default_path}"')


def set(base_path, xml_setter: XmlSetter):
    """ Sets the node given by the (xml_file, xpath_expr) to value """
    (expr, tree, nodes) = get_node(base_path, xml_setter.dest)
    for node in nodes:
        value = str(xml_setter.value).replace('${configFolder}', str(base_path))

        if len(expr) == 1:
            node.text = value
        else:
            node.attrib[expr[1]] = value

        xml_file = Path(base_path) / xml_setter.dest.xml_file
        with open(xml_file, 'wb') as f:
            f.write(etree.tostring(tree, encoding="UTF-8", xml_declaration=True, pretty_print=True))


def get_node(base_path, xml_node: XmlNode):
    """ Gets the node for further processing """
    xml_file = os.path.join(base_path, xml_node.xml_file)

    tree = etree.parse(xml_file)
    expr = xml_node.xpath_expr.split('/@')
    try:
        nodes = tree.xpath(expr[0])
    except IndexError:
        terminate_program(
            f'Unable to execute xpath expression "{xml_node.xpath_expr}" on "{xml_file}"')

    return (expr, tree, nodes)


class XmlUtil:
    """
    Utilily for xml manipulation w.r.t. openPASS configurations
    Also defines some manipulation node presets starting with 'CONFIG_*'
    """
    CONFIG_PATH = XmlNode(
        'Scenarios/XOSC/Scenario.xosc', "//ScenarioObject/ObjectController/Controller/Properties/Property[@name='simulator::paths::config']/@value", str)
    LIB_PATH = XmlNode(
        'Scenarios/XOSC/Scenario.xosc', "//ScenarioObject/ObjectController/Controller/Properties/Property[@name='simulator::paths::lib']/@value", str)
    CONFIG_DURATION = XmlNode(
        'Scenarios/XOSC/Scenario.xosc', "//StopTrigger/ConditionGroup/Condition/ByValueCondition/SimulationTimeCondition/@value", int)

    @staticmethod
    def get(config_under_test, xml_node: XmlNode):
        """"Retrieves the value of the node given by an XPATH expression"""
        expr, _, node = get_node(config_under_test, xml_node)
        datatype = str if xml_node.datatype is None else xml_node.datatype
        return datatype(node.text) if len(expr) == 1 else datatype(node.attrib[expr[1]])

    @staticmethod
    def update(configs_path, config: dict):
        """Updates configuration files to log into CSV and set the values defined by the given dictionary

        Keyword arguments:
        configs_path -- Path to config set, which shall be manipulated
        config       -- dictionary for updating the values
                        Mandatory:  invocations, duration
                        Optional:   random seed
        """
        set(configs_path,
            XmlSetter(XmlUtil.CONFIG_DURATION, config["duration"]))

        #optional fields:
        if XmlUtil.CONFIG_PATH.node_exists(configs_path):
            set(configs_path,
                XmlSetter(XmlUtil.CONFIG_PATH, config["config_path"]))

        if XmlUtil.LIB_PATH.node_exists(configs_path):
            set(configs_path,
                XmlSetter(XmlUtil.LIB_PATH, config["lib_path"]))

    @staticmethod
    def custom_update(configs_path, xml_setter: XmlSetter):
        """Updates a single configuration files based on the given xml_setter

        Keyword arguments:
        configs_path -- Path to config set, which shall be manipulated
        xml_setter   -- A valid xml setter
        """
        set(configs_path, xml_setter)
