################################################################################
# Copyright (c) 2021-2025 Bayerische Motoren Werke Aktiengesellschaft (BMW AG)
#
# This program and the accompanying materials are made available under the
# terms of the Eclipse Public License 2.0 which is available at
# http://www.eclipse.org/legal/epl-2.0.
#
# SPDX-License-Identifier: EPL-2.0
################################################################################

from copy import copy
from dataclasses import dataclass, replace
import logging
import pickle
import shutil
import sys
from typing import List
from pytest_optestrunner.merge_csv2csv import extract_agent_data_csv
from pytest_optestrunner.merge import extract_json_data, write_xml_tree
from pytest_optestrunner.path_manager import PathManager, SimulatorBasePaths, ConfigSource
from pytest_optestrunner.runner import Runner
from pytest_optestrunner.config_modulator import ConfigModulator
from pytest_optestrunner.config_parser import TestItem
from pytest_optestrunner.path_hasher import generate_hash
from pytest_optestrunner.convert_json2xml import convert_json_to_xml
from pytest_optestrunner.user_settings_manipulator import IniFileParser
from pytest_optestrunner.convert_json2xml import convert_json_to_xml
from pytest_optestrunner.merge import write_xml_tree
from pytest_optestrunner.utils import merge_multiple_CoreInformation
from filelock import FileLock
from pathlib import Path

logger = logging.getLogger(__file__)

@dataclass(init=True, frozen=True)
class SimulationResult:
    exit_code: int
    ram_usage: float
    result_path: str

class CacheObjectAlreadyExists(Exception):
    pass

class ResultCache:
    def __init__(self, base_path):
        self.cache_path = Path(base_path / PathManager.PYOPENPASS_CACHE)

    def _get_file(self, key):
        return self.cache_path / key

    def get(self, key) -> SimulationResult:
        cache_file = self._get_file(key)
        if cache_file.exists():
            with open(cache_file, 'rb') as file:
                return pickle.load(file)

    def set(self, key, result: SimulationResult) -> None:
        cache_file = self._get_file(key)
        if cache_file.exists():
            raise CacheObjectAlreadyExists(
                f"Cannot set {cache_file} because it already exists. Clear cache before using it.")
        with open(cache_file, 'wb') as file:
            pickle.dump(result, file)

class Simulator():
    def __init__(self, simulator_base_paths: SimulatorBasePaths, mutual_config_path: str, resources_path: str, fmu_path: str):
        self.simulator_base_paths = simulator_base_paths
        self.resultCache = ResultCache(simulator_base_paths.results)
        self.runner = Runner(simulator_base_paths.base_path, simulator_base_paths.executable)
        self.mutual_config_path = mutual_config_path
        self.resources_path = resources_path
        self.fmu_path = fmu_path

    def _handle_simulation_output(self, paths: PathManager, backup: bool, keep_source: bool):
        sim_output = paths.results / 'simulationOutput.xml'
        core_information = paths.results / 'CoreInformation.json'

        if sim_output.exists() and core_information.exists():
            logger.error('Unable to proceed: Both "simulationOutput.xml" and "CoreInformation.json" were detected')
            sys.exit(1)

        merge_multiple_CoreInformation(paths.results)

        if core_information.exists():
            converted_xml_data = convert_json_to_xml(core_information)
            write_xml_tree(converted_xml_data, paths.results / 'simulationOutput.xml', '  ')

        if sim_output.exists() and backup:
            logger.debug("Creating backup of %s as %s",
                         {sim_output}, {sim_output.with_suffix(".orig")})
            shutil.copyfile(sim_output, sim_output.with_suffix('.orig'))

        for run_id_path in PathManager.get_run_ids(paths.results):
            if run_id_path:
                component_paths = paths.scan_for_additional_controller_output(run_id_path)
                extract_agent_data_csv(paths.results, component_paths, run_id_path.name, backup, keep_source)
                extract_json_data(paths.results, sim_output, run_id_path, keep_source)

    def _apply_config_and_run(self, paths: PathManager, test_item) -> SimulationResult:
        # for the hashing we need the actual configuration, but already modified
        # if it is identical to a config we already know, the result is being reused
        paths.collect_config()
        ConfigModulator.apply(test_item, paths.configs, paths.plugins_path)
        ConfigModulator.write_optestrunner_info(test_item, paths, self.runner.executable)
        config_hash = generate_hash(paths.configs)

        # for keeping reusing results, we keep them in a folder named after the hash
        # telling the pathmanager is necessary for collecting the artifacts later on
        paths.set_results_subfolder(config_hash)
        user_settings_parser = IniFileParser(paths.configs / Path("UserSettings/UserSettings.ini"))
        user_settings_parser.update_path(str(paths.results), 'OutputDirectoryPath')
        user_settings_parser.update_path(f"{{{str(paths.libgecco_path)}}}", 'Plugins')
        # for working with multiple threads (xdist)
        # the thread actually simulating, locks the resource
        with FileLock(paths.cache_lock(config_hash)):
            sim_result = self.resultCache.get(config_hash)
            if not sim_result:
                paths.clear_results()
                runner_result = self.runner.execute(paths.configs, paths.results, test_item.invocations, test_item.random_seed)

                self._handle_simulation_output(paths, backup=True, keep_source=True)

                sim_result = SimulationResult(*runner_result, str(paths.results))
                self.resultCache.set(config_hash, sim_result)

        return sim_result

    def _run_determinism(self, test_item: TestItem) -> List[SimulationResult]:
        simulation_results = [self._run(test_item)]
        for run in range(test_item.invocations):
            single_run_test_item = copy(test_item)
            single_run_test_item.name = f'Rep{run:02d}'
            single_run_test_item.invocations = 1
            single_run_test_item.random_seed_offset = run
            single_run_test_item.random_seed += run
            simulation_results.append(self._run(single_run_test_item))
        return simulation_results

    def _run(self, test_item: TestItem) -> SimulationResult:
        config_source = ConfigSource(
            self.mutual_config_path, self.resources_path, self.fmu_path, test_item.config)
        paths = PathManager(self.simulator_base_paths, config_source, test_item.nodeid, test_item.id)
        simulation_result = self._apply_config_and_run(paths, test_item)
        paths.collect_artifacts()
        return replace(simulation_result, result_path=str(paths.artifacts))

    def run(self, test_item: TestItem) -> List[SimulationResult]:
        if hasattr(test_item, 'determinism') and test_item.determinism:
            return self._run_determinism(test_item)
        else:
            return [self._run(test_item)]
