################################################################################
# Copyright (c) 2021-2025 Bayerische Motoren Werke Aktiengesellschaft (BMW AG)
#
# This program and the accompanying materials are made available under the
# terms of the Eclipse Public License 2.0 which is available at
# http://www.eclipse.org/legal/epl-2.0.
#
# SPDX-License-Identifier: EPL-2.0
################################################################################

import os
from shutil import copyfile, rmtree
import re
from pathlib import Path


class ConfigSource():
    def __init__(self, common_path: str, config_base_path: str, fmu_path: str, config_dir: str):
        self.common = common_path
        self.config_full_path = os.path.join(config_base_path, config_dir)
        if not os.path.exists(self.config_full_path):
            raise Exception(f'Config folder does not exist: {self.config_full_path}')
        self.fmu_path = Path(fmu_path) if fmu_path else None
        self.fmu_name = config_dir


class SimulatorBasePaths():
    LOG_FILE_NAME = 'test.log'

    def __init__(
        self,
        base_path: str,
        executable: str,
        input: str = "",
        output: str = "",
        artifacts: str = "",
        libgecco_path: str = "",
        plugins_path: str = "",
    ):
        self.executable = Path(executable)
        self.base_path = Path(base_path).absolute() if base_path else self.executable.parent
        self.configs = Path(input) if Path(input).is_absolute() else self.base_path / input
        self.results = Path(output) if Path(output).is_absolute() else self.base_path / output
        self.artifacts = Path(artifacts) if Path(artifacts).is_absolute() else self.base_path / artifacts
        self.plugins_path = Path(plugins_path)
        self.libgecco_path = Path(libgecco_path) if libgecco_path else self.plugins_path


class SimulatorPaths():
    def __init__(self, base_paths: SimulatorBasePaths, config_subdirectory, artifacts_subdirectory):
        self.configs = base_paths.configs / config_subdirectory
        self.results = base_paths.results
        self.artifacts = base_paths.artifacts / artifacts_subdirectory


class PathManager:
    PYOPENPASS_CACHE = ".cache"

    def __init__(self, simulation_base_paths: SimulatorBasePaths, config_source: ConfigSource, nodeid: str, id: str):
        simulator_paths = SimulatorPaths(
            simulation_base_paths,
            config_subdirectory=self.as_path(f'{nodeid}::{id}'),
            artifacts_subdirectory=self.as_path(f'{nodeid}::{id}'))
        self.config_source = config_source
        self.results_subfolder = None
        self.results_base_folder = simulator_paths.results
        self.configs = simulator_paths.configs
        self.artifacts = simulator_paths.artifacts
        self.libgecco_path = simulation_base_paths.libgecco_path
        self.plugins_path = simulation_base_paths.plugins_path

    def set_results_subfolder(self, folder):
        if not self.results_subfolder:
            self.results_subfolder = folder
        else:
            raise Exception("results_subfolder already set")

    def cache_lock(self, hash):
        return str(self.results_base_folder / self.PYOPENPASS_CACHE / f"{hash}.lock")

    @property
    def results(self):
        if not self.results_subfolder:
            raise Exception("results_subfolder not set")
        return self.results_base_folder / self.results_subfolder

    @property
    def logfile(self):
        return self.results / SimulatorBasePaths.LOG_FILE_NAME

    @staticmethod
    def as_path(string) -> Path:
        return Path(re.sub(r':+|@r|@+|<+|>+|"+|\\+|\|+|\?+|\*+', '_', string.replace('::', '/')))

    @staticmethod
    def _copydir(source, dest, pattern: str = None):
        """Copy a directory structure overwriting existing files"""
        for root, _, files in os.walk(source):
            if not os.path.isdir(root):
                os.makedirs(root)

            for file in files:
                if pattern and not re.match(pattern, file):
                    continue

                rel_path = root.replace(str(source), '').lstrip(os.sep)
                dest_path = os.path.join(dest, rel_path)

                if not os.path.isdir(dest_path):
                    os.makedirs(dest_path)

                copyfile(os.path.join(root, file),
                         os.path.join(dest_path, file))

    @staticmethod
    def clean_and_create(folder):
        if os.path.isfile(folder) or os.path.islink(folder):
            os.unlink(folder)

        if os.path.exists(folder):
            rmtree(folder, ignore_errors=True)

        os.makedirs(folder)

    def _copy_fmu_resources(self):
        """
        Copies FMU resources from possible source locations to the configuration directory.
        It checks for sources with different priorities:
        1. A specific FMU path derived from the config name (recursive search for gecco FMU's)
        2. A base FMU path for shared FMUs (non-recursive search for SCM FMU).
        The first source found with valid FMU/SSP files will be used.
        """
        destination_fmu_path = self.configs
        possible_sources = [
            (self.config_source.fmu_path / self.config_source.fmu_name, '**/*'),  # Priority 1: Recursive
            (self.config_source.fmu_path, '*')  # Priority 2: Non-recursive
        ]

        for source_path, glob_pattern in possible_sources:
            if source_path and source_path.exists():
                fmu_files = list(source_path.glob(f'{glob_pattern}.fmu')) + \
                            list(source_path.glob(f'{glob_pattern}.ssp'))
                if fmu_files:
                    self._copydir(source_path, destination_fmu_path, pattern=r'.*\.(fmu|ssp)$')
                    return

    def collect_config(self):
        destination = self.configs

        rmtree(destination, ignore_errors=True)
        os.makedirs(destination)

        if self.config_source.common:
            self._copydir(self.config_source.common, destination)
        self._copydir(self.config_source.config_full_path, destination)

        if self.config_source.fmu_path:
            self._copy_fmu_resources()

    def collect_artifacts(self):
        self._copydir(self.configs,
                      self.artifacts)
        self._copydir(self.results,
                      self.artifacts)

        if self.logfile.exists():
            copyfile(
                str(self.logfile),
                str(self.artifacts / self.logfile.name))

    def clear_results(self):
        self.clean_and_create(self.results)

    @staticmethod
    def get_run_ids(results_path : Path):
        run_ids = []
        for directory in sorted(results_path.iterdir()):
            if directory.is_dir() and re.match(r'run\d+', directory.name):
                run_ids.append(directory)
        return run_ids

    @staticmethod
    def scan_for_additional_controller_output(run_id_path: Path):
        components = []
        for entity in sorted(run_id_path.iterdir()):
            if entity.is_dir() and re.match(r'entity\d+', entity.name):
                for component in entity.iterdir():
                    match = re.match(r'(?P<component>[^/]+)', component.name)
                    if match:
                        components.append(component)
        return components

def clear_cache(simulator_path, output_dir):
    PathManager.clean_and_create(
        Path(simulator_path).parent.resolve() / output_dir / PathManager.PYOPENPASS_CACHE)
