################################################################################
# Copyright (c) 2021-2025 Bayerische Motoren Werke Aktiengesellschaft (BMW AG)
#
# This program and the accompanying materials are made available under the
# terms of the Eclipse Public License 2.0 which is available at
# http://www.eclipse.org/legal/epl-2.0.
#
# SPDX-License-Identifier: EPL-2.0
################################################################################
import os

from pathlib import Path
from dataclasses import dataclass
from pytest_optestrunner.openpass_file import OpenPassFile
from pytest_optestrunner.path_manager import SimulatorBasePaths
from os import name as os_name
from pytest_optestrunner.report import Report
import subprocess, os

def addoption(parser):
    group = parser.getgroup("pyopenpass")
    group.addoption(
        "--base-path",
        action="store",
        dest="base_path",
        default=None,
        help=r'base path for the simulation')
    group.addoption(
        "--simulation",
        action="store",
        dest="simulator",
        default=r'C:\OpenPass\bin\opSimulation.exe' if os_name == 'nt' else '/openPass/bin/opSimulation',
        help=r'path to the simulation executable')
    group.addoption(
        "--configs-path",
        action="store",
        dest="input",
        default=r'configs',
        help=r'relative path for providing configs during testing')
    group.addoption(
        "--results-path",
        action="store",
        dest="output",
        default=r'results',
        help=r'relative path for collecting test results during testing')
    group.addoption(
        "--artifacts-path",
        action="store",
        dest="artifacts",
        default=r'artifacts',
        help=r'relative path for collecting test artifacts during testing')
    group.addoption(
        "--mutual",
        action="store",
        dest="mutual",
        default=None,
        help=r'path to mutual configuration files, if existing')
    group.addoption(
        "--resources",
        action="store",
        dest="resources",
        default=None,
        help=r'path to the resources (e.g. /gecco/bin/examples/configs)')
    group.addoption(
        "--plugin-path",
        action="store",
        dest="plugin_path",
        default=None,
        help=r'path to the traffic participant model plugin ex: "gecco" as shared library')
    group.addoption(
        "--fmu-path",
        action="store",
        dest="fmu_path",
        default=None,
        help=r'path to FMUs')
    group.addoption(
        "--report-path",
        action="store",
        dest="report_path",
        default='.',
        help=r'path for html report (e.g. /gecco/bin/examples/report)')
    group.addoption(
        "--allowed-warnings",
        action="append",
        nargs=1,
        dest="allowed_warnings",
        default=[],
        help=r'file path(s) containing allowed warnings. If a line in the specified file(s) matches the substring of a detected warning, the corresponding test will not fail due to that warning')
    group.addoption(
        "--store-artifacts",
        action="store_true",
        dest="store_artifacts",
        default=False,
        help=r'to store artifacts (artifacts are the combination of the config and results for each test case; flag, no value required)')
    group.addoption(
        "--cache-results",
        action="store_true",
        dest="cache_results",
        default=False,
        help=r'when activated, results are cached. If two separate test cases produce the same input configuration, the cached result is reused for evaluating their queries. Note that caching increases the required hard disk space during testing (flag, no value required)')

@dataclass(frozen=True)
class Args:
    base_path: str
    simulation_path: str
    input_path: str
    output_path: str
    artifact_path: str
    mutual: str
    resources: str
    plugin_path: str
    fmu_path: str
    report_path: str
    allowed_warnings: str
    store_artifacts: bool
    cache_results: bool


def parse_arguments(config) -> Args:
    try:
        allowed_warnings = [each_warning for warning in config.getoption('allowed_warnings') for each_warning in warning]

        return Args(
            config.getoption('base_path'),
            config.getoption('simulator'),
            config.getoption('input'),
            config.getoption('output'),
            config.getoption('artifacts'),
            config.getoption('mutual'),
            config.getoption('resources'),
            config.getoption('plugin_path'),
            config.getoption('fmu_path'),
            config.getoption('report_path'),
            allowed_warnings,
            config.getoption('store_artifacts'),
            config.getoption('cache_results'))
    except ValueError:
        raise OSError(f"Missing one or more required options "
                      f"'simulation', 'mutual', 'resources', or 'report-path'")

def is_pyopenpass_testfile(path):
    path = Path(path)
    return path.suffix == ".json" and path.name.startswith("test")

class Reporter(Report):
    def __init__(self, config) -> None:
        args = parse_arguments(config)
        super(Reporter, self).__init__(args.report_path)

    @staticmethod
    def is_openpass_test_result(item):
        return isinstance(item, dict) and "simulation" in item and "analysis" in item

    @staticmethod
    def make_report(item, report):
        '''
        see https://docs.pytest.org/en/latest/example/simple.html#post-process-test-reports-failures
        '''
        if Reporter.is_openpass_test_result(item):
            extra = getattr(report, "extra", [])
            if report.when == "call":
                extra.append(item)
                report.extra = extra

class PyOpenPass:
    def __init__(self, config) -> None:
        self.args = parse_arguments(config)
        simulator_base_paths = SimulatorBasePaths(self.args.base_path, self.args.simulation_path)
        self._check_simulator(simulator_base_paths)
        self.config = config

    @staticmethod
    def _check_simulator(simulator_base_paths: SimulatorBasePaths, flag='--version'):
        simulator_path = simulator_base_paths.executable
        if not simulator_path.is_file() and not os.access(simulator_path, os.X_OK):
            raise Exception(f"'{simulator_path}' does not point to an executable.")
        
        supported_clis = ['astas_cli', 'gtgen_cli']
        if simulator_path.stem.lower() not in supported_clis:
            raise Exception(f'"{simulator_path}" does not point to a supported simulator CLI (currently: {", ".join(supported_clis)})')
        try:
            subprocess.run([simulator_path, flag], capture_output=True, check=True, cwd=simulator_base_paths.base_path)
        except subprocess.CalledProcessError as e:
            raise Exception(f"Cannot execute {simulator_path.stem.lower()} from the provided simulation path")

    def pytest_collect_file(self, parent, path):
        if is_pyopenpass_testfile(path):
            return OpenPassFile.from_parent(parent, path=Path(path), args=self.args)

    def pytest_collection_modifyitems(cls, items):
        selected_items = []
        deselected_items = []

        for item in items:
            if any(map(lambda s: s.lower().startswith('disabled_'), item.nodeid.split('::'))):
                deselected_items.append(item)
            else:
                selected_items.append(item)

        cls.config.hook.pytest_deselected(items=deselected_items)
        items[:] = selected_items
