################################################################################
# Copyright (c) 2025 Bayerische Motoren Werke Aktiengesellschaft (BMW AG)
#
# This program and the accompanying materials are made available under the
# terms of the Eclipse Public License 2.0 which is available at
# http://www.eclipse.org/legal/epl-2.0.
#
# SPDX-License-Identifier: EPL-2.0
################################################################################

import configparser
from pathlib import Path
from typing import Union

class CasePreservingConfigParser(configparser.ConfigParser):
    def optionxform(self, optionstr):
        """
        Override optionxform to preserve the case of options.
        """
        return optionstr

class IniFileParser:
    def __init__(self, file_path):
        """
        Initialize the IniFileParser with the path to the .ini file.
        :param file_path: Path to the .ini file as a pathlib.Path object.
        """
        self.file_path = Path(file_path)
        if not self.file_path.exists():
            raise FileNotFoundError(f"The file '{file_path}' does not exist.")
        
        self.file_path = file_path
        self.config = CasePreservingConfigParser()
        self.config.read(file_path)

    def update_path(self, new_path: Union[str, Path], key):
        """
        Update the key in the .ini file with a new value.
        :param new_path: The value to set for the given key.
        :param key:      The name of the section to be updated
        """
        updated = False

        for section in self.config.sections():
            if self.config.has_option(section, key):
                self.config.set(section, key, new_path)
                updated = True

        if not updated:
            raise KeyError(f"The key {key} was not found in any section of the .ini file.")

        with self.file_path.open('w') as configfile:
            self.config.write(configfile)
