# Commands covered:  error, catch
#
# This file contains a collection of tests for one or more of the Tcl
# built-in commands.  Sourcing this file into Tcl runs the tests and
# generates output for errors.  No output means no errors were found.
#
# Copyright (c) 1991-1993 The Regents of the University of California.
# Copyright (c) 1994-1996 Sun Microsystems, Inc.
# Copyright (c) 1998-1999 by Scriptics Corporation.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# RCS: @(#) $Id: error.tcl,v 1.1 2006/12/29 09:02:53 fourdman Exp $

if {[lsearch [namespace children] ::tcltest] == -1} {
    package require tcltest
    namespace import -force ::tcltest::*
}

proc foo {} {
    global errorInfo
    set a [catch {format [error glorp2]} b]
    error {Human-generated}
}

proc foo2 {} {
    global errorInfo
    set a [catch {format [error glorp2]} b]
    error {Human-generated} $errorInfo
}

# Catch errors occurring in commands and errors from "error" command

test error-1.1 {simple errors from commands} {
    catch {format [string index]} b
} 1

test error-1.2 {simple errors from commands} {
    catch {format [string index]} b
    set b
} {wrong # args: should be "string index string charIndex"}

test error-1.3 {simple errors from commands} {
    catch {format [string index]} b
    set errorInfo
    # this used to return '... while executing ...', but
    # string index is fully compiled as of 8.4a3
} {wrong # args: should be "string index string charIndex"
    while executing
"string index"}

test error-1.4 {simple errors from commands} {
    catch {error glorp} b
} 1

test error-1.5 {simple errors from commands} {
    catch {error glorp} b
    set b
} glorp

test error-1.6 {simple errors from commands} {
    catch {catch a b c} b
} 1

test error-1.7 {simple errors from commands} {
    catch {catch a b c} b
    set b
} {wrong # args: should be "catch command ?varName?"}

test error-1.8 {simple errors from commands} {nonPortable} {
    # This test is non-portable: it generates a memory fault on
    # machines like DEC Alphas (infinite recursion overflows
    # stack?)

    proc p {} {
        uplevel 1 catch p error
    }
    p
} 0

# Check errors nested in procedures.  Also check the optional argument
# to "error" to generate a new error trace.

test error-2.1 {errors in nested procedures} {
    catch foo b
} 1

test error-2.2 {errors in nested procedures} {
    catch foo b
    set b
} {Human-generated}

test error-2.3 {errors in nested procedures} {
    catch foo b
    set errorInfo
} {Human-generated
    while executing
"error {Human-generated}"
    (procedure "foo" line 4)
    invoked from within
"foo"}

test error-2.4 {errors in nested procedures} {
    catch foo2 b
} 1

test error-2.5 {errors in nested procedures} {
    catch foo2 b
    set b
} {Human-generated}

test error-2.6 {errors in nested procedures} {
    catch foo2 b
    set errorInfo
} {glorp2
    while executing
"error glorp2"
    (procedure "foo2" line 3)
    invoked from within
"foo2"}

# Error conditions related to "catch".

test error-3.1 {errors in catch command} {
    list [catch {catch} msg] $msg
} {1 {wrong # args: should be "catch command ?varName?"}}
test error-3.2 {errors in catch command} {
    list [catch {catch a b c} msg] $msg
} {1 {wrong # args: should be "catch command ?varName?"}}
test error-3.3 {errors in catch command} {
    catch {unset a}
    set a(0) 22
    list [catch {catch {format 44} a} msg] $msg
} {1 {couldn't save command result in variable}}
catch {unset a}

# More tests related to errorInfo and errorCode

test error-4.1 {errorInfo and errorCode variables} {
    list [catch {error msg1 msg2 msg3} msg] $msg $errorInfo $errorCode
} {1 msg1 msg2 msg3}
test error-4.2 {errorInfo and errorCode variables} {
    list [catch {error msg1 {} msg3} msg] $msg $errorInfo $errorCode
} {1 msg1 {msg1
    while executing
"error msg1 {} msg3"} msg3}
test error-4.3 {errorInfo and errorCode variables} {
    list [catch {error msg1 {}} msg] $msg $errorInfo $errorCode
} {1 msg1 {msg1
    while executing
"error msg1 {}"} NONE}
test error-4.4 {errorInfo and errorCode variables} {
    set errorCode bogus
    list [catch {error msg1} msg] $msg $errorInfo $errorCode
} {1 msg1 {msg1
    while executing
"error msg1"} NONE}
test error-4.5 {errorInfo and errorCode variables} {
    set errorCode bogus
    list [catch {error msg1 msg2 {}} msg] $msg $errorInfo $errorCode
} {1 msg1 msg2 {}}

# Errors in error command itself

test error-5.1 {errors in error command} {
    list [catch {error} msg] $msg
} {1 {wrong # args: should be "error message ?errorInfo? ?errorCode?"}}
test error-5.2 {errors in error command} {
    list [catch {error a b c d} msg] $msg
} {1 {wrong # args: should be "error message ?errorInfo? ?errorCode?"}}

# Make sure that catch resets error information

test error-6.1 {catch must reset error state} {
    catch {error outer [catch {error inner inner.errorInfo inner.errorCode}]}
    list $errorCode $errorInfo
} {NONE 1}
test error-6.3 {catch must reset error state} {
    set errorCode BUG
    catch {error outer [catch set]}
    list $errorCode $errorInfo
} {NONE 1}
test error-6.4 {catch must reset error state} {
    catch {error [catch {error foo bar baz}] 1}
    list $errorCode $errorInfo
} {NONE 1}
test error-6.7 {catch must reset error state} {
    proc foo {} {
        return -code error -errorinfo [catch {error foo bar baz}]
    }
    catch foo
    list $errorCode
} {NONE}
test error-6.9 {catch must reset error state} {
    proc foo {} {
        return -code error [catch {error foo bar baz}]
    }
    catch foo
    list $errorCode
} {NONE}


# cleanup
catch {rename p ""}
::tcltest::cleanupTests
return 
