/*
 * Copyright (c) 2018, 2021, Oracle and/or its affiliates.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2.0, as
 * published by the Free Software Foundation.
 *
 * This program is also distributed with certain software (including
 * but not limited to OpenSSL) that is licensed under separate terms,
 * as designated in a particular file or component or in included license
 * documentation.  The authors of MySQL hereby grant you an
 * additional permission to link the program and your derivative works
 * with the separately licensed software that they have included with
 * MySQL.
 *
 * Without limiting anything contained in the foregoing, this file,
 * which is part of MySQL Connector/Node.js, is also subject to the
 * Universal FOSS Exception, version 1.0, a copy of which can be found at
 * http://oss.oracle.com/licenses/universal-foss-exception.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License, version 2.0, for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
 */

'use strict';

const rowResult = require('./RowResult');

/**
 * API for raw SQL queries.
 * @module SqlResult
 * @mixes module:RowResult
 */

/**
 * @private
 * @alias module:SqlResult
 * @param {Object} state - result details
 * @returns {module:SqlResult}
 */
function SqlResult (state) {
    state = Object.assign({}, state);

    return Object.assign({}, rowResult(state), {
        /**
         * Retrieve the first <code>AUTO INCREMENT</code> value generated by the operation.
         * @function
         * @name module:SqlResult#getAutoIncrementValue
         * @returns {number} The first value.
         */
        getAutoIncrementValue () {
            return state.generatedInsertId;
        },

        /**
         * Checks if the result set contains additional data.
         * @function
         * @name module:SqlResult#hasData
         * @returns {boolean}
         */
        hasData () {
            return this.getResults().length > 0;
        },

        /**
         * Returns the entire result set (without flushing) as a JavaScript Arrray.
         * @function
         * @name module:SqlResult#toArray
         * @returns {Array}
         */
        toArray () {
            return this.getResults().map(result => result.map(row => row.toArray()));
        }
    });
}

module.exports = SqlResult;
