#include "udm_config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#ifdef HAVE_IO_H
#include <io.h>	/* for Win */
#endif
#ifdef HAVE_DIRECT_H
#include <direct.h> /* for Win */
#endif
#ifdef  HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#include <fcntl.h>
#include <sys/stat.h>
#include <errno.h>

#include "udm_common.h"
#include "udm_utils.h"
#include "udm_parser.h"
#include "udm_xmalloc.h"
#include "udm_log.h"


__INDLIB__ int UdmAddParser(UDM_ENV * Conf,char *from_mime,char *to_mime,char *cmd){
	if(Conf->nparsers){
		Conf->parsers=(UDM_PARSER*)(realloc(Conf->parsers,(Conf->nparsers+1)*sizeof(UDM_PARSER)));
	}else{
		Conf->parsers=(UDM_PARSER*)(malloc(sizeof(UDM_PARSER)));
	}
	Conf->parsers[Conf->nparsers].from_mime=strdup(from_mime);
	Conf->parsers[Conf->nparsers].to_mime=strdup(to_mime);
	Conf->parsers[Conf->nparsers].cmd=strdup(cmd);
	Conf->nparsers++;
	return 0;
}

void UdmFreeParsers(UDM_ENV * Env){
	size_t i;
	for(i=0;i<Env->nparsers;i++){
		UDM_FREE(Env->parsers[i].from_mime);
		UDM_FREE(Env->parsers[i].to_mime);
		UDM_FREE(Env->parsers[i].cmd);
	}
	UDM_FREE(Env->parsers)
	Env->nparsers=0;
}


#ifdef USE_PARSER

/* Parser1: from STDIN to STDOUT */

#if (WIN32|WINNT)
static char *parse1(UDM_AGENT * Agent,char *buf,size_t buflen,char *cmd,size_t maxlen,char *url){
	UdmLog(Agent,UDM_LOG_ERROR,"This type of parser isn't supported");
	return NULL;
}
#else
static char *parse1(UDM_AGENT * Agent,char *buf,size_t buflen,char *cmd,size_t maxlen,char *url){
	char *result = NULL;
	int wr[2];
	int rd[2];    
	pid_t pid;    

	/* Create write and read pipes */
	if (pipe(wr) == -1){
		UdmLog(Agent,UDM_LOG_ERROR,"Cannot make a pipe for a write");
		return NULL;
	}
	if (pipe(rd) == -1){
		UdmLog(Agent,UDM_LOG_ERROR,"Cannot make a pipe for a read");
		return NULL;
	}    

	/* Fork a clild */
	if ((pid = fork())==-1){
		UdmLog(Agent,UDM_LOG_ERROR,"Cannot spawn a child");
		return NULL;
	}

	if (pid>0){
		/* Parent process */
	        char string[UDMSTRSIZ];

		/* Close other pipe ends */
		close(wr[0]);
		close(wr[1]);
		close(rd[1]);

		result=buf;
		memset(result,0,maxlen);
		memset(string,0,sizeof(string));
		while (read(rd[0],string,sizeof(string)-1)>0){
			strncat(result,string,maxlen-strlen(result));
			memset(string,0,sizeof(string));
		}
		close(rd[0]);
		wait(NULL);
	}else{
		/* Child process */
		/* Fork a clild */
		if ((pid = fork())==-1){
			UdmLog(Agent,UDM_LOG_ERROR,"Cannot spawn a child");
			return NULL;
		}

		if (pid>0){
		/* Parent process */
			/* Close other pipe ends */
			close(wr[0]);
			close(rd[0]);
			close(rd[1]);

			/* Send string to be parsed */
			write(wr[1], buf, buflen);
			close(wr[1]);

			exit(0);
		}else{
			/* Child process */
			/* Close other pipe ends */
			close (wr[1]);
			close (rd[0]);

			/* Connect pipe to stdout */
			dup2(rd[1],STDOUT_FILENO);

			/* Connect pipe to stdin */
			dup2(wr[0],STDIN_FILENO);

			system(cmd);
			exit(0);
		}
	}
	return result;
}
#endif

/* Parser2: from FILE to STDOUT */
static char *parse2(UDM_AGENT * Agent,char *buf,char *cmd,size_t maxlen,char *url){
	FILE *f;
	char *result = NULL;

	result=buf;
	memset(result,0,maxlen);

#if (WIN32|WINNT)
	f=_popen(cmd,"rb");
#else
	f=popen(cmd,"r");
#endif
	if(f){
		int fd;
	        char string[UDMSTRSIZ];
		
		fd=fileno(f);
		memset(string,0,sizeof(string));
		while (read(fd,string,sizeof(string)-1)>0){
			strncat(result,string,maxlen-strlen(result));
			memset(string,0,sizeof(string));
		}
#if (WIN32|WINNT)
		_pclose(f);
#else
		pclose(f);
#endif
	}else{
		UdmLog(Agent,UDM_LOG_ERROR,"Error in popen() (parse2)");
		return NULL;
	}
	return result;
}

/* Parser3: from FILE to FILE */
static char *parse3(UDM_AGENT * Agent,char *buf,char *cmd,size_t maxlen,char *url,char *to_file){
	char *result = NULL;
	int fd;

	result=buf;
	memset(result,0,maxlen);

	system(cmd);
	
	if((fd=open(to_file,O_RDONLY|UDM_BINARY))){
		read(fd,result,maxlen);
		close(fd);
	}else{
		UdmLog(Agent,UDM_LOG_ERROR,"Can't open output file (parse3)");
		return NULL;
	}
	return result;
}

/* Parser4: from STDIN to FILE */
static char *parse4(UDM_AGENT * Agent,char *buf,size_t buflen,char *cmd,size_t maxlen,char *url,char *to_file){
	FILE *f;
	char *result = NULL;

#if (WIN32|WINNT)
	f=_popen(cmd,"wb");
#else
	f=popen(cmd,"w");
#endif

	if(f){
		int fd;
		
		fd=fileno(f);
		write(fd, buf, buflen);
#if (WIN32|WINNT)
		_pclose(f);
#else
		pclose(f);
#endif
		result=buf;
		memset(result,0,maxlen);

		if((fd=open(to_file,O_RDONLY|UDM_BINARY))){
			read(fd,result,maxlen);
			close(fd);
		}else{
			UdmLog(Agent,UDM_LOG_ERROR,"Can't open output file (parse4)");
			return NULL;
		}
	}else{
		UdmLog(Agent,UDM_LOG_ERROR,"Error in popen() (parse4)");
		return NULL;
	}
	return result;
}

static char *parse_file (UDM_AGENT * Agent,UDM_PARSER * parser,char *buf, size_t length, size_t maxlen,char * url){
	char cmd[UDMSTRSIZ]="";
	char *result;
	char *arg1pos,*arg2pos;
	int parser_type;
	char fn0[1024]="";
	char fn1[1024]="";
	char * fnames[2];

	arg1pos=strstr(parser->cmd,"$1");
	arg2pos=strstr(parser->cmd,"$2");

	/* Build temp file names and command line */
	tmpnam(fn0);strcpy(fn1,fn0);
	fnames[0]=strcat(fn0,".in");
	fnames[1]=strcat(fn1,".out");
	UdmBuildParamStr(cmd,sizeof(cmd),parser->cmd,fnames,2);

	if(arg1pos){
		int fd;

		/* Create temporary file */
		umask((mode_t)022);

		fd=open(fnames[0],O_RDWR|O_CREAT|UDM_BINARY,UDM_IWRITE);
		/* Write to the temporary file */
		write(fd,buf,length);
		close(fd);
	}

	if(arg1pos&&arg2pos)parser_type=3;
	else{
		if(arg1pos)parser_type=2;
		else
		if(arg2pos)parser_type=4; 
		else parser_type=1;
	}
	
	/*fprintf(stderr,"cmd='%s' parser_type=%d\n",cmd,parser_type);*/
	UdmLog(Agent,UDM_LOG_EXTRA,"Starting external parser: '%s'",cmd);
	UdmSetEnv("UDM_URL",url);
	switch(parser_type){
		case 1: result=parse1(Agent,buf,length,cmd,maxlen,url); break;
		case 2: result=parse2(Agent,buf,cmd,maxlen,url); break;
		case 3: result=parse3(Agent,buf,cmd,maxlen,url,fnames[1]); break;
		case 4: result=parse4(Agent,buf,length,cmd,maxlen,url,fnames[1]); break;
	}
	UdmUnsetEnv("UDM_URL");

	/* Remove temporary file */
	if(arg1pos)unlink(fnames[0]);
	if(arg2pos)unlink(fnames[1]);

	return result;
}

char *UdmExecParser(UDM_AGENT * Agent,char *from_mime, int *to_mime, char *buf, size_t length, size_t maxlen,char * url){
	size_t i;

	for(i=0;i<Agent->Conf->nparsers;i++){
		if(!UdmStrCaseMatch(from_mime,Agent->Conf->parsers[i].from_mime)){
			*to_mime=i;
 			return parse_file(Agent,&(Agent->Conf->parsers[i]),buf,length,maxlen,url);
		}
    	}
    	return NULL;
}

#endif
