#include "udm_config.h"

#ifdef HAVE_SQL

/*
#define DEBUG_SQL
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "udm_common.h"
#include "udm_spell.h"
#include "udm_robots.h"
#include "udm_utils.h"
#include "udm_mutex.h"
#include "udm_db.h"
#include "udm_charset.h"
#include "udm_parseurl.h"
#include "udm_log.h"
#include "udm_proto.h"
#include "udm_conf.h"
#include "udm_crc32.h"
#include "udm_xmalloc.h"

#if (WIN32|WINNT)
#include <windef.h>
#include <time.h>
#else
#include <unistd.h>
#include <sys/time.h>
#endif

#if (HAVE_MYSQL)
#include "mysql.h"
#define DBVER "MySQL"
#define DB_DEFAULT	"mysql"
#define HAVE_SQL_LIMIT
#define HAVE_SQL_IN
#elif (HAVE_PGSQL)
#include "libpq-fe.h"
#define DBVER "PgSQL"
#define DB_DEFAULT	"pgsql"
#define HAVE_SQL_LIMIT
#define HAVE_SQL_IN
#elif HAVE_MSQL
#define DBVER "mSQL"
#define DB_DEFAULT	"msql"
#include "msql.h"
#elif HAVE_IODBC
#define DBVER "iODBC"
#define DB_DEFAULT	""
#include "iodbc.h"
#include "isql.h"
#include "isqlext.h"
#elif HAVE_EASYSOFT
#define DBVER "EasySoft"
#define DB_DEFAULT	""
#include "sql.h"
#include "sqlext.h"
#elif HAVE_VIRT
#define DBVER "Virtuoso"
#define DB_DEFAULT	"virt"
#include "iodbc.h"
#include "isql.h"
#include "isqlext.h"
#elif HAVE_UNIXODBC
#define DBVER "unixODBC"
#define DB_DEFAULT	""
#include "sql.h"
#include "sqlext.h"
#elif HAVE_SOLID
#define DBVER "Solid"
#define DB_DEFAULT	"solid"
#include "cli0cli.h"
#elif HAVE_IBASE
#include "ibase.h"
#define DBVER "InterBase"
#define DB_DEFAULT	"ibase"
#elif HAVE_ORACLE8
#include "oci.h"
#define DBVER "Oracle8"
#define DB_DEFAULT	"oracle"
#define HAVE_SQL_IN
#elif HAVE_ORACLE7
#include "ocidfn.h"
#include "oratypes.h"
#define HAVE_SQL_IN
#ifdef __STDC__
#include <ociapr.h>
#else
#include <ocikpr.h>
#endif
#define DBVER "Oracle7"
#define DB_DEFAULT	"oracle"
#define HAVE_SQL_IN
#else
#define DBVER ""
#define DB_DEFAULT	""
#endif

#ifdef DEBUG_SQL
#include <sys/times.h>
#undef CLOCKS_PER_SEC
#define CLOCKS_PER_SEC (sysconf(_SC_CLK_TCK))
static unsigned long start_timer(void){
#if (WIN32|WINNT)
	return clock();
#else
	struct tms tms_tmp;
	return times(&tms_tmp);
#endif /* (WIN32|WINNT) */
}
#endif /* DEBUG_SQL */

int UdmInitDB(){
	UdmSetDBType(DB_DEFAULT);
	return(0);
}

/* Multi-dict mode defines */
#define NDICTS	18
#define MINDICT	2
#define MAXDICT	NDICTS
static int dictlen[NDICTS]={2,2,2,3,4,5,6,7,8,9,10,11,12,16,16,16,16,32};
#define	DICTNUM(l)	((l>16)?dictlen[17]:dictlen[l])

#define	URL_DELETE_CACHE 50
#define	URL_SELECT_CACHE 320
#define URL_LOCK_TIME    4*60*60

static  char catstr[UDMSTRSIZ]="";
static  char langstr[UDMSTRSIZ]="";
static  char tagstr[UDMSTRSIZ]="";
static  char statusstr[UDMSTRSIZ]="";
static  char urlstr[UDMSTRSIZ]="";
static  char timestr[UDMSTRSIZ]="";
static  char category[UDMSTRSIZ]="";
static	UDM_STOPWORD *stoplist=NULL;
static  int nstoplist=0;
static	int currow=0;
static	int nrows=0;

__INDLIB__ char * UdmVersion(){
static char udmver[128];
	sprintf(udmver,"%s/%s",VERSION,DBVER);
	return(udmver);
}

static time_t now(){ /* !!! is it reentrant? */
time_t tloc;
	return(time(&tloc));
}

char * escstr(char *x,char *y){
char *s;
#ifdef HAVE_MYSQL
	if(DBType==UDM_DB_MYSQL){
		mysql_escape_string(x,y,strlen(y));
		return(x);
	}
#endif
	s=x;
	if (DBType == UDM_DB_ORACLE || DBType == UDM_DB_ORACLE8  ||
	    DBType == UDM_DB_MSSQL || DBType == UDM_DB_ORACLE7)
	 {
	    while(*y){
		switch(*y){
			case '\'': *x=*y;x++;*x=*y;
			default:*x=*y;
		}
		x++;y++;
	    }
	} else {
	    while(*y){
		switch(*y){
			case '\'':
			case '\\':
				*x='\\';x++;
			default:*x=*y;
		}
		x++;y++;
	    }
	}
	*x=0;return(s);
}


/************************ MYSQL **************************************/
#if   (HAVE_MYSQL)
#define ER_DUP_ENTRY            1062
#define ER_DUP_KEY		1022
#define CR_SERVER_LOST          2013
#define CR_SERVER_GONE_ERROR    2006
#define ER_SERVER_SHUTDOWN      1053
typedef struct struct_db {
	MYSQL mysql;
	MYSQL_RES *res;
	MYSQL_RES *res1;
	int connected;
	int res_limit;
	int commit_fl;
	int errcode;
	char errstr[UDMSTRSIZ];
} DB;
typedef MYSQL_RES UDB_RES;
#define SQL_NUM_ROWS(x)	mysql_num_rows(x)
#define SQL_FREE(x)	mysql_free_result(x)

static int InitDB(DB*db){
	if (!(mysql_connect(&(db->mysql),DBHost,DBUser,DBPass))){
		db->errcode=1;
		return(1);
	}
	if ((mysql_select_db(&(db->mysql),DBName?DBName:"udmsearch"))){
		db->errcode=1;
		return(1);
	}
	db->connected=1;
	return(0);
}
static char * sql_value(MYSQL_RES *res,int i,int j){
MYSQL_ROW row;
	if(i>=mysql_num_rows(res))return(0);		
	mysql_data_seek(res,i);
	row=mysql_fetch_row(res);
	if(row)return(row[j]?row[j]:"");
	else	return(0);
}
static void CloseDB(DB *db){
	if(db->connected)mysql_close(&(db->mysql));
	db->connected=0;
}
static void displayError(DB * db){
	sprintf(db->errstr,"#%d: %s",
		mysql_errno(&((DB*)db)->mysql),
		mysql_error(&((DB*)db)->mysql));
}
static int safe_mysql_query(DB * db,char *query){
int err,i;
	if(!db->connected){
		InitDB(db);
		if(db->errcode)
			return(db->errcode);
	}
	for(i=0;i<2;i++){
		if((err=mysql_query(&db->mysql,query))){
			if((mysql_errno(&db->mysql)==CR_SERVER_LOST)||
			(mysql_errno(&db->mysql)==CR_SERVER_GONE_ERROR)||
			(mysql_errno(&db->mysql)==ER_SERVER_SHUTDOWN)){
				UDMSLEEP(5);
			}else
				return(err);
		}else
			return(err);
	}
	return(err);
}
static MYSQL_RES * sql_query(DB * db,char *query){
int res;
#ifdef DEBUG_SQL
double t;
unsigned long ticks;
	ticks=start_timer();
#endif
	db->errcode=0;
	res=safe_mysql_query(db,query);
#ifdef DEBUG_SQL
	ticks=start_timer()-ticks;
	t=(float)ticks/CLOCKS_PER_SEC;
	fprintf(stderr,"[%d] SQL %.2fs: %s\n",getpid(),t,query);
#endif
	if(res){
		displayError(db);
		if((mysql_errno(&(db->mysql))!=ER_DUP_ENTRY) &&
			(mysql_errno(&(db->mysql))!=ER_DUP_KEY)){
			db->errcode=1;
		}
		return(0);
	}
	return(mysql_store_result(&db->mysql));
}
/*********************************** POSTGRESQL *********************/
#elif (HAVE_PGSQL)
typedef struct struct_db {
	PGconn     *pgsql;
	PGresult   *res;
	PGresult   *res1;
	int res_limit;
	int commit_fl;
	int connected;
	int errcode;
	char errstr[UDMSTRSIZ];
} DB;
typedef PGresult UDB_RES;
#define SQL_NUM_ROWS(x)	PQntuples(x)
#define SQL_FREE(x)	PQclear(x)
#define sql_value(x,y,z) PQgetvalue(x,y,z)

static int InitDB(DB *db){
char port[8];
	sprintf(port,"%d",DBPort);
	db->pgsql = PQsetdbLogin(DBHost, DBPort?port:0, 0, 0, DBName, DBUser, DBPass);
	if (PQstatus(db->pgsql) == CONNECTION_BAD){
		db->errcode=1;
		return(1);
	}
	db->connected=1;
	return(0);
}
static void CloseDB(DB *db){
	if(db->connected)PQfinish(db->pgsql);
	db->connected=0;
}
static void displayError(DB * db){
	sprintf(db->errstr, "%s", PQerrorMessage(db->pgsql));
}
static PGresult * safe_pgsql_query(DB *db,char *q){
	if(!(db->connected)){
		InitDB(db);
		if(db->errcode)return(0);
	}
	return(PQexec(db->pgsql,q));
}
static PGresult * sql_query(void *db,char *q){
PGresult * res;
#ifdef DEBUG_SQL
double t;
unsigned long ticks;
	ticks=start_timer();
#endif

	((DB*)db)->errcode=0;
	res=safe_pgsql_query((DB*)db,q);
	if(!res){
		displayError(db);
		((DB*)db)->errcode=1;
		return(0);
	}
	if((PQresultStatus(res)!=PGRES_COMMAND_OK)&&(PQresultStatus(res)!=PGRES_TUPLES_OK)){
		displayError(db);
		if(!strstr(PQerrorMessage(((DB*)db)->pgsql),"duplicate"))
			((DB*)db)->errcode=1;
		return(0);
	}
#ifdef DEBUG_SQL
	ticks=start_timer()-ticks;
	t=(float)ticks/CLOCKS_PER_SEC;
	fprintf(stderr,"[%d] SQL %.2fs: %s\n",getpid(),t,q);
#endif
	return(res);
}
/*****************************  miniSQL *******************************/
#elif (HAVE_MSQL)
typedef struct struct_db {
	int		msql;
	m_result	*res;
	int res_limit;
	int commit_fl;
	int connected;
	int errcode;
	char errstr[UDMSTRSIZ];
} DB;
typedef m_result UDB_RES;
#define SQL_NUM_ROWS(x)	msqlNumRows(x)
#define SQL_FREE(x)	msqlFreeResult(x)

static char * sql_value(m_result *res,int i,int j){
m_row row;

	if(i>=msqlNumRows(res))return("");
	msqlDataSeek(res,i);
	row=msqlFetchRow(res);
	if(row)return(row[j]?row[j]:"");
	else	return(0);
}
static int InitDB(DB *db){
char *host;
	if((host=DBHost))if(!strcmp(DBHost,"localhost"))host=NULL;
	if((db->msql=msqlConnect(host))<0){
		db->errcode=1;
		return(1);
	}
	if(msqlSelectDB(db->msql,DBName)==-1){
		db->errcode=1;
		return(1);
	}
	((DB*)db)->connected=1;
	return(0);
}
static void CloseDB(DB *db){
	if(db->connected)msqlClose(db->msql);
	db->connected=0;
}
static void displayError(DB *db){
	sprintf(db->errstr,"%s",msqlErrMsg);
}
static int safe_msql_query(DB *db,char *query){
int err;

	if(!(db->connected)){
		InitDB(db);
		if(db->errcode)
			return(db->errcode);
	}
	if((err=msqlQuery(db->msql,query))==-1){
		if(!strstr(msqlErrMsg,"Non unique")){
			db->errcode=1;
			return(err);
		}
		return(0);
	}
	return(0);
}
static m_result * sql_query(void *db,char *query){
int err;
	((DB*)db)->errcode=0;
	if((err=safe_msql_query((DB*)db,query))){
		displayError(db);
		return(0);
	}
	return(msqlStoreResult());
}
/************************** ODBC ***********************************/
#elif (HAVE_IODBC || HAVE_UNIXODBC || HAVE_SOLID || HAVE_VIRT || HAVE_EASYSOFT)
typedef struct or_struct{
	int nRows;
	int nCols;
	int freeme;
	char **items;
} iODBC_RES;
typedef iODBC_RES UDB_RES;
typedef struct struct_db {
	HDBC	hDbc;
	HENV	hEnv;
	HSTMT	hstmt;
	iODBC_RES * res;
	int res_limit;
	int commit_fl;
	int connected;
	int errcode;
	char errstr[UDMSTRSIZ];
} DB;

#define SQL_NUM_ROWS(x)	(x?x->nRows:0)
#define SQL_FREE(x)	iodbc_free_result(x)
#define sql_val(x,y,z)	x->items[y*x->nCols+z]
#define sql_value(x,y,z) x?(sql_val(x,y,z)?sql_val(x,y,z):""):""
#define SQL_OK(rc)	((rc==SQL_SUCCESS)||(rc==SQL_SUCCESS_WITH_INFO))

static int iodbc_free_result(iODBC_RES *res){
int i;
	if(res){
		if(res->freeme){
			if(res->items){
				for(i=0;i<res->nCols*res->nRows;i++)
					if(res->items[i])
						free(res->items[i]);
				free(res->items);
			}
			free(res);
		}
	}
	return(0);
}
static int displayError(DB * db){
	UCHAR szSqlState[6];
	UCHAR szErrMsg[256];
	SDWORD naterr;
	SWORD length;
	
	db->errstr[0]=0;
	while (SQL_SUCCESS == SQLError(db->hEnv,db->hDbc,db->hstmt,szSqlState,&naterr,szErrMsg,sizeof(szErrMsg),&length))
		strcat(db->errstr, szErrMsg);
	return(0);
}
static iODBC_RES * execDB(DB*db,char* sqlstr){
	RETCODE rc;
	SWORD iResColumns;
	SDWORD iRowCount;
	int i,res_count;
	char* p;
	UCHAR szColName[32];
	SWORD pcbColName;
	SWORD pfSQLType;
	UDWORD pcbColDef;
	SWORD pibScale;
	SWORD pfNullable;
	iODBC_RES *result;
	SDWORD	pcbValue;
	static char	bindbuf[(int)(32L * 1024L - 16L)];

	/* -------- 
	p=sqlstr;
	while(*p){
		if(*p=='?')*p='.';
		p++;
	}
	*/

	rc=SQLAllocStmt(db->hDbc, &(db->hstmt));
	if (!SQL_OK(rc)){
		db->errcode=1;
		return(NULL);
	}
	rc=SQLExecDirect(db->hstmt,(UCHAR*)sqlstr, SQL_NTS);
	if (!SQL_OK(rc)){
		db->errcode=1;
		return(NULL); 
	}
	rc=SQLNumResultCols(db->hstmt, &iResColumns);
	if(!SQL_OK(rc)){
		db->errcode=1;
		return(NULL);
	}
	if(!iResColumns) {
		rc=SQLRowCount(db->hstmt, &iRowCount);
		if (!SQL_OK(rc)){
			db->errcode=1;
			return(NULL);
		}
		result=NULL;
	}else{
		result=(iODBC_RES*)malloc(sizeof(iODBC_RES));
		result->nRows = 0;
		result->nCols = iResColumns;
		result->items=NULL;
		result->freeme=1;

		rc = SQL_NO_DATA_FOUND;
		for (res_count=0;(db->res_limit?(res_count<db->res_limit):1);res_count++){
			rc=SQLFetch(db->hstmt);
			if (!SQL_OK(rc)) {
				if (rc!=SQL_NO_DATA_FOUND){
					db->errcode=1;
				}
				break;
			}
			if(!result->nRows){
				result->items=(char **)malloc(1*iResColumns*sizeof(char*));
			}else{
				result->items=(char **)realloc(result->items,((result->nRows+1)*iResColumns*sizeof(char*)));
			}
			for (i = 0; i < iResColumns; i++) {
				SQLDescribeCol(db->hstmt, i+1, szColName, sizeof(szColName),
					&pcbColName, &pfSQLType, &pcbColDef,&pibScale, &pfNullable);
				SQLGetData(db->hstmt,i+1,SQL_C_CHAR,bindbuf,sizeof(bindbuf),&pcbValue);
				if (pcbValue==SQL_NULL_DATA) p = NULL;
				else p = bindbuf;
				if(p)UdmRTrim(p," ");
				sql_val(result,result->nRows,i)=strdup(p?p:"");
			}
			result->nRows++;
		}
	}
	db->res_limit=0;
	SQLFreeStmt(db->hstmt, SQL_DROP);
	db->errcode=0;
	return(result);
}
static int InitDB(DB*db){
char DSN[UDMSTRSIZ]="";
#if (HAVE_SOLID)
	sprintf(DSN,"tcp %s %d",DBHost?DBHost:"localhost",DBPort?DBPort:1313);
#else
	strcpy(DSN,DBName?DBName:"");
#endif
	db->errcode = SQLAllocEnv( &(db->hEnv) );
	if( SQL_SUCCESS != db->errcode )return -2;
	db->errcode = SQLAllocConnect( db->hEnv, &(db->hDbc) );
	if( SQL_SUCCESS != db->errcode )return -3;
	db->errcode = SQLSetConnectOption( db->hDbc, SQL_AUTOCOMMIT, SQL_AUTOCOMMIT_ON);
	if( SQL_SUCCESS != db->errcode )return -4;
	db->errcode = SQLConnect( db->hDbc, DSN, SQL_NTS, DBUser, SQL_NTS, DBPass, SQL_NTS);
	if( !SQL_OK(db->errcode)) return(-5);
	else db->errcode=0;
	db->connected=1;
	return 0;
}
static void CloseDB(DB * db){
	if(db->connected){
		db->connected=0;
		db->errcode = SQLTransact( db->hEnv, db->hDbc, SQL_COMMIT);
		if( SQL_SUCCESS != db->errcode )return;
		db->errcode = SQLDisconnect( db->hDbc );
		if( SQL_SUCCESS != db->errcode )return;
		db->errcode = SQLFreeConnect( db->hDbc );
		if( SQL_SUCCESS != db->errcode )return;
		else	db->hDbc = SQL_NULL_HDBC;
		db->errcode = SQLFreeEnv( db->hEnv );
		if( SQL_SUCCESS != db->errcode )return;
		else	db->hEnv = SQL_NULL_HENV;
	}
}
static iODBC_RES * sql_query(DB * db,char *qbuf){
	iODBC_RES * res;
	
	if(LockProc)LockProc(UDM_LOCK,UDM_LOCK_QUERY);
	if(!db->connected){
		InitDB(db);
		if(db->errcode){
			displayError(db);
			if(LockProc)LockProc(UDM_UNLOCK,UDM_LOCK_QUERY);
			return(0);
		}else{
			db->connected=1;
		}
	}
	res=execDB(db,qbuf);
	if((db->errcode)){
		displayError(db);
		if(strstr(db->errstr,"uplicat")){ /* PgSQL,MySQL*/
			db->errcode=0;
		}else
		if(strstr(db->errstr,"nique")){ /* Solid, Virtuoso */
			db->errcode=0;
		}else{
			db->errcode=1;
			res=NULL;
		}
		SQLFreeStmt(db->hstmt, SQL_DROP);
	}
	if(LockProc)LockProc(UDM_UNLOCK,UDM_LOCK_QUERY);
	return(res);
}
#elif HAVE_IBASE

typedef struct or_struct{
	int nRows;
	int nCols;
	int freeme;
	char **items;
} iBASE_RES;
typedef iBASE_RES UDB_RES;

#define SQL_NUM_ROWS(x)	(x->nRows)
#define SQL_FREE(x)	ibase_free_result(x)
#define sql_val(x,y,z) x->items[y*x->nCols+z]
#define sql_value(x,y,z) sql_val(x,y,z)?sql_val(x,y,z):""

#define SQL_VARCHAR(len) struct {short vary_length; char vary_string[(len)+1];}
typedef struct ibase_db {
	int		connected;
	int		errcode;
	isc_db_handle	DBH;		/* database handle */
	ISC_STATUS	status[20];	/* status vector */
	char		errstr[UDMSTRSIZ]; /* error buffer */
	UDB_RES		* res;
	int		res_limit;
	int		commit_fl;
} DB;

static void displayError(DB * db){
char *s;
ISC_STATUS * stat;
	stat=(db->status);
	s=db->errstr;
	while(isc_interprete(s ,&stat)){
		strcat(db->errstr," ");
		s=db->errstr+strlen(db->errstr);
	}
}

static int ibase_free_result(iBASE_RES *res){
int i;
	if(res){
		if(res->freeme){
			if(res->items){
				for(i=0;i<res->nCols*res->nRows;i++)
					if(res->items[i])
						free(res->items[i]);
				free(res->items);
			}
			free(res);
		}
	}
	return(0);
}
static int InitDB(DB * db){
	
	char dpb_buffer[256], *dpb, *p;
	int dpb_length, len;

	dpb = dpb_buffer;

	*dpb++ = isc_dpb_version1;

	if (DBUser != NULL && (len = strlen(DBUser))) {
		*dpb++ = isc_dpb_user_name;
		*dpb++ = len;
		for (p = DBUser; *p;) {
			*dpb++ = *p++;
		}
	}

	if (DBPass != NULL && (len = strlen(DBPass))) {
		*dpb++ = isc_dpb_password;
		*dpb++ = strlen(DBPass);
		for (p = DBPass; *p;) {
			*dpb++ = *p++;
		}
	}
	/*
	if (charset != NULL && (len = strlen(charset))) {
		*dpb++ = isc_dpb_lc_ctype;
		*dpb++ = strlen(charset);
		for (p = charset; *p;) {
			*dpb++ = *p++;
		}
	}
#ifdef isc_dpb_sql_role_name
	if (role != NULL && (len = strlen(role))) {
		*dpb++ = isc_dpb_sql_role_name;
		*dpb++ = strlen(role);
		for (p = role; *p;) {
			*dpb++ = *p++;
		}
	}
#endif
	*/

	dpb_length = dpb - dpb_buffer;

	if(isc_attach_database(db->status, strlen(DBName), DBName, &(db->DBH), dpb_length, dpb_buffer)){
		db->errcode=1;
		return(1);
	}
	return(0);
}
static void CloseDB(DB*db){
	if(db->connected){
		if (isc_detach_database(db->status, &(db->DBH))){
			db->errcode=1;
		}
	}
}
static iBASE_RES empty_res;
static iBASE_RES * sql_ibase_query(DB * db, char *query){
	int i, coltype;
	static char query_info[] = { isc_info_sql_stmt_type };
	char info_buffer[18];
	short l;
	long query_type;
	XSQLDA *osqlda=NULL;
	isc_tr_handle tr_handle = NULL;
	isc_stmt_handle query_handle = NULL;
	long fetch_stat;
	XSQLVAR *var;
	iBASE_RES *result=NULL;

	if(!db->connected){
		InitDB(db);
		if(db->errcode){
			displayError(db);
			db->errcode=1;
			return NULL;
		}else{
			db->connected=1;
		}
	}
	if (isc_start_transaction(db->status, &tr_handle, 1, &(db->DBH), 0, NULL)){
		db->errcode=1;
		return(0);
	}
	if (isc_dsql_allocate_statement(db->status, &(db->DBH), &query_handle)){
		db->errcode=1; 
		return(0);
	}
	if (isc_dsql_prepare(db->status, &tr_handle, &query_handle, 0, query, 1, osqlda)){
		db->errcode=1; 
		return(0);
	}
	if (!isc_dsql_sql_info(db->status, &query_handle, sizeof(query_info), query_info, sizeof(info_buffer), info_buffer)) {
		l = (short) isc_vax_integer((char ISC_FAR *) info_buffer + 1, 2);
		query_type = isc_vax_integer((char ISC_FAR *) info_buffer + 3, l);
	}

	/* Find out what kind of query is to be executed */
	if (query_type == isc_info_sql_stmt_select || query_type == isc_info_sql_stmt_select_for_upd) {
		/*
		 * Select, need to allocate output sqlda and and prepare it for use.
		 */
		osqlda = (XSQLDA *) malloc(XSQLDA_LENGTH(0));
		osqlda->sqln = 0;
		osqlda->version = SQLDA_VERSION1;

		if (isc_dsql_describe(db->status, &query_handle, 1, osqlda)) {
			free(osqlda);
			db->errcode=1;
			return NULL;
		}

		if (osqlda->sqld) {
			osqlda = (XSQLDA *) realloc(osqlda, XSQLDA_LENGTH(osqlda->sqld));
			osqlda->sqln = osqlda->sqld;
			osqlda->version = SQLDA_VERSION1;
			if (isc_dsql_describe(db->status, &query_handle, 1, osqlda)) {
				free(osqlda);
				db->errcode=1;
				return NULL;
			}
		}
		for (i = 0; i < osqlda->sqld; i++) {
			osqlda->sqlvar[i].sqlind = (short *) malloc(sizeof(short));
			coltype = osqlda->sqlvar[i].sqltype & ~1;
			switch(coltype)
				{
				case SQL_TEXT:
					osqlda->sqlvar[i].sqldata = (char *) malloc(sizeof(char)*(osqlda->sqlvar[i].sqllen));
					break;
				case SQL_VARYING:
					osqlda->sqlvar[i].sqldata = (char *) malloc(sizeof(char)*(osqlda->sqlvar[i].sqllen+2));
					break;
				case SQL_SHORT:
					osqlda->sqlvar[i].sqldata = (char *) malloc(sizeof(short));
					break;
				case SQL_LONG:
					osqlda->sqlvar[i].sqldata = (char *) malloc(sizeof(long));
					break;
				case SQL_FLOAT:
					osqlda->sqlvar[i].sqldata = (char *) malloc(sizeof(float));
					break;
				case SQL_DOUBLE:
					osqlda->sqlvar[i].sqldata = (char *) malloc(sizeof(double));
					break;
				case SQL_DATE:
				case SQL_BLOB:
				case SQL_ARRAY:
					osqlda->sqlvar[i].sqldata = (char *) malloc(sizeof(ISC_QUAD));
					break;
				}
		}
		if (isc_dsql_execute(db->status, &tr_handle, &query_handle, 1, NULL)) {
			free(osqlda);
			db->errcode=1;
			return NULL;
		}
		result=(iBASE_RES*)malloc(sizeof(iBASE_RES));
		result->nRows = 0;
		result->nCols = osqlda->sqld;
		result->items=NULL;
		result->freeme=1;

		while ((fetch_stat = isc_dsql_fetch(db->status, &query_handle, 1, osqlda)) == 0){
			int i;char buf[UDMSTRSIZ]="";
			var=osqlda->sqlvar;

			if(!result->nRows){
				result->items=(char **)malloc(1*osqlda->sqld*sizeof(char*));
			}else{
				result->items=(char **)realloc(result->items,((result->nRows+1)*osqlda->sqld*sizeof(char*)));
			}
			for(i=0;i<osqlda->sqld; i++, var++){
				if(*var->sqlind==-1)
					 buf[0]=0;  /* NULL data */
				else
				switch(var->sqltype & ~1){
					case SQL_TEXT:
					strncpy(buf,(char*)var->sqldata,var->sqllen);
					buf[var->sqllen]=0;
					break;
					case SQL_LONG:
					sprintf(buf,"%ld",*(long*)(var->sqldata));
					break;
					case SQL_SHORT:
					sprintf(buf,"%d",*(short*)(var->sqldata));
					break;
					default:
					sprintf(buf,"Unknown SQL type\n");
					buf[0]=0;
					break; 
				}
				UdmRTrim(buf," ");
				sql_val(result,result->nRows,i)=strdup(buf);
			}
			result->nRows++;
		}
		free(osqlda);

		if (fetch_stat != 100L){
			db->errcode=1; 
			return NULL;
		}
		if (isc_dsql_free_statement(db->status, &query_handle, DSQL_close)){
			db->errcode=1; 
			return NULL;
		}
		
	} else {
		/* Not select */
		if (isc_dsql_execute(db->status, &tr_handle, &query_handle, 1, osqlda)) {
			db->errcode=1;
			return NULL;
		}
		empty_res.nRows=0;
		empty_res.nCols=0;
		empty_res.items=NULL;
		empty_res.freeme=0;
		result=&empty_res;
	}
	if (isc_commit_transaction(db->status, &tr_handle)){
		db->errcode=1; 
		return NULL;
	}
	return result;
}
static iBASE_RES * sql_query(DB * db, char *query){
iBASE_RES * res;
	res=sql_ibase_query(db,query);
	if(db->errcode){
		displayError(db);
		if(strstr(db->errstr,"uplicat")){
			db->errcode=0;
		}
		return(NULL);
	}
	return(res);
}

#elif HAVE_ORACLE8
/******************** Oracle8 OCI - native support driver ******************/
/* (C) copyleft 2000 Anton Zemlyanov, az@hotmail.ru */
/* TODO: efficient transactions, multi-row fetch, limits stuff */
 
#define	MAX_COLS_IN_TABLE	32
typedef struct or_struct {
	int	nRows;
	int	nCols;
	int	freeme;
	char    *defbuff[MAX_COLS_IN_TABLE]; /* Buffers for OCIStmtFetch */
	sb2	indbuff[MAX_COLS_IN_TABLE];  /* Indicators for NULLs */
	char	**items;
} ORACLE_RES;
typedef ORACLE_RES UDB_RES;

typedef struct struct_db {
	OCIEnv     *envhp;
	OCIError   *errhp;
	OCISvcCtx  *svchp;
	OCIStmt	   *stmthp;
	OCIParam   *param;
	OCIDefine  *defb[MAX_COLS_IN_TABLE];
	ORACLE_RES *res;
	int        res_limit;
	int	commit_fl;
	int        connected;
	ub4        errcode;
	char       errstr[UDMSTRSIZ];
} DB;
#define BUF_OUT_SIZE	512
#define MAX_BIND_PARAM	3
int out_rec;
int out_pos[MAX_BIND_PARAM];
int out_pos_1[BUF_OUT_SIZE];
int out_pos_2[BUF_OUT_SIZE];
int out_pos_3[BUF_OUT_SIZE];

#define SQL_NUM_ROWS(x)	(x?x->nRows:0)
#define SQL_FREE(x)	oci_free_result(x)
#define sql_val(x,y,z)	x->items[y*x->nCols+z]
#define sql_value(x,y,z) x?(sql_val(x,y,z)?sql_val(x,y,z):""):""
#define SQL_OK(rc)	((rc==OCI_SUCCESS)||(rc==OCI_SUCCESS_WITH_INFO))

static int oci_free_result(ORACLE_RES *res)
{
	int i;
	if(res){
		if(res->freeme){
			if(res->items){
				for(i=0;i<res->nCols*res->nRows;i++)
					if(res->items[i])
						free(res->items[i]);
				free(res->items);
			}
			if(res->defbuff){
				for(i=0;i<res->nCols;i++)
					if(res->defbuff[i])
						free(res->defbuff[i]);
			}
			free(res);
		}
	}
	return(0);
}

static int displayError(DB *db)
{
	sb4	errcode=0;
	text	errbuf[512];
	char	*ptr;

	db->errstr[0]='\0';

	switch (db->errcode)
	{
	case OCI_SUCCESS:
		sprintf(db->errstr,"Oracle - OCI_SUCCESS");
		break;
	case OCI_SUCCESS_WITH_INFO:
		sprintf(db->errstr,"Oracle - OCI_SUCCESS_WITH_INFO");
		break;
	case OCI_NEED_DATA:
		sprintf(db->errstr,"Oracle - OCI_NEED_DATA");
		break;
	case OCI_NO_DATA:
		sprintf(db->errstr,"Oracle - OCI_NODATA");
		break;
	case OCI_ERROR:
		OCIErrorGet((dvoid *)db->errhp, (ub4) 1, (text *) NULL, &errcode,
			errbuf, (ub4) sizeof(errbuf), OCI_HTYPE_ERROR);
		ptr=errbuf;
		while(*ptr) {
			if(*ptr=='\n')
				*ptr='!';
			++ptr;
		}
		sprintf(db->errstr,"Oracle - %.*s", 512, errbuf);
		break;
	case OCI_INVALID_HANDLE:
		sprintf(db->errstr,"Oracle - OCI_INVALID_HANDLE");
		break;
	case OCI_STILL_EXECUTING:
		sprintf(db->errstr,"Oracle - OCI_STILL_EXECUTE");
		break;
	case OCI_CONTINUE:
		sprintf(db->errstr,"Oracle - OCI_CONTINUE");
		break;
	default:
		sprintf(db->errstr,"Oracle - unknown internal bug");
		break;
	}
	return 0;
}

static int InitDB(DB *db)
{
	OCIInitialize( OCI_DEFAULT, NULL, NULL, NULL, NULL );
	OCIEnvInit( &(db->envhp), OCI_DEFAULT, 0, NULL );
	OCIHandleAlloc( db->envhp, (dvoid **)&(db->errhp), OCI_HTYPE_ERROR,
		0, NULL );
	db->errcode=OCILogon(db->envhp, db->errhp, &(db->svchp),
		DBUser, strlen(DBUser),
		DBPass, strlen(DBPass),
		DBName, strlen(DBName) );
	if(db->errcode!=OCI_SUCCESS)
		return -1;

	db->errcode=0;
	db->connected=1;
	db->commit_fl=0;
	out_rec=0;

/*        sql_query(((DB*)(db)), "ALTER SESSION SET SQL_TRACE=TRUE"); */

	return 0;
}

static void CloseDB(DB *db)
{
	if(!db->connected)
		return;
	db->errcode=OCILogoff(db->svchp,db->errhp);
	db->connected=0;
}

static int bind_by_pos(DB *db, OCIBind *bndhp[], int *out, int pos){
int rc;
	/* Bind url_id,word_id,intag)*/
	if ( (rc = OCIBindByPos(db->stmthp, &bndhp[pos], db->errhp, (ub4) pos,
                      (dvoid *) out, (sb4) sizeof(*out), SQLT_INT,
                      (dvoid *) 0, (ub2 *)0, (ub2 *)0,
                      (ub4) 0, (ub4 *) 0, (ub4) OCI_DEFAULT))){

		return rc;
	}
	return 0;
}

static int bind_array(DB *db, OCIBind *bndhp[], int *out, int pos){
int rc;
	/*  bind array  */
	if ((rc = OCIBindArrayOfStruct(bndhp[pos], db->errhp, sizeof(*out), 0, 0, 0))){
		return rc;
	}
	return 0;
}

/* Empty set for non-select queries */
static ORACLE_RES empty_res;
static ORACLE_RES* sql_oracle_query(DB *db, char *qbuf)
{
	ORACLE_RES *result;
	sword	rc;
	ub2	stmt_type;
	int	cnt;

	int	colcnt;
	ub2	coltype;
	ub2	colsize;
	int	coldefsize=0;
	int	oci_fl;
	sb4	errcode=0;
	text	errbuf[512];
	int	*out, pos, num_pos, num_rec, i;
	OCIBind *bndhp[MAX_BIND_PARAM+1];

	rc=OCIHandleAlloc(db->envhp, (dvoid *)&(db->stmthp), OCI_HTYPE_STMT,0,NULL);
	if(!SQL_OK(rc)){
		db->errcode=rc;
		return NULL;
	}
	rc=OCIStmtPrepare(db->stmthp,db->errhp,qbuf,strlen(qbuf),
			OCI_NTV_SYNTAX,OCI_DEFAULT);
	if(!SQL_OK(rc)){
		db->errcode=rc;
		return NULL;
	}

if (out_rec){
	for (i=0; i<MAX_BIND_PARAM+1; i++)
		bndhp[i] = (OCIBind *) 0;

	num_pos= 0;
	for( i=0; i<MAX_BIND_PARAM; i++){
		if (!out_pos[i])
			break;
		pos = out_pos[i];
		num_pos ++;
		switch(pos){
			case 1:
				out = &out_pos_1[0];
				break;
			case 2:
				out = &out_pos_2[0];
				break;
			case 3:
				out = &out_pos_3[0];
				break;
		}
		if ((rc=bind_by_pos(db, bndhp, out, pos))){
			db->errcode=rc;
			return NULL;
		}
    
		if ((rc=bind_array(db, bndhp, out, pos))){
			db->errcode=rc;
			return NULL;
		}
	}
	num_rec = out_rec;
}else
	num_rec = 1;

out_rec=0;
    
	rc=OCIAttrGet(db->stmthp,OCI_HTYPE_STMT,&stmt_type,0,
			OCI_ATTR_STMT_TYPE,db->errhp);
	if(!SQL_OK(rc)){
		db->errcode=rc;
		return NULL;
	}

	if(stmt_type!=OCI_STMT_SELECT) {
		/* non-select statements */
		/* COMMIT_ON_SUCCESS in inefficient */
		if (db->commit_fl)
			oci_fl=OCI_DEFAULT;
		else
			oci_fl=OCI_COMMIT_ON_SUCCESS;

		rc=OCIStmtExecute(db->svchp,db->stmthp,db->errhp, num_rec,0,
				NULL,NULL,oci_fl);

		if (num_rec>1)
			  (void) OCITransCommit(db->svchp, db->errhp, (ub4) 0);

		if(!SQL_OK(rc)){
			db->errcode=rc;
			OCIErrorGet((dvoid *)db->errhp, (ub4) 1, (text *) NULL, &errcode,
				errbuf, (ub4) sizeof(errbuf), OCI_HTYPE_ERROR);
			if(strncmp(errbuf,"ORA-00001",9)) /* ignore ORA-00001 */
				return NULL;
			else 
				db->errcode=OCI_SUCCESS;
		}
		/* OCI_ATTR_ROW_COUNT of db->stmthp - Rows affected */
		rc=OCIHandleFree(db->stmthp,OCI_HTYPE_STMT);
		if(!SQL_OK(rc)){
			db->errcode=rc;
			return NULL;
		}
		empty_res.nRows=0;
		empty_res.nCols=0;
		empty_res.items=0;
		empty_res.freeme=0;
		return (ORACLE_RES*)&empty_res;
	}

	/* select statements */
	/*Allocate result Set*/
	result=(ORACLE_RES*)malloc(sizeof(ORACLE_RES));
	result->nRows=0;
	result->nCols=0;
	result->items=NULL;
	result->freeme=1;

	rc=OCIStmtExecute(db->svchp,db->stmthp,db->errhp,0,0,
			NULL,NULL,OCI_DEFAULT);
	if(!SQL_OK(rc)){
		db->errcode=rc;
		return NULL;
	}

	/*describe the select list and define buffers for the select list */
	colcnt=0;
	while(OCIParamGet(db->stmthp,OCI_HTYPE_STMT,db->errhp,(dvoid *)&(db->param),
			colcnt+1 ) == OCI_SUCCESS) {
		rc=OCIAttrGet(db->param,OCI_DTYPE_PARAM,&(coltype),0,
				OCI_ATTR_DATA_TYPE,db->errhp);
		if(!SQL_OK(rc)){
			db->errcode=rc;
			return NULL;
		}
		rc=OCIAttrGet(db->param,OCI_DTYPE_PARAM,&colsize,0,
			OCI_ATTR_DATA_SIZE,db->errhp);
		if(!SQL_OK(rc)){
			db->errcode=rc;
			return NULL;
		}

		/* OCIStmtFetch do not terminate data with \0 -
		add a byte for terminator to define buffers - insurance*/

		switch(coltype) {
		case SQLT_CHR: /* variable length string */
		case SQLT_AFC: /* fixed length string */
			coldefsize=colsize+1;
			result->defbuff[colcnt]=(char *)malloc(coldefsize);
			result->defbuff[colcnt][coldefsize-1]='\0';
			break;
		case SQLT_NUM: /* numbers up to 14 digits now */
			coldefsize=15;
			result->defbuff[colcnt]=(char *)malloc(coldefsize);
			result->defbuff[colcnt][coldefsize-1]='\0';
			break;
		default:
			printf("<P>Unknown datatype: %d\n",coltype);
			return (ORACLE_RES*)NULL;
		}
		rc=OCIDefineByPos(db->stmthp,&(db->defb[colcnt]),db->errhp,
			colcnt+1,result->defbuff[colcnt],coldefsize-1,SQLT_CHR,
			&(result->indbuff[colcnt]),0,0,OCI_DEFAULT);
		if(!SQL_OK(rc)){
			db->errcode=rc;
	 		return NULL;
		}
		colcnt++;
	}
	result->nCols=colcnt;

	/* Now fetching the selected rows into the memory */
	while(1) {
		if (db->res_limit)
			if (db->res_limit == result->nRows)
				break;

		rc=OCIStmtFetch(db->stmthp,db->errhp,1,OCI_FETCH_NEXT,OCI_DEFAULT);
		if(rc==OCI_NO_DATA)
			break;
		if(!SQL_OK(rc)) {
			db->errcode=rc;
			return NULL;
		}
		if(!result->nRows)  /* first row - allocate */
			result->items=(char **)malloc(1*result->nCols*sizeof(char*));
		else
			result->items=(char **)realloc(result->items,
				((result->nRows+1)*result->nCols*sizeof(char*)) );
		for(cnt=0; cnt<result->nCols; ++cnt) {
			if(result->indbuff[cnt]==OCI_IND_NULL) {
				sql_val(result,result->nRows,cnt) = strdup("");
			} else {
				sql_val(result,result->nRows,cnt)=
					strdup( UdmRTrim(result->defbuff[cnt]," ") );
			}
		}
		result->nRows++;
	}
	rc=OCIHandleFree(db->stmthp,OCI_HTYPE_STMT);
	db->res_limit = 0;
	if(!SQL_OK(rc)){
		db->errcode=rc;
		return NULL;
	}

	return (ORACLE_RES*)result;
}

static ORACLE_RES* sql_query(DB *db, char *qbuf) {
ORACLE_RES *res;
#ifdef DEBUG_SQL
int row_num;
double t;
unsigned long ticks;
	ticks=start_timer();
	row_num = out_rec;
#endif
	if(!db->connected) {
		InitDB(db);
		if(db->errcode) {
			displayError(db);
			return NULL;
		} else
			db->connected=1;
 	}
	res=sql_oracle_query(db,qbuf);

#ifdef DEBUG_SQL
	ticks=start_timer()-ticks;
	t=(float)ticks/CLOCKS_PER_SEC;
	fprintf(stderr,"[%d] SQL %.2fs: %s ->%d\n",getpid(),t,qbuf, row_num);
#endif


	if(db->errcode) {
		displayError(db);
		return NULL;
	}

	return (ORACLE_RES*)res;
}

#elif HAVE_ORACLE7
/******************** Oracle7 OCI - native support driver ******************
 * (C) 2000 Kir Maximov, maxkir@email.com 
 * 
 * Without transactions ...
 */

#define MAX_COLS_IN_TABLE    32
#define HDA_SIZE 256
#define MAX_ITEM_BUFFER_SIZE 32

/* Query result structure */
typedef struct or_struct {
    int nRows;
    int nCols;
    int dont_free_me;
    char    **items;
} ORACLE_RES;
typedef ORACLE_RES UDB_RES;

/* For results of non-SELECT queries: */
static ORACLE_RES empty_res;

/* Database structure */
typedef struct struct_db {
    ORACLE_RES *res;
    int        res_limit;
    int        commit_fl;
    int        connected;
    ub4        errcode;
    char       errstr[UDMSTRSIZ];

    Lda_Def     lda;
    ub1 hda[HDA_SIZE];

    Cda_Def     cursor;
    
} DB;

#define UDM_OCI_UNIQUE 1

#define SQL_NUM_ROWS(x) (x?x->nRows:0)
#define SQL_FREE(x) oci_free_result(x)
#define sql_val(x,y,z)  x->items[y*x->nCols+z]
#define sql_value(x,y,z) (x?(sql_val(x,y,z)?sql_val(x,y,z):""):"")

#define ORA_VERSION_7 2

/*  some SQL and OCI function codes */
#define FT_INSERT                3
#define FT_SELECT                4
#define FT_UPDATE                5
#define FT_DELETE                9

/* Declare structures for query information. */
struct describe
{
    sb4             dbsize;
    sb2             dbtype;
    sb1             buf[MAX_ITEM_BUFFER_SIZE];
    sb4             buflen;
    sb4             dsize;
};

struct define 
{
    ub1             buf[UDMSTRSIZ];
    sb2             indp;
    ub2             col_retlen, col_retcode;
};
    

/*-------------------------------------------------------*/
static int oci_free_result(ORACLE_RES *res)
{
    int i;

    if(res && !res->dont_free_me)
    {
        if(res->items)
        {
            for(i=0;i<res->nCols*res->nRows;i++)
            {
                if(res->items[i])
                    free(res->items[i]);
            }
            free(res->items);
        }
        free(res);
    }
    return(0);
}
/*-------------------------------------------------------*/
static int displayError(DB *db)
{
    sword n;
    text msg[512];

    db->errstr[0]='\0';
    if (!db->errcode) 
        return 0; 

    n = oerhms(&db->lda, db->cursor.rc, msg, (sword) sizeof msg);
    
    strcpy(db->errstr, "ORACLE - ");
    strcat(db->errstr, msg);

    printf("displayError: %p %s\n", db, msg);

    return 0;
}
/*-------------------------------------------------------*/
static int InitDB(DB *db)
{
    Cda_Def *cda = &db->cursor;
    Lda_Def *lda = &db->lda;
    char buff[1024];

    udm_snprintf(buff, sizeof(buff)-1, "%s/%s@%s", DBUser, DBPass, DBName);
    
    db->errcode = olog(lda, db->hda,
                       buff, -1,
                       (text *)0, -1,
                       0, -1, OCI_LM_DEF);

    if ( db->errcode )
    {
        /* Error happened */
        return -1;
    }

    /* Now, open a cursor to be used in all queries*/
    oopen(cda, lda, (text*)0, -1, -1, (text*)0, -1);
    db->errcode = cda->rc;
    if (db->errcode)
    {
        displayError(db);
        return 0;
    }
    
    db->errcode = 0;
    db->connected = 1;

    empty_res.dont_free_me = 1;
   
    return 0;
}
/*-------------------------------------------------------*/
static void CloseDB(DB *db)
{
    if(!db->connected)
        return;
    
    /* First, close a cursor */
    oclose(&db->cursor);

    /* Logout */
    db->errcode = ologof(&db->lda);
    db->connected = 0;
}
/*-------------------------------------------------------*/
static ORACLE_RES* fetch_data(DB *db)
{
    /* Real feching of data for SELECT statements */
    ORACLE_RES *res = NULL;
    int col, deflen;
    struct describe desc[MAX_COLS_IN_TABLE];
    struct define   def[MAX_COLS_IN_TABLE];
    Cda_Def *cda = &db->cursor;
    char *buf_ptr;
   

    /* Now, get column desriptions */
    for(col = 0; col < MAX_COLS_IN_TABLE; col ++)
    {
        desc[col].buflen = MAX_ITEM_BUFFER_SIZE;
        if ( odescr( cda, col+1, &desc[col].dbsize,
                     &desc[col].dbtype, desc[col].buf,
                     &desc[col].buflen, &desc[col].dsize,
                     0,0,0
                   )
           )
        {
            if ( cda->rc == 1007 ) /* No more variables */
                break;
            db->errcode = cda->rc;
            if (db->errcode)
            {
                displayError(db);
                return NULL;
            }
            
        }
        
        switch(desc[col].dbtype)
        {
        case 2: /* NUMBER */
            deflen = 14;
            break;
        case 96: /* CHAR */
        case 1:  /* VARCHAR2 */
        default:
            deflen = (desc[col].dbsize > UDMSTRSIZ ? UDMSTRSIZ : desc[col].dbsize + 1);
        }
       
        if (odefin(cda, col + 1,
                    def[col].buf, deflen, 5, /* 5 - null terminated string */
                    -1, &def[col].indp, (text*)0, -1, -1,
                    &def[col].col_retlen,
                    &def[col].col_retcode ))
        {
            db->errcode = cda->rc;
            if (db->errcode)
            {
                displayError(db);
                return NULL;
            }
        }
    }
    /* Now col contains a number of columns */

    /* Get memory for resulting data */
    res = (ORACLE_RES*)malloc(sizeof(ORACLE_RES));
    if (!res)
        return NULL;

    /* Clear all structure data */
    memset(res, 0, sizeof(ORACLE_RES));

    res->nCols = col;

    /* Now, fetching the data */
    for (;;)
    {
        /* Fetch a row, break on end of fetch, */
        /* disregard null fetch "error" */
        
        if (ofetch(cda))
        {
            if ( cda->rc == 1403 ) /* No data found */
                break;
            if ( cda->rc != 1405 && cda->rc ) /* Null value returned */
            {
               SQL_FREE(res);
               db->errcode = cda->rc;
               displayError(db);
               return NULL;
            }
        }
        /* Next row: */
        
        /* [Re]allocate memory for data */
        if (!res->items)
        {
            res->items = (char**) malloc( res->nCols * sizeof(char*));
        }
        else
        {
            res->items = (char**) realloc(res->items, 
                             (res->nRows + 1) * res->nCols * sizeof(char*));
        }
                
        for (col = 0; col < res->nCols; col++)
        {
            if (def[col].indp < 0) /* NULL value */
            {
                buf_ptr = strdup("");
            }
            else
            {
                buf_ptr = strdup((char*)def[col].buf);
                UdmRTrim(buf_ptr, " ");
            }
            /* sql_val(res, res->nRows, col)*/
            res->items[res->nRows*res->nCols + col] = buf_ptr;
            /*printf( "\n<P>Field %d Val '%s' Null '%d'",col,sql_value(res,res->nRows,col), def[col].indp );*/

        }
        res->nRows ++;
    }

    return res;    
}
/*-------------------------------------------------------*/
static ORACLE_RES* sql_oracle_query(DB *db, char *query)
{
    /* Make real query and store result */
    ORACLE_RES* res = NULL;
    Cda_Def *cda = &db->cursor;
    
    /* Parse SQL statement */
    oparse(cda, (text*) query,
                            (sb4)-1, (sword) 0, (ub4)ORA_VERSION_7 );
    if ( cda->rc )
    {
        db->errcode = cda->rc;
        displayError(db);
        return NULL;
    }
    /* Save sql function */

    /* Exec */
    oexec(cda);
    if (cda->rc)
    {
        if (cda->rc != UDM_OCI_UNIQUE)  /* ignore unique constraint violation */
        {
            displayError(db);
            return NULL;
        }
        db->errcode = 0;
    }
    
    switch (cda->ft)
     {
        case FT_DELETE:
        case FT_INSERT:
        case FT_UPDATE:
            /* Empty result */
            res = &empty_res;

            /* commit */
            ocom(&db->lda);

            break;
        case FT_SELECT:
            /* Fetch data needed */
            res = fetch_data(db);
            break;
        default:
            res = NULL;
            break;
    }
    
    return res;
}

/*-------------------------------------------------------*/
static ORACLE_RES* sql_query(DB *db, char *qbuf) 
{
    ORACLE_RES *res;
    
    if(!db->connected) 
    {
        InitDB(db);
        if(db->errcode) 
        {
            displayError(db);
            return NULL;
        }
    }
    
    /* Make real query */
    res = sql_oracle_query(db, qbuf);
/*    printf("\nQ: '%s'\n", qbuf); */
    
    if(!res) 
    {
        if (db->errcode)
        {
            displayError(db);
        }
        else
        {
            fprintf(stderr, "sql_query: this should never happen.\n");   
        }
    }
    return res;
}


/******************************** Template  ************************/
#else
typedef int UDB_RES;
typedef struct struct_db {
	int   *res;
	int   *res1;
	int res_limit;
	int commit_fl;
	int connected;
	int errcode;
	char errstr[UDMSTRSIZ];
} DB;
#define SQL_NUM_ROWS(x)	0
#define SQL_FREE(x)
#define sql_value(x,y,z) 0

static int InitDB(DB *db){
	db->connected=1;return(0);
}
static int * sql_query(DB * db,char *query){
	return(0);
}
void CloseDB(DB * db){
}
#endif

/********************************************************************/
/* --------------------------------------------------------------- */
/* Almost Unified code for all supported SQL backends              */
/* --------------------------------------------------------------- */
/*******************************************************************/

static UDB_RES * urlres;

void * UdmAllocDB(int mode){
DB * db;
	db=(DB*)malloc(sizeof(DB));
#if (HAVE_IODBC || HAVE_UNIXODBC || HAVE_SOLID || HAVE_VIRT || HAVE_EASYSOFT)
	db->hDbc=SQL_NULL_HDBC;
	db->hEnv=SQL_NULL_HENV;
	db->hstmt=SQL_NULL_HSTMT;
#endif
#if (HAVE_IBASE)
	db->DBH=NULL;
#endif
	db->res=NULL;
	db->res_limit=0;
	db->connected=0;
	db->errcode=0;
	db->errstr[0]=0;
	db->commit_fl=0;
	return((void*)db);
}
void UdmFreeDB(void * db){
	if(db){
		CloseDB((DB*)db);
		free(db);
	}
}
int UdmDBErrorCode(void *db){
	return(((DB*)db)->errcode);
}
char * UdmDBErrorMsg(void *db){
	return(((DB*)db)->errstr);
}


/******************** Limits stuff *********************************/

__INDLIB__ int UdmAddTagLimit(char * tag){
	if(*tagstr)strcpy(UDM_STREND(tagstr)-1," OR ");
	else	strcat(tagstr," AND (");
	if(DBType==UDM_DB_PGSQL)
		sprintf(UDM_STREND(tagstr),"(url.tag || '') LIKE '%s')",tag);
	else
		sprintf(UDM_STREND(tagstr),"url.tag LIKE '%s')",tag);
	return(0);
}

__INDLIB__ int UdmAddStatusLimit(int status){
#ifdef  HAVE_SQL_IN
	if(*statusstr)sprintf(UDM_STREND(statusstr)-1,",%d)",status);
	else	sprintf(statusstr," AND url.status IN (%d)",status);
#else
	if(*statusstr)strcpy(UDM_STREND(statusstr)-1," OR ");
	else	strcat(statusstr," AND (");
	sprintf(UDM_STREND(statusstr),"url.status=%d)",status);
#endif
	return(0);
}

__INDLIB__ int UdmAddURLLimit(char *URL){
	if(*urlstr)strcpy(UDM_STREND(urlstr)-1," OR ");
	else	strcat(urlstr," AND (");
	if(DBType==UDM_DB_PGSQL)
		sprintf(UDM_STREND(urlstr),"(url.url || '') LIKE '%s')",URL);
	else
		sprintf(UDM_STREND(urlstr),"url.url LIKE '%s')",URL);
	return(0);
}

__INDLIB__ int UdmAddLangLimit(char * lang){
	if(*langstr)strcpy(UDM_STREND(langstr)-1," OR ");
	else	strcat(langstr," AND (");
	sprintf(UDM_STREND(langstr),"url.lang LIKE '%s')",lang);
	return(0);
}

__INDLIB__ int UdmAddCatLimit(char * cat){
	strcpy(cat,category?cat:"");
	if(*catstr)strcpy(UDM_STREND(catstr)-1," OR ");
	else    strcat(catstr," AND (");
	sprintf(UDM_STREND(catstr),"url.category LIKE '%s%%')",cat);
	return(0);
}



__INDLIB__ int UdmAddTimeLimit(struct udm_stl_info_t * stl){
	char lt_gt;
	
	switch(stl->type){
		case 1: case -1:
			if (stl->type==1)
				lt_gt='>';
			else
				lt_gt='<';
			sprintf(timestr, " AND (url.last_mod_time%c%li)", lt_gt, stl->t1);
			break;
		case 2: /* between */
			sprintf(timestr, " AND (url.last_mod_time BETWEEN %li AND %li)", stl->t1, stl->t2);
			break;
		default:
			return -1;
	}
/*	printf("<BR>timestr=%s<BR>\n", timestr);*/
	return(0);
}

__INDLIB__ int UdmClearLimits(){
	urlstr[0]='\0';
	tagstr[0]='\0';
	statusstr[0]='\0';
	langstr[0]='\0';
	timestr[0]='\0';
	catstr[0]='\0';
	return(0);
}
__INDLIB__ int UdmClearURLLimit(){
	urlstr[0]=0;
	return(0);
}

/************* robots.txt stuff *********************************/
int  UdmLoadRobots(UDM_INDEXER *Indexer){
int i,nrobots;
	if(LockProc)LockProc(UDM_LOCK,UDM_LOCK_ROBOTS);
	if(Robots)UdmFreeRobots(Indexer);
	((DB*)(Indexer->db))->res=sql_query(((DB*)(Indexer->db)),
		"SELECT hostinfo,path FROM robots");
	if(UdmDBErrorCode(Indexer->db)){
		if(LockProc)LockProc(UDM_UNLOCK,UDM_LOCK_ROBOTS);
		return(IND_ERROR);
	}
	nrobots=SQL_NUM_ROWS(((DB*)(Indexer->db))->res);
	if(nrobots){
		Robots=(UDM_ROBOT *)malloc((nrobots+1)*sizeof(UDM_ROBOT));
		for(i=0;i<nrobots;i++){
			Robots[i].hostinfo=strdup(sql_value(((DB*)(Indexer->db))->res,i,0));
			Robots[i].path=strdup(sql_value(((DB*)(Indexer->db))->res,i,1));
		}
		Robots[i].hostinfo=Robots[i].path=NULL;
	}
	SQL_FREE(((DB*)(Indexer->db))->res);
	if(LockProc)LockProc(UDM_UNLOCK,UDM_LOCK_ROBOTS);
	return(IND_OK);
}
int UdmDeleteRobotsFromHost(UDM_INDEXER *Indexer,char *hostinfo){
char qbuf[UDMSTRSIZ];
	sprintf(qbuf,"DELETE FROM robots WHERE hostinfo='%s'",hostinfo);
	sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	return(IND_OK);
}
int UdmAddRobotsToHost(UDM_INDEXER *Indexer,char *hostinfo,char *s){
char qbuf[UDMSTRSIZ];
	sprintf(qbuf,"INSERT INTO robots (hostinfo,path) VALUES ('%s','%s')",hostinfo,s);
	sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	return(IND_OK);
}
int UdmDeleteAllFromRobots(UDM_INDEXER *Indexer){
char qbuf[UDMSTRSIZ];
	sprintf(qbuf,"DELETE FROM robots");
	sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db)){
		return(IND_ERROR);
	}
	return(IND_OK);
}


/************************* stopwords *******************************/

static int cmpstop(const void *s1,const void *s2){
	return(strcmp(((UDM_STOPWORD*)s1)->word,((UDM_STOPWORD*)s2)->word));
}
int  UdmLoadStopList(void * db){
int i,j,cur=0;

	((DB*)db)->res=sql_query((DB*)db,"SELECT word,lang FROM stopword");
	if(((DB*)db)->errcode)return(IND_ERROR);

	nstoplist=SQL_NUM_ROWS(((DB*)db)->res);
	stoplist=(UDM_STOPWORD *)malloc((nstoplist)*sizeof(UDM_STOPWORD));
	for(i=0;i<nstoplist;i++){
		char *newword, *newlang;
		int found;

		newword=sql_value(((DB*)db)->res,i,0);
		newlang=sql_value(((DB*)db)->res,i,1);
		found=0;

		/* If the word is already in list     */
		/* We will not add it again           */
		/* But mark it as "international word"*/
		/* i.e. the word without language     */
		/* It will allow to avoid troubles    */
		/* with language guesser              */
		for(j=0;j<cur;j++){
			if(!strcmp(stoplist[j].word,newword)){
				stoplist[j].lang[0]=0;
				found=1;
				break;
			}
		}
		/* Add new stopword */
		if(!found){
			stoplist[cur].word=strdup(newword);
			strncpy(stoplist[cur].lang,newlang,2);
			stoplist[cur].lang[2]=0;
			cur++;
		}
	}
	SQL_FREE(((DB*)db)->res);
	/* If there was some inretnational words */
	/* We have to realloc memory to actually */
	/* required size                         */
	if(cur<nstoplist){
		nstoplist=cur;
		stoplist=(UDM_STOPWORD *)realloc(stoplist,(nstoplist)*sizeof(UDM_STOPWORD));
	}
	/* Sort it to run binary search later */
	qsort((void*)stoplist,nstoplist,sizeof(UDM_STOPWORD),cmpstop);
	return(IND_OK);
}
UDM_STOPWORD * UdmIsStopWord(char *word){
int low = 0;
int high = nstoplist - 1;
int middle, match;

	if(!stoplist)return(0);
	while (low <= high) {
		middle = (low + high) / 2;
		match = strcmp(stoplist[middle].word, word);
		if (match == 0)
			return (&stoplist[middle]);
		if (match < 0)
			low = middle + 1;
		else
			high = middle - 1;
	}
	return(NULL);
}


/************************ URLs ***********************************/

int UdmAddURL(UDM_INDEXER *Indexer,char *url,int referrer,int hops, char * msg_id){
char *e_url;
int next_url_id=0;
char qbuf[UDMSTRSIZ]="AddURL";

	e_url=(char*)malloc(strlen(url)*2);escstr(e_url,url);

	switch(DBType){
	case UDM_DB_MSQL:
		/* miniSQL has _seq as autoincrement value */
		((DB*)(Indexer->db))->res=sql_query((DB*)(Indexer->db),"SELECT _seq FROM url");
		if(UdmDBErrorCode(Indexer->db)){
			free(e_url);
			return(IND_ERROR);
		}
		next_url_id=atoi(sql_value(((DB*)(Indexer->db))->res,0,0));
		SQL_FREE(((DB*)(Indexer->db))->res);
#ifdef NEWS_EXT
		sprintf(qbuf,"INSERT INTO url (url,referrer,hops,rec_id,crc32,last_index_time,next_index_time,status,msg_id) VALUES ('%s',%d,%d,%d,0,%d,%d,0,'%s')",e_url,referrer,hops,next_url_id,(int)now(),(int)now(),msg_id);
#else
		sprintf(qbuf,"INSERT INTO url (url,referrer,hops,rec_id,crc32,last_index_time,next_index_time,status) VALUES ('%s',%d,%d,%d,0,%d,%d,0)",e_url,referrer,hops,next_url_id,(int)now(),(int)now());
#endif
		break;
		
	case UDM_DB_SOLID:
	case UDM_DB_ORACLE:
	case UDM_DB_ORACLE7:
	case UDM_DB_ORACLE8:
		if (strlen(e_url)>UDM_URLSIZE)
			e_url[UDM_URLSIZE]=0;
#ifdef NEWS_EXT
		sprintf(qbuf,"INSERT INTO url (url,referrer,hops,rec_id,crc32,last_index_time,next_index_time,status,msg_id) VALUES ('%s',%d,%d,next_url_id.nextval,0,%d,%d,0,'%s')",e_url,referrer,hops,(int)now(),(int)now(),msg_id);
#else
		sprintf(qbuf,"INSERT INTO url (url,referrer,hops,rec_id,crc32,last_index_time,next_index_time,status) VALUES ('%s',%d,%d,next_url_id.nextval,0,%d,%d,0)",e_url,referrer,hops,(int)now(),(int)now());
#endif
		break;
	default:
#ifdef NEWS_EXT
		sprintf(qbuf,"INSERT INTO url (url,referrer,hops,crc32,last_index_time,next_index_time,status,msg_id) VALUES ('%s',%d,%d,0,%d,%d,0,'%s')",e_url,referrer,hops,(int)now(),(int)now(),msg_id);
#else
		sprintf(qbuf,"INSERT INTO url (url,referrer,hops,crc32,last_index_time,next_index_time,status) VALUES ('%s',%d,%d,0,%d,%d,0)",e_url,referrer,hops,(int)now(),(int)now());
#endif
	}
	free(e_url);
	sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))
		return(IND_ERROR);
	else
		return(IND_OK);
}
int UdmDeleteUrl(UDM_INDEXER *Indexer,int url_id){
char qbuf[UDMSTRSIZ];
int res;
	if((res=UdmDeleteWordFromURL(Indexer,url_id))!=IND_OK)return(res);
	sprintf(qbuf,"DELETE FROM url WHERE rec_id=%d",url_id);
	sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	return(IND_OK);
}
int UdmMarkForReindex(UDM_INDEXER *Indexer){
char qbuf[UDMSTRSIZ];
	sprintf(qbuf,"UPDATE url SET next_index_time=%d WHERE rec_id>=0 %s%s%s%s%s",(int)now(),tagstr,urlstr,statusstr,langstr,catstr);
	sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	return(IND_OK);
}
int UdmUpdateUrl(UDM_INDEXER *Indexer,int url_id,int status,int period){
char qbuf[UDMSTRSIZ];
#ifdef HAVE_ORACLE8
	sprintf(qbuf,"UPDATE url SET status=:1,next_index_time=:2 WHERE rec_id=:3");
        out_rec = 1;
        UDMMEMZERO(out_pos, sizeof(out_pos));
        out_pos[0] = 1; out_pos[1] = 2; out_pos[2] = 3;
        out_pos_1[0] = status;
        out_pos_2[0] = (int)(now()+period);
        out_pos_3[0] = url_id;
#else
	sprintf(qbuf,"UPDATE url SET status=%d,next_index_time=%d WHERE rec_id=%d",status,(int)(now()+period),url_id);
#endif
	sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	return(IND_OK);
}

#ifdef NEWS_EXT
int UdmRegisterChild(UDM_INDEXER *Indexer, int parent_id, int child_id)
{
	char qbuf[UDMSTRSIZ];
	sprintf(qbuf,"insert into thread values(%d,%d)",parent_id,child_id);
	sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	return(IND_OK);
	return 0;
}
int UdmFindMsgID(UDM_INDEXER *Indexer, const char * msg_id)
{
#ifdef HEIKODEBUG
FILE * inn_f = NULL;
#endif
	int i;
	int num_rows;
	char * idstring;
	char qbuf[UDMSTRSIZ];
	char msg_id_stripped[strlen(msg_id)];
	int rec_id = -1;
	strncpy(msg_id_stripped,msg_id+1,strlen(msg_id)-2);
	msg_id_stripped[strlen(msg_id)-2]=0;
	/* msg_id still has the leading < and trailing > characters */
	sprintf(qbuf,"SELECT rec_id from url where msg_id='%s'",msg_id_stripped);
#ifdef HEIKODEBUG	
if((inn_f = fopen("/tmp/inn_qbuf.tmp","a")))
{
	fprintf (inn_f,"\n%s\n",qbuf);
	fclose(inn_f);
}
else
{
	fprintf(stderr,"\nCould not open file\n");
	fflush(stderr);
}
#endif
	((DB*)(Indexer->db))->res=sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))return(0);
	num_rows = SQL_NUM_ROWS(((DB*)(Indexer->db))->res);
	for(i = 0; i < num_rows; i++)
		if((idstring=sql_value(((DB*)(Indexer->db))->res,i,0)))
			rec_id=atoi(idstring);
	SQL_FREE(((DB*)(Indexer->db))->res);
	return(rec_id);
}
int UdmDeleteAllFromThread(UDM_INDEXER *Indexer){
char qbuf[UDMSTRSIZ];
	sprintf(qbuf,"DELETE FROM thread");
	sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	return(IND_OK);
}
int UdmLongUpdateUrl(UDM_INDEXER *Indexer,
int url_id,int status,int changed,int size,int period,
char *tag,int index,
time_t last_mod_time,
char *text_escaped,
char *title_escaped,
char *content_type,
char *keywords_escaped,
char *descript_escaped,
udmcrc32_t crc32,
char *lang,
char *category,
char *hd_date,
char *hd_subj,
char *hd_from,
char *hd_group,
char *hd_ref,
char *msg_id
){
/* I know, writing 8192 there is not cool... :-| */
int BUFFSIZE = UDMSTRSIZ + 8192 + strlen(text_escaped);
char qbuf[BUFFSIZE];
char last_index_time[64]="";

#ifdef HEIKODEBUG
FILE * inn_f = NULL;
#endif

	if(changed)sprintf(last_index_time,",last_index_time=%d",(int)now());
sprintf(qbuf,"\
UPDATE url SET \
status=%d,last_mod_time=%li,\
next_index_time=%d,\
tag='%s',txt='%s',title='%s',content_type='%s',docsize=%d,\
keywords='%s',description='%s',crc32=%d,lang='%s',\
header_date='%s',header_subj='%s',header_from='%s',header_group='%s',\
header_refs='%s',msg_id='%s' \
WHERE rec_id=%d",
	status,last_mod_time,(int)(now()+period),
	tag,text_escaped,title_escaped,content_type,size,
	keywords_escaped,descript_escaped,crc32,lang,
	hd_date,hd_subj,hd_from,hd_group,hd_ref,msg_id,
	url_id);
#ifdef HEIKODEBUG	
	if((inn_f = fopen("/tmp/inn_qbuf.tmp","a")))
	{
		fprintf (inn_f,"\n-------------------------\n%s\n",qbuf);
		fclose(inn_f);
	}
	else
	{
		fprintf(stderr,"\nCould not open file\n");
		fflush(stderr);
	}
#endif
	sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	return(IND_OK);
}
#else
int UdmLongUpdateUrl(UDM_INDEXER *Indexer,
int url_id,int status,int changed,int size,int period,
char * tag,int index,
time_t last_mod_time,
char *text_escaped,
char *title_escaped,
char *content_type,
char *keywords_escaped,
char *descript_escaped,
udmcrc32_t crc32,
char *lang,
char *category){
char qbuf[UDMSTRSIZ];
char last_index_time[64]="";

	if(changed)sprintf(last_index_time,",last_index_time=%d",(int)now());
sprintf(qbuf,"\
UPDATE url SET \
status=%d,last_mod_time=%li,\
next_index_time=%d,\
tag='%s',txt='%s',title='%s',content_type='%s',docsize=%d,\
keywords='%s',description='%s',crc32=%d,lang='%s',category='%s' \
WHERE rec_id=%d",
	status,last_mod_time,(int)(now()+period),
	tag,text_escaped,title_escaped,content_type,size,
	keywords_escaped,descript_escaped,crc32,lang,category,url_id);
	sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	return(IND_OK);
}
#endif

/********************** Words ***********************************/

int UdmDeleteWordFromURL(UDM_INDEXER *Indexer,int url_id){
char qbuf[UDMSTRSIZ];
int i,last=0;
	switch(DBMode){
	
	case UDM_DBMODE_MULTI:
		for(i=MINDICT;i<MAXDICT;i++){
			if(last!=DICTNUM(i)){
				sprintf(qbuf,"DELETE FROM dict%d WHERE url_id=%d",
					DICTNUM(i),url_id);
				sql_query(((DB*)(Indexer->db)),qbuf);
				if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
				last=DICTNUM(i);
			}
		}
		break;
	case UDM_DBMODE_MULTI_CRC:
		for(i=MINDICT;i<MAXDICT;i++){
			if(last!=DICTNUM(i)){
#ifdef HAVE_ORACLE8
			    sprintf(qbuf,"DELETE FROM ndict%d WHERE url_id=:1", DICTNUM(i));
				out_rec = 1;
				UDMMEMZERO(out_pos, sizeof(out_pos));
				out_pos[0] = 1;
				out_pos_1[0] = url_id;
#else
				sprintf(qbuf,"DELETE FROM ndict%d WHERE url_id=%d",
					DICTNUM(i),url_id);
#endif
				sql_query(((DB*)(Indexer->db)),qbuf);
				if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
				last=DICTNUM(i);
			}
		}
		break;
	case UDM_DBMODE_SINGLE_CRC:
		sprintf(qbuf,"DELETE FROM ndict WHERE url_id=%d",url_id);
		sql_query(((DB*)(Indexer->db)),qbuf);
		if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
		break;
	default:  /* UDM_DBMODE_SINGLE */
		sprintf(qbuf,"DELETE FROM dict WHERE url_id=%d",url_id);
		sql_query(((DB*)(Indexer->db)),qbuf);
		if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
		break;
	}
	return(IND_OK);
}

static int StoreWordsMulti(UDM_INDEXER * Indexer,int url_id){
int i,wcur,len,res;
UDM_WORD *Word;
char qbuf[UDMSTRSIZ];
char tablename[64]="dict";

	wcur=Indexer->nwords;
	Word=Indexer->Word;
	if(DBUseLock){
		switch(DBType){
			case UDM_DB_ORACLE:
				sql_query((DB*)(Indexer->db),"COMMIT");
				sql_query((DB*)(Indexer->db),"SET TRANSACTION READ WRITE");
				((DB*)(Indexer->db))->commit_fl = 1;
				break;
			default:
				break;
		}
		if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	}

	if(IND_OK!=(res=UdmDeleteWordFromURL(Indexer,url_id)))return(res);
	if(DBUseLock){
		switch(DBType){
			case UDM_DB_ORACLE:
				sql_query((DB*)(Indexer->db),"COMMIT");
				break;
			default:
				break;
		}
		if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	}

	for(i=0;i<wcur;i++){
		if(Word[i].count){
			if(DBMode==UDM_DBMODE_MULTI){
				len=strlen(Word[i].word);
				len=DICTNUM(len);
				if(len)sprintf(tablename,"dict%d",len);
			}
			sprintf(qbuf,"INSERT INTO %s (url_id,word,intag) VALUES(%d,'%s',%d)",tablename,url_id,Word[i].word,Word[i].count);
			sql_query(((DB*)(Indexer->db)),qbuf);
			if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
		}
	}
	if(DBUseLock){
		switch(DBType){
			case UDM_DB_ORACLE:
				sql_query((DB*)(Indexer->db),"COMMIT");
				((DB*)(Indexer->db))->commit_fl = 0;
				break;
			default:
				break;
		}
		if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	}

	return(IND_OK);
}

static UDM_WORD * findword(int wcur,UDM_WORD *Word,char *w){
int i;
	for(i=0;i<wcur;i++)
		if(!strcmp(Word[i].word,w))
			return(&Word[i]);
	return(0);
}
static int StoreWordsSingle(UDM_INDEXER * Indexer,int url_id){
int i,old,new,flush,wcur;
int were,added,deleted,updated;
char *s;UDM_WORD *w,*Word;
char *e;
char qbuf[UDMSTRSIZ];
	wcur=Indexer->nwords;
	Word=Indexer->Word;
	flush=were=added=deleted=updated=0;
	sprintf(qbuf,"SELECT word,intag FROM dict WHERE url_id=%d",url_id);
	((DB*)(Indexer->db))->res=sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	if(DBUseLock){
		switch(DBType){
			case UDM_DB_MYSQL:
				sql_query((DB*)(Indexer->db),"LOCK TABLES dict WRITE");
				break;
			case UDM_DB_PGSQL:
				sql_query((DB*)(Indexer->db),"BEGIN WORK");
				break;
			case UDM_DB_ORACLE:
				sql_query((DB*)(Indexer->db),"COMMIT");
				sql_query((DB*)(Indexer->db),"SET TRANSACTION READ WRITE");
				((DB*)(Indexer->db))->commit_fl = 1;
				break;
			default:
				break;
		}
		if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	}

	were=SQL_NUM_ROWS(((DB*)(Indexer->db))->res);

	for(i=0;i<were;i++){
		e=s=sql_value(((DB*)(Indexer->db))->res,i,0);
		old=atoi(sql_value(((DB*)(Indexer->db))->res,i,1));
		UdmRTrim(s," ");
		if((w=findword(wcur,Word,s))){
			new=w->count;
			if((new)&&(old)&&(new!=old)){
				sprintf(qbuf,"UPDATE dict SET intag=%d WHERE word='%s' AND url_id='%d'",new,s,url_id);
				sql_query(((DB*)(Indexer->db)),qbuf);
				if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
				updated++;
				flush++;
			}
			w->count=0;
		}else{
			sprintf(qbuf,"DELETE FROM dict WHERE url_id=%d AND word='%s'",url_id,s);
			sql_query(((DB*)(Indexer->db)),qbuf);
			if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
			deleted++;flush++;
		}
		if(flush>1024){
			if(DBUseLock){
				switch(DBType){
					case UDM_DB_MYSQL:
						sql_query((DB*)(Indexer->db),"UNLOCK TABLES");
						break;
					case UDM_DB_PGSQL:
						sql_query((DB*)(Indexer->db),"END WORK");
						break;
					case UDM_DB_ORACLE:
						((DB*)(Indexer->db))->commit_fl = 0;
						sql_query((DB*)(Indexer->db),"COMMIT");
						break;
					default:
						break;
				}
				if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
				switch(DBType){
					case UDM_DB_MYSQL:
						sql_query((DB*)(Indexer->db),"LOCK TABLES dict WRITE");
						break;
					case UDM_DB_PGSQL:
						sql_query((DB*)(Indexer->db),"BEGIN WORK");
						break;
					case UDM_DB_ORACLE:
						((DB*)(Indexer->db))->commit_fl = 1;
						sql_query((DB*)(Indexer->db),"COMMIT");
						sql_query((DB*)(Indexer->db),"SET TRANSACTION READ WRITE");
						break;
					default:
						break;
				}
				if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
				
			}
			flush=0;
		}
	}
	SQL_FREE(((DB*)(Indexer->db))->res);
	for(i=0;i<wcur;i++){
		if(Word[i].count){
			sprintf(qbuf,"INSERT INTO dict (url_id,word,intag) VALUES(%d,'%s',%d)",url_id,Word[i].word,Word[i].count);
			sql_query(((DB*)(Indexer->db)),qbuf);
			if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
			flush++;added++;
			if(flush>1024){
				if(DBUseLock){
					switch(DBType){
						case UDM_DB_MYSQL:
							sql_query((DB*)(Indexer->db),"UNLOCK TABLES");
							break;
						case UDM_DB_PGSQL:
							sql_query((DB*)(Indexer->db),"END WORK");
							break;
				    		case UDM_DB_ORACLE:
							((DB*)(Indexer->db))->commit_fl = 0;
							sql_query((DB*)(Indexer->db),"COMMIT");
							break;
						default:
							break;
					}
					if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
					switch(DBType){
						case UDM_DB_MYSQL:
							sql_query((DB*)(Indexer->db),"LOCK TABLES dict WRITE");
							break;
						case UDM_DB_PGSQL:
							sql_query((DB*)(Indexer->db),"BEGIN WORK");
							break;
						case UDM_DB_ORACLE:
							((DB*)(Indexer->db))->commit_fl = 1;
							sql_query((DB*)(Indexer->db),"COMMIT");
							sql_query((DB*)(Indexer->db),"SET TRANSACTION READ WRITE");
							break;
						default:
							break;
					}
					if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
				}
				flush=0;
			}
		}
	}
	if(DBUseLock){
		switch(DBType){
			case UDM_DB_MYSQL:
				sql_query((DB*)(Indexer->db),"UNLOCK TABLES");
				break;
			case UDM_DB_PGSQL:
				sql_query((DB*)(Indexer->db),"END WORK");
				break;
			case UDM_DB_ORACLE:
				((DB*)(Indexer->db))->commit_fl = 0;
				sql_query((DB*)(Indexer->db),"COMMIT");
				break;
			default:
				break;
		}
		if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	}
	return(IND_OK);
}

static int StoreWordsSingleCRC(UDM_INDEXER * Indexer,int url_id){
int i,wcur,res,flush=0;
UDM_WORD *Word;
char tablename[64]="ndict";
udmcrc32_t crc;
char qbuf[UDMSTRSIZ];

	wcur=Indexer->nwords;
	Word=Indexer->Word;
	
	if(DBUseLock){
		switch(DBType){
			case UDM_DB_MYSQL:
				sql_query((DB*)(Indexer->db),"LOCK TABLES ndict WRITE");
				break;
			case UDM_DB_PGSQL:
				sql_query((DB*)(Indexer->db),"BEGIN WORK");
				break;
			case UDM_DB_ORACLE:
				((DB*)(Indexer->db))->commit_fl = 1;
				sql_query((DB*)(Indexer->db),"COMMIT");
				sql_query((DB*)(Indexer->db),"SET TRANSACTION READ WRITE");
				break;
			default:
				break;
		}
		if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	}

	if(IND_OK!=(res=UdmDeleteWordFromURL(Indexer,url_id)))return(res);

	if(DBUseLock){
		switch(DBType){
			case UDM_DB_ORACLE:
				sql_query((DB*)(Indexer->db),"COMMIT");
				break;
			default:
				break;
		}
		if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	}

	for(i=0;i<wcur;i++){
#ifdef HAVE_ORACLE8
		if(Word[i].count){
			crc=UdmStrCRC32(Word[i].word);
			out_pos_1[flush] = url_id;
			out_pos_2[flush] = crc;
			out_pos_3[flush] = Word[i].count;
			flush ++;
		}
		
		if (flush && (flush == BUF_OUT_SIZE || i+1==wcur)){
			sprintf(qbuf,"INSERT INTO %s (url_id,word_id,intag) VALUES(:1, :2, :3)", tablename);
			out_rec = flush;
			UDMMEMZERO(out_pos, sizeof(out_pos));
			out_pos[0] = 1; out_pos[1] = 2; out_pos[2] = 3;
			sql_query(((DB*)(Indexer->db)),qbuf);
			if(UdmDBErrorCode(Indexer->db))
				return(IND_ERROR);
			flush = 0;
		}
#else
		if(Word[i].count){

			crc=UdmStrCRC32(Word[i].word);
			sprintf(qbuf,"INSERT INTO %s (url_id,word_id,intag) VALUES(%d,%d,%d)",tablename,url_id,crc,Word[i].count);
			sql_query(((DB*)(Indexer->db)),qbuf);
			if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);

		}
#endif
	}

	if(DBUseLock){
		switch(DBType){
			case UDM_DB_MYSQL:
				sql_query((DB*)(Indexer->db),"UNLOCK TABLES");
				break;
			case UDM_DB_PGSQL:
				sql_query((DB*)(Indexer->db),"END WORK");
				break;
			case UDM_DB_ORACLE:
				((DB*)(Indexer->db))->commit_fl = 0;
				sql_query((DB*)(Indexer->db),"COMMIT");
				break;
			default:
				break;
		}
		if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	}
	return(IND_OK);
}

int findcrc(int wcur, udmcrc32_t *crc_ar, udmcrc32_t crc){
int i;
	if (!crc_ar)
		return -1;
        for(i=0;i<wcur;i++){
                if(crc_ar[i] == crc)
                        return(i);
	}
        return(-1);
}

static int StoreWordsMultiCRC(UDM_INDEXER * Indexer,int url_id, int status){
int i,wcur, n, dict_prev, len;
UDM_WORD *Word;
char tablename[64]="ndict";
udmcrc32_t crc;
char qbuf[UDMSTRSIZ];
int flush, old_intag, nwords, ind;
char tbl_nm[64];
udmcrc32_t old_word_id, *crc_ar=NULL;


	wcur=Indexer->nwords;
	Word=Indexer->Word;

	/* Fill crc */
	if (wcur)
		crc_ar = (udmcrc32_t*)UdmXmalloc(wcur*sizeof(udmcrc32_t));
	for( i=0; i<wcur; i++)
		crc_ar[i]=UdmStrCRC32(Word[i].word);

	if(DBUseLock){
		switch(DBType){
			case UDM_DB_ORACLE:
				((DB*)(Indexer->db))->commit_fl = 1;
				((DB*)(Indexer->db))->res_limit=0;
				sql_query((DB*)(Indexer->db),"COMMIT");
				sql_query((DB*)(Indexer->db),"SET TRANSACTION READ WRITE");
				break;
			default:
				break;
		}
		if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	}

	for( n=0; n<NDICTS; n++){
		if (dict_prev == dictlen[n])
			continue;
		dict_prev = dictlen[n];
		udm_snprintf(tbl_nm, 64, "%s%d", tablename, dictlen[n]);
#ifdef HAVE_ORACLE8
		if (status){

			sprintf(qbuf,"SELECT word_id, intag FROM %s WHERE url_id=:1",tbl_nm);
			out_rec = 1;
			UDMMEMZERO(out_pos, sizeof(out_pos));
			out_pos[0] = 1;
			out_pos_1[0] = url_id;
#else
		if (status){
			sprintf(qbuf,"SELECT word_id, intag FROM %s WHERE url_id=%d",tbl_nm, url_id);
#endif
			((DB*)(Indexer->db))->res=sql_query(((DB*)(Indexer->db)),qbuf);
			if(UdmDBErrorCode(Indexer->db))
				return(IND_ERROR);
			nwords=SQL_NUM_ROWS(((DB*)(Indexer->db))->res);
		}else
			nwords=0;

		flush = 0;
		/* Delete old words */
		for (i=0; i<nwords; i++){
			old_word_id =(udmcrc32_t)atoi(sql_value(((DB*)(Indexer->db))->res,i,0));
			old_intag = atoi(sql_value(((DB*)(Indexer->db))->res,i,1));
			if ((ind=findcrc(wcur, crc_ar, old_word_id)) !=-1){
	    			if(Word[ind].count && (old_intag)&&(Word[ind].count!=old_intag)){
#ifdef HAVE_ORACLE8
					sprintf(qbuf,"UPDATE %s SET intag=:1 WHERE word_id=:2 AND url_id=:3", tbl_nm);
					out_rec = 1;
					UDMMEMZERO(out_pos, sizeof(out_pos));
					out_pos[0] = 1; out_pos[1] = 2; out_pos[2] = 3;
					out_pos_1[0] = Word[ind].count;
					out_pos_2[0] = old_word_id;
					out_pos_3[0] = url_id;
#else
					sprintf(qbuf,"UPDATE %s SET intag=%d WHERE word_id=%d AND url_id=%d",
						    tbl_nm, Word[ind].count, old_word_id, url_id);
#endif
					sql_query(((DB*)(Indexer->db)),qbuf);
					if(UdmDBErrorCode(Indexer->db))
						return(IND_ERROR);
				}
				Word[ind].count=0;
			}else{
#ifdef HAVE_ORACLE8
				out_pos_1[flush] = url_id;
				out_pos_2[flush] = old_word_id;
			        flush++;
			}
			if (flush && (flush == BUF_OUT_SIZE || i+1==nwords)){
				sprintf(qbuf,"DELETE FROM %s WHERE url_id=:1 AND word_id=:2",tbl_nm);
				out_rec = flush;
				UDMMEMZERO(out_pos, sizeof(out_pos));
				out_pos[0] = 1;out_pos[1] = 2;
#else
				sprintf(qbuf,"DELETE FROM %s WHERE url_id=%d AND word_id=%d",
					    tbl_nm, url_id, old_word_id);
#endif
				sql_query(((DB*)(Indexer->db)),qbuf);
				if(UdmDBErrorCode(Indexer->db))
					return(IND_ERROR);
				flush = 0;
			}
		}
		if (status)
			SQL_FREE(((DB*)(Indexer->db))->res);
		flush = 0;
		/* Insert new words */
	        for(i=0;i<wcur;i++){
			len=strlen(Word[i].word);
			if (Word[i].count && (DICTNUM(len) == dictlen[n])){
				crc=crc_ar[i];
#ifdef HAVE_ORACLE8
				out_pos_1[flush] = url_id;
				out_pos_2[flush] = crc;
				out_pos_3[flush] = Word[i].count;
				flush ++;
				Word[i].count = 0;
			}
			if (flush && (flush == BUF_OUT_SIZE || i+1==wcur)){
				sprintf(qbuf,"INSERT INTO %s (url_id,word_id,intag) VALUES(:1, :2, :3)",tbl_nm);
				out_rec = flush;
				UDMMEMZERO(out_pos, sizeof(out_pos));
				out_pos[0] = 1;out_pos[1] = 2; out_pos[2] = 3;
#else
				sprintf(qbuf,"INSERT INTO %s (url_id,word_id,intag) VALUES(%d,%d,%d)",
					tbl_nm, url_id,crc,Word[i].count);
#endif
				sql_query(((DB*)(Indexer->db)),qbuf);
				if(UdmDBErrorCode(Indexer->db))
					return(IND_ERROR);
				flush = 0;
			}
		}
	}
	UDM_FREE(crc_ar);
	if(DBUseLock){
		switch(DBType){
			case UDM_DB_ORACLE:
				((DB*)(Indexer->db))->commit_fl = 0;
				sql_query((DB*)(Indexer->db),"COMMIT");
				break;
			default:
				break;
		}
		if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	}

	return(IND_OK);
}


int UdmStoreWords(UDM_INDEXER * Indexer,int url_id, int status){
int res;

	switch(DBMode){
		case UDM_DBMODE_MULTI:
			res=StoreWordsMulti(Indexer,url_id);
			break;
		case UDM_DBMODE_SINGLE_CRC:
			res=StoreWordsSingleCRC(Indexer,url_id);
			break;
		case UDM_DBMODE_MULTI_CRC:
			res=StoreWordsMultiCRC(Indexer,url_id, status);
			break;
		default:
			res=StoreWordsSingle(Indexer,url_id);
			break;
	}

	return(res);
}


int UdmDeleteAllFromUrl(UDM_INDEXER *Indexer){
char qbuf[UDMSTRSIZ];
#ifdef HAVE_ORACLE8
	sprintf(qbuf,"TRUNCATE TABLE url");
#else
	sprintf(qbuf,"DELETE FROM url");
#endif
	sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	return(IND_OK);
}

int UdmDeleteAllFromDict(UDM_INDEXER *Indexer){
char qbuf[UDMSTRSIZ];
int i,last=0;
	switch(DBMode){
	case UDM_DBMODE_MULTI:
		for(i=MINDICT;i<MAXDICT;i++){
			if(last!=DICTNUM(i)){
				if (DBType == UDM_DB_ORACLE)
					sprintf(qbuf,"TRUNCATE TABLE dict%d",DICTNUM(i));
				else
					sprintf(qbuf,"DELETE FROM dict%d",DICTNUM(i));

				sql_query(((DB*)(Indexer->db)),qbuf);
				if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
				last=DICTNUM(i);
			}
		}
		break;
	case UDM_DBMODE_MULTI_CRC:
		for(i=MINDICT;i<MAXDICT;i++){
			if(last!=DICTNUM(i)){
				if (DBType == UDM_DB_ORACLE)
					sprintf(qbuf,"TRUNCATE TABLE ndict%d",DICTNUM(i));
				else
					sprintf(qbuf,"DELETE FROM ndict%d",DICTNUM(i));
	
				sql_query(((DB*)(Indexer->db)),qbuf);
				if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
				last=DICTNUM(i);
			}
		}
		break;
	case UDM_DBMODE_SINGLE_CRC:
		if (DBType == UDM_DB_ORACLE)
			sprintf(qbuf,"TRUNCATE TABLE ndict");
		else
			sprintf(qbuf,"DELETE FROM ndict");
    	
		sql_query(((DB*)(Indexer->db)),qbuf);
		if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
		break;
	default:
		if (DBType == UDM_DB_ORACLE)
			sprintf(qbuf,"TRUNCATE TABLE dict");
		else
			sprintf(qbuf,"DELETE FROM dict");

		sql_query(((DB*)(Indexer->db)),qbuf);
		if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
		break;
	}
	return(IND_OK);
}


/************************ Clones stuff ***************************/
int UdmFindOrigin(UDM_INDEXER *Indexer, udmcrc32_t crc32, int size){
	int i=0,origin=0;
	char *o;
	char qbuf[UDMSTRSIZ];
	
	if (crc32==0)
		return 0;
#ifdef HAVE_ORACLE8
	sprintf(qbuf,"SELECT rec_id FROM url WHERE crc32=:1 AND status=200 AND docsize=:2");
        out_rec = 1;
        UDMMEMZERO(out_pos, sizeof(out_pos));
        out_pos[0] = 1; out_pos[1] = 2;
        out_pos_1[0] = crc32;
        out_pos_2[0] = size;
#else
	sprintf(qbuf,"SELECT rec_id FROM url WHERE crc32=%d AND status=200 AND docsize=%d",crc32, size);
#endif
	((DB*)(Indexer->db))->res=sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))return(0);
	for(i=0;i<SQL_NUM_ROWS(((DB*)(Indexer->db))->res);i++)
		if((o=sql_value(((DB*)(Indexer->db))->res,i,0)))
			if((!origin)||(origin>atoi(o)))
				origin=atoi(o);
	SQL_FREE(((DB*)(Indexer->db))->res);
	return(origin);
}


int  UdmUpdateClone(UDM_INDEXER *Indexer, int url_id, int status, int period,
		char *content_type, time_t last_mod_time, udmcrc32_t crc32){

	char qbuf[UDMSTRSIZ];
	
sprintf(qbuf,
"UPDATE url \
SET crc32=%d,status=%d,content_type='%s',\
last_mod_time=%li,next_index_time=%d \
WHERE rec_id=%d",crc32,status,content_type,\
last_mod_time,(int)(now()+period),url_id);
	sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	return(IND_OK);
}


UDM_DOCUMENT * UdmGetDocInfo(UDM_INDEXER *Indexer,int index_flags){
char urlin[UDMSTRSIZ]="";
char sortstr[64]="";
char updstr[64]="";
char lmtstr[64]="";
int i=0,url_num;
UDM_DOCUMENT * Result=NULL;
char qbuf[UDMSTRSIZ];

	if(currow>=nrows){
		if(urlres)SQL_FREE(urlres);

		url_num=URL_SELECT_CACHE;

		/*
		if(MaxURLNumber>0){
			url_num=MaxURLNumber-GetIndexedNumber();
			if(url_num>URL_SELECT_CACHE)
				url_num=URL_SELECT_CACHE;
		}
		*/

		if(DBUseLock){
			switch(DBType){
				case UDM_DB_MYSQL:
					sql_query((DB*)(Indexer->db),"LOCK TABLES url WRITE");
					break;
				case UDM_DB_PGSQL:
					sql_query((DB*)(Indexer->db),"BEGIN WORK");
					sql_query((DB*)(Indexer->db),"LOCK url");
					break;
				case UDM_DB_ORACLE:
					sprintf(updstr, " FOR UPDATE ");
					sprintf(lmtstr, " AND ROWNUM <=:2");
					break;
				default:
					break;
			}
			if(UdmDBErrorCode(Indexer->db))return(NULL);
		}


		sprintf(sortstr,(index_flags&UDM_FLAG_EXP_FIRST)?" ORDER BY next_index_time":"");
		if (index_flags & UDM_FLAG_SORT_HOPS) {
			if (strlen(sortstr) == 0) {
				sprintf(sortstr," ORDER BY hops ASC ");
			} else {
				strcat(sortstr,", hops ASC ");
			}
		}

#ifdef HAVE_SQL_LIMIT
		sprintf(qbuf,
		"SELECT url,rec_id,docsize,status,last_index_time,hops,crc32,last_mod_time FROM url WHERE next_index_time<=%d %s%s%s%s%s%s LIMIT %d",
			(int)now(),tagstr,urlstr,statusstr,langstr,catstr,sortstr,url_num);
#else
		((DB*)(Indexer->db))->res_limit=url_num;
#if HAVE_ORACLE8
		out_rec = 1;
		UDMMEMZERO(out_pos, sizeof(out_pos));
		out_pos[0] = 1; out_pos[1] = 2;
		out_pos_1[0] = (int)now();
		out_pos_2[0] = url_num;

		sprintf(qbuf,
		"SELECT url,rec_id,docsize,status,last_index_time,hops,crc32,last_mod_time FROM url WHERE next_index_time<=:1 %s%s%s%s%s%s%s%s",
			tagstr,urlstr,statusstr,langstr,lmtstr,sortstr,catstr,updstr);
#else
		sprintf(qbuf,
		"SELECT url,rec_id,docsize,status,last_index_time,hops,crc32,last_mod_time FROM url WHERE next_index_time<=%d %s%s%s%s%s%s%s%s",
			(int)now(),tagstr,urlstr,statusstr,langstr,lmtstr,sortstr,catstr,updstr);
#endif
#endif
		urlres=sql_query(((DB*)(Indexer->db)),qbuf);
		if(UdmDBErrorCode(Indexer->db)){
			return(NULL);
		}
		nrows=SQL_NUM_ROWS(urlres);
		currow=0;
		if(!nrows){
			if(DBUseLock){
				switch(DBType){
					case UDM_DB_MYSQL:
						sql_query((DB*)(Indexer->db),"UNLOCK TABLES");
						break;
					case UDM_DB_PGSQL:
						sql_query((DB*)(Indexer->db),"END WORK");
						break;
					default:
						break;
				}
			}
			return(NULL);
		}
		urlin[0]=0;
		
		for(i=0;i<nrows;i++){
#ifdef  HAVE_SQL_IN
			if(urlin[0])strcat(urlin,",");
			strcat(urlin,sql_value(urlres,i,1));
#else
			sprintf(qbuf,"UPDATE url SET next_index_time=%d WHERE rec_id=%d",(int)(now()+URL_LOCK_TIME),
				atoi(sql_value(urlres,i,1)));
			sql_query(((DB*)(Indexer->db)),qbuf);
			if(UdmDBErrorCode(Indexer->db)){
				return(NULL);
			}
#endif
		}
#ifdef HAVE_SQL_IN
		sprintf(qbuf,"UPDATE url SET next_index_time=%d WHERE rec_id in (%s)",(int)(now()+URL_LOCK_TIME),urlin);
		sql_query(((DB*)(Indexer->db)),qbuf);
		if(UdmDBErrorCode(Indexer->db)){
			return(NULL);
		}
#endif
		if(DBUseLock){
			switch(DBType){
				case UDM_DB_MYSQL:
					sql_query((DB*)(Indexer->db),"UNLOCK TABLES");
					break;
				case UDM_DB_PGSQL:
					sql_query((DB*)(Indexer->db),"END WORK");
					break;
				default:
					break;
			}
			if(UdmDBErrorCode(Indexer->db))return(NULL);
		}
	}
	Result=(UDM_DOCUMENT *)malloc(sizeof(UDM_DOCUMENT));
	Result->content_type=NULL;
	Result->title=NULL;
	Result->text=NULL;
	Result->last_index_time=NULL;
	Result->next_index_time=NULL;
	Result->keywords=NULL;
	Result->description=NULL;
	Result->crc32=0;
	Result->url=strdup(sql_value(urlres,currow,0));
	Result->url_id=atoi(sql_value(urlres,currow,1));
	Result->size=atoi(sql_value(urlres,currow,2));
	Result->status=atoi(sql_value(urlres,currow,3));
	Result->last_index_time=strdup(sql_value(urlres,currow,4));
	Result->hops=atoi(sql_value(urlres,currow,5));
	Result->crc32=strtol(sql_value(urlres,currow,6), NULL, 10);
	Result->last_mod_time=atol(sql_value(urlres,currow,7));
	currow++;
	return(Result);
}



/************************************************************/
/* Misc functions                                           */
/************************************************************/

typedef struct stat_struct {
	int status;
	int expired;
	int total;
} UDM_STAT;

static UDM_SPELL t_Spell;

int UdmDBImportAffixes(void *db){
int suffixes,i;

    ((DB*)db)->res=sql_query((DB*)db,"SELECT flag,lang,mask,find,repl FROM affix WHERE type='s'");
    if(((DB*)db)->errcode)return(((DB*)db)->errcode);

    suffixes=SQL_NUM_ROWS(((DB*)db)->res);
    for(i=0;i<suffixes;i++){
	char *t_flag;
	char *t_lang;
	char *t_mask;
	char *t_find;
	char *t_repl;
		
	t_flag=sql_value(((DB*)db)->res,i,0);		
	t_lang=sql_value(((DB*)db)->res,i,1);
	t_mask=sql_value(((DB*)db)->res,i,2);
	t_find=sql_value(((DB*)db)->res,i,3);
	t_repl=sql_value(((DB*)db)->res,i,4);
	
	UdmAddAffix(*t_flag,t_lang,t_mask,t_find,t_repl);
    }
    SQL_FREE(((DB*)db)->res);
    return(0);
}

UDM_SPELL * UdmFindWordDB(char *word){
char qbuf[UDMSTRSIZ];
int found;
int i;
void * db;
char *c_flag=NULL;
char *c_lang=NULL;
char *t_flag=NULL;
char *t_lang=NULL;

	db=UdmAllocDB(UDM_OPEN_MODE_READ);
	sprintf(qbuf,"SELECT flag,lang FROM spell WHERE word='%s'", word);
	((DB*)db)->res=sql_query((DB*)db,qbuf);
	if(((DB*)db)->errcode) {
	    printf("Warning: %s<br>Ispell mode cannot work properly<br>\n",UdmDBErrorMsg(db));
	    return(NULL);
	}

	found=SQL_NUM_ROWS(((DB*)db)->res);
	
	for (i=0; i<found; i++) {
	    c_flag=sql_value(((DB*)db)->res,i,0);		
	    c_lang=sql_value(((DB*)db)->res,i,1);
	    if (strlen(c_flag) || (!i)) {
		t_flag=c_flag;
		t_lang=c_lang;
	    }
	}
	    
	SQL_FREE(((DB*)db)->res);
	UdmFreeDB(db);
	    
	if (found) {
	    t_Spell.word=strdup(word);
	    strncpy(t_Spell.flag,t_flag,10);
	    strncpy(t_Spell.lang,t_lang,2);
	    return (&t_Spell);
	} else return(NULL);
}

int UdmInsertAffix(UDM_INDEXER *Indexer,char flag,char *lang,char *mask,char *find,char *repl,char *type){
char qbuf[UDMSTRSIZ];
char escmask[UDMSTRSIZ];
char escfind[UDMSTRSIZ];
char escrepl[UDMSTRSIZ];

	escstr(escmask,mask);
	escstr(escfind,find);
	escstr(escrepl,repl);

	sprintf(qbuf,"INSERT INTO affix (flag,type,lang,mask,find,repl) VALUES ('%c','%s','%s','%s$','%s','%s')",flag,type,lang,escmask,escfind,escrepl);
	sql_query(((DB*)(Indexer->db)),qbuf);return(UdmDBErrorCode(Indexer->db));
}

int UdmInsertSpell(UDM_INDEXER *Indexer,char *flag,char *lang,char *word){
char qbuf[UDMSTRSIZ];
char escword[UDMSTRSIZ];

	escstr(escword,word);

	sprintf(qbuf,"INSERT INTO spell (word,flag,lang) VALUES ('%s','%s','%s')",escword,flag,lang);
	sql_query(((DB*)(Indexer->db)),qbuf);return(UdmDBErrorCode(Indexer->db));
}

__INDLIB__ int UdmGetStatistics(UDM_INDEXER *Indexer){
int i,j;
char qbuf[UDMSTRSIZ];
int expired_total=0;
int total_total=0;

	if(!StatInfo)return(IND_OK);

#if (HAVE_MYSQL|HAVE_PGSQL)
#if   (HAVE_MYSQL)
	sprintf(qbuf,"SELECT status,sum(next_index_time<=%d),count(*) FROM url WHERE 1=1 %s%s%s%s%s GROUP BY status",(int)now(),tagstr,urlstr,statusstr,langstr,catstr);
#elif (HAVE_PGSQL)
	sprintf(qbuf,"SELECT status,sum(case when next_index_time<=%d then 1 else 0 end),count(*) FROM url WHERE 1=1 %s%s%s%s%s GROUP BY status",(int)now(),tagstr,urlstr,statusstr,langstr,catstr);
#endif
	((DB*)(Indexer->db))->res=sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	if((j=SQL_NUM_ROWS(((DB*)(Indexer->db))->res))){
		for(i=0;i<j;i++){
			int status,expired,total;
			status=atoi(sql_value(((DB*)(Indexer->db))->res,i,0));
			expired=atoi(sql_value(((DB*)(Indexer->db))->res,i,1));
			total=atoi(sql_value(((DB*)(Indexer->db))->res,i,2));
			total_total+=total;
			expired_total+=expired;
			StatInfo(Indexer->handle,status,expired,total,
				UdmHTTPErrMsg(status));
		}
	}
	SQL_FREE(((DB*)(Indexer->db))->res);
#elif (HAVE_IBASE||HAVE_MSQL || HAVE_IODBC || HAVE_UNIXODBC || HAVE_SOLID || HAVE_VIRT || HAVE_ORACLE8 || HAVE_ORACLE7 || HAVE_EASYSOFT)
	sprintf(qbuf,"SELECT status,next_index_time FROM url WHERE rec_id>=0 %s%s%s%s%s",tagstr,urlstr,statusstr,langstr,catstr);
	((DB*)(Indexer->db))->res=sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	if(SQL_NUM_ROWS(((DB*)(Indexer->db))->res)){
		UDM_STAT *stat=NULL;
		int n,num=0;
		
		n=now();
		for(i=0;i<SQL_NUM_ROWS(((DB*)(Indexer->db))->res);i++){
			for(j=0;j<num;j++){
				if(stat[j].status==atoi(sql_value(((DB*)(Indexer->db))->res,i,0))){
					if(atoi(sql_value(((DB*)(Indexer->db))->res,i,1))<=n)
						stat[j].expired++;
					stat[j].total++;
					break;
				}
			}
			if(j==num){
			
				if(!num)
					stat=(UDM_STAT *)malloc(sizeof(UDM_STAT)*(num+1));
				else
					stat=(UDM_STAT *)realloc(stat,sizeof(UDM_STAT)*(num+1));
				stat[j].status=atoi(sql_value(((DB*)(Indexer->db))->res,i,0));
				stat[j].expired=0;
				if(atoi(sql_value(((DB*)(Indexer->db))->res,i,1))<=n)
					stat[j].expired++;
				stat[j].total=1;
				num++;
			}
		}
		for(i=0; i<num; i++){
			StatInfo(Indexer->handle,
				stat->status,stat->expired,stat->total,
				UdmHTTPErrMsg(stat->status));
			total_total+=stat->total;
			expired_total+=stat->expired;
			stat++;
		}
	}
	SQL_FREE(((DB*)(Indexer->db))->res);
#endif		
	StatInfo(Indexer->handle,-1,expired_total,total_total,"");
	return(IND_OK);
}



__INDLIB__ int UdmGetReferers(UDM_INDEXER *Indexer){
int i,j;
char qbuf[UDMSTRSIZ];
	sprintf(qbuf,"SELECT url.status,url2.url,url.url FROM url,url url2 WHERE url.referrer=url2.rec_id %s%s%s%s%s",tagstr,urlstr,statusstr,langstr,catstr);
	((DB*)(Indexer->db))->res=sql_query(((DB*)(Indexer->db)),qbuf);
	if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
	j=SQL_NUM_ROWS(((DB*)(Indexer->db))->res);
	for(i=0;i<j;i++){
		if(RefInfo)RefInfo(
			atoi(sql_value(((DB*)(Indexer->db))->res,i,0)),
			sql_value(((DB*)(Indexer->db))->res,i,2),
			sql_value(((DB*)(Indexer->db))->res,i,1)
		);
	}
	SQL_FREE(((DB*)(Indexer->db))->res);
	return(IND_OK);
}

int UdmClearDB(UDM_INDEXER *Indexer){
int i,j,err;
char qbuf[UDMSTRSIZ];

	if(!tagstr[0]&&!urlstr[0]&&!statusstr[0]&&!langstr[0]&&!catstr[0]){
#ifdef NEWS_EXT
		if((IND_OK!=(err=UdmDeleteAllFromThread(Indexer))))return(err);
#endif
		if((IND_OK!=(err=UdmDeleteAllFromDict(Indexer))))return(err);
		if((IND_OK!=(err=UdmDeleteAllFromUrl(Indexer))))return(err);
		if((IND_OK!=(err=UdmDeleteAllFromRobots(Indexer))))return(err);
	}else{
		j=0;
		while(1){
			sprintf(qbuf,"SELECT rec_id FROM url WHERE rec_id>=0 %s%s%s%s%s LIMIT %d",tagstr,urlstr,statusstr,langstr,catstr,URL_DELETE_CACHE);
			((DB*)(Indexer->db))->res=sql_query(((DB*)(Indexer->db)),qbuf);
			if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
			if(SQL_NUM_ROWS(((DB*)(Indexer->db))->res)){
#ifdef HAVE_SQL_IN
				int last=0;
				char urlin[UDMSTRSIZ]="";
				urlin[0]=0;
				for(i=0;i<SQL_NUM_ROWS(((DB*)(Indexer->db))->res);i++){
					if(i)strcat(urlin,",");
					strcat(urlin,sql_value(((DB*)(Indexer->db))->res,i,0));
				}
				j+=i;
				switch(DBMode){
				case UDM_DBMODE_MULTI:
					for(i=MINDICT;i<MAXDICT;i++){
						if(last!=DICTNUM(i)){
							sprintf(qbuf,"DELETE FROM dict%d WHERE url_id in (%s)",DICTNUM(i),urlin);
							sql_query(((DB*)(Indexer->db)),qbuf);
							if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
							last=DICTNUM(i);
						}
					}
					break;
				case UDM_DBMODE_MULTI_CRC:
					for(i=MINDICT;i<MAXDICT;i++){
						if(last!=DICTNUM(i)){
							sprintf(qbuf,"DELETE FROM ndict%d WHERE url_id in (%s)",
								DICTNUM(i),urlin);
							sql_query(((DB*)(Indexer->db)),qbuf);
							if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
							last=DICTNUM(i);
						}
					}
					break;
				default:
					sprintf(qbuf,"DELETE FROM dict WHERE url_id in (%s)",urlin);
					sql_query(((DB*)(Indexer->db)),qbuf);
					if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);
					break;
				}
				sprintf(qbuf,"DELETE FROM url WHERE rec_id in (%s)",urlin);
				sql_query(((DB*)(Indexer->db)),qbuf);
				if(UdmDBErrorCode(Indexer->db))return(IND_ERROR);

				SQL_FREE(((DB*)(Indexer->db))->res);
#else
				for(i=0;i<SQL_NUM_ROWS(((DB*)(Indexer->db))->res);i++)
					if(IND_OK!=UdmDeleteUrl(Indexer,atoi(sql_value(((DB*)(Indexer->db))->res,i,0))))
						return(IND_ERROR);
				j+=i;
#endif
				sprintf(qbuf,"%d",j);
			}else{
				SQL_FREE(((DB*)(Indexer->db))->res);
				break;
			}
		}
	}
	return(IND_OK);
}

/********************* Categories ************************************/
UDM_CATEGORY * UdmCatList(void * db,char * addr){
UDM_CATEGORY * r=NULL;
int i;
char qbuf[UDMSTRSIZ];

	if(!addr)addr="";
	sprintf(qbuf,"SELECT rec_id,path,link,name FROM categories WHERE path LIKE '%s__'",addr);
	((DB*)db)->res=sql_query((DB*)db,qbuf);
	if(UdmDBErrorCode(db))return(NULL);
	if(SQL_NUM_ROWS(((DB*)db)->res)){
		r=(UDM_CATEGORY*)malloc(sizeof(UDM_CATEGORY)*(SQL_NUM_ROWS(((DB*)db)->res)+1));
		for(i=0;i<SQL_NUM_ROWS(((DB*)db)->res);i++){
			r[i].rec_id=atoi(sql_value(((DB*)db)->res,i,0));
			strcpy(r[i].path,sql_value(((DB*)db)->res,i,1));
			strcpy(r[i].link,sql_value(((DB*)db)->res,i,2));
			strcpy(r[i].name,sql_value(((DB*)db)->res,i,3));
			r[i+1].rec_id=0;
		}
	}
	SQL_FREE(((DB*)db)->res);
	return(r);
}

UDM_CATEGORY * UdmCatPath(void * db,char * addr){
UDM_CATEGORY * r=NULL;
int i,l;
char qbuf[UDMSTRSIZ];

	if(!addr)addr="";
	l=(strlen(addr)/2)+1;

	r=(UDM_CATEGORY*)malloc(sizeof(UDM_CATEGORY)*(l+1));
	r[0].rec_id=0;

	for(i=0;i<l;i++){
		char head[128];
		r[i+1].rec_id=0;
		strncpy(head,addr,i*2);head[i*2]=0;
		sprintf(qbuf,"SELECT rec_id,path,link,name FROM categories WHERE path='%s'",head);
		((DB*)db)->res=sql_query((DB*)db,qbuf);
		if(UdmDBErrorCode(db))return(NULL);
		if(SQL_NUM_ROWS(((DB*)db)->res)){
			r[i].rec_id=atoi(sql_value(((DB*)db)->res,0,0));
			strcpy(r[i].path,sql_value(((DB*)db)->res,0,1));
			strcpy(r[i].link,sql_value(((DB*)db)->res,0,2));
			strcpy(r[i].name,sql_value(((DB*)db)->res,0,3));
		}
		SQL_FREE(((DB*)db)->res);
	}
	return(r);
}


/******************* Search stuff ************************************/

UDM_DOCUMENT * UdmCloneList(void * db,udmcrc32_t crc32){
UDM_DOCUMENT * r=NULL;
int i;
char qbuf[UDMSTRSIZ];

	sprintf(qbuf,"SELECT rec_id,url,content_type,last_mod_time FROM url WHERE crc32=%d AND status=200",crc32);
	((DB*)db)->res=sql_query((DB*)db,qbuf);
	if(UdmDBErrorCode(db))return(NULL);
	r=(UDM_DOCUMENT*)malloc(sizeof(UDM_DOCUMENT)*(SQL_NUM_ROWS(((DB*)db)->res)+1));
	for(i=0;i<SQL_NUM_ROWS(((DB*)db)->res);i++){
		r[i].url_id=atoi(sql_value(((DB*)db)->res,i,0));
		r[i].url=strdup(sql_value(((DB*)db)->res,i,1));
		r[i].content_type=strdup(sql_value(((DB*)db)->res,i,2));
		r[i].last_mod_time=atol(sql_value(((DB*)db)->res,i,3));
		r[i+1].url_id=0;
	}
	SQL_FREE(((DB*)db)->res);
	return(r);
}
/********************************************************/

typedef struct wrd{
	int  url_id;
	int  count;
	int  weight;
} SEARCHWORD;

static int cmpword(const void *s1,const void *s2){
int res;
	if(!(res=(((SEARCHWORD*)s2)->count)-(((SEARCHWORD*)s1)->count)))
		if(!(res=(((SEARCHWORD*)s2)->weight-(((SEARCHWORD*)s1)->weight))))
			if(!(res=(((SEARCHWORD*)s1)->url_id-(((SEARCHWORD*)s2)->url_id))));
	return(res);
}
static int cmpurlid(const void *s1,const void *s2){
	return(((SEARCHWORD*)s2)->url_id-((SEARCHWORD*)s1)->url_id);
}

UDM_DOCUMENT * UdmFind(void * db,char *q,int np,int ps,int search_mode,int sort_order,char *wordinfo,int *found){
UDM_DOCUMENT * r=NULL;
char *w;
int i,j,num_words=0,first,last,num;
SEARCHWORD * wrd=NULL;
char qbuf[UDMSTRSIZ];
char q_esc[UDMSTRSIZ];
char *lasttok;
int url_id;
int weight=0;
int nwrd=0;
int curnum=0;
#ifdef HAVE_MYSQL
MYSQL_ROW row;
#endif
char instr[UDMSTRSIZ]="";

	UdmLoadStopList(db);
	if(UdmDBErrorCode(db))return(NULL);

	*wordinfo=0;

	/* To track it later */
	escstr(q_esc,q);

	w=UdmGetWord(q,&lasttok,UdmGetDefaultCharset());
	while(w){
		int len;
		char ** ww;
		char *rw;
		char tablename[32]="dict";

		ww=UdmNormalizeWord(w);
		rw=ww?*ww:w;

		if(UdmIsStopWord(rw)){
			sprintf(UDM_STREND(wordinfo)," %s :stopword",rw);
			w=UdmGetWord(NULL,&lasttok,UdmGetDefaultCharset());
			continue;
		}
		num_words++;
		switch(DBMode){
		case UDM_DBMODE_MULTI:
			len=strlen(rw);len=DICTNUM(len);
			sprintf(tablename,"dict%d",len);
			break;
		case UDM_DBMODE_MULTI_CRC:
			len=strlen(rw);len=DICTNUM(len);
			sprintf(tablename,"ndict%d",len);
			break;
		case UDM_DBMODE_SINGLE_CRC:
			strcpy(tablename,"ndict");
			break;
		default:
			break;
		}
		if((DBMode==UDM_DBMODE_SINGLE_CRC)||
			(DBMode==UDM_DBMODE_MULTI_CRC)){
			udmcrc32_t crc;
			crc=UdmStrCRC32(rw);
			if(*tagstr || *statusstr || *urlstr || *langstr || *timestr || *catstr)
				sprintf(qbuf,"\
SELECT %s.url_id,%s.intag \
FROM %s,url \
WHERE %s.word_id=%d \
AND url.rec_id=%s.url_id %s%s%s%s%s%s",
				tablename,tablename,tablename,tablename,crc,
				tablename,tagstr,statusstr,urlstr,langstr,
				timestr,catstr);
			else
				sprintf(qbuf,"\
SELECT url_id,intag \
FROM %s \
WHERE word_id=%d",
				tablename,crc);
		}else{
			if(*tagstr||*statusstr||*urlstr||*langstr||*catstr)
		
				sprintf(qbuf,"\
SELECT %s.url_id,%s.intag \
FROM %s,url \
WHERE %s.word='%s' \
AND url.rec_id=%s.url_id %s%s%s%s%s%s",
				tablename,tablename,tablename,tablename,
				rw,tablename,tagstr,statusstr,urlstr,langstr,
				timestr,catstr);
			else
				sprintf(qbuf,"\
SELECT url_id,intag \
FROM %s \
WHERE word='%s'",
				tablename,rw);
		}
		((DB*)db)->res=sql_query((DB*)db,qbuf);
		if(UdmDBErrorCode(db))return(NULL);
		num=SQL_NUM_ROWS(((DB*)db)->res);
		sprintf(UDM_STREND(wordinfo)," %s : %d",rw,num);

		if(!nwrd){
			nwrd=num;
			wrd=(SEARCHWORD*)malloc(nwrd*sizeof(SEARCHWORD));
		}else{
			nwrd+=num;
			wrd=(SEARCHWORD*)realloc(wrd,nwrd*sizeof(SEARCHWORD));
		}
		
		for(i=0;i<num;i++){
#ifdef HAVE_MYSQL
			/* mysql_data_seek is slow */
			/* We will use sequential fetch instead*/
			row=mysql_fetch_row(((DB*)db)->res);
			url_id=atoi(row[0]);
			weight=atoi(row[1]);
#else
			url_id=atoi(sql_value(((DB*)db)->res,i,0));
			weight=atoi(sql_value(((DB*)db)->res,i,1));
#endif
			wrd[curnum].url_id=url_id;
			wrd[curnum].count=1;
			wrd[curnum].weight=weight;
			curnum++;
		}
		SQL_FREE(((DB*)db)->res);
		w=UdmGetWord(NULL,&lasttok,UdmGetDefaultCharset());
	}

	if(nwrd){
		qsort((void*)wrd,nwrd,sizeof(SEARCHWORD),cmpurlid);
		j=0;
		for(i=1;i<nwrd;i++){
			if(wrd[j].url_id==wrd[i].url_id){
				wrd[j].weight+=wrd[i].weight;
				wrd[j].count++;
			}else{
				if((search_mode==UDM_MOD_ALL)&&(wrd[j].count<num_words)){
					/* Skip this result */
				}else{
					j++;
				}
				wrd[j].url_id=wrd[i].url_id;
				wrd[j].count=wrd[i].count;
				wrd[j].weight=wrd[i].weight;
			}
		}
		if((search_mode==UDM_MOD_ALL)&&(wrd[j].count<num_words))
			nwrd=j;
		else
			nwrd=j+1;

		qsort((void*)wrd,nwrd,sizeof(SEARCHWORD),cmpword);
	}
	if(TrackMode&UDM_TRACK_QUERIES){
		sprintf(qbuf,"INSERT INTO qtrack (qwords,qtime,found) VALUES ('%s',%d,%d)",q_esc,(int)now(),nwrd);
		((DB*)db)->res=sql_query((DB*)db,qbuf);
		if(UdmDBErrorCode(db))return(NULL);
	}

	if(!nwrd)return(0);

	first=np*ps;
	*found=nwrd;
	if(first>=nwrd)first=nwrd-1;
	last=first+ps;
	if(last>nwrd)last=nwrd;
	nwrd=last-first;
	r=(UDM_DOCUMENT*)malloc(sizeof(UDM_DOCUMENT)*(nwrd+1));
	
	instr[0]=0;
#ifdef HAVE_SQL_IN
	for(i=0;i<nwrd;i++){
		sprintf(UDM_STREND(instr),"%s%d",i?",":"",wrd[i+first].url_id);
	}
	for(i=0;i<nwrd+1;i++)
		r[i].url_id=0;
	sprintf(qbuf,"\
SELECT rec_id,url,content_type,last_mod_time,title,txt,docsize,last_index_time,next_index_time,referrer,keywords,description,crc32 \
FROM url \
WHERE rec_id IN (%s) ORDER BY rec_id",
	instr);

	((DB*)db)->res=sql_query(((DB*)db),qbuf);
	if(UdmDBErrorCode(db))return(NULL);
	for(i=0;i<SQL_NUM_ROWS(((DB*)db)->res);i++) {
		for(j=0;j<nwrd;j++) {
			if(wrd[j+first].url_id == atoi(sql_value(((DB*)db)->res,i,0))) {
				r[j].url_id=wrd[i+first].url_id;
				r[j].rating=wrd[i+first].count;
				r[j].url=strdup(sql_value(((DB*)db)->res,i,1));
				r[j].content_type=strdup(sql_value(((DB*)db)->res,i,2));
				r[j].last_mod_time=atol(sql_value(((DB*)db)->res,i,3));
				r[j].title=strdup(sql_value(((DB*)db)->res,i,4));
				r[j].text=strdup(sql_value(((DB*)db)->res,i,5));
				r[j].size=atoi(sql_value(((DB*)db)->res,i,6));
				r[j].last_index_time=strdup(sql_value(((DB*)db)->res,i,7));
				r[j].next_index_time=strdup(sql_value(((DB*)db)->res,i,8));
				r[j].referrer=atoi(sql_value(((DB*)db)->res,i,9));
				r[j].keywords=strdup(sql_value(((DB*)db)->res,i,10));
				r[j].description=strdup(sql_value(((DB*)db)->res,i,11));
				r[j].crc32=(udmcrc32_t)strtol(sql_value(((DB*)db)->res,i,12), NULL, 10);
			}
		}
	}
	SQL_FREE(((DB*)db)->res);
#else
	for(i=0;i<nwrd;i++){

		r[i].url_id=0;
		sprintf(qbuf,"SELECT \
rec_id,url,content_type,last_mod_time,title,txt, \
docsize,last_index_time,next_index_time, \
referrer,keywords,description,crc32 FROM url \
WHERE rec_id=%d",wrd[i+first].url_id);
		((DB*)db)->res=sql_query(((DB*)db),qbuf);
		if(UdmDBErrorCode(db))return(NULL);
		if(SQL_NUM_ROWS(((DB*)db)->res)){
			r[i].url_id=wrd[i+first].url_id;
			r[i].rating=wrd[i+first].count;
			r[i].url=strdup(sql_value(((DB*)db)->res,0,1));
			r[i].content_type=strdup(sql_value(((DB*)db)->res,0,2));
			r[i].last_mod_time=atol(sql_value(((DB*)db)->res,0,3));
			r[i].title=strdup(sql_value(((DB*)db)->res,0,4));
			r[i].text=strdup(sql_value(((DB*)db)->res,0,5));
			r[i].size=atoi(sql_value(((DB*)db)->res,0,6));
			r[i].last_index_time=strdup(sql_value(((DB*)db)->res,0,7));
			r[i].next_index_time=strdup(sql_value(((DB*)db)->res,0,8));
			r[i].referrer=atoi(sql_value(((DB*)db)->res,0,9));
			r[i].keywords=strdup(sql_value(((DB*)db)->res,0,10));
			r[i].description=strdup(sql_value(((DB*)db)->res,0,11));
			r[i].crc32=(udmcrc32_t)strtol(sql_value(((DB*)db)->res,0,12), NULL, 10);
		}
		SQL_FREE(((DB*)db)->res);
	}
#endif
	r[nwrd].url_id=0;
	return(r);
}


/***********************************************************/
/*   Indexing of database content                          */
/***********************************************************/


#ifdef USE_HTDB
static char * get_path_part(char *path,char *dst,int part){
char *s,*e;
int i=0;

	s=path;
	while(*s){
		if(i==part){
			if((e=strchr(s,'/')))strncpy(dst,s,e-s);
			else	strcpy(dst,s);
			return(dst);
		}
		if(*s=='/')i++;
		s++;
	}
	*dst=0;
	return(dst);
}

static char * include_params(char *src,char *path,char *dst){
char *s,*e;
int i;
	s=src;e=dst;*e=0;
	while(*s){
		if((*s=='\\')){
			*e=*(s+1);e++;*e=0;s+=2;
			continue;
		}
		if(*s!='$'){
			*e=*s;
			s++;e++;*e=0;
			continue;
		}
		s++;i=atoi(s);
		while((*s>='0')&&(*s<='9'))s++;
		get_path_part(path,e,i);
		while(*e)e++;
	}
	return(dst);
}


int UdmHTDBGet(UDM_INDEXER * Indexer,char *path,char *filename,char * htdb_list, char * htdb_doc,
		char *buf,int maxsize){
int i;
char qbuf[UDMSTRSIZ]="";
DB * db;


	db=(DB*)Indexer->db;
	buf[0]=0;
	if(*filename){
		char real_path[UDMSTRSIZ]="";
		sprintf(real_path,"%s%s",path,filename);
		if(!htdb_doc)htdb_doc="";
		include_params(htdb_doc,real_path,qbuf);
		db->res=sql_query(db,qbuf);
		if(UdmDBErrorCode(db))return(0);
		if(SQL_NUM_ROWS(db->res)==1){
			sprintf(buf,"%s",sql_value(db->res,0,0));
		}else{
			sprintf(buf,"HTTP/1.0 404 Not Found\r\n\r\n");
		}
		SQL_FREE(db->res);
	}else{
		char *s;
		int len;
		include_params(htdb_list,path,qbuf);
		db->res=sql_query(db,qbuf);
		if(UdmDBErrorCode(db))return(0);
		sprintf(buf,"HTTP/1.0 200 OK\r\nContent-type: text/html\r\n\r\n<HTML><BODY>\n");
		s=UDM_STREND(buf);
		len=SQL_NUM_ROWS(db->res);
		for(i=0;i<SQL_NUM_ROWS(db->res);i++){
#ifdef HAVE_MYSQL
			MYSQL_ROW row;
			row=mysql_fetch_row(db->res);
			sprintf(s,"<a href=\"%s\">%s</a><br>\n",*row,*row);
			s=UDM_STREND(s);
#else
			sprintf(UDM_STREND(buf),"<a href=\"%s\">%s</a><br>\n",
				sql_value(db->res,i,0),
				sql_value(db->res,i,0));
#endif
		}
		SQL_FREE(db->res);
		strcat(buf,"</BODY></HTML>\n");
	}
	return(strlen(buf));
}
#endif



#endif /* HAVE_SQL */
