/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

// Bdb.cpp: implementation of the Bdb class.
//
//////////////////////////////////////////////////////////////////////

#include "Engine.h"
#include "BDB.h"
#include "Cache.h"
#include "Interlock.h"
#include "PagePrecedence.h"
#include "PageWriter.h"
#include "Thread.h"
#include "SQLError.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Bdb::Bdb()
{
	cache = NULL;
	next = NULL;
	prior = NULL;
	hash = NULL;
	buffer = NULL;
	flags = 0;
	pageNumber = -1;
	useCount = 0;
	age = 0;
	higher = lower = NULL;
	markingThread = NULL;
	priorDirty = nextDirty = NULL;
	flushIt = false;
}

Bdb::~Bdb()
{
	PagePrecedence *precedence;

	while ( (precedence = higher) )
		cache->clearPrecedence (precedence);

	while ( (precedence = lower) )
		cache->clearPrecedence (precedence);

}


void Bdb::mark(TransId transId)
{
	ASSERT (useCount > 0);
	ASSERT (lockType == Exclusive);
	transactionId = transId;
	//cache->validateCache();
	
	if (!markingThread)
		{
		markingThread = syncObject.getExclusiveThread();
		++markingThread->pageMarks;
		}

	if (!(flags & BDB_dirty))
		{
		flags |= BDB_dirty;
		cache->markDirty (this);
		}
}

void Bdb::addRef(LockType lType)
{
	incrementUseCount();
	syncObject.lock (NULL, lType);
	lockType = lType;
}

void Bdb::release()
{
	ASSERT (useCount > 0);
	decrementUseCount();

	if (markingThread)
		{
		//cache->validateCache();
		--markingThread->pageMarks;
		markingThread = NULL;
		}

	if (flags & BDB_register)
		{
		cache->pageWriter->writePage(dbb, pageNumber, transactionId);
		flags &= ~BDB_register;
		}

	syncObject.unlock (NULL, lockType);

	if (cache->panicShutdown)
		{
		Thread *thread = Thread::getThread("Cache::fetchPage");
		
		if (thread->pageMarks == 0)
			throw SQLError(RUNTIME_ERROR, "Emergency shut is underway");
		}

}

void Bdb::downGrade(LockType lType)
{
	ASSERT (lockType == Exclusive);
	lockType = lType;
	syncObject.downGrade (lType);
}

void Bdb::incrementUseCount()
{
	INTERLOCKED_INCREMENT (useCount);
}

void Bdb::decrementUseCount()
{
	ASSERT (useCount > 0);
	INTERLOCKED_DECREMENT (useCount);
}

void Bdb::setPrecedence(int32 priorPage)
{
	cache->setPrecedence (this, priorPage);
}

bool Bdb::isHigher(Bdb *bdb)
{
	if (this == bdb)
		return true;

	for (PagePrecedence *prec = higher; prec; prec = prec->nextHigh)
		if (prec->higher->isHigher (bdb))
			return true;

	return false;
}

void Bdb::setWriter()
{
	flags |= BDB_writer | BDB_register;
}
