/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

// IndexPage.cpp: implementation of the IndexPage class.
//
//////////////////////////////////////////////////////////////////////

#include <memory.h>
#include <string.h>
#include <stdio.h>
#include "Engine.h"
#include "IndexPage.h"
#include "Dbb.h"
#include "BDB.h"
#include "Validation.h"
#include "InversionPage.h"
#include "IndexNode.h"
#include "Log.h"
#include "Debug.h"
#include "IndexKey.h"
#include "SerialLog.h"
#include "SerialLogControl.h"
#include "SQLError.h"


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

AddNodeResult IndexPage::addNode(Dbb *dbb, IndexKey *indexKey, int32 recordNumber)
{
	if (dbb->debug)
		{
		Btn::printKey ("new key: ", indexKey, 0, false);
		printPage (this, 0, false);
		}

	// Find the insertion point.  In the process, compute the prior key

	//validate(NULL);
	UCHAR *key = indexKey->key;
	UCHAR	nextKey [MAX_PHYSICAL_KEY_LENGTH];
	IndexKey priorKey;

	IndexNode node(findInsertionPoint(indexKey, recordNumber, &priorKey));

	if (node.getNumber() == recordNumber && 
		 indexKey->keyLength == node.offset + node.length &&
		 memcmp (priorKey.key, key, node.offset) == 0 &&
		 memcmp (node.key, key + node.offset, node.length) == 0)
		return Duplicate;

	// We got the prior key value.  Get key value for insertion point

	memcpy(nextKey, priorKey.key, priorKey.keyLength);
	int nextLength = node.offset + node.length;

	// Compute delta size of new node

	int offset1 = computePrefix (&priorKey, indexKey);
	int length1 = indexKey->keyLength - offset1;
	int delta = IndexNode::nodeLength(offset1, length1, recordNumber);
	int32 nextNumber;
	int offset2;
	
	if ((UCHAR*) node.node == (UCHAR*) this + length)
		offset2 = -1;
	else
		{
		node.expandKey (nextKey);
		offset2 = computePrefix(indexKey->keyLength, indexKey->key, node.offset + node.length, nextKey);
		int deltaOffset = offset2 - node.offset;
		
		if (node.length >= 128 && (node.length - deltaOffset) < 128)
			--delta;

		if (node.offset < 128 && (node.offset + deltaOffset) >= 128)
			++delta;
			
		delta -= deltaOffset;
		nextNumber = node.getNumber();
		}

	// If new node doesn't fit on page, split the page now

	if (length + delta > dbb->pageSize)
		{
		if ((char*) node.nextNode < (char*) this + length)
			return SplitMiddle;

		if (node.getNumber() == END_LEVEL)
			return SplitEnd;

		if (offset2 == -1 || keyCompare(node.keyLength(), nextKey, indexKey->keyLength, indexKey->key) >= 0)
			return NextPage;

		return SplitEnd;
		}

	/***
	Log::debug ("insert index %d, record %d, page offset %d\n",
			index, recordNumber, (CHAR*) node - (CHAR*) page);
	***/

	// Slide tail of bucket forward to make room

	if (offset2 >= 0)
		{
		UCHAR *tail = (UCHAR*) node.nextNode;
		int tailLength = (UCHAR*) this + length - tail;
		ASSERT (tailLength >= 0);
		
		if (tailLength > 0)
			memmove (tail + delta, tail, tailLength);
		}

	// Insert new node

	IndexNode newNode;
	newNode.insert (node.node, offset1, length1, key, recordNumber);

	// If necessary, rebuild next node

	if (offset2 >= 0)
		newNode.insert (newNode.nextNode, offset2, nextLength - offset2, nextKey, nextNumber);

	length += delta;
	//validate(NULL);

	if (dbb->debug)
		printPage (this, 0, false);

	return NodeAdded;
}

Btn* IndexPage::findNode(IndexKey *indexKey, IndexKey *expandedKey)
{
	uint offset = 0;
	uint priorLength = 0;
	UCHAR *key = indexKey->key;
	UCHAR *keyEnd = key + indexKey->keyLength;
	Btn *bucketEnd = (Btn*) ((char*) this + length);

	if (indexKey->keyLength == 0)
		{
		if (expandedKey)
			expandedKey->keyLength = 0;

		return nodes;
		}

	IndexNode node (this);

	for (;; node.getNext())
		{
		if (node.node >= bucketEnd)
			{
			if (node.node != bucketEnd)
				FATAL ("Index findNode: trashed btree page");
				
			if (expandedKey)
				expandedKey->keyLength = priorLength;
				
			return bucketEnd;
			}
			
		int32 number = node.getNumber();
		
		if (number == END_LEVEL)
			break;
			
		if (node.offset < offset)
			break;
			
		if (node.offset > offset || node.length == 0)
			continue;
			
		UCHAR *p = key + node.offset;
		UCHAR *q = node.key;
		UCHAR *nodeEnd = q + node.length;
		
		for (;;)
			{
			if (p == keyEnd)
				goto exit;
				
			if (q == nodeEnd)
				{
				if (indexKey->checkTail(p) < 0)
					goto exit;
				break;
				}
				
			if (*p > *q)
				break;
				
			if (*p++ < *q++)
				goto exit;
			}
			
		offset = p - key;
		
		if (expandedKey)
			{
			node.expandKey (expandedKey);
			priorLength = node.offset + node.length;
			}
		}

	exit:

	if (expandedKey)
		expandedKey->keyLength = priorLength;

	return node.node;
}

int IndexPage::computePrefix(int l1, UCHAR * v1, int l2, UCHAR * v2)
{
	int		n, max;

	for (n = 0, max = MIN (l1, l2); n < max; ++n)
		if (*v1++ != *v2++)
			return n;

	return n;
}

int IndexPage::computePrefix(IndexKey *key1, IndexKey *key2)
{
	int max = MIN(key1->keyLength, key2->keyLength);
	int n;
	UCHAR *v1 = key1->key;
	UCHAR *v2 = key2->key;

	for (n = 0; n < max; ++n)
		if (*v1++ != *v2++)
			return n;

	return n;
}

Bdb* IndexPage::splitIndexPageMiddle(Dbb * dbb, Bdb *bdb, IndexKey *splitKey, TransId transId)
{
	if (dbb->debug)
		printPage (this, 0, false);

	// Fudge page end to handle the all-duplicates problem

	UCHAR *key = splitKey->key;
	Btn *pageEnd = (Btn*) ((char*) this + dbb->pageSize - sizeof(Btn) - 256);
	Btn *midpoint = (Btn*) ((UCHAR*) nodes + 
					  (length - OFFSET (IndexPage*, nodes)) / 2);

	/* Find midpoint.  Don't worry about duplicates.
	   Use the node that crosses the midpoint of what is 
	   currently on the page.*/

	IndexNode node (this);
	IndexNode prevNode = node;
	Btn *chain = node.node;

	for (; node.node < pageEnd; node.getNext())
		{
		int l = node.expandKey(key);

		if (node.offset || node.length)
			{
			if (node.nextNode > midpoint)
				break;

			chain = node.node;
			}
		}

	// The split node should never be the last node on the page
	if ((UCHAR*) node.nextNode > (UCHAR*) this + length)
		node = prevNode;

	int tailLength = ((UCHAR*) this + length) - (UCHAR*) node.nextNode;

	if (tailLength < 0)
		{
		node.parseNode(nodes);
		tailLength = ((UCHAR*) this + length) - (UCHAR*) node.nextNode;
		}

	// If we didn't find a break, use the last one.  This may be the first node

	int kLength = splitKey->keyLength = node.offset + node.length;

	// Allocate and format new page.  Link forward and backward to siblings.

	Bdb *splitBdb = splitPage(dbb, bdb, transId);
	IndexPage *split = (IndexPage*) splitBdb->buffer;

	/* Copy midpoint node to new page.  Compute length, then
	   copy tail of page to new page */

	int32 recordNumber = node.getNumber();
	IndexNode newNode;
	newNode.insert(split->nodes, 0, kLength, key, recordNumber);

	memcpy(newNode.nextNode, node.nextNode, tailLength);

	/* Now we get tricky.  We need to add a "end bucket" marker with the value
	 * of the next bucket.  So copy a little more (it's already compressed)
	 * and zap the node "number" to "END_BUCKET" */

	//node.setNumber (END_BUCKET);
	node.insert(node.node, node.offset, node.length, key, END_BUCKET);
	length = (UCHAR*) node.nextNode - (UCHAR*) this;

	//validate(split);
	split->length = ((UCHAR*) newNode.nextNode + tailLength) - (UCHAR*) split;
	//split->validate(this);

	if (level == 0)
		splitKey->appendRecordNumber(recordNumber);
		
	if (dbb->debug)
		{
		printPage (this, 0, false);
		printPage (splitBdb, false);
		}

	logIndexPage(splitBdb, transId);


	return splitBdb;
}

Btn* IndexPage::findIndex(IndexKey *indexKey)
{
	uint offset = 0;
	UCHAR *keyEnd = indexKey->keyEnd();
	Btn *bucketEnd = (Btn*) ((UCHAR*) this + length);
	Btn *prior = nodes;
	UCHAR *key = indexKey->key;

	for (IndexNode node (this); node.node < bucketEnd; prior = node.node, node.getNext())	
		{
		UCHAR *q = node.key;

		if (node.offset < offset)
			return prior;

		if (node.getNumber() == END_LEVEL)
			return prior;

		if (node.offset > offset || node.length == 0)
			continue;

		UCHAR *p = key + node.offset;
		UCHAR *nodeEnd = q + node.length;

		if (level > 0)
			nodeEnd -= node.getNumberLength(); //sizeof(int32);

		for (;;)
			{
			if (p == keyEnd)
				return prior;

			if (q == nodeEnd || *p > *q)
				break;

			if (*p++ < *q++)
				return prior;
			}

		offset = p - key;
		}

	return prior;
}

Btn* IndexPage::findIndexNode(IndexKey *indexKey, int32 keyRecordNumber)
{
	uint offset = 0;
	UCHAR *keyEnd = indexKey->keyEnd() - indexKey->getNumberLength(); //sizeof(int32);
	Btn *bucketEnd = (Btn*) ((UCHAR*) this + length);
	Btn *prior = nodes;
	UCHAR *key = indexKey->key;

	for (IndexNode node (this); node.node < bucketEnd; prior = node.node, node.getNext())	
		{
		UCHAR *q = node.key;

		if (node.offset < offset)
			return prior;

		if (node.getNumber() == END_LEVEL)
			return prior;

		if (node.offset > offset || node.length == 0)
			continue;

		UCHAR *p = key + node.offset;
		UCHAR *nodeEnd = q + node.length - node.getNumberLength(); //sizeof(int32);

		for (;;)
			{
			if (p == keyEnd)
				{
				if (q < nodeEnd)
					return prior;

				int32 nodeRecordNumber = getRecordNumber(nodeEnd);
				
				if (keyRecordNumber < nodeRecordNumber)
					return prior;
				}

			if (q == nodeEnd || *p > *q)
				break;

			if (*p++ < *q++)
				return prior;
			}

		offset = p - key;
		}

	return prior;
}

void IndexPage::validate(void *before)
{
	Btn *bucketEnd = (Btn*) ((char*) this + length);
	UCHAR key [MAX_PHYSICAL_KEY_LENGTH];
	int keyLength = 0;

	for (IndexNode node (this);; node.getNext())
		{
		if (node.node >= bucketEnd)
			{
			if (node.node != bucketEnd)
				{
				if (before)
					printPage ((IndexPage*) before, 0, false);
					
				printPage (this, 0, false);
				FATAL ("Index findNode: trashed btree page");
				}
				
			break;
			}
			
		int l = MIN(keyLength, node.keyLength());
		int recordNumber = node.getNumber();
		
		if (recordNumber == END_BUCKET)
			ASSERT(nextPage != 0);

		for (UCHAR *p = node.key, *q = key + node.offset, *end = key + l; q < end;)
			if (*p++ > *q++)
				break;
			else if (*p++ < *q++)
				{
				if (before)
					printPage ((IndexPage*) before, 0, false);
					
				printPage(this, 0, false);
				FATAL("Mal-formed index page");
				}

		keyLength = node.expandKey(key);
		}
}

void IndexPage::printPage(Bdb * bdb, bool inversion)
{
	printPage ((IndexPage*) bdb->buffer, bdb->pageNumber, inversion);
}

Bdb* IndexPage::createNewLevel(Dbb * dbb, int level, int version, int32 page1, int32 page2, IndexKey* key2, TransId transId)
{
	Bdb *parentBdb = dbb->allocPage (PAGE_btree, transId);
	IndexPage *parentPage = (IndexPage*) parentBdb->buffer;
	parentPage->level = level;
	parentPage->version = version;
	IndexKey dummy(key2->index);
	dummy.keyLength = 0;
	parentPage->length = OFFSET (IndexPage*, nodes);
	parentPage->addNode (dbb, &dummy, END_LEVEL);
	parentPage->addNode (dbb, &dummy, page1);
	parentPage->addNode (dbb, key2, page2);
	//parentPage->validate(NULL);

	return parentBdb;
}

Bdb* IndexPage::findLevel(Dbb * dbb, Bdb *bdb, int level, IndexKey *indexKey, int32 recordNumber)
{
	IndexPage *page = (IndexPage*) bdb->buffer;
	
	while (page->level > level)
		{
		//IndexNode node (page->findIndexNode(indexKey, recordNumber));
		IndexNode node (page->findIndex(indexKey));
		bdb = dbb->handoffPage (bdb, node.getNumber(), PAGE_btree, Exclusive);
		page = (IndexPage*) bdb->buffer;
		}

	return bdb;
}

int IndexPage::deleteNode(Dbb * dbb, IndexKey *indexKey, int32 recordNumber)
{
	UCHAR *key = indexKey->key;
	uint keyLength = indexKey->keyLength;

	if (dbb->debug)
		{
		Btn::printKey ("delete key: ", indexKey, 0, false);
		printPage (this, 0, false);
		}

	UCHAR	nextKey [MAX_PHYSICAL_KEY_LENGTH];
	IndexKey priorKey;
	IndexNode node (findNode (indexKey, &priorKey));
	Btn *bucketEnd = (Btn*) ((char*) this + length);
	int offset = (char*) node.node - (char*) this;

	// Make sure we've got an exact hit on key

	if (node.offset + node.length != keyLength || 
		memcmp (priorKey.key, key, node.offset) != 0 ||
		memcmp (node.key, key + node.offset, node.length))
		return -1;

	// We've got the start of the duplicate chain.  Find specific node

	for (bool first = true; node.node < bucketEnd; node.getNext(), first = false)
		{
		offset = (char*) node.node - (char*) this;
		
		if (!first && (node.offset != keyLength || node.length != 0))
			return -1;
			
		int32 number = node.getNumber();
		
		if (number == recordNumber)
			{
			IndexNode next (node.nextNode);
			if (next.node >= bucketEnd)
				length = (char*) node.node - (char*) this;
			else
				{
				// Reconstruct following key for recompression

				memcpy (nextKey, key, next.offset);
				memcpy (nextKey + next.offset, next.key, next.length);
				int nextLength = next.offset + next.length;

				Btn *tail = next.nextNode;
				int tailLength = (char*) bucketEnd - (char*) tail;

				// Compute new prefix length; rebuild node

				int prefix = computePrefix (priorKey.keyLength, priorKey.key, nextLength, nextKey);
				int32 num = next.getNumber();
				node.insert (node.node, prefix, nextLength - prefix, nextKey, num);

				// Compute length of remainder of bucket (no compression issues)

				Btn *newTail = node.nextNode;
				if (tailLength > 0)
					memcpy (newTail, tail, tailLength);

				length = (char*) newTail + tailLength - (char*) this;
				//validate(NULL);
				}
				
			if (dbb->debug)
				printPage (this, 0, false);
				
			return 1;
			}
			
		node.expandKey (&priorKey);
		}

	return 0;
}

void IndexPage::printPage(IndexPage * page, int32 pageNumber, bool inversion)
{
	Sync sync(&Log::syncObject, "IndexPage::printPage");
	sync.lock(Exclusive);
	UCHAR	key [MAX_PHYSICAL_KEY_LENGTH];
	int		length;

	Log::debug ("Btree Page %d, level %d, length %d, prior %d, next %d, parent %d\n", 
			pageNumber,
			page->level,
			page->length,
			page->priorPage,
			page->nextPage,
			page->parentPage);

	Btn *end = (Btn*) ((UCHAR*) page + page->length);
	IndexNode node (page);

	for (; node.node < end; node.getNext())
		{
		Log::debug ("   %d. offset %d, length %d, %d: ",
				(char*) node.node - (char*) page, 
				node.offset, node.length, node.getNumber());
		node.expandKey (key);
		node.printKey ("", key, inversion);
		}

	length = (UCHAR*) node.node - (UCHAR*) page;

	if (length != page->length)
		Log::debug ("**** bad bucket length -- should be %d ***\n", length);
}

void IndexPage::validate(Dbb *dbb, Validation *validation, Bitmap *pages, int32 parentPageNumber)
{
	if (pageType == PAGE_inversion)
		{
		((InversionPage*) this)->validate (dbb, validation, pages);
		return;
		}

	// Take care of earlier level first

	Bitmap children;

	if (level)
		{
		Btn *end = (Btn*) ((UCHAR*) this + length);
		
		if (nodes < end)
			{
			IndexNode node(this);
			int32 pageNumber = node.getNumber();
			
			if (pageNumber > 0)
				{
				Bdb *bdb = dbb->fetchPage (pageNumber, PAGE_any, Shared);
				
				if (level == 1 && bdb->buffer->pageType == PAGE_inversion)
					{
					children.set (pageNumber);
					validation->inUse (pageNumber, "InversionPage");
					InversionPage *inversionPage = (InversionPage*) bdb->buffer;
					inversionPage->validate (dbb, validation, &children);
					}
				else if (validation->isPageType (bdb, PAGE_btree, "intermediate index page a, index id %d", validation->indexId))
					{
					children.set (pageNumber);
					validation->inUse (pageNumber, "IndexPage");
					IndexPage *indexPage = (IndexPage*) bdb->buffer;
					indexPage->validate (dbb, validation, &children, pageNumber);
					}
					
				bdb->release();
				}
			}
		}

	if (level)
		validateNodes (dbb, validation, &children, parentPageNumber);

	// Next, loop through siblings

	for (int32 pageNumber = nextPage; pageNumber;)
		{
		Bdb *bdb = dbb->fetchPage (pageNumber, PAGE_any, Shared);
		
		if (validation->isPageType (bdb, PAGE_btree, "intermediate index page b, index id %d", validation->indexId))
			{
			pages->set (pageNumber);
			validation->inUse (pageNumber, "IndexPage");
			IndexPage *indexPage = (IndexPage*) bdb->buffer;
			
			if (indexPage->level)
				indexPage->validateNodes(dbb, validation, &children, pageNumber);
				
			pageNumber = indexPage->nextPage;
			}
		else
			pageNumber = 0;
			
		bdb->release();
		}
}


void IndexPage::validateNodes(Dbb *dbb, Validation *validation, Bitmap *children, int32 parentPageNumber)
{
	Btn *bucketEnd = (Btn*) ((char*) this + length);

	for (IndexNode node (this); node.node < bucketEnd; node.getNext())
		{
		int32 pageNumber = node.getNumber();
		
		if (pageNumber == END_LEVEL || pageNumber == END_BUCKET)
			{
			if (node.nextNode != bucketEnd)
				validation->error ("bucket end not last node");
				
			break;
			}
			
		if (!children->isSet (pageNumber))
			{
			validation->error ("lost index child page %d, index id %d", pageNumber, validation->indexId);
			printPage (this, parentPageNumber, false);
			Bdb *bdb = dbb->fetchPage (pageNumber, PAGE_any, Shared);
			
			if (level == 1 && bdb->buffer->pageType == PAGE_inversion)
				{
				InversionPage *inversionPage = (InversionPage*) bdb->buffer;
				inversionPage->validate (dbb, validation, children);
				}
			else if (validation->isPageType (bdb, PAGE_btree, "IndexPage"))
				{
				IndexPage *indexPage = (IndexPage*) bdb->buffer;
				
				if (indexPage->parentPage == parentPageNumber)
					indexPage->validate (dbb, validation, children, bdb->pageNumber);
				else
					validation->error ("lost index child page %d, index id %d has wrong parent", pageNumber, validation->indexId);
				}
				
			bdb->release();
			}
		}
}

/***
void IndexPage::validateInsertion(int keyLength, UCHAR *key, int32 recordNumber)
{
	Btn *bucketEnd = (Btn*) ((char*) this + length);
	IndexNode node (this);

	for (; node.node < bucketEnd; node.getNext())
		{
		int32 number = node.getNumber();
		if (number == recordNumber)
			return;
		if (number == END_LEVEL || number == END_BUCKET)
			break;
		}

	Log::log ("IndexPage::validateInsertion insert of record %d failed\n", recordNumber);
	Btn::printKey ("insert key: ", keyLength, key, 0, false);
	printPage (this, 0, false);
	//ASSERT (false);
}
***/

void IndexPage::analyze(int pageNumber)
{
	Btn *bucketEnd = (Btn*) ((char*) this + length);
	int count = 0;
	int prefixTotal = 0;
	int lengthTotal = 0;

	for (IndexNode node (this); node.node < bucketEnd; node.getNext())
		{
		++count;
		prefixTotal += node.length;
		lengthTotal += node.length + node.offset;
		}
	
	Log::debug ("Index page %d, level %d, %d nodes, %d byte compressed, %d bytes uncompressed\n",
				pageNumber, level, count, prefixTotal, lengthTotal);
}

bool IndexPage::isLastNode(Btn *indexNode)
{
	IndexNode node (indexNode);

	return (char*) node.nextNode == (char*) this + length;
}

Bdb* IndexPage::splitIndexPageEnd(Dbb *dbb, Bdb *bdb, TransId transId, IndexKey *insertKey, int recordNumber)
{
	if (dbb->debug)
		{
		printPage (this, 0, false);
		Btn::printKey("insert key", insertKey, 0, false);
		}

	Btn *pageEnd = (Btn*) ((char*) this + length);
	IndexNode node (this);
	Btn *chain = node.node;
	uint priorLength = 0;
	IndexKey tempKey(insertKey->index);
	Btn *priorNode;

	// Lets start by finding a place to split the buck near the end

	for (; node.nextNode < pageEnd; node.getNext())
		{
		priorNode = node.node;
		node.expandKey (&tempKey);

		if (node.offset != priorLength || node.length)
			{
			chain = node.node;
			priorLength = node.length;
			}
		}

	// Allocate and format new page.  Link forward and backward to siblings.

	Bdb *splitBdb = splitPage(dbb, bdb, transId);
	IndexPage *split = (IndexPage*) splitBdb->buffer;

	// Build new page from what's left over from last page

	IndexKey rolloverKey(&tempKey);
	node.expandKey(&rolloverKey);
	int rolloverNumber = node.getNumber();
	int insertKeyLength = insertKey->keyLength;
	int offset = computePrefix(insertKeyLength, insertKey->key, tempKey.keyLength, tempKey.key);

	if ((char*) node.nextNode - node.length + insertKeyLength - offset > (char*) this + dbb->pageSize)
		{
		//printPage(this, 0, false);
		node.parseNode(priorNode);
		int number = node.getNumber();
		//node.setNumber(END_BUCKET);
		node.insert(node.node, node.offset, node.length, tempKey.key, END_BUCKET);
		length = (char*) node.nextNode - (char*) this;
		split->appendNode(&tempKey, number, dbb->pageSize);
		}
	else
		{
		int offset = computePrefix(&tempKey, insertKey);
		node.insert(node.node, offset, insertKeyLength - offset, insertKey->key, END_BUCKET);
		length = (char*)(node.nextNode) - (char*) this;
		}

	split->appendNode(insertKey, recordNumber, dbb->pageSize);
	split->appendNode(&rolloverKey, rolloverNumber, dbb->pageSize);
	//printf ("splitIndexPage: level %d, %d -> %d\n", level, bdb->pageNumber, nextPage);
	//validate( splitBdb->buffer);

	if (dbb->debug)
		{
		printPage (this, 0, false);
		printPage (splitBdb, false);
		}

	logIndexPage(splitBdb, transId);

	return splitBdb;
}

Btn* IndexPage::appendNode(int newLength, UCHAR *newKey, int32 recordNumber, int pageSize)
{
	UCHAR key [MAX_PHYSICAL_KEY_LENGTH];
	Btn *end = (Btn*)((char*) this + length);
	IndexNode node (nodes);
	int keyLength = 0;

	for (;node.node < end; node.getNext())
		keyLength = node.expandKey(key);

	int offset = computePrefix(newLength, newKey, keyLength, key);

	if (length + keyLength - offset >= pageSize)
		throw SQLError (INDEX_OVERFLOW, "Cannot split page; expanded key will not fit");

	node.insert(node.node, offset, newLength - offset, newKey, recordNumber);
	length = (char*)(node.nextNode) - (char*) this;

	return node.node;
}

Btn* IndexPage::appendNode(IndexKey *newKey, int32 recordNumber, int pageSize)
{
	IndexKey key(newKey->index);
	key.keyLength = 0;
	Btn *end = (Btn*)((char*) this + length);
	IndexNode node (nodes);

	for (;node.node < end; node.getNext())
		node.expandKey(&key);

	int offset = computePrefix(newKey, &key);

	if (length + (int) newKey->keyLength - offset >= pageSize)
		throw SQLError (INDEX_OVERFLOW, "Cannot split page; expanded key will not fit");

	node.insert(node.node, offset, newKey->keyLength - offset, newKey->key, recordNumber);
	length = (char*)(node.nextNode) - (char*) this;

	return node.node;
}

int IndexPage::keyCompare(int length1, UCHAR *key1, int length2, UCHAR *key2)
{
	// return -1 if key1 is greater, +1 if key2 is greater

	for (;length1 && length2; --length1, --length2)
		if (*key1++ != *key2++)
			return (key1 [-1] > key2 [-1]) ? -1 : 1;

	if (!length1 && !length2)
		return 0;

	return (length1) ? -1 : 1;
}

Bdb* IndexPage::splitPage(Dbb *dbb, Bdb *bdb, TransId transId)
{
	Bdb *splitBdb = dbb->allocPage (PAGE_btree, transId);
	IndexPage *split = (IndexPage*) splitBdb->buffer;

	split->level = level;
	split->version = version;
	split->priorPage = bdb->pageNumber;
	split->parentPage = parentPage;
	split->length = OFFSET(IndexPage*, nodes);
	
	// Link page into right sibling

	if ((split->nextPage = nextPage))
		{
		Bdb *nextBdb = dbb->fetchPage (split->nextPage, PAGE_btree, Exclusive);
		IndexPage *next = (IndexPage*) nextBdb->buffer;
		dbb->setPrecedence(bdb, splitBdb->pageNumber);
		nextBdb->mark(transId);
		next->priorPage = splitBdb->pageNumber;
		nextBdb->release();
		}

	nextPage = splitBdb->pageNumber;

	return splitBdb;
}

Btn* IndexPage::findInsertionPoint(IndexKey *indexKey, int32 recordNumber, IndexKey *expandedKey)
{
	Btn *bucketEnd = (Btn*) ((char*) this + length);

	return findInsertionPoint(level, indexKey, recordNumber, expandedKey, nodes, bucketEnd);

	/***
	UCHAR *key = indexKey->key;
	const UCHAR *keyEnd = key + indexKey->keyLength;
	Btn *bucketEnd = (Btn*) ((char*) this + length);
	IndexNode node (this);
	uint offset = 0;
	uint priorLength = 0;

	for (;; node.getNext())
		{
		if (node.node >= bucketEnd)
			{
			if (node.node != bucketEnd)
				{
				Btn::printKey ("indexKey: ", indexKey, 0, false);
				Btn::printKey ("expandedKey: ", expandedKey, 0, false);
				printPage(this, 0, false);
				FATAL ("Index findNode: trashed btree page");
				}
				
			expandedKey->keyLength = priorLength;
				
			return bucketEnd;
			}
			
		int32 number = node.getNumber();
		
		if (number == END_LEVEL || number == END_BUCKET)
			break;
			
		if (node.offset < offset)
			break;
		
		if (node.offset > offset)
			continue;
		
		const UCHAR *p = key + node.offset;
		const UCHAR *q = node.key;
		const UCHAR *nodeEnd = q + node.length;
		
		for (;;)
			{
			if (p == keyEnd)
				{
				if (level == 0 && recordNumber > number && (node.offset + node.length == indexKey->keyLength))
					break;

				goto exit;
				}
				
			if (q == nodeEnd || *p > *q)
				break;
				
			if (*p++ < *q++)
				goto exit;
			}
			
		offset = p - key;
		node.expandKey (expandedKey);
		priorLength = node.offset + node.length;
		}

	exit:

	expandedKey->keyLength = priorLength;

	return node.node;
	***/
}

int32 IndexPage::getRecordNumber(const UCHAR *ptr)
{
	int32 number = 0;
	
	for (int n = 0; n < 4; ++n)
		number = number << 8 | *ptr++;
	
	return number;
}

void IndexPage::logIndexPage(Bdb *bdb, TransId transId)
{
	Dbb *dbb = bdb->dbb;

	if (dbb->recovering)
		return;

	ASSERT(bdb->useCount > 0);
	IndexPage *indexPage = (IndexPage*) bdb->buffer;
	dbb->serialLog->logControl->indexPage.append(transId, bdb->pageNumber, indexPage->level, 
												 indexPage->parentPage, indexPage->priorPage, indexPage->nextPage,  
												 indexPage->length - OFFSET (IndexPage*, nodes), 
												 (const UCHAR*) indexPage->nodes);
}

Btn* IndexPage::findInsertionPoint(int level, IndexKey* indexKey, int32 recordNumber, IndexKey* expandedKey, Btn* nodes, Btn* bucketEnd)
{
	UCHAR *key = indexKey->key;
	const UCHAR *keyEnd = key + indexKey->keyLength;
	IndexNode node(nodes);
	uint offset = 0;
	uint priorLength = 0;

	if (node.offset)
		{
		for (; offset < node.offset && indexKey->key[offset] == expandedKey->key[offset]; ++offset)
			;
		
		priorLength = expandedKey->keyLength;
		}

	for (;; node.getNext())
		{
		if (node.node >= bucketEnd)
			{
			if (node.node != bucketEnd)
				{
				Btn::printKey ("indexKey: ", indexKey, 0, false);
				Btn::printKey ("expandedKey: ", expandedKey, 0, false);
				FATAL ("Index findNode: trashed btree page");
				}
				
			expandedKey->keyLength = priorLength;
				
			return bucketEnd;
			}
			
		int32 number = node.getNumber();
		
		if (number == END_LEVEL || number == END_BUCKET)
			break;
			
		if (node.offset < offset)
			break;
		
		if (node.offset > offset)
			continue;
		
		const UCHAR *p = key + node.offset;
		const UCHAR *q = node.key;
		const UCHAR *nodeEnd = q + node.length;
		
		for (;;)
			{
			if (p == keyEnd)
				{
				if (level == 0 && recordNumber > number && (node.offset + node.length == indexKey->keyLength))
					break;

				goto exit;
				}
				
			if (q == nodeEnd || *p > *q)
				break;
				
			if (*p++ < *q++)
				goto exit;
			}
			
		offset = p - key;
		node.expandKey (expandedKey);
		priorLength = node.offset + node.length;
		}

	exit:

	expandedKey->keyLength = priorLength;

	return node.node;
}
