/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

// PageInventoryPage.cpp: implementation of the PageInventoryPage class.
//
//////////////////////////////////////////////////////////////////////

#include "Engine.h"
#include "PageInventoryPage.h"
#include "Dbb.h"
#include "BDB.h"
#include "Validation.h"
#include "Bitmap.h"
#include "RecordLocatorPage.h"
#include "IndexPage.h"
#include "DataPage.h"
#include "DataOverflowPage.h"
#include "SectionPage.h"
#include "Transaction.h"
#include "Log.h"

//#define STOP_PAGE	98

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

/***
PageInventoryPage::PageInventoryPage()
{

}

PageInventoryPage::~PageInventoryPage()
{

}
***/

void PageInventoryPage::create(Dbb * dbb, TransId transId)
{
	Bdb *bdb = createInventoryPage (dbb, PIP_PAGE, transId);
	PageInventoryPage *space = (PageInventoryPage*) bdb->buffer;
	space->freePages [0] &= ~(MASK (PIP_PAGE) | 
							  MASK (SECTION_ROOT) |
							  MASK (HEADER_PAGE) | 
							  MASK (INDEX_ROOT));
	bdb->release();
}

Bdb* PageInventoryPage::createInventoryPage(Dbb * dbb, int32 pageNumber, TransId transId)
{
	Bdb *bdb = dbb->fakePage (pageNumber, PAGE_inventory, transId);
	PageInventoryPage *page = (PageInventoryPage*) bdb->buffer;

	for (int n = 0; n < dbb->pipSlots; ++n)
		page->freePages [n] = -1;

	return bdb;
}

Bdb* PageInventoryPage::allocPage(Dbb * dbb, PageType pageType, TransId transId)
{
	for (int32 pip = dbb->lastPageAllocated / dbb->pagesPerPip;; ++pip)
		{
		int32 pipPageNumber = (pip == 0) ? PIP_PAGE : pip * dbb->pagesPerPip - 1;
		Bdb *bdb = dbb->fetchPage (pipPageNumber, PAGE_inventory, Exclusive);
		PageInventoryPage *page = (PageInventoryPage*) bdb->buffer;
		int32 mask;
		
		for (int slot = 0; slot < dbb->pipSlots; ++slot)
			if ( (mask = page->freePages [slot]) )
				for (int n = 0; n < PIP_BITS; ++n)
					if (mask & MASK (n))
						{
						bdb->mark(transId);
						page->freePages [slot] &= ~MASK (n);
						int32 pageNumber = n + slot * PIP_BITS + pip * dbb->pagesPerPip;
						
						if (n + slot * PIP_BITS == dbb->pagesPerPip - 1)
							createInventoryPage (dbb, pageNumber, transId)->release();
						else
							{
							bdb->release();
							dbb->lastPageAllocated = pageNumber;
							Bdb *newBdb = dbb->fakePage (pageNumber, pageType, transId);
							newBdb->setPrecedence (pipPageNumber);

#ifdef STOP_PAGE
							if (pageNumber == STOP_PAGE)
								Log::debug("page %d allocated\n", pageNumber);
#endif

							return newBdb;
							}
						}
						
		bdb->release();
		}
}

void PageInventoryPage::freePage(Dbb *dbb, int32 pageNumber, TransId transId)
{
#ifdef STOP_PAGE
	if (pageNumber == STOP_PAGE)
		Log::debug("page %d released\n", pageNumber);
#endif

	dbb->freePage (pageNumber);
	int32 pip = pageNumber / dbb->pagesPerPip;
	int32 n = pageNumber % dbb->pagesPerPip;

	if (pip == 0)
		pip = PIP_PAGE;
	else
		pip = pip * dbb->pagesPerPip - 1;

	Bdb *bdb = dbb->fetchPage (pip, PAGE_inventory, Exclusive);
	PageInventoryPage *page = (PageInventoryPage*) bdb->buffer;
	bdb->mark(transId);
	page->freePages [n / PIP_BITS] |= MASK (n % PIP_BITS);
	bdb->release();
	dbb->lastPageAllocated = MIN (pageNumber - 1, dbb->lastPageAllocated);
}

void PageInventoryPage::markPageInUse(Dbb *dbb, int32 pageNumber, TransId transId)
{
	int32 pip = pageNumber / dbb->pagesPerPip;
	int32 n = pageNumber % dbb->pagesPerPip;

	if (pip == 0)
		pip = PIP_PAGE;
	else
		pip = pip * dbb->pagesPerPip - 1;

	Bdb *bdb = dbb->fetchPage (pip, PAGE_inventory, Exclusive);
	PageInventoryPage *page = (PageInventoryPage*) bdb->buffer;
	bdb->mark(transId);
	page->freePages [n / PIP_BITS] &= ~MASK (n % PIP_BITS);
	bdb->release();
}

bool PageInventoryPage::isPageInUse(Dbb *dbb, int32 pageNumber)
{
	int32 pip = pageNumber / dbb->pagesPerPip;
	int32 n = pageNumber % dbb->pagesPerPip;

	if (pip == 0)
		pip = PIP_PAGE;
	else
		pip = pip * dbb->pagesPerPip - 1;

	Bdb *bdb = dbb->fetchPage (pip, PAGE_inventory, Shared);
	PageInventoryPage *page = (PageInventoryPage*) bdb->buffer;
	bool used = !(page->freePages [n / PIP_BITS] & MASK (n % PIP_BITS));
	bdb->release();

	return used;
}

void PageInventoryPage::validate(Dbb *dbb, Validation *validation)
{
	int lastPage = dbb->pagesPerPip - 1;
	int lastSlot = lastPage / PIP_BITS;
	int lastBit = MASK (lastPage % PIP_BITS);

	for (int32 pageNumber = PIP_PAGE, sequence = 0; pageNumber; ++sequence)
		{
		Bdb *bdb = dbb->fetchPage (pageNumber, PAGE_any, Shared);
		
		if (validation->isPageType (bdb, PAGE_inventory, "PageInventoryPage"))
			{
			validation->inUse (bdb, "PageInventoryPage");
			PageInventoryPage *page = (PageInventoryPage*) bdb->buffer;
			
			if (page->freePages [lastSlot] & lastBit)
				pageNumber = 0;
			else
				pageNumber = (sequence + 1) * dbb->pagesPerPip - 1;
			}
		else
			pageNumber = 0;
			
		bdb->release();
		}
}

void PageInventoryPage::validateInventory(Dbb *dbb, Validation *validation)
{
	int lastPage = dbb->pagesPerPip - 1;
	int lastSlot = lastPage / PIP_BITS;
	int lastBit = MASK (lastPage % PIP_BITS);
	Bitmap *usedPages = &validation->pages;

	for (int32 pageNumber = PIP_PAGE, sequence = 0; pageNumber; ++sequence)
		{
		Bdb *pipBdb = dbb->fetchPage (pageNumber, PAGE_any, Exclusive);
		
		if (validation->isPageType (pipBdb, PAGE_inventory, "PageInventoryPage"))
			{
			PageInventoryPage *page = (PageInventoryPage*) pipBdb->buffer;
			int32 base = dbb->pagesPerPip * sequence;
			
			for (int n = 0; n < dbb->pagesPerPip; ++n)
				{
				bool used = usedPages->isSet (base + n);
				
				if (page->freePages [n / PIP_BITS] & MASK (n % PIP_BITS))
					{
					if (used)
						{
						validation->error ("unallocated page %d in use", base + n);
						
						if (validation->isRepair())
							{
							pipBdb->mark(NO_TRANSACTION);
							page->freePages [n / PIP_BITS] &= ~MASK (n % PIP_BITS);
							}
						}
					}
				else
					if (!used)
						{
						int32 pageNumber = base + n;
						//Bdb *bdb = dbb->fetchPage (pageNumber, PAGE_any, Shared);
						Bdb *bdb = dbb->trialFetch(pageNumber, PAGE_any, Shared);
						
						if (bdb)
							{
							Page *page = bdb->buffer;
							
							switch (page->pageType)
								{
								case PAGE_record_locator:
									validation->error ("orphan section index page %d, section %d, seq %d", 
													pageNumber, 
													((RecordLocatorPage*) page)->section,
													((RecordLocatorPage*) page)->sequence);
									break;

								case PAGE_btree:
									{
									IndexPage *ipg = (IndexPage*) page;
									validation->error ("orphan index page %d, level %d, parent %d, prior %d, next %d", 
													pageNumber, 
													ipg->level, 
													ipg->parentPage, 
													ipg->priorPage, 
													ipg->nextPage);
									}
									break;

								case PAGE_data:
									{
									DataPage *pg = (DataPage*) page;
									validation->error ("orphan data page %d, maxLine %d", pageNumber, pg->maxLine);
									}
									break;

								case PAGE_data_overflow:
									{
									DataOverflowPage *pg = (DataOverflowPage*) page;
									validation->error ("orphan data overflow page %d, section=%d, next=%d", pageNumber, pg->section, pg->nextPage);
									}
									break;

								case PAGE_sections:
									{
									SectionPage *pg = (SectionPage*) page;
									validation->error ("orphan section page %d, section=%d, seq=%d, level=%d, flgs=%d", 
														pageNumber, pg->section, pg->level, pg->flags);
									}
									break;

								case PAGE_free:
									validation->error ("orphan free page %d", pageNumber);
									break;

								default:
									validation->error ("orphan page %d, type %d", 
													pageNumber, page->pageType);
								}
							}
						else
							validation->error("possible unwritten orphan page %d", pageNumber);
							
						if (validation->isRepair())
							{
							pipBdb->release();
							
							if (bdb)
								dbb->freePage (bdb, 0);
							else
								freePage(dbb, pageNumber, NO_TRANSACTION);
								
							pipBdb = dbb->fetchPage (pageNumber, PAGE_any, Shared);
							page = (PageInventoryPage*) pipBdb->buffer;
							}
						else if (bdb)
							bdb->release();
						}
				}
				
			if (page->freePages [lastSlot] & lastBit)
				pageNumber = 0;
			else
				pageNumber = (sequence + 1) * dbb->pagesPerPip - 1;
			}
		else
			pageNumber = 0;

		pipBdb->release();
		}
}

int32 PageInventoryPage::getLastPage(Dbb *dbb)
{
	for (int32 pip = dbb->lastPageAllocated / dbb->pagesPerPip;; ++pip)
		{
		Bdb *bdb = dbb->fetchPage (
						(pip == 0) ? PIP_PAGE : pip * dbb->pagesPerPip - 1,
						 PAGE_inventory, Shared);
		PageInventoryPage *page = (PageInventoryPage*) bdb->buffer;
		int32 mask;
		
		for (int slot = dbb->pipSlots - 1; slot >= 0; --slot)
			if ((mask = page->freePages [slot]) != -1)
				for (int n = PIP_BITS - 1; n >= 0; --n)
					if (!(mask & MASK (n)))
						{
						int32 pageNumber = n + slot * PIP_BITS + pip * dbb->pagesPerPip;
						
						if (n + slot * PIP_BITS == dbb->pagesPerPip - 1)
							goto nextPip;
							
						bdb->release();
						return pageNumber + 1;
						}
						
		nextPip:
		bdb->release();
		}
}

void PageInventoryPage::reallocPage(Dbb *dbb, int32 pageNumber)
{
#ifdef STOP_PAGE
	if (pageNumber == STOP_PAGE)
		Log::debug("page %d reallocated\n", pageNumber);
#endif

	int32 pip = pageNumber / dbb->pagesPerPip;
	int32 n = pageNumber % dbb->pagesPerPip;

	if (pip == 0)
		pip = PIP_PAGE;
	else
		pip = pip * dbb->pagesPerPip - 1;

	Bdb *bdb = dbb->fetchPage (pip, PAGE_inventory, Exclusive);
	PageInventoryPage *page = (PageInventoryPage*) bdb->buffer;
	int slot = n / PIP_BITS;

	if (page->freePages [slot] & MASK(n % PIP_BITS))
		{
		bdb->mark(NO_TRANSACTION);
		page->freePages [slot] &= ~MASK(n % PIP_BITS);
		}

	bdb->release();
}

void PageInventoryPage::analyzePages(Dbb* dbb, PagesAnalysis* pagesAnalysis)
{
	for (int32 pip = 0;; ++pip)
		{
		Bdb *bdb = dbb->fetchPage (
						(pip == 0) ? PIP_PAGE : pip * dbb->pagesPerPip - 1,
						 PAGE_inventory, Shared);
		PageInventoryPage *page = (PageInventoryPage*) bdb->buffer;
		pagesAnalysis->totalPages += dbb->pagesPerPip;
		int clump;
		int maxPage = 0;
			
		for (int slot = 0; slot < dbb->pipSlots; ++slot)
			{
			clump = page->freePages[slot];
			int count = 0;
			
			if (clump == 0)
				{
				count = PIP_BITS;
				maxPage = pip * dbb->pagesPerPip + (slot + 1) * PIP_BITS - 1;
				}
			else if (clump != -1)
				for (int n = 0; n < PIP_BITS; ++n)
					if (!(clump & MASK(n)))
						{
						++count;
						maxPage = pip * dbb->pagesPerPip + (slot) * PIP_BITS + n;
						}
			
			pagesAnalysis->allocatedPages += count;
			
			if (maxPage > pagesAnalysis->maxPage)
				pagesAnalysis->maxPage = maxPage;
			}
		
		bdb->release();
		
		if (clump & MASK(PIP_BITS - 1))
			break;
		}
}
